package com.biz.crm.availablelist.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.biz.crm.annotation.Klock;
import com.biz.crm.availablelist.entity.CusAvailablelistEntity;
import com.biz.crm.availablelist.repositories.CusAvailablelistRepositories;
import com.biz.crm.availablelist.service.CusAvailablelistService;
import com.biz.crm.availablelist.utils.CusAvailablelistUtil;
import com.biz.crm.availablelistrule.advise.Advise;
import com.biz.crm.availablelistrule.advise.strategy.CustomerChangeAdviseTargetStrategy;
import com.biz.crm.availablelistrule.service.AvailableListRuleService;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.config.SpringApplicationContextUtil;
import com.biz.crm.customer.service.MdmCustomerMsgService;
import com.biz.crm.dms.promotion.PromotionFeign;
import com.biz.crm.dms.promotion.PromotionPolicyFeign;
import com.biz.crm.dms.promotion.PromotionV2Feign;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.YesNoEnum.YesNoCodeNumberEnum;
import com.biz.crm.nebular.dms.npromotion.vo.PromotionQueryReq;
import com.biz.crm.nebular.dms.npromotion.vo.PromotionSaleProductVo;
import com.biz.crm.nebular.mdm.availablelist.AvalibleListProductLevelChangeDto;
import com.biz.crm.nebular.mdm.availablelist.CusAvailablelistVo;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgRespVo;
import com.biz.crm.nebular.mdm.pricesetting.req.MdmPriceSearchReqVo;
import com.biz.crm.nebular.mdm.pricesetting.resp.MdmPriceResp;
import com.biz.crm.nebular.mdm.product.resp.MdmProductAdviseRespVo;
import com.biz.crm.pricesetting.service.IMdmPriceSettingService;
import com.biz.crm.product.service.MdmProductService;
import com.biz.crm.productlevel.service.MdmProductLevelService;
import com.biz.crm.util.*;
import com.biz.crm.utils.MdmConstant;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.aggregations.Aggregation;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;
import org.springframework.data.elasticsearch.core.query.DeleteQuery;
import org.springframework.data.elasticsearch.core.query.NativeSearchQueryBuilder;
import org.springframework.data.elasticsearch.core.query.SearchQuery;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Description:
 * @Author: zhangyuzhu
 * @Date: 2020/10/9 15:02
 **/
@Slf4j
@ConditionalOnMissingBean(name = "cusAvailablelistServiceExpandImpl")
@Service(value = "cusAvailablelistService")
public class CusAvailablelistServiceImpl<M extends BaseMapper<T>,T> implements CusAvailablelistService {

    @Autowired
    private ElasticsearchRestTemplate elasticsearchRestTemplate;

    @Autowired
    private RedissonUtil redissonUtil;

    @Autowired
    private CusAvailablelistRepositories cusAvailablelistRepositories;

    @Resource
    private MdmCustomerMsgService mdmCustomerMsgService;

    @Resource
    private AvailableListRuleService availableListRuleService;

    @Resource
    private MdmProductService mdmProductService;

    @Resource
    private Advise advise;

    @Resource
    private CustomerChangeAdviseTargetStrategy customerChangeAdviseTargetStrategy;

    @Autowired
    private ContextUtil contextUtil;

    @Resource
    private IMdmPriceSettingService iMdmPriceSettingService;

    @Autowired
    private PromotionPolicyFeign promotionPolicyFeign;

    @Autowired
    private MdmProductLevelService mdmProductLevelService;

    @Autowired
    private PromotionV2Feign promotionV2Feign;

    @Autowired
    private PromotionFeign promotionFeign;


    /**
     * 1、验证
     * 2、查询并删除
     * 3、查询促销
     * 4、批量添加
     * @param goods 商品列表
     * @param cusCode 客户编码
     */
    @Override
    public void reFresh(List<MdmProductAdviseRespVo> goods, String cusCode) {
        log.info("可购清单刷新[{}],cusCode[{}]", goods, cusCode);
        //1
        ValidateUtils.validate(cusCode,"客户编码不能为空！");
        //查询经销商信息
        MdmCustomerMsgRespVo customerMsgRespVo = mdmCustomerMsgService.query(null,cusCode);
        ValidateUtils.validate(customerMsgRespVo,"该客户不存在,客户编码:"+cusCode);

        //2
        List<String> ofenGoodsCodes = new ArrayList<>();//常购清单商品code
        if(EsUtil.indexExsit(elasticsearchRestTemplate, CusAvailablelistEntity.class,redissonUtil)){
            List<CusAvailablelistEntity> ofenGoods = cusAvailablelistRepositories.findByCusCodeAndOfenFlag(cusCode,YesNoEnum.YesNoCodeNumberEnum.YES.getCode());
            if(!CollectionUtils.isEmpty(ofenGoods)){
                ofenGoodsCodes = ofenGoods.stream().map(CusAvailablelistEntity :: getProductCode).collect(Collectors.toList());
            }
            DeleteQuery deleteQuery = new DeleteQuery();
            BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
            boolQueryBuilder.must(QueryBuilders.termQuery("cusCode.keyword",cusCode));
            deleteQuery.setQuery(boolQueryBuilder);
            elasticsearchRestTemplate.delete(deleteQuery,CusAvailablelistEntity.class);
            elasticsearchRestTemplate.refresh(MdmConstant.ES_CUSAVAILABLELIST_INDEXNAME);
            log.info("执行删除[{}]***************\r\n", cusCode);
        }
        log.info("执行[{}],[{}]***************\r\n", cusCode, goods.size());
        if(CollectionUtils.isEmpty(goods)){
            return;
        }

        //3
        PromotionQueryReq req = new PromotionQueryReq();
        req.setAccountType("customer");
        req.setAccountCode(cusCode);
        Map<String, Boolean> promotionMap = ApiResultUtil.objResult(promotionFeign.findAvailablePromotionMapByCus(req
                ), true);//查询促销

//        Map<String, Boolean> promotionMap = Maps.newHashMap();

        //4
        cusAvailablelistRepositories.saveAll(CusAvailablelistUtil.packageEntities(goods,customerMsgRespVo,ofenGoodsCodes,promotionMap));
        elasticsearchRestTemplate.refresh(MdmConstant.ES_CUSAVAILABLELIST_INDEXNAME);
        log.info("执行新增[{}]***************\r\n", cusCode);
    }

    /**
     * 1、查询客户信息
     * 2、查询客户可够商品code（去重后）
     * 3、查询商品详情信息
     * 4、刷新
     * @param cusCode 客户code
     */
    @Klock(keys = {MdmConstant.CUSAVAILBLELIST_LOCK ,"#cusCode"},waitTime = 30,leaseTime = 10)
    @Override
    public void reFresh(String cusCode) {
        //1
        Map<String, List<String>> cusList = mdmCustomerMsgService.findOrgCodeListByCustomerCodeList(Lists.newArrayList(cusCode));
        log.info("\r\n******cusList*****map******[{}]**************\r\n", cusList);
        if(CollectionUtils.isEmpty(cusList)){
            log.info("该次产品层级变动没有涉及到的经销商！33333");
            return;
        }

        //2
        List<String> goodsCodes = availableListRuleService.findGoodsByCusCodeAndOrgCode(cusCode,cusList.get(cusCode));
        log.info("\r\n******goodsCodes***********[{}]**************\r\n", goodsCodes);

        //3
        List<MdmProductAdviseRespVo> goods = new ArrayList<>();
        if(!CollectionUtils.isEmpty(goodsCodes)){
            List<String> currentGoodsCodes = new ArrayList<>();
            int i = 0;
            for(String goodsCode : goodsCodes){
                i ++;
                currentGoodsCodes.add(goodsCode);
                if(currentGoodsCodes.size() == 500 || i == goodsCodes.size()){
                    goods.addAll(mdmProductService.adviseQuery(currentGoodsCodes));
                    currentGoodsCodes.clear();
                }
            }
        }else {
            log.info("该经销商没有可够商品！");
        }

        //4
        goods = goods.stream().filter(vo -> !StringUtils.isEmpty(vo.getProductCode())).collect(Collectors.toList());
        this.reFresh(goods,cusCode);
    }
    /**
     * 同步新增的可购商品清单
     *
     * 1、验证
     * 2、查询并删除
     * 3、查询促销
     * 4、批量添加
     * @param goods 商品列表
     * @param cusCode 客户编码
     */
    @Override
    public void syncCusAvailableList(List<MdmProductAdviseRespVo> goods, String cusCode,String ruleCode) {
        log.info("开始同步可购清单[{}],cusCode[{}]", goods, cusCode);
        //1
        ValidateUtils.validate(cusCode,"客户编码不能为空！");
        //查询经销商信息
        MdmCustomerMsgRespVo customerMsgRespVo = mdmCustomerMsgService.query(null,cusCode);
        ValidateUtils.validate(customerMsgRespVo,"该客户不存在,客户编码:"+cusCode);

        //2、删除es中的客户商品清单
        List<String> ofenGoodsCodes = new ArrayList<>();//常购清单商品code
        if(EsUtil.indexExsit(elasticsearchRestTemplate, CusAvailablelistEntity.class,redissonUtil)){
            List<CusAvailablelistEntity> ofenGoods = cusAvailablelistRepositories.findByCusCodeAndOfenFlag(cusCode,YesNoEnum.YesNoCodeNumberEnum.YES.getCode());
            if(!CollectionUtils.isEmpty(ofenGoods)){
                ofenGoodsCodes = ofenGoods.stream().map(CusAvailablelistEntity :: getProductCode).collect(Collectors.toList());
            }
//            DeleteQuery deleteQuery = new DeleteQuery();
//            BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
//            boolQueryBuilder.must(QueryBuilders.termQuery("cusCode.keyword",cusCode));
//            deleteQuery.setQuery(boolQueryBuilder);
//            elasticsearchRestTemplate.delete(deleteQuery,CusAvailablelistEntity.class);
//            elasticsearchRestTemplate.refresh(MdmConstant.ES_CUSAVAILABLELIST_INDEXNAME);
//            log.info("执行删除[{}]***************\r\n", cusCode);
//        }
//        log.info("执行[{}],[{}]***************\r\n", cusCode, goods.size());
//        if(CollectionUtils.isEmpty(goods)){
//            return;
        }

        //3、查询促销
        PromotionQueryReq req = new PromotionQueryReq();
        req.setAccountType("customer");
        req.setAccountCode(cusCode);
        Map<String, Boolean> promotionMap = ApiResultUtil.objResult(promotionFeign.findAvailablePromotionMapByCus(req
        ), true);//查询促销

//        Map<String, Boolean> promotionMap = Maps.newHashMap();

        //4、添加商品信息
//        List<String> ofenGoodsCodes=Lists.newArrayList();
        List<CusAvailablelistEntity> cusAvailablelistEntities = CusAvailablelistUtil.packageEntities(goods, customerMsgRespVo, ofenGoodsCodes, promotionMap);
        for (CusAvailablelistEntity cusAvailablelistEntity : cusAvailablelistEntities) {
            cusAvailablelistEntity.setAvailableRuleCode(ruleCode);
        }
        cusAvailablelistRepositories.saveAll(cusAvailablelistEntities);
        elasticsearchRestTemplate.refresh(MdmConstant.ES_CUSAVAILABLELIST_INDEXNAME);
        log.info("执行新增[{}]***************\r\n", cusCode);
    }

    @Override
    public void editSyncCusAvailableList(List<MdmProductAdviseRespVo> goods, String cusCode,String ruleCode) {
        log.info("开始同步可购清单[{}],cusCode[{}]", goods, cusCode);
        //1
        ValidateUtils.validate(cusCode,"客户编码不能为空！");
        //查询经销商信息
        MdmCustomerMsgRespVo customerMsgRespVo = mdmCustomerMsgService.query(null,cusCode);
        ValidateUtils.validate(customerMsgRespVo,"该客户不存在,客户编码:"+cusCode);

        //2、查询并删除
        List<String> ofenGoodsCodes = new ArrayList<>();//常购清单商品code
        if(EsUtil.indexExsit(elasticsearchRestTemplate, CusAvailablelistEntity.class,redissonUtil)){
            List<CusAvailablelistEntity> ofenGoods = cusAvailablelistRepositories.findByCusCodeAndOfenFlag(cusCode,YesNoEnum.YesNoCodeNumberEnum.YES.getCode());
            if(!CollectionUtils.isEmpty(ofenGoods)){
                ofenGoodsCodes = ofenGoods.stream().map(CusAvailablelistEntity :: getProductCode).collect(Collectors.toList());
            }
//            DeleteQuery deleteQuery = new DeleteQuery();
//            BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
//            System.out.println(ruleCode);
//            boolQueryBuilder.must(QueryBuilders.termQuery("availableRuleCode.keyword", ruleCode));
//            deleteQuery.setQuery(boolQueryBuilder);
//            elasticsearchRestTemplate.delete(deleteQuery,CusAvailablelistEntity.class);
//            elasticsearchRestTemplate.refresh(MdmConstant.ES_CUSAVAILABLELIST_INDEXNAME);
//            log.info("执行删除[{}]***************\r\n", ruleCode);
        }
        log.info("执行[{}],[{}]***************\r\n", ruleCode, goods.size());
        if(CollectionUtils.isEmpty(goods)){
            return;
        }

        //3、查询促销
        PromotionQueryReq req = new PromotionQueryReq();
        req.setAccountType("customer");
        req.setAccountCode(cusCode);
        Map<String, Boolean> promotionMap = ApiResultUtil.objResult(promotionFeign.findAvailablePromotionMapByCus(req
        ), true);//查询促销

//        Map<String, Boolean> promotionMap = Maps.newHashMap();

        //4、添加商品信息
        List<CusAvailablelistEntity> cusAvailablelistEntities = CusAvailablelistUtil.packageEntities(goods, customerMsgRespVo, ofenGoodsCodes, promotionMap);
        for (CusAvailablelistEntity cusAvailablelistEntity : cusAvailablelistEntities) {
            cusAvailablelistEntity.setAvailableRuleCode(ruleCode);
        }
        cusAvailablelistRepositories.saveAll(cusAvailablelistEntities);
        elasticsearchRestTemplate.refresh(MdmConstant.ES_CUSAVAILABLELIST_INDEXNAME);
        log.info("执行新增[{}]***************\r\n", cusCode);
    }


    @Override
    public void reFresh(List<String> cusCodes) {
        if(CollectionUtils.isEmpty(cusCodes)){
            log.info("该次产品层级变动没有涉及到的经销商2222！");
            return;
        }
        //加上catch 避免某一个客户刷新失败影响所有需要刷新的客户
        CusAvailablelistService cusAvailablelistService = SpringApplicationContextUtil.getApplicationContext().getBean(CusAvailablelistService.class);
        for(String cusCode : cusCodes){
            try{
                cusAvailablelistService.reFresh(cusCode);
            }catch (Exception e){
                log.error(new StringBuilder("可够清单刷新失败，客户:").append(cusCode).toString(),e);
            }
        }
    }

    /**
     * 1、验证
     * 2、查询可够商品
     * 3、查询促销
     * 4、更新标志
     * 5、保存
     * @param cusCode
     */
    @Klock(keys = {MdmConstant.CUSAVAILBLELIST_LOCK ,"#cusCode"},waitTime = 30,leaseTime = 10)
    @Override
    public void reFreshPromotion(String cusCode) {
        //1
        if(StringUtils.isEmpty(cusCode)){
            log.info("客户列表为空222222");
            return;
        }
        if(!EsUtil.indexExsit(elasticsearchRestTemplate,CusAvailablelistEntity.class,redissonUtil)){
            log.info("暂无es索引");
            return ;
        }

        //2
        List<CusAvailablelistEntity> entities = cusAvailablelistRepositories.findByCusCode(cusCode);
        if(CollectionUtils.isEmpty(entities)){
            log.info("该客户无可够清单");
            return;
        }

        //3
        PromotionQueryReq req = new PromotionQueryReq();
        req.setAccountType("customer");
        req.setAccountCode(cusCode);
        Result<Map<String, Boolean>> result = promotionFeign.findAvailablePromotionMapByCus(req);
        if(!result.isSuccess()){
            throw new BusinessException("商品服务正在重启，请联系管理员");
        }
        Map<String, Boolean> promotionMap = result.getResult();//查询促销
        if(CollectionUtils.isEmpty(promotionMap)){
            promotionMap = new HashMap<>();
        }

        //4
        for(CusAvailablelistEntity cusAvailablelistEntity : entities){
//            if(null == promotionMap.get(cusAvailablelistEntity.getProductCode())){
//                log.info("{}的商品{}没有促销",cusCode,cusAvailablelistEntity.getProductCode());
//                cusAvailablelistEntity.setPromotionFlag(YesNoEnum.YesNoCodeNumberEnum.NO.getCode());
//            }else {
//                cusAvailablelistEntity.setPromotionFlag(YesNoEnum.YesNoCodeNumberEnum.YES.getCode());
//            }
            boolean f = promotionMap.getOrDefault(cusAvailablelistEntity.getProductCode(),false);
            log.info("\r\n*******[{}],[{}]******\r\n",cusAvailablelistEntity.getProductCode(),f);
            cusAvailablelistEntity.setPromotionFlag(
                YesNoCodeNumberEnum.NO.getCode());
            if(f){
                cusAvailablelistEntity.setPromotionFlag(YesNoEnum.YesNoCodeNumberEnum.YES.getCode());
            }
        }

        //5
        cusAvailablelistRepositories.saveAll(entities);
    }

    @Override
    public void reFreshPromotion(List<String> cusCodes) {
        if(CollectionUtils.isEmpty(cusCodes)){
            log.info("客户列表为空11111");
            return;
        }
        //加上catch 避免某一个客户刷新失败影响所有需要刷新的客户
        CusAvailablelistService cusAvailablelistService = (CusAvailablelistService) contextUtil.getBean(CusAvailablelistService.class);
        for(String cusCode : cusCodes){
            try{
                cusAvailablelistService.reFreshPromotion(cusCode);
            }catch (Exception e){
                log.error(new StringBuilder("可够清单促销标志刷新失败，客户:").append(cusCode).toString(),e);
            }
        }
    }

    /**
     * 1、查询包含这些产品层级的规则【有效的】的所有客户【去重之后】
     * 2、批量刷新客户可够清单
     * @param productLevelCode
     */
    @Override
    public void reFreshByProductLevel(List<String> productLevelCode) {
        if(CollectionUtils.isEmpty(productLevelCode)){
            log.info("产品层级为空！");
            return;
        }
        //1
        List<String> cusCodes = availableListRuleService.listCusCodesByProductLevel(productLevelCode);

        //2
        if(CollectionUtils.isEmpty(cusCodes)){
            log.info("该次产品层级变动没有涉及到的经销商！");
            return;
        }
        this.reFresh(cusCodes);
    }

    /**
     * 1、根据商品查询设计到的客户
     * 2、循环刷新客户的可够商品
     * @param dtoMap
     */
    @Override
    public void reFreshProductLevel(Map<String,AvalibleListProductLevelChangeDto> dtoMap) {
        if(CollectionUtils.isEmpty(dtoMap)){
            return;
        }
        //1
        List<CusAvailablelistEntity> entityList = cusAvailablelistRepositories.findByProductCodeIn(new ArrayList<>(dtoMap.keySet()));
        if(CollectionUtils.isEmpty(entityList)){
            log.info("暂无设计到该商品的可够清单");
            return;
        }

        //2
        //加上catch 避免某一个客户刷新失败影响所有需要刷新的客户
        CusAvailablelistService cusAvailablelistService = (CusAvailablelistService) contextUtil.getBean(CusAvailablelistService.class);
        for(CusAvailablelistEntity entity : entityList){
            try{
                cusAvailablelistService.reFreshProductLevelByCus(entity.getCusCode(),dtoMap);
            }catch (Exception e){
                log.error(new StringBuilder("可够清单商品层级刷新失败，产品:").append(entity.getProductCode()).toString(),e);
            }
        }
    }

    /**
     * 根据客户刷新商品层级
     * 1、查询
     * 2、组装
     * 3、保存
     * @param cusCode
     * @param dtoMap
     */
    @Override
    @Klock(keys = {MdmConstant.CUSAVAILBLELIST_LOCK ,"#cusCode"},waitTime = 30,leaseTime = 10)
    public void reFreshProductLevelByCus(String cusCode,Map<String,AvalibleListProductLevelChangeDto> dtoMap){
        //1
        List<CusAvailablelistEntity> entityList = cusAvailablelistRepositories.findByCusCodeAndProductCodeIn(cusCode,new ArrayList<>(dtoMap.keySet()));
        if(CollectionUtils.isEmpty(entityList)){
            return;
        }

        //2
        List<CusAvailablelistEntity> saveList = new ArrayList<>();
        for(CusAvailablelistEntity entity : entityList){
            AvalibleListProductLevelChangeDto dto = dtoMap.get(entity.getProductCode());
            if(null != dto){
                entity.setProductLevelCode(dto.getNewLevelCode());
                entity.setProductLevelName(dto.getNewLevelName());
                saveList.add(entity);
            }
        }

        //3
        if(!CollectionUtils.isEmpty(saveList)){
            cusAvailablelistRepositories.saveAll(saveList);
            //elasticsearchRestTemplate.refresh(MdmConstant.ES_CUSAVAILABLELIST_INDEXNAME);
        }
    }


    public BoolQueryBuilder buildBoolQueryBuilder(CusAvailablelistVo cusAvailablelistVo){
        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        if(!StringUtils.isEmpty(cusAvailablelistVo.getCusCode())){
            boolQueryBuilder.must(QueryBuilders.termQuery("cusCode.keyword",cusAvailablelistVo.getCusCode()));
        }
        if(!StringUtils.isEmpty(cusAvailablelistVo.getIsShelf())){
            boolQueryBuilder.must(QueryBuilders.termQuery("isShelf.keyword",cusAvailablelistVo.getIsShelf()));
        }
        if(!StringUtils.isEmpty(cusAvailablelistVo.getCusName())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("cusName",cusAvailablelistVo.getCusName()));
        }

        if(!StringUtils.isEmpty(cusAvailablelistVo.getCusOrgCode())){
            boolQueryBuilder.must(QueryBuilders.termQuery("cusOrgCode.keyword",cusAvailablelistVo.getCusOrgCode()));
        }
        if(!StringUtils.isEmpty(cusAvailablelistVo.getCusOrgName())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("cusOrgName",cusAvailablelistVo.getCusOrgName()));
        }
        if(!StringUtils.isEmpty(cusAvailablelistVo.getCusChannelCode())){
            boolQueryBuilder.must(QueryBuilders.termQuery("cusChannelCode.keyword",cusAvailablelistVo.getCusChannelCode()));
        }
        if(!StringUtils.isEmpty(cusAvailablelistVo.getCusChannelName())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("cusChannelName",cusAvailablelistVo.getCusChannelName()));
        }

        if(!StringUtils.isEmpty(cusAvailablelistVo.getProductType())){
            boolQueryBuilder.must(QueryBuilders.termQuery("productType.keyword",cusAvailablelistVo.getProductType()));
        }

        if(!StringUtils.isEmpty(cusAvailablelistVo.getPromotionFlag())){
            boolQueryBuilder.must(QueryBuilders.termQuery("promotionFlag",cusAvailablelistVo.getPromotionFlag()));
        }
        return boolQueryBuilder;
    }


    @CrmDictMethod
    @Override
    public PageResult<CusAvailablelistVo> list(CusAvailablelistVo cusAvailablelistVo) {
        PageResult<CusAvailablelistVo> pageResult = new PageResult<CusAvailablelistVo>();
        pageResult.setCount(Long.parseLong("0"));
        pageResult.setData(new ArrayList<>());
        if(!EsUtil.indexExsit(elasticsearchRestTemplate,CusAvailablelistEntity.class,redissonUtil)){
            return pageResult;
        }
        BoolQueryBuilder boolQueryBuilder = buildBoolQueryBuilder(cusAvailablelistVo);
        if(!StringUtils.isEmpty(cusAvailablelistVo.getProductLevelCode())){
            List<String> productLevelCodes = mdmProductLevelService.getAllChildrenProductLevelCodeList(null, cusAvailablelistVo.getProductLevelCode());
            if(CollectionUtils.isEmpty(productLevelCodes)){
                return pageResult;
            }
            BoolQueryBuilder productLevelCodeBoolQueryBuilder = QueryBuilders.boolQuery();
            //查询
            for (String productLevelCode: productLevelCodes){
                productLevelCodeBoolQueryBuilder.should(QueryBuilders.termQuery("productLevelCode.keyword",productLevelCode));
            }
            boolQueryBuilder.must(productLevelCodeBoolQueryBuilder);
        }
        if(!StringUtils.isEmpty(cusAvailablelistVo.getProductCode())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("productCode",cusAvailablelistVo.getProductCode()));
        }
        if(!StringUtils.isEmpty(cusAvailablelistVo.getProductName())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("productName",cusAvailablelistVo.getProductName()));
        }
        if(null != cusAvailablelistVo.getOfenFlag()){
            boolQueryBuilder.must(QueryBuilders.termQuery("ofenFlag",cusAvailablelistVo.getOfenFlag().toString()));
        }

        SearchQuery searchQuery = new NativeSearchQueryBuilder()
                .withIndices(MdmConstant.ES_CUSAVAILABLELIST_INDEXNAME)//索引名
                .withQuery(boolQueryBuilder)//查询条件
                .withSort(SortBuilders.fieldSort("promotionFlag").unmappedType("keyword").order(SortOrder.DESC))
                .withPageable(PageRequest.of(cusAvailablelistVo.getPageNum().intValue()-1,cusAvailablelistVo.getPageSize()))//分页参数
                .build();
        Page<CusAvailablelistEntity> scroll = elasticsearchRestTemplate.queryForPage(searchQuery,CusAvailablelistEntity.class);
        List<CusAvailablelistEntity> list = scroll.getContent();
        if(CollectionUtils.isEmpty(list)){
            return pageResult;
        }
        List<CusAvailablelistVo> reList = new ArrayList<>(list.size());
        for(CusAvailablelistEntity entity : list){
            CusAvailablelistVo vo = new CusAvailablelistVo();
            BeanUtils.copyProperties(entity,vo);
            reList.add(vo);
        }

        //查询价格
        if(!StringUtils.isEmpty(cusAvailablelistVo.getCusCode())){
            MdmPriceSearchReqVo mdmPriceSearchReqVo = new MdmPriceSearchReqVo();
            mdmPriceSearchReqVo.setProductCodeList(
                    reList.stream()
                            .map(CusAvailablelistVo :: getProductCode).collect(Collectors.toList())
            );
            mdmPriceSearchReqVo.setCustomerCode(cusAvailablelistVo.getCusCode());
            mdmPriceSearchReqVo.setConditionTypeCodeList(
                    new ArrayList<String>(){{
                        add("sale_price");
                    }}
            );
            Map<String, Map<String, MdmPriceResp>> priceMap = iMdmPriceSettingService.search(mdmPriceSearchReqVo);
            for(CusAvailablelistVo vo : reList){
                vo.setPrice(null);
                if(null != priceMap.get(vo.getProductCode()) &&  null != priceMap.get(vo.getProductCode()).get("sale_price")){
                    vo.setPrice(priceMap.get(vo.getProductCode()).get("sale_price").getPrice());
                }
            }
        }
        pageResult.setData(reList);
        pageResult.setCount(scroll.getTotalElements());
        return pageResult;
    }


    /**
     * 1、验证
     * 2、删除
     * @param id
     */
    @Override
    public void delById(String id) {
        //1
        if(StringUtils.isEmpty(id)){
            return;
        }
        if(!EsUtil.indexExsit(elasticsearchRestTemplate,CusAvailablelistEntity.class,redissonUtil)){
            return;
        }
        Optional<CusAvailablelistEntity> optionalCusAvailablelistEntity = cusAvailablelistRepositories.findById(id);
        if(!optionalCusAvailablelistEntity.isPresent()){
            return;
        }

        //2
        CusAvailablelistService cusAvailablelistService = (CusAvailablelistService) contextUtil.getBean(CusAvailablelistService.class);
        cusAvailablelistService.delByIdAndCusCode(id,optionalCusAvailablelistEntity.get().getCusCode());
    }

    @Klock(keys = {MdmConstant.CUSAVAILBLELIST_LOCK ,"#cusCode"},waitTime = 30,leaseTime = 10)
    @Override
    public void delByIdAndCusCode(String id, String cusCode) {
        cusAvailablelistRepositories.deleteById(id);
        elasticsearchRestTemplate.refresh(MdmConstant.ES_CUSAVAILABLELIST_INDEXNAME);
    }

    /**
     * 1、验证
     * 2、删除
     * @param ids
     */
    @Override
    public void delByIds(List<String> ids) {
        //1
        if(CollectionUtils.isEmpty(ids)){
            return;
        }
        if(!EsUtil.indexExsit(elasticsearchRestTemplate,CusAvailablelistEntity.class,redissonUtil)){
            return;
        }
        List<CusAvailablelistEntity> entities = cusAvailablelistRepositories.findByIdIn(ids);
        Map<String, List<CusAvailablelistEntity>> entityMap = entities.stream().collect(Collectors.groupingBy(CusAvailablelistEntity::getCusCode));

        //2
        CusAvailablelistService cusAvailablelistService = (CusAvailablelistService) contextUtil.getBean(CusAvailablelistService.class);
        for(Map.Entry<String, List<CusAvailablelistEntity>> entry : entityMap.entrySet()){
            List<CusAvailablelistEntity> handleEntities = entry.getValue();
            if(CollectionUtils.isEmpty(handleEntities)){
                continue;
            }
            try{
                cusAvailablelistService.delByIdsAndCusCode(handleEntities.stream().map(CusAvailablelistEntity :: getId).collect(Collectors.toList())
                        ,entry.getKey());
            }catch (Exception e){
                log.error(new StringBuilder("可够清单删除失败，客户:").append(entry.getKey()).toString(),e);
            }
        }

    }

    @Klock(keys = {MdmConstant.CUSAVAILBLELIST_LOCK ,"#cusCode"},waitTime = 30,leaseTime = 10)
    @Override
    public void delByIdsAndCusCode(List<String> ids, String cusCode) {
        if(CollectionUtils.isEmpty(ids)){
            return;
        }
        cusAvailablelistRepositories.deleteByIdIn(ids);
        elasticsearchRestTemplate.refresh(MdmConstant.ES_CUSAVAILABLELIST_INDEXNAME);
    }

    /**
     * 1、验证
     * 2、查询受影响的客户
     * 3、更新客户可够清单
     * @param mdmProductAdviseRespVos
     */
    @Override
    public void updateByGoodsCodes(List<MdmProductAdviseRespVo> mdmProductAdviseRespVos) {
        //1
        if(CollectionUtils.isEmpty(mdmProductAdviseRespVos)){
            log.info("参数为空");
            return;
        }
        if(!EsUtil.indexExsit(elasticsearchRestTemplate,CusAvailablelistEntity.class,redissonUtil)){
            log.info("暂无es索引");
            return;
        }

        //2
        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        List<String> goodsCodes = mdmProductAdviseRespVos.stream().map(MdmProductAdviseRespVo :: getProductCode).collect(Collectors.toList());
        boolQueryBuilder.must(QueryBuilders.termsQuery("productCode.keyword",
                goodsCodes
        ));
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("cusCodes").field("cusCode.keyword").size(10000);
        SearchQuery searchQuery = new NativeSearchQueryBuilder()
                .withIndices(MdmConstant.ES_CUSAVAILABLELIST_INDEXNAME)//索引名
                .withQuery(boolQueryBuilder)//查询条件
                .withPageable(PageRequest.of(0,10))
                .addAggregation(aggregationBuilder)//聚合
                .build();
        Aggregations aggregations = elasticsearchRestTemplate.query(searchQuery, response -> response.getAggregations());
        Aggregation tags = aggregations.asMap().get("cusCodes");
        Terms teamSum= (Terms)tags;
        List<? extends Terms.Bucket> buckets = teamSum.getBuckets();
        List<String> cusCodes = new ArrayList<>();
        for (Terms.Bucket bucket : buckets) {
            cusCodes.add(bucket.getKeyAsString());
        }
        if(CollectionUtils.isEmpty(cusCodes)){
            log.info("暂无涉及到的客户");
            return;
        }

        //3
        //加上catch 避免某一个客户刷新失败影响所有需要刷新的客户
        CusAvailablelistService cusAvailablelistService = (CusAvailablelistService) contextUtil.getBean(CusAvailablelistService.class);
        for(String cusCode : cusCodes){
            try{
                cusAvailablelistService.updateGoodsByCusCode(mdmProductAdviseRespVos,cusCode);
            }catch (Exception e){
                log.error(new StringBuilder("可够清单更新商品失败，客户:").append(cusCode).toString(),e);
            }
        }
    }

    /**
     * 1、验证
     * 2、查询受影响的客户
     * 3、删除客户可够清单
     * @param goodsCodes
     */
    @Override
    public void delByGoodsCodes(List<String> goodsCodes) {
        //1
        if(CollectionUtils.isEmpty(goodsCodes)){
            log.info("参数为空");
            return;
        }
        if(!EsUtil.indexExsit(elasticsearchRestTemplate,CusAvailablelistEntity.class,redissonUtil)){
            log.info("es索引为空");
            return;
        }

        //2
        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        boolQueryBuilder.must(QueryBuilders.termsQuery("productCode.keyword",
                goodsCodes
        ));
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("cusCodes").field("cusCode.keyword").size(10000);
        SearchQuery searchQuery = new NativeSearchQueryBuilder()
                .withIndices(MdmConstant.ES_CUSAVAILABLELIST_INDEXNAME)//索引名
                .withQuery(boolQueryBuilder)//查询条件
                .withPageable(PageRequest.of(0,10))
                .addAggregation(aggregationBuilder)//聚合
                .build();
        Aggregations aggregations = elasticsearchRestTemplate.query(searchQuery, response -> response.getAggregations());
        Aggregation tags = aggregations.asMap().get("cusCodes");
        Terms teamSum= (Terms)tags;
        List<? extends Terms.Bucket> buckets = teamSum.getBuckets();
        List<String> cusCodes = new ArrayList<>();
        for (Terms.Bucket bucket : buckets) {
            cusCodes.add(bucket.getKeyAsString());
        }
        if(CollectionUtils.isEmpty(cusCodes)){
            log.info("暂无相关客户");
        }

        //3
        //加上catch 避免某一个客户刷新失败影响所有需要刷新的客户
        CusAvailablelistService cusAvailablelistService = (CusAvailablelistService) contextUtil.getBean(CusAvailablelistService.class);
        for(String cusCode : cusCodes){
            try{
                cusAvailablelistService.delByGoodsCodesAndCusCode(goodsCodes,cusCode);
            }catch (Exception e){
                log.error(new StringBuilder("可够清单删除失败，客户:").append(cusCode).toString(),e);
            }
        }
    }


    @Klock(keys = {MdmConstant.CUSAVAILBLELIST_LOCK ,"#cusCode"},waitTime = 30,leaseTime = 10)
    @Override
    public void delByGoodsCodesAndCusCode(List<String> goodsCodes, String cusCode) {
        cusAvailablelistRepositories.deleteByCusCodeAndProductCodeIn(cusCode,goodsCodes);
    }

    @Override
    public PageResult<CusAvailablelistVo> listForM(CusAvailablelistVo cusAvailablelistVo) {
        PageResult<CusAvailablelistVo> pageResult = new PageResult<CusAvailablelistVo>();
        pageResult.setCount(Long.parseLong("0"));
        pageResult.setData(new ArrayList<>());
        if(!EsUtil.indexExsit(elasticsearchRestTemplate,CusAvailablelistEntity.class,redissonUtil)){
            return pageResult;
        }

        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        if(!StringUtils.isEmpty(cusAvailablelistVo.getCusCode())){
            boolQueryBuilder.must(QueryBuilders.wildcardQuery("cusCode.keyword","*"+cusAvailablelistVo.getCusCode()+"*"));
        }
        if(!StringUtils.isEmpty(cusAvailablelistVo.getCusName())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("cusName",cusAvailablelistVo.getCusName()));
        }

        if(!StringUtils.isEmpty(cusAvailablelistVo.getCusOrgCode())){
            boolQueryBuilder.must(QueryBuilders.wildcardQuery("cusOrgCode.keyword","*"+cusAvailablelistVo.getCusOrgCode()+"*"));
        }
        if(!StringUtils.isEmpty(cusAvailablelistVo.getCusOrgName())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("cusOrgName",cusAvailablelistVo.getCusOrgName()));
        }
        if(!StringUtils.isEmpty(cusAvailablelistVo.getCusChannelCode())){
            boolQueryBuilder.must(QueryBuilders.termQuery("cusChannelCode.keyword",cusAvailablelistVo.getCusChannelCode()));
        }
        if(!StringUtils.isEmpty(cusAvailablelistVo.getCusChannelName())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("cusChannelName",cusAvailablelistVo.getCusChannelName()));
        }

        if(!StringUtils.isEmpty(cusAvailablelistVo.getProductType())){
            boolQueryBuilder.must(QueryBuilders.termQuery("productType.keyword",cusAvailablelistVo.getProductType()));
        }

        if(!StringUtils.isEmpty(cusAvailablelistVo.getPromotionFlag())){
            boolQueryBuilder.must(QueryBuilders.termQuery("promotionFlag",cusAvailablelistVo.getPromotionFlag()));
        }
        if(!StringUtils.isEmpty(cusAvailablelistVo.getProductLevelCode())){
            List<String> productLevelCodes = mdmProductLevelService.getAllChildrenProductLevelCodeList(null, cusAvailablelistVo.getProductLevelCode());
            if(CollectionUtils.isEmpty(productLevelCodes)){
                return pageResult;
            }
            BoolQueryBuilder productLevelCodeBoolQueryBuilder = QueryBuilders.boolQuery();
            //查询
            for (String productLevelCode: productLevelCodes){
                productLevelCodeBoolQueryBuilder.should(QueryBuilders.termQuery("productLevelCode.keyword",productLevelCode));
            }
            boolQueryBuilder.must(productLevelCodeBoolQueryBuilder);
        }
        if(!StringUtils.isEmpty(cusAvailablelistVo.getProductCode())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("productCode",cusAvailablelistVo.getProductCode()));
        }
        if(!StringUtils.isEmpty(cusAvailablelistVo.getProductName())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("productName",cusAvailablelistVo.getProductName()));
        }
        if(null != cusAvailablelistVo.getOfenFlag()){
            boolQueryBuilder.must(QueryBuilders.termQuery("ofenFlag",cusAvailablelistVo.getOfenFlag().toString()));
        }

        SearchQuery searchQuery = new NativeSearchQueryBuilder()
                .withIndices(MdmConstant.ES_CUSAVAILABLELIST_INDEXNAME)//索引名
                .withQuery(boolQueryBuilder)//查询条件
                .withSort(SortBuilders.fieldSort("promotionFlag").unmappedType("keyword").order(SortOrder.DESC))
                .withPageable(PageRequest.of(cusAvailablelistVo.getPageNum().intValue()-1,cusAvailablelistVo.getPageSize()))//分页参数
                .build();
        Page<CusAvailablelistEntity> scroll = elasticsearchRestTemplate.queryForPage(searchQuery,CusAvailablelistEntity.class);
        List<CusAvailablelistEntity> list = scroll.getContent();
        if(CollectionUtils.isEmpty(list)){
            return pageResult;
        }
        List<CusAvailablelistVo> reList = new ArrayList<>(list.size());
        for(CusAvailablelistEntity entity : list){
            CusAvailablelistVo vo = new CusAvailablelistVo();
            BeanUtils.copyProperties(entity,vo);
            reList.add(vo);
        }

        // 根据创建时间从前往后排序
        if(reList.size()>0) {
            Collections.sort(reList, new Comparator<CusAvailablelistVo>() {
                @Override
                public int compare(CusAvailablelistVo o1, CusAvailablelistVo o2) {
                    if (DateUtil.str2Timestamp(o1.getCreateDateAll()).compareTo(DateUtil.str2Timestamp(o2.getCreateDateAll())) > 0) {
                        return 1;
                    } else if (DateUtil.str2Timestamp(o1.getCreateDateAll()).compareTo(DateUtil.str2Timestamp(o2.getCreateDateAll())) == 0) {
                        return 0;
                    } else {
                        return -1;
                    }
                }
            });
        }
        pageResult.setData(reList);
        pageResult.setCount(scroll.getTotalElements());
        return pageResult;
    }

    /**
     * 1、查询
     * 2、更新
     * @param ids
     */
    @Klock(keys = {MdmConstant.CUSAVAILBLELIST_LOCK ,"#cusCode"},waitTime = 30,leaseTime = 10)
    @Override
    public void ofenIds(List<String> ids, YesNoEnum.YesNoCodeNumberEnum ofenflag, String cusCode) {
        //1
        List<CusAvailablelistEntity> entities = cusAvailablelistRepositories.findByIdIn(ids);
        if(CollectionUtils.isEmpty(entities)){
            return;
        }

        //2
        for(CusAvailablelistEntity entity : entities){
            entity.setOfenFlag(ofenflag.getCode());
        }
        cusAvailablelistRepositories.saveAll(entities);
    }

    /**
     * 1、验证
     * 2、查询
     * 3、查询价格并组装返回
     * @param goodsCodes
     * @param cusCode
     * @return
     */
    @Override
    public Map<String, CusAvailablelistVo> findGoodsByCusAndGoodsCodes(List<String> goodsCodes, String cusCode) {
        //1
        if(StringUtils.isEmpty(cusCode) || CollectionUtils.isEmpty(goodsCodes)){
            return new HashMap<>();
        }

        //2
        List<CusAvailablelistEntity> entities = cusAvailablelistRepositories.findByCusCodeAndProductCodeIn(cusCode,goodsCodes);

        //3
        Map<String,CusAvailablelistEntity> entityMap = new HashMap<>();
        //价格map
        Map<String, Map<String, MdmPriceResp>> priceMap = new HashMap<>();
        if(!CollectionUtils.isEmpty(entities)){
            MdmPriceSearchReqVo mdmPriceSearchReqVo = new MdmPriceSearchReqVo();
            mdmPriceSearchReqVo.setProductCodeList(
                    entities.stream()
                            .map(CusAvailablelistEntity :: getProductCode).collect(Collectors.toList())
            );
            mdmPriceSearchReqVo.setCustomerCode(cusCode);
            mdmPriceSearchReqVo.setConditionTypeCodeList(
                     new ArrayList<String>(){{
                        add("sale_price");
                    }}
            );
            priceMap = iMdmPriceSettingService.search(mdmPriceSearchReqVo);
        }

        entityMap = entities.stream().collect(
                Collectors.toMap(CusAvailablelistEntity::getProductCode, Function.identity(), (key1, key2) -> key2)
        );
        Map<String,CusAvailablelistVo> reMap = new HashMap<>(goodsCodes.size());
        for(String goodsCode : goodsCodes){
            CusAvailablelistEntity entity = entityMap.get(goodsCode);
            CusAvailablelistVo vo = null;
            if(entity != null){
                vo = new CusAvailablelistVo();
                BeanUtils.copyProperties(entity,vo);
                if(priceMap.get(goodsCode) == null || null == priceMap.get(goodsCode).get("sale_price")){
                    vo.setPrice(null);
                }else {
                    vo.setPrice(priceMap.get(goodsCode).get("sale_price").getPrice());
                }
            }
            reMap.put(goodsCode,vo);
        }
        return reMap;
    }

    @Override
    public CusAvailablelistVo findById(String id) {
        Optional<CusAvailablelistEntity> optionalCusAvailablelistEntity = cusAvailablelistRepositories.findById(id);
        if(!optionalCusAvailablelistEntity.isPresent()){
            return null;
        }
        CusAvailablelistEntity entity = optionalCusAvailablelistEntity.get();
        CusAvailablelistVo vo = new CusAvailablelistVo();
        BeanUtils.copyProperties(entity,vo);
        return vo;
    }

    /**
     * 1、验证
     * 2、查询商品信息
     * 3、查询价格
     * 4、
     * @param goodsCode
     * @param cusCode
     * @return
     */
    @Override
    public Map<String, Object> findGoodsDetailAndPromotionByCode(String goodsCode, String cusCode) {
        //1
        ValidateUtils.validate(cusCode,"请登录!");
        Map<String, Object> reMap = new HashMap<>();
        reMap.put("goods",null);
        reMap.put("promotion",null);
        if(StringUtils.isEmpty(goodsCode)){
            return reMap;
        }

        //2
        List<CusAvailablelistEntity> entities = cusAvailablelistRepositories.findByCusCodeAndProductCode(cusCode,goodsCode);
        if(CollectionUtils.isEmpty(entities)){
            return reMap;
        }
        CusAvailablelistVo vo = new CusAvailablelistVo();
        BeanUtils.copyProperties(entities.get(0),vo);
        vo.setSaleUnitName(DictUtil.dictValue("product_sale_unit",vo.getSaleUnit()));

        List<MdmProductAdviseRespVo> mdmProductAdviseRespVos = mdmProductService.adviseQuery(Collections.singletonList(goodsCode));
        if(CollectionUtil.listNotEmptyNotSizeZero(mdmProductAdviseRespVos) && mdmProductAdviseRespVos.size() == 1){
            MdmProductAdviseRespVo mdmProductAdviseRespVo = mdmProductAdviseRespVos.get(0);
            CrmBeanUtil.copyPropertiesIgnoreEmpty(mdmProductAdviseRespVo,vo);
        }
        reMap.put("goods",vo);

        //3
        MdmPriceSearchReqVo mdmPriceSearchReqVo = new MdmPriceSearchReqVo();
        mdmPriceSearchReqVo.setProductCodeList(
                Arrays.asList(goodsCode)
        );
        mdmPriceSearchReqVo.setCustomerCode(cusCode);
        mdmPriceSearchReqVo.setConditionTypeCodeList(
                new ArrayList<String>(){{
                    add("sale_price");
                }}
        );
        Map<String, Map<String, MdmPriceResp>> priceMap = iMdmPriceSettingService.search(mdmPriceSearchReqVo);
        vo.setPrice(null);
        if(null != priceMap.get(goodsCode) && null != priceMap.get(goodsCode).get("sale_price")){
            vo.setPrice(priceMap.get(goodsCode).get("sale_price").getPrice());
        }

        //4
        PromotionQueryReq req = new PromotionQueryReq();
        req.setAccountType("customer");
        req.setAccountCode(cusCode);
        List<PromotionSaleProductVo> saleProductVos = new ArrayList<>();
        PromotionSaleProductVo product = new PromotionSaleProductVo();
        product.setProductCode(goodsCode);
        saleProductVos.add(product);
        req.setSaleProductVos(saleProductVos);
        reMap.put("promotion",ApiResultUtil.objResult(
                promotionV2Feign.findAvailablePromotionsByAccount(req)
        ));

        return reMap;
    }

    /**
     * 1、验证
     * 2、查询商品信息
     * 3、查询价格
     * @param goodsCode
     * @param cusCode
     * @return
     */
    @Override
    public CusAvailablelistVo findGoodsDetailByCode(String goodsCode, String cusCode) {
        //1
        ValidateUtils.validate(cusCode,"请登录!");
        if(StringUtils.isEmpty(goodsCode)){
            return null;
        }

        //2
        List<CusAvailablelistEntity> entities = cusAvailablelistRepositories.findByCusCodeAndProductCode(cusCode,goodsCode);
        if(CollectionUtils.isEmpty(entities)){
            return null;
        }
        CusAvailablelistVo vo = new CusAvailablelistVo();
        BeanUtils.copyProperties(entities.get(0),vo);
        vo.setSaleUnitName(DictUtil.dictValue("product_sale_unit",vo.getSaleUnit()));

        //3
        MdmPriceSearchReqVo mdmPriceSearchReqVo = new MdmPriceSearchReqVo();
        mdmPriceSearchReqVo.setProductCodeList(
                Arrays.asList(goodsCode)
        );
        mdmPriceSearchReqVo.setCustomerCode(cusCode);
        mdmPriceSearchReqVo.setConditionTypeCodeList(
                new ArrayList<String>(){{
                    add("sale_price");
                }}
        );
        Map<String, Map<String, MdmPriceResp>> priceMap = iMdmPriceSettingService.search(mdmPriceSearchReqVo);
        vo.setPrice(null);
        if(null != priceMap.get(goodsCode) && priceMap.get(goodsCode).get("sale_price") != null){
            vo.setPrice(priceMap.get(goodsCode).get("sale_price").getPrice());
        }
        return vo;
    }

    /**
     * 更新客户可够清单
     * @param mdmProductAdviseRespVos
     * @param cusCode
     */
    @Klock(keys = {MdmConstant.CUSAVAILBLELIST_LOCK ,"#cusCode"},waitTime = 30,leaseTime = 10)
    public void updateGoodsByCusCode(List<MdmProductAdviseRespVo> mdmProductAdviseRespVos,String cusCode){
        if(!EsUtil.indexExsit(elasticsearchRestTemplate,CusAvailablelistEntity.class,redissonUtil)){
            log.info("暂无es索引");
            return;
        }
        mdmProductAdviseRespVos = mdmProductAdviseRespVos.stream().filter(vo -> !StringUtils.isEmpty(vo.getProductCode())).collect(Collectors.toList());
        List<CusAvailablelistEntity> cusAvailablelistEntities = cusAvailablelistRepositories.findByProductCodeInAndCusCode(
                mdmProductAdviseRespVos.stream().map(MdmProductAdviseRespVo :: getProductCode).collect(Collectors.toList())
                ,cusCode
        );
        if(CollectionUtils.isEmpty(cusAvailablelistEntities)){
            log.info("暂无设计到的可够清单");
            return;
        }

        //2
        Map<String, MdmProductAdviseRespVo> goodsMap = mdmProductAdviseRespVos.stream().collect(
                Collectors.toMap(MdmProductAdviseRespVo::getProductCode, a -> a, (MdmProductAdviseRespVo k1, MdmProductAdviseRespVo k2) -> {
                    return (MdmProductAdviseRespVo) k1;
                })
        );

        int pageSize = 500;
        int total = cusAvailablelistEntities.size();
        List<CusAvailablelistEntity> entities = new ArrayList<>(pageSize);
        for(int i = 0 ; i< total ; i++){
            CusAvailablelistEntity entity = cusAvailablelistEntities.get(i);
            MdmProductAdviseRespVo vo = goodsMap.get(entity.getProductCode());
            if(null != vo){
                entity = CusAvailablelistUtil.copyGoodsVo(entity,vo);
                entities.add(entity);
            }else {
                log.info("{}产品没有查询到",entity.getProductCode());
            }
            if(entities.size() > 0 && (entities.size() == pageSize || i == total-1)){
                cusAvailablelistRepositories.deleteByIdIn(entities.stream().map(CusAvailablelistEntity :: getId).collect(Collectors.toList()));
                cusAvailablelistRepositories.saveAll(entities);
                entities.clear();
            }
        }
        //elasticsearchRestTemplate.refresh(MdmConstant.ES_CUSAVAILABLELIST_INDEXNAME);
    }
}
