package com.biz.crm.availablelist.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.biz.crm.annotation.Klock;
import com.biz.crm.availablelist.entity.TerminalAvailablelistEntity;
import com.biz.crm.availablelist.repositories.TerminalAvailablelistRepositories;
import com.biz.crm.availablelist.service.TerminalAvailablelistService;
import com.biz.crm.availablelist.utils.TerminalAvailablelistUtil;
import com.biz.crm.availablelistrule.advise.Advise;
import com.biz.crm.availablelistrule.advise.strategy.TerminalChangeAdviseTargetStrategy;
import com.biz.crm.availablelistrule.service.AvailableListRuleService;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.nebular.mdm.availablelist.AvalibleListProductLevelChangeDto;
import com.biz.crm.nebular.mdm.availablelist.TerminalAvailablelistVo;
import com.biz.crm.nebular.mdm.product.resp.MdmProductAdviseRespVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalVo;
import com.biz.crm.product.service.MdmProductService;
import com.biz.crm.productlevel.service.MdmProductLevelService;
import com.biz.crm.terminal.service.MdmTerminalService;
import com.biz.crm.util.ContextUtil;
import com.biz.crm.util.EsUtil;
import com.biz.crm.util.RedissonUtil;
import com.biz.crm.util.ValidateUtils;
import com.biz.crm.utils.MdmConstant;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.aggregations.Aggregation;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;
import org.springframework.data.elasticsearch.core.query.DeleteQuery;
import org.springframework.data.elasticsearch.core.query.NativeSearchQueryBuilder;
import org.springframework.data.elasticsearch.core.query.SearchQuery;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @Description:
 * @Author: zhangyuzhu
 * @Date: 2020/10/9 15:02
 **/
@Slf4j
@ConditionalOnMissingBean(name = "terminalAvailablelistServiceExpandImpl")
@Service(value = "terminalAvailablelistService")
public class TerminalAvailablelistServiceImpl<M extends BaseMapper<T>,T> implements TerminalAvailablelistService {

    @Autowired
    private ElasticsearchRestTemplate elasticsearchRestTemplate;

    @Autowired
    private RedissonUtil redissonUtil;

    @Autowired
    private TerminalAvailablelistRepositories terminalAvailablelistRepositories;

    @Resource
    private MdmTerminalService mdmTerminalService;

    @Resource
    private MdmProductService mdmProductService;

    @Resource
    private AvailableListRuleService availableListRuleService;

    @Resource
    private Advise advise;

    @Resource
    private TerminalChangeAdviseTargetStrategy terminalChangeAdviseTargetStrategy;

    @Autowired
    private ContextUtil contextUtil;

    @Autowired
    private MdmProductLevelService mdmProductLevelService;


    /**
     * 1、验证
     * 2、删除
     * 3、批量添加
     * @param goods 商品列表
     * @param terminalCode 终端编码
     */
    @Override
    public void reFresh(List<MdmProductAdviseRespVo> goods, String terminalCode) {
        //1
        ValidateUtils.validate(terminalCode,"终端编码不能为空！");
        MdmTerminalVo terminalReqVo = mdmTerminalService.query(null,terminalCode);
        ValidateUtils.validate(terminalReqVo,"该终端不存在,终端编码:"+terminalCode);

        //2
        if(EsUtil.indexExsit(elasticsearchRestTemplate, TerminalAvailablelistEntity.class,redissonUtil)){
            DeleteQuery deleteQuery = new DeleteQuery();
            BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
            boolQueryBuilder.must(QueryBuilders.termQuery("terminalCode.keyword",terminalCode));
            deleteQuery.setQuery(boolQueryBuilder);
            elasticsearchRestTemplate.delete(deleteQuery,TerminalAvailablelistEntity.class);
            elasticsearchRestTemplate.refresh(MdmConstant.ES_TERMINALAVAILABLELIST_INDEXNAME);
        }

        //3
        if(CollectionUtils.isEmpty(goods)){
            return;
        }
        terminalAvailablelistRepositories.saveAll(TerminalAvailablelistUtil.packageEntities(goods,terminalReqVo));
        elasticsearchRestTemplate.refresh(MdmConstant.ES_TERMINALAVAILABLELIST_INDEXNAME);
    }

    /**
     * 1、查询终端信息
     * 2、查询终端可够商品code（去重后）
     * 3、查询商品详情信息
     * 4、刷新
     * @param terminalCode 终端code
     */
    @Klock(keys = {MdmConstant.TERMINALAVAILBLELIST_LOCK ,"#terminalCode"},waitTime = 30,leaseTime = 10)
    @Override
    public void reFresh(String terminalCode) {
        //1
        Map<String, List<String>> terminalList = mdmTerminalService.findOrgCodeListByTerminalCodeList(Lists.newArrayList(terminalCode));
        if(CollectionUtils.isEmpty(terminalList)){
            log.info("终端查询失败");
            return;
        }

        //2
        List<String> goodsCodes = availableListRuleService.findGoodsByTerminalCodeAndOrgCode(terminalCode,terminalList.get(terminalCode));

        //3
        List<MdmProductAdviseRespVo> goods = new ArrayList<>(0);
        if(!CollectionUtils.isEmpty(goodsCodes)){
            List<String> currentGoodsCodes = new ArrayList<>();
            int i = 0;
            for(String goodsCode : goodsCodes){
                i ++;
                currentGoodsCodes.add(goodsCode);
                if(currentGoodsCodes.size() == 500 || i == goodsCodes.size()){
                    goods.addAll(mdmProductService.adviseQuery(currentGoodsCodes));
                    currentGoodsCodes.clear();
                }
            }
        }else {
            log.info("该终端没有查询到可够商品");
        }

        //4
        this.reFresh(goods,terminalCode);
    }

    @Override
    public void reFresh(List<String> terminalCodes) {
        if(CollectionUtils.isEmpty(terminalCodes)){
            log.info("终端编码为空");
            return;
        }
        //加上catch 避免某一个终端刷新失败影响所有需要刷新的终端
        TerminalAvailablelistService terminalAvailablelistService = (TerminalAvailablelistService) contextUtil.getBean(TerminalAvailablelistService.class);
        for(String terminalCode : terminalCodes){
            try{
                terminalAvailablelistService.reFresh(terminalCode);
            }catch (Exception e){
                log.error(new StringBuilder("可够清单刷新失败，终端:").append(terminalCode).toString(),e);
            }
        }
    }

    /**
     * 1、查询包含这些产品层级的规则【有效的】的所有终端【去重之后】
     * 2、批量刷新终端可够清单
     * @param productLevelCode
     */
    @Override
    public void reFreshByProductLevel(List<String> productLevelCode) {
        if(CollectionUtils.isEmpty(productLevelCode)){
            log.info("产品层级code列表为空11111!");
            return;
        }
        //1
        List<String> terminalCodes = availableListRuleService.listTerminalCodesByProductLevel(productLevelCode);

        //2
        if(CollectionUtils.isEmpty(terminalCodes)){
            log.info("该次变动没有涉及到的终端!");
            return;
        }
        this.reFresh(terminalCodes);
    }

    /**
     * 1、根据商品查询设计到的终端
     * 2、循环刷新终端的可够商品
     * @param dtoMap
     */
    @Override
    public void reFreshProductLevel(Map<String, AvalibleListProductLevelChangeDto> dtoMap) {
        if(CollectionUtils.isEmpty(dtoMap)){
            return;
        }
        //1
        List<TerminalAvailablelistEntity> entityList = terminalAvailablelistRepositories.findByProductCodeIn(new ArrayList<>(dtoMap.keySet()));
        if(CollectionUtils.isEmpty(entityList)){
            log.info("暂无设计到该商品的可够清单");
            return;
        }

        //2
        //加上catch 避免某一个终端刷新失败影响所有需要刷新的客户
        TerminalAvailablelistService terminalAvailablelistService = (TerminalAvailablelistService) contextUtil.getBean(TerminalAvailablelistService.class);
        for(TerminalAvailablelistEntity entity : entityList){
            try{
                terminalAvailablelistService.reFreshProductLevelByCus(entity.getTerminalCode(),dtoMap);
            }catch (Exception e){
                log.error(new StringBuilder("终端可够清单商品层级刷新失败，产品:").append(entity.getProductCode()).toString(),e);
            }
        }
    }

    /**
     * 根据客户刷新商品层级
     * 1、查询
     * 2、组装
     * 3、保存
     * @param terminalCode
     * @param dtoMap
     */
    @Override
    @Klock(keys = {MdmConstant.CUSAVAILBLELIST_LOCK ,"#terminalCode"},waitTime = 30,leaseTime = 10)
    public void reFreshProductLevelByCus(String terminalCode, Map<String, AvalibleListProductLevelChangeDto> dtoMap) {
        //1
        List<TerminalAvailablelistEntity> entityList = terminalAvailablelistRepositories.findByTerminalCodeAndProductCodeIn(terminalCode,new ArrayList<>(dtoMap.keySet()));
        if(CollectionUtils.isEmpty(entityList)){
            return;
        }

        //2
        List<TerminalAvailablelistEntity> saveList = new ArrayList<>();
        for(TerminalAvailablelistEntity entity : entityList){
            AvalibleListProductLevelChangeDto dto = dtoMap.get(entity.getProductCode());
            if(null != dto){
                entity.setProductLevelCode(dto.getNewLevelCode());
                entity.setProductLevelName(dto.getNewLevelName());
                saveList.add(entity);
            }
        }

        //3
        if(!CollectionUtils.isEmpty(saveList)){
            terminalAvailablelistRepositories.saveAll(saveList);
            //elasticsearchRestTemplate.refresh(MdmConstant.ES_TERMINALAVAILABLELIST_INDEXNAME);
        }
    }

    /**
     * 1、验证
     * 2、查询受影响的客户
     * 3、更新客户可够清单
     * @param mdmProductAdviseRespVos
     */
    @Override
    public void updateByGoodsCodes(List<MdmProductAdviseRespVo> mdmProductAdviseRespVos) {
        //1
        if(CollectionUtils.isEmpty(mdmProductAdviseRespVos)){
            log.info("参数为空");
            return;
        }
        if(!EsUtil.indexExsit(elasticsearchRestTemplate, TerminalAvailablelistEntity.class,redissonUtil)){
            log.info("暂无es索引");
            return;
        }

        //2
        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        boolQueryBuilder.must(QueryBuilders.termsQuery("productCode.keyword",
                mdmProductAdviseRespVos.stream().map(MdmProductAdviseRespVo :: getProductCode).collect(Collectors.toList())
        ));
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("terminalCodes").field("terminalCode.keyword").size(10000);
        SearchQuery searchQuery = new NativeSearchQueryBuilder()
                .withIndices(MdmConstant.ES_TERMINALAVAILABLELIST_INDEXNAME)//索引名
                .withQuery(boolQueryBuilder)//查询条件
                .withPageable(PageRequest.of(0,10))
                .addAggregation(aggregationBuilder)//聚合
                .build();
        Aggregations aggregations = elasticsearchRestTemplate.query(searchQuery, response -> response.getAggregations());
        Aggregation tags = aggregations.asMap().get("terminalCodes");
        Terms teamSum= (Terms)tags;
        List<? extends Terms.Bucket> buckets = teamSum.getBuckets();
        List<String> terminalCodes = new ArrayList<>();
        for (Terms.Bucket bucket : buckets) {
            terminalCodes.add(bucket.getKeyAsString());
        }
        if(CollectionUtils.isEmpty(terminalCodes)){
            log.info("暂无涉及到的在终端");
            return;
        }

        //3
        //加上catch 避免某一个终端刷新失败影响所有需要刷新的终端
        TerminalAvailablelistService terminalAvailablelistService = (TerminalAvailablelistService) contextUtil.getBean(TerminalAvailablelistService.class);
        for(String terminalCode : terminalCodes){
            try{
                terminalAvailablelistService.updateByGoodsCodesAndTerminalCode(mdmProductAdviseRespVos,terminalCode);
            }catch (Exception e){
                log.error(new StringBuilder("可够清单商品删除失败，终端:").append(terminalCode).toString(),e);
            }
        }
    }

    @Klock(keys = {MdmConstant.TERMINALAVAILBLELIST_LOCK ,"#terminalCode"},waitTime = 30,leaseTime = 10)
    @Override
    public void updateByGoodsCodesAndTerminalCode(List<MdmProductAdviseRespVo> mdmProductAdviseRespVos, String terminalCode) {
        if(!EsUtil.indexExsit(elasticsearchRestTemplate, TerminalAvailablelistEntity.class,redissonUtil)){
            log.info("暂无es索引");
            return;
        }
        mdmProductAdviseRespVos = mdmProductAdviseRespVos.stream().filter(vo -> !StringUtils.isEmpty(vo.getProductCode())).collect(Collectors.toList());
        List<TerminalAvailablelistEntity> terminalAvailablelistEntities = terminalAvailablelistRepositories.findByTerminalCodeAndProductCodeIn(
                terminalCode,
                mdmProductAdviseRespVos.stream().map(MdmProductAdviseRespVo :: getProductCode).collect(Collectors.toList())
        );
        if(CollectionUtils.isEmpty(terminalAvailablelistEntities)){
            log.info("暂无相关可够清单");
            return;
        }

        //2
        Map<String, MdmProductAdviseRespVo> goodsMap = mdmProductAdviseRespVos.stream().collect(
                Collectors.toMap(MdmProductAdviseRespVo::getProductCode, a -> a, (MdmProductAdviseRespVo k1, MdmProductAdviseRespVo k2) -> {
                    return (MdmProductAdviseRespVo) k1;
                })
        );

        int pageSize = 500;
        int total = terminalAvailablelistEntities.size();
        List<TerminalAvailablelistEntity> entities = new ArrayList<>(pageSize);
        for(int i = 0 ; i< total ; i++){
            TerminalAvailablelistEntity entity = terminalAvailablelistEntities.get(i);
            MdmProductAdviseRespVo vo = goodsMap.get(entity.getProductCode());
            if(null != vo){
                entity = TerminalAvailablelistUtil.copyGoodsVo(entity,vo);
                entities.add(entity);
            }else {
                log.info("{}产品没有查询到",entity.getProductCode());
            }
            if(entities.size() == pageSize || i == total-1){
                terminalAvailablelistRepositories.deleteByIdIn(entities.stream().map(TerminalAvailablelistEntity:: getId).collect(Collectors.toList()));
                terminalAvailablelistRepositories.saveAll(entities);
                entities.clear();
            }
        }
        //elasticsearchRestTemplate.refresh(MdmConstant.ES_TERMINALAVAILABLELIST_INDEXNAME);
    }

    /**
     * 1、验证
     * 2、查询受影响的终端
     * 3、删除终端可够清单
     * @param goodsCodes
     */
    @Override
    public void delByGoodsCodes(List<String> goodsCodes) {
        if(CollectionUtils.isEmpty(goodsCodes)){
            log.info("参数为空");
            return;
        }
        if(!EsUtil.indexExsit(elasticsearchRestTemplate, TerminalAvailablelistEntity.class,redissonUtil)){
            log.info("es索引不存在");
            return;
        }
        //2
        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        boolQueryBuilder.must(QueryBuilders.termsQuery("productCode.keyword",
                goodsCodes
        ));
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("terminalCodes").field("terminalCode.keyword").size(10000);
        SearchQuery searchQuery = new NativeSearchQueryBuilder()
                .withIndices(MdmConstant.ES_TERMINALAVAILABLELIST_INDEXNAME)//索引名
                .withQuery(boolQueryBuilder)//查询条件
                .withPageable(PageRequest.of(0,10))
                .addAggregation(aggregationBuilder)//聚合
                .build();
        Aggregations aggregations = elasticsearchRestTemplate.query(searchQuery, response -> response.getAggregations());
        Aggregation tags = aggregations.asMap().get("terminalCodes");
        Terms teamSum= (Terms)tags;
        List<? extends Terms.Bucket> buckets = teamSum.getBuckets();
        List<String> terminalCodes = new ArrayList<>();
        for (Terms.Bucket bucket : buckets) {
            terminalCodes.add(bucket.getKeyAsString());
        }
        if(CollectionUtils.isEmpty(terminalCodes)){
            log.info("暂无相关终端");
        }

        //3
        //加上catch 避免某一个终端刷新失败影响所有需要刷新的终端
        TerminalAvailablelistService terminalAvailablelistService = (TerminalAvailablelistService) contextUtil.getBean(TerminalAvailablelistService.class);
        for(String terminalCode : terminalCodes){
            try{
                terminalAvailablelistService.delByGoodsCodesAndTerminalCode(goodsCodes,terminalCode);
            }catch (Exception e){
                log.error(new StringBuilder("可够清单商品删除失败，终端:").append(terminalCode).toString(),e);
            }
        }
    }

    @Klock(keys = {MdmConstant.TERMINALAVAILBLELIST_LOCK ,"#terminalCode"},waitTime = 30,leaseTime = 10)
    @Override
    public void delByGoodsCodesAndTerminalCode(List<String> goodsCodes, String terminalCode) {
        terminalAvailablelistRepositories.deleteByTerminalCodeAndProductCodeIn(terminalCode,goodsCodes);
    }

    @CrmDictMethod
    @Override
    public PageResult<TerminalAvailablelistVo> list(TerminalAvailablelistVo terminalAvailablelistVo) {
        PageResult<TerminalAvailablelistVo> pageResult = new PageResult<TerminalAvailablelistVo>();
        pageResult.setCount(Long.parseLong("0"));
        pageResult.setData(new ArrayList<>());
        if(!EsUtil.indexExsit(elasticsearchRestTemplate,TerminalAvailablelistEntity.class,redissonUtil)){
            return pageResult;
        }

        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getTerminalCode())){
            boolQueryBuilder.must(QueryBuilders.termQuery("terminalCode.keyword",terminalAvailablelistVo.getTerminalCode()));
        }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getTerminalName())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("terminalName",terminalAvailablelistVo.getTerminalName()));
        }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getTerminalOrgCode())){
            boolQueryBuilder.must(QueryBuilders.termQuery("terminalOrgCode.keyword",terminalAvailablelistVo.getTerminalOrgCode()));
        }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getTerminalOrgName())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("terminalOrgName",terminalAvailablelistVo.getTerminalOrgName()));
        }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getTerminalChannelCode())){
            boolQueryBuilder.must(QueryBuilders.termQuery("terminalChannelCode.keyword",terminalAvailablelistVo.getTerminalChannelCode()));
        }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getTerminalChannelName())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("terminalChannelName",terminalAvailablelistVo.getTerminalChannelName()));
        }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getProductType())){
            boolQueryBuilder.must(QueryBuilders.termQuery("productType.keyword",terminalAvailablelistVo.getProductType()));
        }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getProductLevelCode())){
            List<String> productLevelCodes = mdmProductLevelService.getAllChildrenProductLevelCodeList(null, terminalAvailablelistVo.getProductLevelCode());
            if(CollectionUtils.isEmpty(productLevelCodes)){
                return pageResult;
            }
            BoolQueryBuilder productLevelCodeBoolQueryBuilder = QueryBuilders.boolQuery();
            //查询
            for (String productLevelCode: productLevelCodes){
                productLevelCodeBoolQueryBuilder.should(QueryBuilders.termQuery("productLevelCode.keyword",productLevelCode));
            }
            boolQueryBuilder.must(productLevelCodeBoolQueryBuilder);
        }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getProductCode())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("productCode",terminalAvailablelistVo.getProductCode()));
        }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getProductName())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("productName",terminalAvailablelistVo.getProductName()));
        }

        SearchQuery searchQuery = new NativeSearchQueryBuilder()
                .withIndices(MdmConstant.ES_TERMINALAVAILABLELIST_INDEXNAME)//索引名
                .withQuery(boolQueryBuilder)//查询条件
//                .withSort(SortBuilders.fieldSort("fictitiousId").order(SortOrder.DESC))
                .withPageable(PageRequest.of(terminalAvailablelistVo.getPageNum().intValue()-1,terminalAvailablelistVo.getPageSize()))//分页参数
                .build();
        Page<TerminalAvailablelistEntity> scroll = elasticsearchRestTemplate.queryForPage(searchQuery,TerminalAvailablelistEntity.class);
        List<TerminalAvailablelistEntity> list = scroll.getContent();
        if(CollectionUtils.isEmpty(list)){
            return pageResult;
        }
        List<TerminalAvailablelistVo> reList = new ArrayList<>(list.size());
        for(TerminalAvailablelistEntity entity : list){
            TerminalAvailablelistVo vo = new TerminalAvailablelistVo();
            BeanUtils.copyProperties(entity,vo);
            reList.add(vo);
        }
        pageResult.setData(reList);
        pageResult.setCount(scroll.getTotalElements());
        return pageResult;
    }

    @Override
    public PageResult<TerminalAvailablelistVo> listForM(TerminalAvailablelistVo terminalAvailablelistVo) {
        PageResult<TerminalAvailablelistVo> pageResult = new PageResult<TerminalAvailablelistVo>();
        pageResult.setCount(Long.parseLong("0"));
        pageResult.setData(new ArrayList<>());
        if(!EsUtil.indexExsit(elasticsearchRestTemplate,TerminalAvailablelistEntity.class,redissonUtil)){
            return pageResult;
        }

        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getTerminalCode())){
            boolQueryBuilder.must(QueryBuilders.wildcardQuery("terminalCode.keyword","*"+terminalAvailablelistVo.getTerminalCode()+"*"));
        }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getIsShelf())){
            boolQueryBuilder.must(QueryBuilders.termQuery("isShelf.keyword",terminalAvailablelistVo.getIsShelf()));
        }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getTerminalName())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("terminalName",terminalAvailablelistVo.getTerminalName()));
        }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getTerminalOrgCode())){
            boolQueryBuilder.must(QueryBuilders.wildcardQuery("terminalOrgCode.keyword","*"+terminalAvailablelistVo.getTerminalOrgCode()+"*"));
                    }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getTerminalOrgName())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("terminalOrgName",terminalAvailablelistVo.getTerminalOrgName()));
        }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getTerminalChannelCode())){
            boolQueryBuilder.must(QueryBuilders.termQuery("terminalChannelCode.keyword",terminalAvailablelistVo.getTerminalChannelCode()));
        }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getTerminalChannelName())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("terminalChannelName",terminalAvailablelistVo.getTerminalChannelName()));
        }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getProductType())){
            boolQueryBuilder.must(QueryBuilders.termQuery("productType.keyword",terminalAvailablelistVo.getProductType()));
        }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getProductLevelCode())){
            List<String> productLevelCodes = mdmProductLevelService.getAllChildrenProductLevelCodeList(null, terminalAvailablelistVo.getProductLevelCode());
            if(CollectionUtils.isEmpty(productLevelCodes)){
                return pageResult;
            }
            BoolQueryBuilder productLevelCodeBoolQueryBuilder = QueryBuilders.boolQuery();
            //查询
            for (String productLevelCode: productLevelCodes){
                productLevelCodeBoolQueryBuilder.should(QueryBuilders.termQuery("productLevelCode.keyword",productLevelCode));
            }
            boolQueryBuilder.must(productLevelCodeBoolQueryBuilder);
        }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getProductCode())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("productCode",terminalAvailablelistVo.getProductCode()));
        }
        if(!StringUtils.isEmpty(terminalAvailablelistVo.getProductName())){
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("productName",terminalAvailablelistVo.getProductName()));
        }

        SearchQuery searchQuery = new NativeSearchQueryBuilder()
                .withIndices(MdmConstant.ES_TERMINALAVAILABLELIST_INDEXNAME)//索引名
                .withQuery(boolQueryBuilder)//查询条件
//                .withSort(SortBuilders.fieldSort("fictitiousId").order(SortOrder.DESC))
                .withPageable(PageRequest.of(terminalAvailablelistVo.getPageNum().intValue()-1,terminalAvailablelistVo.getPageSize()))//分页参数
                .build();
        Page<TerminalAvailablelistEntity> scroll = elasticsearchRestTemplate.queryForPage(searchQuery,TerminalAvailablelistEntity.class);
        List<TerminalAvailablelistEntity> list = scroll.getContent();
        if(CollectionUtils.isEmpty(list)){
            return pageResult;
        }
        List<TerminalAvailablelistVo> reList = new ArrayList<>(list.size());
        for(TerminalAvailablelistEntity entity : list){
            TerminalAvailablelistVo vo = new TerminalAvailablelistVo();
            BeanUtils.copyProperties(entity,vo);
            reList.add(vo);
        }
        pageResult.setData(reList);
        pageResult.setCount(scroll.getTotalElements());
        return pageResult;
    }

    /**
     * 1、验证
     * 2、删除
     * @param id
     */
    @Override
    public void delById(String id) {
        if(StringUtils.isEmpty(id)){
            return;
        }
        if(!EsUtil.indexExsit(elasticsearchRestTemplate, TerminalAvailablelistEntity.class,redissonUtil)){
            return;
        }
        Optional<TerminalAvailablelistEntity> optionalTerminalAvailablelistEntity = terminalAvailablelistRepositories.findById(id);
        if(!optionalTerminalAvailablelistEntity.isPresent()){
            return;
        }
        //2
        TerminalAvailablelistService terminalAvailablelistService = (TerminalAvailablelistService) contextUtil.getBean(TerminalAvailablelistService.class);
        terminalAvailablelistService.delByIdAndTerminalCode(id,optionalTerminalAvailablelistEntity.get().getTerminalCode());
    }

    @Klock(keys = {MdmConstant.TERMINALAVAILBLELIST_LOCK ,"#terminalCode"},waitTime = 30,leaseTime = 10)
    @Override
    public void delByIdAndTerminalCode(String id, String terminalCode) {
        terminalAvailablelistRepositories.deleteById(id);
        elasticsearchRestTemplate.refresh(MdmConstant.ES_TERMINALAVAILABLELIST_INDEXNAME);
    }

    /**
     * 1、验证
     * 2、删除
     * @param ids
     */
    @Override
    public void delByIds(List<String> ids) {
        if(CollectionUtils.isEmpty(ids)){
            return;
        }
        if(!EsUtil.indexExsit(elasticsearchRestTemplate, TerminalAvailablelistEntity.class,redissonUtil)){
            return;
        }
        List<TerminalAvailablelistEntity> entities = terminalAvailablelistRepositories.findByIdIn(ids);
        Map<String, List<TerminalAvailablelistEntity>> entityMap = entities.stream().collect(Collectors.groupingBy(TerminalAvailablelistEntity::getTerminalCode));

        //2
        TerminalAvailablelistService terminalAvailablelistService = (TerminalAvailablelistService) contextUtil.getBean(TerminalAvailablelistService.class);
        for(Map.Entry<String, List<TerminalAvailablelistEntity>> entry : entityMap.entrySet()){
            List<TerminalAvailablelistEntity> handleEntities = entry.getValue();
            if(CollectionUtils.isEmpty(handleEntities)){
                continue;
            }
            try{
                terminalAvailablelistService.delByIdsAndTerminalCode(handleEntities.stream().map(TerminalAvailablelistEntity :: getId).collect(Collectors.toList())
                        ,entry.getKey());
            }catch (Exception e){
                log.error(new StringBuilder("可够清单删除失败，终端:").append(entry.getKey()).toString(),e);
            }
        }

    }

    @Klock(keys = {MdmConstant.TERMINALAVAILBLELIST_LOCK ,"#terminalCode"},waitTime = 30,leaseTime = 10)
    @Override
    public void delByIdsAndTerminalCode(List<String> ids, String terminalCode) {
        if(CollectionUtils.isEmpty(ids)){
            return;
        }
        terminalAvailablelistRepositories.deleteByIdIn(ids);
        elasticsearchRestTemplate.refresh(MdmConstant.ES_TERMINALAVAILABLELIST_INDEXNAME);
    }

    @Override
    public TerminalAvailablelistVo findById(String id) {
        Optional<TerminalAvailablelistEntity> optionalTerminalAvailablelistEntity = terminalAvailablelistRepositories.findById(id);
        if(!optionalTerminalAvailablelistEntity.isPresent()){
            return null;
        }
        TerminalAvailablelistEntity entity = optionalTerminalAvailablelistEntity.get();
        TerminalAvailablelistVo vo = new TerminalAvailablelistVo();
        BeanUtils.copyProperties(entity,vo);
        return vo;
    }
}
