package com.biz.crm.coderule.service.impl;

import com.biz.crm.annotation.Klock;
import com.biz.crm.coderule.mapper.MdmCodeRuleMapper;
import com.biz.crm.coderule.model.MdmCodeRuleEntity;
import com.biz.crm.coderule.service.MdmCodeRuleGeneratorService;
import com.biz.crm.coderule.service.MdmCodeRuleService;
import com.biz.crm.util.StringUtils;
import com.biz.crm.utils.MdmConstant;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * 编码生成service
 *
 * @author Tao.Chen
 * @date 2020-11-05 10:47:13
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmCodeRuleGeneratorServiceImpl")
public class MdmCodeRuleGeneratorServiceImpl implements MdmCodeRuleGeneratorService {

    @Resource
    private MdmCodeRuleMapper mdmCodeRuleMapper;
    @Resource
    private MdmCodeRuleService mdmCodeRuleService;

    @Override
    @Klock(keys = {MdmConstant.CODE_GENERATOR, "#ruleCode"}, waitTime = 10, leaseTime = 5)
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRES_NEW)
    public List<String> generateCodeList(String ruleCode, Integer number) {
        Assert.hasText(ruleCode, "编码规则不能为空");
        Assert.notNull(number, "编码个数不能为空");
        Assert.isTrue(number >= 1, "至少生成一个编码");

        List<String> list = new ArrayList<>();
        MdmCodeRuleEntity rule = mdmCodeRuleService.lambdaQuery().eq(MdmCodeRuleEntity::getRuleCode, ruleCode).one();
        Assert.notNull(rule, "未找到编码规则：" + ruleCode);

        //当前时间
        Date now = new Date();
        String prefix = rule.getPrefix() == null ? "" : rule.getPrefix();
        String dateStr = "";
        if (StringUtils.isNotEmpty(rule.getDateFormat())) {
            SimpleDateFormat sdf = new SimpleDateFormat(rule.getDateFormat());
            dateStr = sdf.format(now);
        }
        Integer currentValue = rule.getCurrentValue();
        int numberLength = rule.getCodeLength() - prefix.length() - dateStr.length();
        for (int i = 1; i <= number; i++) {
            currentValue++;
            list.add(prefix + dateStr + String.format("%0" + numberLength + "d", currentValue));
        }
        rule.setCurrentValue(currentValue);

        //记录最后一次生成时间
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        rule.setGenerateDate(sdf.format(now));

        mdmCodeRuleService.updateById(rule);
        return list;
    }
}
