package com.biz.crm.cusorg.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.annotation.Klock;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.cusorg.entity.MdmCusOrgEntity;
import com.biz.crm.cusorg.mapper.MdmCusOrgMapper;
import com.biz.crm.cusorg.service.MdmCusOrgService;
import com.biz.crm.customer.model.MdmCustomerEntity;
import com.biz.crm.customer.service.MdmCustomerMsgService;
import com.biz.crm.eunm.CodeRuleEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.nebular.mdm.CrmTreeVo;
import com.biz.crm.nebular.mdm.common.BaseTreeReqVo;
import com.biz.crm.nebular.mdm.common.BaseTreeRespVo;
import com.biz.crm.nebular.mdm.cusorg.*;
import com.biz.crm.terminal.model.MdmTerminalEntity;
import com.biz.crm.terminal.service.MdmTerminalService;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Description:
 * @Author: zhangyuzhu
 * @Date: 2020/8/26 11:33
 **/
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmCusOrgServiceExpandImpl")
public class MdmCusOrgServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmCusOrgMapper, MdmCusOrgEntity> implements MdmCusOrgService {

    @Resource
    private MdmCusOrgMapper cusOrgMapper;
    @Resource
    private MdmCustomerMsgService mdmCustomerMsgService;
    @Resource
    private MdmTerminalService mdmTerminalService;
    @Resource
    private CrmLogSendUtil crmLogSendUtil;


    @Override
    @Klock(leaseTime = 1, keys = "customer_org")
    @Transactional(rollbackFor = Exception.class)
    public void add(CusOrgVo cusOrgVo) {
        Assert.hasText(cusOrgVo.getCustomerOrgName(), "客户组织名称不能为空");
        if (StringUtils.isEmpty(cusOrgVo.getCustomerOrgCode())) {
            cusOrgVo.setCustomerOrgCode(CodeUtil.generateCode(CodeRuleEnum.MDM_CUSTOMER_ORG.getCode()));
        } else {
            MdmCusOrgEntity one = this.lambdaQuery()
                    .eq(MdmCusOrgEntity::getCustomerOrgCode, cusOrgVo.getCustomerOrgCode())
                    .select(MdmCusOrgEntity::getId)
                    .one();
            Assert.isNull(one, "客户组织编码已存在");
        }
        MdmCusOrgEntity name = this.lambdaQuery()
                .eq(MdmCusOrgEntity::getCustomerOrgName, cusOrgVo.getCustomerOrgName())
                .select(MdmCusOrgEntity::getId)
                .one();
        Assert.isNull(name, "客户组织名称已存在");
        MdmCusOrgEntity entity = CrmBeanUtil.copy(cusOrgVo, MdmCusOrgEntity.class);
        CrmTreeVo treeVo = generateRuleCodeByParentCode(entity.getParentCode());
        updateCurAndChildren(entity.getCustomerOrgCode(), treeVo.getRuleCode(), treeVo.getLevelNum());
        entity.setRuleCode(treeVo.getRuleCode());
        entity.setLevelNum(treeVo.getLevelNum());
        this.save(entity);
        CustomerOrgUtil.deleteAllCache();
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            crmLogSendUtil.sendForAdd(((String) menuObject), entity.getId(), entity.getCustomerOrgName(), cusOrgVo);
        }
    }

    @Override
    public CusOrgVo query(String id, String customerOrgCode) {
        MdmCusOrgEntity one = this.lambdaQuery()
                .eq(!StringUtils.isEmpty(customerOrgCode), MdmCusOrgEntity::getCustomerOrgCode, customerOrgCode)
                .eq(!StringUtils.isEmpty(id), MdmCusOrgEntity::getId, id)
                .last(DatabaseTypeUtil.SEGMENT)
                .one();
        return CrmBeanUtil.copy(one, CusOrgVo.class);
    }


    @Override
    @Klock(leaseTime = 1, keys = "customer_org")
    @Transactional(rollbackFor = Exception.class)
    public void edit(CusOrgVo cusOrgVo) {
        String id = cusOrgVo.getId();
        Assert.hasText(id, "请指定要编辑的客户组织!");
        MdmCusOrgEntity entity = cusOrgMapper.selectById(id);
        Assert.notNull(entity, "该客户组织不存在或被删除");
        MdmCusOrgEntity one = this.lambdaQuery()
                .eq(MdmCusOrgEntity::getCustomerOrgCode, cusOrgVo.getCustomerOrgCode())
                .select(MdmCusOrgEntity::getId)
                .one();
        Assert.isTrue(one == null || id.equals(one.getId()), "客户组织编码已存在");
        MdmCusOrgEntity name = this.lambdaQuery()
                .eq(MdmCusOrgEntity::getCustomerOrgName, cusOrgVo.getCustomerOrgName())
                .select(MdmCusOrgEntity::getId)
                .one();
        Assert.isTrue(name == null || id.equals(name.getId()), "客户组织名称已存在");

        String entityRuleCode = entity.getRuleCode();
        Assert.hasText(entityRuleCode, "当前客户组织ruleCode不能为空");
        if (!StringUtils.isEmpty(cusOrgVo.getParentCode())) {
            Assert.isTrue(!cusOrgVo.getParentCode().equals(cusOrgVo.getCustomerOrgCode()), "上级客户组织不能与客户组织相同");
            QueryWrapper<MdmCusOrgEntity> wrapper = Wrappers.<MdmCusOrgEntity>query()
                    .eq("customer_org_code", cusOrgVo.getParentCode());
            MdmCusOrgEntity parent = this.getOne(wrapper, false);
            Assert.notNull(parent, "上级客户组织不存在");
            if (!parent.getCustomerOrgCode().equals(entity.getParentCode())) {
                String ruleCode = parent.getRuleCode();
                Assert.hasText(ruleCode, "上级客户组织ruleCode不能为空");
                Assert.isTrue(!ruleCode.startsWith(entityRuleCode), "上级客户组织不能是当前客户组织或当前客户组织的下级客户组织");
            }
        }
        CrmBeanUtil.copyProperties(cusOrgVo, entity);
        this.updateById(entity);
        CrmTreeVo treeVo = generateRuleCodeByParentCode(entity.getParentCode());
        updateCurAndChildren(entity.getCustomerOrgCode(), treeVo.getRuleCode(), treeVo.getLevelNum());
        CustomerOrgUtil.deleteAllCache();
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            crmLogSendUtil.sendForUpdate(((String) menuObject), entity.getId(), entity.getCustomerOrgName(), CrmBeanUtil.copy(entity, CusOrgVo.class), cusOrgVo);
        }
    }


    @Override
    public PageResult<CusOrgVo> list(CusOrgVo cusOrgVo) {
        Page<CusOrgVo> page = PageUtil.buildPage(cusOrgVo.getPageNum(), cusOrgVo.getPageSize());
        if (!StringUtils.isEmpty(cusOrgVo.getCode())) {
            //查询出来为空不抛异常
            MdmCusOrgEntity one = this.lambdaQuery()
                    .eq(MdmCusOrgEntity::getCustomerOrgCode, cusOrgVo.getCode())
                    .select(MdmCusOrgEntity::getRuleCode)
                    .one();
            if (one != null) {
                cusOrgVo.setRuleCode(one.getRuleCode());
            }
//            String ruleCode = getRuleCode(cusOrgVo.getCode());
//            cusOrgVo.setRuleCode(ruleCode);
        }

        List<CusOrgVo> list = cusOrgMapper.list(page, cusOrgVo);
        return PageResult.<CusOrgVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 搜索客户组织list
     * @param cusOrgVo
     * @return
     */
    @Override
    public List<CusOrgVo> findList(CusOrgVo cusOrgVo) {
        return cusOrgMapper.list(cusOrgVo);
    }

    @Override
    public CusOrgVo findById(String id) {
        return CrmBeanUtil.copy(this.getById(id), CusOrgVo.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void ableBatch(List<String> ids) {
        Assert.notEmpty(ids, "空参数");
        List<MdmCusOrgEntity> list = this.lambdaQuery()
                .in(MdmCusOrgEntity::getId, ids)
                .select(MdmCusOrgEntity::getId, MdmCusOrgEntity::getEnableStatus, MdmCusOrgEntity::getRuleCode)
                .list()
                .stream().filter(x -> !CrmEnableStatusEnum.ENABLE.getCode().equals(x.getEnableStatus())).collect(Collectors.toList());
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            List<String> ruleCodeList = list.stream().map(MdmCusOrgEntity::getRuleCode).collect(Collectors.toList());
            Set<String> parentRuleCodesExcludeSelf = TreeRuleCodeUtil.splitParentRuleCodesExcludeAnySelf(ruleCodeList);
            if (!parentRuleCodesExcludeSelf.isEmpty()) {
                List<MdmCusOrgEntity> parentList = this.lambdaQuery()
                        .in(MdmCusOrgEntity::getRuleCode, parentRuleCodesExcludeSelf)
                        .select(MdmCusOrgEntity::getId, MdmCusOrgEntity::getEnableStatus)
                        .list()
                        .stream().filter(x -> !CrmEnableStatusEnum.ENABLE.getCode().equals(x.getEnableStatus())).collect(Collectors.toList());
                Assert.isTrue(CollectionUtil.listEmpty(parentList), "存在未启用的上级客户组织");
            }
            this.lambdaUpdate()
                    .in(MdmCusOrgEntity::getId, list.stream().map(MdmCusOrgEntity::getId).collect(Collectors.toSet()))
                    .set(MdmCusOrgEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                    .update();
            CustomerOrgUtil.deleteAllCache();
        }
    }

    @Override
    @Klock(leaseTime = 1, keys = "customer_org")
    @Transactional(rollbackFor = Exception.class)
    public void stateBatch(List<String> ids) {
        Assert.notEmpty(ids, "id不能为空");
        List<MdmCusOrgEntity> list = this.lambdaQuery()
                .in(MdmCusOrgEntity::getId, ids)
                .select(MdmCusOrgEntity::getCustomerOrgCode)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            List<String> customerOrgCodeList = list.stream().map(MdmCusOrgEntity::getCustomerOrgCode).collect(Collectors.toList());
            List<MdmCusOrgEntity> childrenExcludeSelf = this.lambdaQuery()
                    .in(MdmCusOrgEntity::getParentCode, customerOrgCodeList)
                    .notIn(MdmCusOrgEntity::getId, ids)
                    .select(MdmCusOrgEntity::getId)
                    .list();
            Assert.isTrue(CollectionUtil.listEmpty(childrenExcludeSelf), "存在下级客户组织，不能删除");
            List<MdmCustomerEntity> customerList = mdmCustomerMsgService.lambdaQuery()
                    .in(MdmCustomerEntity::getCustomerOrgCode, customerOrgCodeList)
                    .select(MdmCustomerEntity::getId)
                    .list();
            Assert.isTrue(CollectionUtil.listEmpty(customerList), "客户组织已经关联客户，不能删除");
            List<MdmTerminalEntity> terminalList = mdmTerminalService.lambdaQuery()
                    .in(MdmTerminalEntity::getCustomerOrgCode, customerOrgCodeList)
                    .select(MdmTerminalEntity::getId)
                    .list();
            Assert.isTrue(CollectionUtil.listEmpty(terminalList), "客户组织已经关联终端，不能删除");
            this.removeByIds(ids);
            CustomerOrgUtil.deleteAllCache();
            Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
            if (menuObject != null) {
                for (String id : ids) {
                    CusOrgVo query = query(id, null);
                    if (query != null) {
                        crmLogSendUtil.sendForDel(((String) menuObject), id, query.getCustomerOrgName(), query);
                    }
                }
            }
        }
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        Assert.notEmpty(ids, "空参数");
        List<MdmCusOrgEntity> list = this.lambdaQuery()
                .in(MdmCusOrgEntity::getId, ids)
                .select(MdmCusOrgEntity::getId, MdmCusOrgEntity::getEnableStatus, MdmCusOrgEntity::getRuleCode)
                .list()
                .stream().filter(x -> !CrmEnableStatusEnum.DISABLE.getCode().equals(x.getEnableStatus())).collect(Collectors.toList());
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            List<String> ruleCodeList = list.stream().map(MdmCusOrgEntity::getRuleCode).collect(Collectors.toList());
            //TODO 后续改成不是循环的
            ruleCodeList.forEach(ruleCode -> {
                this.lambdaUpdate()
                        .likeRight(MdmCusOrgEntity::getRuleCode, ruleCode)
                        .set(MdmCusOrgEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
                        .update();
            });
            CustomerOrgUtil.deleteAllCache();
        }
    }

    @Override
    public List<MdmCustomerOrgSelectRespVo> customerOrgSelect(CusOrgVo cusOrgVo) {
        String currentCustomerOrgCode = cusOrgVo.getCurrentCustomerOrgCode();
        if (!StringUtils.isEmpty(currentCustomerOrgCode)) {
            //查询非当前客户及下级的客户组织
            cusOrgVo.setCustomerOrgCode(currentCustomerOrgCode);
            return CrmBeanUtil.copyList(findUnSubList(cusOrgVo), MdmCustomerOrgSelectRespVo.class);
        }
        List<MdmCusOrgEntity> list = this.lambdaQuery()
                .like(!StringUtils.isEmpty(cusOrgVo.getCustomerOrgName()), MdmCusOrgEntity::getCustomerOrgName, cusOrgVo.getCustomerOrgName())
                .eq(!StringUtils.isEmpty(cusOrgVo.getCustomerOrgLevel()), MdmCusOrgEntity::getCustomerOrgLevel, cusOrgVo.getCustomerOrgLevel())
                .eq(MdmCusOrgEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .select(MdmCusOrgEntity::getId, MdmCusOrgEntity::getCustomerOrgCode, MdmCusOrgEntity::getCustomerOrgName)
                .list();
        return CrmBeanUtil.copyList(list, MdmCustomerOrgSelectRespVo.class);
    }

    @Override
    public List<MdmCustomerOrgSelectRespVo> findCustomerOrgSelectList(MdmCustomerOrgSelectReqVo reqVo) {
        reqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        List<MdmCustomerOrgSelectRespVo> list = new ArrayList<>();
        Integer pageSize = reqVo.getPageSize();
        if (pageSize == null || pageSize == 0) {
            pageSize = Integer.valueOf(ParamUtil.getParameterValue(ParameterParam.SELECT_BOX_OPTION_SIZE));
        }
        Set<String> codeList = new HashSet<>(16);
        if (org.apache.commons.lang3.StringUtils.isNotEmpty(reqVo.getSelectedCode())) {
            codeList.add(reqVo.getSelectedCode());
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSelectedCodeList())) {
            codeList.addAll(reqVo.getSelectedCodeList());
        }
        if (!codeList.isEmpty()) {
            reqVo.setSelectedCodeList(new ArrayList<>(codeList));
        }
        if (pageSize > 0) {
            list.addAll(cusOrgMapper.findCustomerOrgSelectList(new Page<>(1L, pageSize, false), reqVo));
        }
        return list;
    }


    @Override
    public List<BaseTreeRespVo> findBaseTreeConditionList(BaseTreeReqVo baseTreeReqVo) {
        return cusOrgMapper.findBaseTreeConditionList(baseTreeReqVo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @Klock
    public void setUpRuleCode() {
        long l = System.currentTimeMillis();
        log.info("-----------重置客户组织降维编码_" + l + "_开始-----------");
        cusOrgMapper.copyIdToRuleCode();
        log.info("-----------重置客户组织降维编码_" + l + "_编码重置成id成功-----------");
        cusOrgMapper.setNullNotExistParentCode();
        log.info("-----------重置客户组织降维编码_" + l + "_清除无效上级编码成功-----------");
        List<MdmCusOrgEntity> list = this.lambdaQuery()
                .isNull(MdmCusOrgEntity::getParentCode)
                .or()
                .eq(MdmCusOrgEntity::getParentCode, "")
                .list();
        for (int i = 0; i < list.size(); i++) {
            updateCurAndChildren(list.get(i).getCustomerOrgCode(), TreeRuleCodeUtil.numToSingleCode(i + 1), 1);
        }
        log.info("-----------重置客户组织降维编码_" + l + "_编码重新生成成功-----------");
        CustomerOrgUtil.deleteAllCache();
        log.info("-----------重置客户组织降维编码_" + l + "_缓存清除成功-----------");
        log.info("-----------重置客户组织降维编码_" + l + "_结束-----------");

//        List<MdmCusOrgEntity> list = this.lambdaQuery()
//                .select(MdmCusOrgEntity::getId,MdmCusOrgEntity::getCustomerOrgCode, MdmCusOrgEntity::getParentCode, MdmCusOrgEntity::getEnableStatus,MdmCusOrgEntity::getLevelNum)
//                .list();
//        if (CollectionUtils.isEmpty(list)) {
//            return;
//        }
//        updateBatchById(list.stream().peek(x -> x.setRuleCode(UUID.randomUUID().toString())).collect(Collectors.toList()));
//        Set<String> set = list.stream().map(MdmCusOrgEntity::getCustomerOrgCode).collect(Collectors.toSet());
//        List<String> rootCodeList = list.stream()
//                .filter(x -> StringUtils.isEmpty(x.getParentCode()) || !set.contains(x.getParentCode()))
//                .map(MdmCusOrgEntity::getCustomerOrgCode)
//                .collect(Collectors.toList());
//        if (CollectionUtils.isEmpty(rootCodeList)) {
//            return;
//        }
//        int i = 1;
//        for (MdmCusOrgEntity mdmCusOrgEntity : list) {
//            if (rootCodeList.contains(mdmCusOrgEntity.getCustomerOrgCode())) {
//                Assert.isTrue(i < 1000, "每层组织仅支持最大数量999");
//                mdmCusOrgEntity.setRuleCode(String.format("%03d", i++));
//                mdmCusOrgEntity.setLevelNum(mdmCusOrgEntity.getRuleCode().length()/3);
//                setUpRuleCode(mdmCusOrgEntity, list);
//            }
//        }
//        this.updateBatchById(list);
    }

    @Override
    public List<CusOrgVo> findUnSubList(CusOrgVo cusOrgVo) {
        String customerOrgCode = cusOrgVo.getCustomerOrgCode();
        if (StringUtils.isEmpty(customerOrgCode)) {
            return new ArrayList<>();
        }
        String ruleCode = getRuleCode(customerOrgCode);
        Assert.hasText(ruleCode, "ruleCode不能为空");
        cusOrgVo.setRuleCode(ruleCode);
        return cusOrgMapper.ruleNotLikeRight(cusOrgVo);
    }

    /**
     * 查询当前及所有上级客户组织
     *
     * @param customerOrgCode customerOrgCode
     * @return list
     */
    @Override
    public List<CusOrgVo> findAllParentList(String customerOrgCode) {
        if (StringUtils.isEmpty(customerOrgCode)) {
            return new ArrayList<>();
        }
        String ruleCode = getRuleCode(customerOrgCode);
        Assert.hasText(ruleCode, "ruleCode不能为空");
        Set<String> split = TreeRuleCodeUtil.splitParentRuleCodes(ruleCode);
        return listCondition(new MdmCustomerOrgConditionReqVo().setRuleCodeList(new ArrayList<>(split)));
    }

    /**
     * @param customerOrgCode 客户组织编码
     * @return ruleCode
     */
    private String getRuleCode(String customerOrgCode) {
        if (StringUtils.isEmpty(customerOrgCode)) {
            return customerOrgCode;
        }
        MdmCusOrgEntity one = this.lambdaQuery()
                .eq(MdmCusOrgEntity::getCustomerOrgCode, customerOrgCode)
                .select(MdmCusOrgEntity::getRuleCode)
                .one();
        Assert.notNull(one, "客户组织[" + customerOrgCode + "]不存在");
        return one.getRuleCode();
    }

    @Override
    public List<CusOrgVo> listCondition(MdmCustomerOrgConditionReqVo mdmCustomerOrgConditionReqVo) {
        return cusOrgMapper.listCondition(mdmCustomerOrgConditionReqVo);
    }

    @Override
    public List<MdmCustomerOrgRedisVo> findRedisListByCustomerOrgCodeList(List<String> customerOrgCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(customerOrgCodeList)) {
            List<MdmCusOrgEntity> list = this.lambdaQuery()
                    .in(MdmCusOrgEntity::getCustomerOrgCode, customerOrgCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                Map<String, String> codeNameMap = new HashMap<>(16);
                Set<String> parentCodeSet = list.stream().filter(item -> StringUtils.isNotEmpty(item.getParentCode())).map(MdmCusOrgEntity::getParentCode).collect(Collectors.toSet());
                if (!parentCodeSet.isEmpty()) {
                    List<MdmCusOrgEntity> parentCodeNameList = this.lambdaQuery()
                            .in(MdmCusOrgEntity::getCustomerOrgCode, parentCodeSet)
                            .select(MdmCusOrgEntity::getCustomerOrgCode, MdmCusOrgEntity::getCustomerOrgName)
                            .list();
                    if (CollectionUtil.listNotEmptyNotSizeZero(parentCodeNameList)) {
                        codeNameMap.putAll(parentCodeNameList.stream().collect(Collectors.toMap(MdmCusOrgEntity::getCustomerOrgCode, MdmCusOrgEntity::getCustomerOrgName)));
                    }
                }
                return list.stream().map(item -> {
                    MdmCustomerOrgRedisVo copy = CrmBeanUtil.copy(item, MdmCustomerOrgRedisVo.class);
                    if (codeNameMap.containsKey(copy.getCustomerOrgCode())) {
                        copy.setCustomerOrgName(codeNameMap.get(copy.getCustomerOrgCode()));
                    }
                    return copy;
                }).collect(Collectors.toList());
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmCustomerOrgParentChildrenRedisVo> findRedisCustomerOrgParentListByCodeList(List<String> customerOrgCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(customerOrgCodeList)) {
            List<MdmCusOrgEntity> list = this.lambdaQuery()
                    .in(MdmCusOrgEntity::getCustomerOrgCode, customerOrgCodeList)
                    .select(MdmCusOrgEntity::getCustomerOrgCode, MdmCusOrgEntity::getRuleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                Set<String> parentRuleCodes = TreeRuleCodeUtil.splitParentRuleCodes(list.stream().map(MdmCusOrgEntity::getRuleCode).collect(Collectors.toList()));
                List<MdmCusOrgEntity> parentList = this.lambdaQuery()
                        .in(MdmCusOrgEntity::getRuleCode, parentRuleCodes)
                        .list();
                Set<String> collect = parentList.stream().filter(item -> StringUtils.isNotEmpty(item.getParentCode())).map(MdmCusOrgEntity::getParentCode).collect(Collectors.toSet());
                Map<String, String> parentCustomerOrgCodeNameMap = new HashMap<>(16);
                if (!collect.isEmpty()) {
                    parentCustomerOrgCodeNameMap.putAll(this.lambdaQuery()
                            .in(MdmCusOrgEntity::getCustomerOrgCode, collect)
                            .select(MdmCusOrgEntity::getCustomerOrgCode, MdmCusOrgEntity::getCustomerOrgName)
                            .list()
                            .stream()
                            .collect(Collectors.toMap(MdmCusOrgEntity::getCustomerOrgCode, MdmCusOrgEntity::getCustomerOrgName)));
                }
                List<MdmCustomerOrgRedisVo> parentRedisVoList = parentList.stream().map(item -> {
                    MdmCustomerOrgRedisVo copy = CrmBeanUtil.copy(item, MdmCustomerOrgRedisVo.class);
                    if (StringUtils.isNotEmpty(copy.getParentCode())) {
                        if (parentCustomerOrgCodeNameMap.containsKey(copy.getParentCode())) {
                            copy.setParentName(parentCustomerOrgCodeNameMap.get(copy.getParentCode()));
                        }
                    }
                    return copy;
                }).collect(Collectors.toList());
                List<MdmCustomerOrgParentChildrenRedisVo> result = new ArrayList<>();
                for (MdmCusOrgEntity item :
                        parentList) {
                    MdmCustomerOrgParentChildrenRedisVo redisVo = new MdmCustomerOrgParentChildrenRedisVo();
                    redisVo.setCustomerOrgCode(item.getCustomerOrgCode());
                    redisVo.setList(parentRedisVoList.stream().filter(x -> item.getRuleCode().startsWith(x.getRuleCode())).collect(Collectors.toList()));
                    result.add(redisVo);
                }
                return result;
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmCustomerOrgParentChildrenRedisVo> findRedisCustomerOrgChildrenListByCodeList(List<String> customerOrgCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(customerOrgCodeList)) {
            List<MdmCusOrgEntity> customerOrgList = this.lambdaQuery()
                    .in(MdmCusOrgEntity::getCustomerOrgCode, customerOrgCodeList)
                    .select(MdmCusOrgEntity::getCustomerOrgCode, MdmCusOrgEntity::getRuleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(customerOrgList)) {
                List<String> likeRightRuleCodeList = customerOrgList.stream().map(item -> item.getRuleCode() + "%").collect(Collectors.toList());
                List<CusOrgVo> childrenList = cusOrgMapper.findChildrenByRuleCodeList(likeRightRuleCodeList, null);
                if (CollectionUtil.listNotEmptyNotSizeZero(childrenList)) {
                    List<MdmCustomerOrgRedisVo> redisList = CrmBeanUtil.copyList(childrenList, MdmCustomerOrgRedisVo.class);
                    List<MdmCustomerOrgParentChildrenRedisVo> resultList = new ArrayList<>();
                    for (MdmCusOrgEntity item :
                            customerOrgList) {
                        MdmCustomerOrgParentChildrenRedisVo redisVo = new MdmCustomerOrgParentChildrenRedisVo();
                        redisVo.setCustomerOrgCode(item.getCustomerOrgCode());
                        redisVo.setList(redisList.stream().filter(x -> x.getRuleCode().startsWith(item.getRuleCode())).collect(Collectors.toList()));
                        resultList.add(redisVo);
                    }
                    return resultList;
                }
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmCustomerOrgReloadRedisVo> findRedisReloadList() {
        final List<MdmCusOrgEntity> all = this.lambdaQuery().list();
        if (CollectionUtil.listNotEmptyNotSizeZero(all)) {
            final Map<String, String> customerOrgCodeNameMap = all.stream().collect(Collectors.toMap(MdmCusOrgEntity::getCustomerOrgCode, MdmCusOrgEntity::getCustomerOrgName));
            final List<MdmCustomerOrgRedisVo> redisList = all.stream().map(item -> {
                MdmCustomerOrgRedisVo copy = CrmBeanUtil.copy(item, MdmCustomerOrgRedisVo.class);
                if (StringUtils.isNotEmpty(copy.getParentCode())) {
                    copy.setParentName(customerOrgCodeNameMap.get(copy.getParentCode()));
                }
                return copy;
            }).collect(Collectors.toList());
            return redisList.stream().map(item -> {
                MdmCustomerOrgReloadRedisVo redisVo = new MdmCustomerOrgReloadRedisVo();
                redisVo.setCustomerOrgCode(item.getCustomerOrgCode());
                redisVo.setSelf(item);
                redisVo.setParentList(redisList.stream().filter(x -> item.getRuleCode().startsWith(x.getRuleCode())).collect(Collectors.toList()));
                redisVo.setChildrenList(redisList.stream().filter(x -> x.getRuleCode().startsWith(item.getRuleCode())).collect(Collectors.toList()));
                return redisVo;
            }).collect(Collectors.toList());
        }
        return new ArrayList<>();
    }
    
    /**
     * 根据上级编码生成新的降维编码和层级
     *
     * @param parentCode
     * @return
     */
    private CrmTreeVo generateRuleCodeByParentCode(String parentCode) {
        String ruleCode = "";
        int levelNum = 1;
        boolean top = true;
        String prefix = "";
        if (!StringUtils.isEmpty(parentCode)) {
            MdmCusOrgEntity parent = this.lambdaQuery().eq(MdmCusOrgEntity::getCustomerOrgCode, parentCode).one();
            if (parent != null) {
                top = false;
                prefix = parent.getRuleCode();
                levelNum = parent.getLevelNum() + 1;
            }
        }
        List<MdmCusOrgEntity> list = null;
        if (top) {
            list = this.lambdaQuery().isNull(MdmCusOrgEntity::getParentCode).or().eq(MdmCusOrgEntity::getParentCode, "").select(MdmCusOrgEntity::getRuleCode).list();
        } else {
            list = this.lambdaQuery().eq(MdmCusOrgEntity::getParentCode, parentCode).list();
        }
        if (CollectionUtil.listEmpty(list)) {
            ruleCode = prefix + TreeRuleCodeUtil.numToSingleCode(1);
        } else {
            Set<Integer> set = new LinkedHashSet<>();
            for (MdmCusOrgEntity item :
                    list) {
                if (!StringUtils.isEmpty(item.getRuleCode())) {
                    try {
                        Integer integer = TreeRuleCodeUtil.getCurLevelCodeValue(item.getRuleCode());
                        set.add(integer);
                    } catch (BusinessException e) {
                        throw new BusinessException("客户组织" + e.getMsg());
                    }
                }
            }
            int maxRuleCodeValueInLevel = TreeRuleCodeUtil.getMaxRuleCodeValueInLevel();
            for (int i = 1; i <= maxRuleCodeValueInLevel; i++) {
                if (i == maxRuleCodeValueInLevel) {
                    throw new BusinessException("降维编码越界，请联系管理员处理");
                }
                if (!set.contains(i)) {
                    ruleCode = prefix + TreeRuleCodeUtil.numToSingleCode(i);
                    break;
                }
            }
        }
        CrmTreeVo treeVo = new CrmTreeVo();
        treeVo.setRuleCode(ruleCode);
        treeVo.setLevelNum(levelNum);
        return treeVo;
    }

    /**
     * 更新当前及下级降维编码和层级
     *
     * @param customerOrgCode 当前编码
     * @param ruleCode        当前降维编码
     * @param levelNum        当前层级
     */
    private void updateCurAndChildren(String customerOrgCode, String ruleCode, int levelNum) {

        //更新当前
        this.lambdaUpdate()
                .eq(MdmCusOrgEntity::getCustomerOrgCode, customerOrgCode)
                .set(MdmCusOrgEntity::getRuleCode, ruleCode)
                .set(MdmCusOrgEntity::getLevelNum, levelNum)
                .update();

        //查询下一层
        List<MdmCusOrgEntity> list = this.lambdaQuery()
                .eq(MdmCusOrgEntity::getParentCode, customerOrgCode)
                .select(MdmCusOrgEntity::getCustomerOrgCode)
                .list();

        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            //遍历下级
            for (int i = 0; i < list.size(); i++) {
                //递归调用
                updateCurAndChildren(list.get(i).getCustomerOrgCode(), ruleCode + TreeRuleCodeUtil.numToSingleCode(i + 1), levelNum + 1);
            }
        }
    }


}
