package com.biz.crm.customer.service.impl;


import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.PageResult;
import com.biz.crm.customer.mapper.MdmCustomerContactMapper;
import com.biz.crm.customer.model.MdmCustomerContactEntity;
import com.biz.crm.customer.service.MdmCustomerContactService;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.nebular.mdm.customer.MdmCustomerContactReqVo;
import com.biz.crm.nebular.mdm.customer.MdmCustomerContactRespVo;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.DatabaseTypeUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.List;
import java.util.stream.Collectors;

/**
 * 接口实现
 *
 * @author zxw
 * @date 2020-09-02 14:10:26
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="MdmCustomerContactServiceExpandImpl")
public class MdmCustomerContactServiceImpl<M extends BaseMapper<T>,T> extends ServiceImpl<MdmCustomerContactMapper, MdmCustomerContactEntity> implements MdmCustomerContactService {

    private final MdmCustomerContactMapper mdmCustomerContactMapper;

    public MdmCustomerContactServiceImpl(MdmCustomerContactMapper mdmCustomerContactMapper) {
        this.mdmCustomerContactMapper = mdmCustomerContactMapper;
    }

    @Override
    public PageResult<MdmCustomerContactRespVo> findList(MdmCustomerContactReqVo mdmCustomerContactReqVo){
        Page<MdmCustomerContactRespVo> page = new Page<>(mdmCustomerContactReqVo.getPageNum(), mdmCustomerContactReqVo.getPageSize());
        List<MdmCustomerContactRespVo> list = mdmCustomerContactMapper.findList(page, mdmCustomerContactReqVo);
        return PageResult.<MdmCustomerContactRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public List<MdmCustomerContactRespVo> list(MdmCustomerContactReqVo mdmCustomerContactReqVo) {
        String customerCode = mdmCustomerContactReqVo.getCustomerCode();
        Assert.hasText(customerCode, "客户编码不能为空");
        List<MdmCustomerContactEntity> list = this.lambdaQuery()
                .eq(!StringUtils.isEmpty(mdmCustomerContactReqVo.getId()), MdmCustomerContactEntity::getId, mdmCustomerContactReqVo.getId())
                .eq(!StringUtils.isEmpty(mdmCustomerContactReqVo.getCustomerCode()), MdmCustomerContactEntity::getCustomerCode, mdmCustomerContactReqVo.getCustomerCode())
                .last(DatabaseTypeUtil.SEGMENT_ITEM)
                .list();
        return CrmBeanUtil.copyList(list, MdmCustomerContactRespVo.class);
    }

    @Override
    public MdmCustomerContactRespVo query(MdmCustomerContactReqVo mdmCustomerContactReqVo){
        MdmCustomerContactEntity one = this.lambdaQuery()
                .eq(!StringUtils.isEmpty(mdmCustomerContactReqVo.getCustomerCode()), MdmCustomerContactEntity::getCustomerCode, mdmCustomerContactReqVo.getCustomerCode())
                .last(DatabaseTypeUtil.SEGMENT)
                .one();
        return CrmBeanUtil.copy(one, MdmCustomerContactRespVo.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmCustomerContactReqVo mdmCustomerContactReqVo){
        this.save(CrmBeanUtil.copy(mdmCustomerContactReqVo,MdmCustomerContactEntity.class));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmCustomerContactReqVo mdmCustomerContactReqVo){
        this.updateById(CrmBeanUtil.copy(mdmCustomerContactReqVo,MdmCustomerContactEntity.class));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(MdmCustomerContactReqVo mdmCustomerContactReqVo){
        this.removeById(mdmCustomerContactReqVo.getId());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(MdmCustomerContactReqVo mdmCustomerContactReqVo){
        String customerCode = mdmCustomerContactReqVo.getCustomerCode();
        if (!StringUtils.isEmpty(customerCode)) {
            this.lambdaUpdate()
                    .eq(MdmCustomerContactEntity::getCustomerCode, customerCode)
                    .set(MdmCustomerContactEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                    .update();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(MdmCustomerContactReqVo mdmCustomerContactReqVo){
        String customerCode = mdmCustomerContactReqVo.getCustomerCode();
        if (!StringUtils.isEmpty(customerCode)) {
            this.lambdaUpdate()
                    .eq(MdmCustomerContactEntity::getCustomerCode, customerCode)
                    .set(MdmCustomerContactEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
                    .update();
        }
    }

    @Override
    public void setUpContact(List<MdmCustomerContactReqVo> mdmCustomerContactReqVos,String customerCode) {
        Assert.hasText(customerCode, "客户编码不能为空");
        this.lambdaUpdate()
                .eq(MdmCustomerContactEntity::getCustomerCode, customerCode)
                .remove();
        if (CollectionUtil.listEmpty(mdmCustomerContactReqVos)) {
            return;
        }
        List<MdmCustomerContactReqVo> collect = mdmCustomerContactReqVos.stream().filter(item -> StringUtils.isNotEmpty(item.getContactName())).peek(item->item.setCustomerCode(customerCode)).collect(Collectors.toList());
        if (CollectionUtil.listEmpty(collect)) {
            return;
        }
        long count = collect.stream().filter(x -> x.getContactMain() != null && YesNoEnum.yesNoEnum.ONE.getValue().equals(x.getContactMain())).count();
        Assert.isTrue(count <= 1, "客户信息：主联系人只能有一个");
        Assert.isTrue(count == 1, "客户信息：主联系人必须有一个");
        List<MdmCustomerContactEntity> mdmCustomerContactEntities = CrmBeanUtil.copyList(collect, MdmCustomerContactEntity.class);
        this.saveBatch(mdmCustomerContactEntities);
    }
}
