package com.biz.crm.dict.controller;

import com.biz.crm.aop.CrmGlobalLog;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.dict.service.MdmDictTypeService;
import com.biz.crm.nebular.mdm.dict.req.MdmDictTypeReqVo;
import com.biz.crm.nebular.mdm.dict.resp.DictGroupRedisVo;
import com.biz.crm.nebular.mdm.dict.resp.MdmDictTypeRespVo;
import com.biz.crm.nebular.mdm.sync.MdmDictEngineSyncVo;
import com.biz.crm.nebular.mdm.sync.MdmSyncStandardDictReqVo;
import com.biz.crm.util.Result;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 数据字典类型表
 *
 * @author Tao.Chen
 * @date 2020-11-20 15:00:54
 */
@Slf4j
@RestController
@RequestMapping("/mdmdicttype")
@Api(tags = "MDM-数据字典分类")
@CrmGlobalLog
public class MdmDictTypeController {

    @Autowired
    private MdmDictTypeService mdmDictTypeService;

    @ApiOperation(value = "查询列表（分页）")
    @PostMapping("/list")
    @CrmDictMethod
    public Result<PageResult<MdmDictTypeRespVo>> list(@RequestBody MdmDictTypeReqVo reqVo) {
        return Result.ok(mdmDictTypeService.findList(reqVo));
    }

    @ApiOperation(value = "详情", httpMethod = "GET", notes = "只查询数据字典类型")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "ID", required = false, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "dictTypeCode", value = "字典类型编码", required = false, dataType = "String", paramType = "query")
    })
    @GetMapping("/query")
    @CrmDictMethod
    public Result<MdmDictTypeRespVo> query(@RequestParam(value = "id", required = false) String id, @RequestParam(value = "dictTypeCode", required = false) String dictTypeCode) {
        return Result.ok(mdmDictTypeService.query(id, dictTypeCode));
    }

    @ApiOperation(value = "新增")
    @PostMapping("/save")
    public Result save(@RequestBody MdmDictTypeReqVo reqVo) {
        mdmDictTypeService.save(reqVo);
        return Result.ok();
    }

    @ApiOperation(value = "更新")
    @PostMapping("/update")
    public Result update(@RequestBody MdmDictTypeReqVo reqVo) {
        mdmDictTypeService.update(reqVo);
        return Result.ok("修改成功");
    }

    @ApiOperation(value = "删除", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/delete")
    public Result delete(@RequestBody List<String> ids) {
        mdmDictTypeService.deleteBatch(ids);
        return Result.ok("删除成功");
    }

    @ApiOperation(value = "清除全部缓存", httpMethod = "POST")
    @PostMapping("/deleteAllCache")
    public Result deleteAllCache() {
        mdmDictTypeService.deleteAllCache();
        return Result.ok("删除成功");
    }

    @ApiOperation(value = "根据id集合批量删除缓存", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/deleteCacheByIds")
    public Result deleteCacheByIds(@RequestBody List<String> ids) {
        mdmDictTypeService.deleteCacheByIds(ids);
        return Result.ok("删除成功");
    }

    @ApiOperation(value = "根据字典类型编码集合批量删除缓存", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "codeList", value = "类型编码集合", required = true, paramType = "body")
    })
    @PostMapping("/deleteCacheByCodeList")
    public Result deleteCacheByCodeList(@RequestBody List<String> codeList) {
        mdmDictTypeService.deleteCacheByCodeList(codeList);
        return Result.ok("删除成功");
    }

    @PostMapping("/getDictGroupRedisList")
    @ApiOperation("请勿调用，字典工具类查询构建缓存数据")
    public Result<List<DictGroupRedisVo>> getDictGroupRedisList(@RequestBody List<String> dictTypeCodeList) {
        return Result.ok(mdmDictTypeService.getDictGroupRedisList(dictTypeCodeList));
    }

    @ApiOperation("仅数据同步查询使用")
    @PostMapping("/obtainSyncList")
    public Result<MdmDictEngineSyncVo> obtainSyncList(@RequestBody MdmSyncStandardDictReqVo mdmSyncStandardDictReqVo) {
        return Result.ok(mdmDictTypeService.obtainSyncList(mdmSyncStandardDictReqVo));
    }
}
