package com.biz.crm.dict.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.PageResult;
import com.biz.crm.dict.mapper.MdmDictTypeMapper;
import com.biz.crm.dict.model.MdmDictAttrConfEntity;
import com.biz.crm.dict.model.MdmDictDataEntity;
import com.biz.crm.dict.model.MdmDictTypeEntity;
import com.biz.crm.dict.service.MdmDictAttrConfService;
import com.biz.crm.dict.service.MdmDictDataService;
import com.biz.crm.dict.service.MdmDictTypeService;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.nebular.mdm.dict.req.MdmDictTypeReqVo;
import com.biz.crm.nebular.mdm.dict.resp.*;
import com.biz.crm.nebular.mdm.sync.MdmDictEngineSyncVo;
import com.biz.crm.nebular.mdm.sync.MdmSyncStandardDictReqVo;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 数据字典类型表接口实现
 *
 * @author Tao.Chen
 * @date 2020-11-20 15:00:54
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmDictTypeServiceExpandImpl")
public class MdmDictTypeServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmDictTypeMapper, MdmDictTypeEntity> implements MdmDictTypeService {

    @Resource
    private MdmDictTypeMapper mdmDictTypeMapper;
    @Resource
    private MdmDictDataService mdmDictDataService;
    @Resource
    private MdmDictAttrConfService mdmDictAttrConfService;

    @Override
    public PageResult<MdmDictTypeRespVo> findList(MdmDictTypeReqVo reqVo) {
        Page<MdmDictTypeRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmDictTypeRespVo> list = mdmDictTypeMapper.findList(page, reqVo);
        return PageResult.<MdmDictTypeRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public MdmDictTypeRespVo query(String id, String dictTypeCode) {
        if (StringUtils.isEmpty(dictTypeCode) && StringUtils.isEmpty(id)) {
            return null;
        }
        MdmDictTypeEntity one = this.lambdaQuery()
                .eq(StringUtils.isNotEmpty(id), MdmDictTypeEntity::getId, id)
                .eq(StringUtils.isNotEmpty(dictTypeCode), MdmDictTypeEntity::getDictTypeCode, dictTypeCode)
                .one();
        if (one != null) {
            return CrmBeanUtil.copy(one, MdmDictTypeRespVo.class);
        }
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmDictTypeReqVo reqVo) {
        Assert.hasText(reqVo.getDictTypeCode(), "字典类型编码不能为空");
        Assert.isTrue(!reqVo.getDictTypeCode().contains(DictUtil.PATH_SPLIT), "字典类型编码不能包含字符“" + DictUtil.PATH_SPLIT + "”");
        Assert.isTrue(!reqVo.getDictTypeCode().contains(","), "字典类型编码不能包含字符“,”");
        Assert.hasText(reqVo.getDictTypeName(), "字典类型名称不能为空");
        Assert.hasText(reqVo.getDictTypeModule(), "模块不能为空");
        List<MdmDictTypeEntity> list = this.lambdaQuery()
                .eq(MdmDictTypeEntity::getDictTypeCode, reqVo.getDictTypeCode())
                .list();
        Assert.isTrue(CollectionUtil.listEmpty(list), "字典类型编码已经存在");
        MdmDictTypeEntity entity = CrmBeanUtil.copy(reqVo, MdmDictTypeEntity.class);
        entity.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        entity.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        this.save(entity);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmDictTypeReqVo reqVo) {
        Assert.hasText(reqVo.getId(), "id不能为空");
        Assert.hasText(reqVo.getDictTypeCode(), "字典类型编码不能为空");
        Assert.hasText(reqVo.getDictTypeName(), "字典类型名称不能为空");
        Assert.hasText(reqVo.getDictTypeModule(), "模块编码不能为空");
        MdmDictTypeEntity entity = this.getById(reqVo.getId());
        Assert.notNull(entity, "id无效");
        Assert.isTrue(entity.getDictTypeCode().equals(reqVo.getDictTypeCode()), "字典类型编码不能修改");
        CrmBeanUtil.copyProperties(reqVo, entity);
        this.updateById(entity);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        Assert.notEmpty(ids, "请选择一条数据");
        List<MdmDictTypeEntity> entities = this.listByIds(ids);
        entities.forEach(item -> {
            //删除字典值
            mdmDictDataService.deleteByDictTypeCode(item.getDictTypeCode());
            //删除字典扩展
            mdmDictAttrConfService.deleteByDictTypeCode(item.getDictTypeCode());
            //删除缓存
            DictUtil.deleteRedisCache(item.getDictTypeCode());
        });
        //删除类型
        this.removeByIds(ids);
    }

    @Override
    public void deleteAllCache() {
        List<MdmDictTypeEntity> list = this.lambdaQuery()
                .select(MdmDictTypeEntity::getDictTypeCode)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            this.deleteCacheByCodeList(list.stream().map(MdmDictTypeEntity::getDictTypeCode).collect(Collectors.toList()));
        }
    }

    @Override
    public void deleteCacheByIds(List<String> ids) {
        if (CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            List<MdmDictTypeEntity> list = this.lambdaQuery()
                    .in(MdmDictTypeEntity::getId, ids)
                    .select(MdmDictTypeEntity::getDictTypeCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                this.deleteCacheByCodeList(list.stream().map(MdmDictTypeEntity::getDictTypeCode).collect(Collectors.toList()));
            }
        }
    }

    @Override
    public void deleteCacheByCodeList(List<String> codeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(codeList)) {
            codeList.forEach(DictUtil::deleteRedisCache);
        }
    }

    @Override
    public List<DictGroupRedisVo> getDictGroupRedisList(List<String> dictTypeCodeList) {
        List<DictGroupRedisVo> list = new ArrayList<>();
        dictTypeCodeList.forEach(dictTypeCode -> {
            DictGroupRedisVo group = this.getDictGroupRedis(dictTypeCode);
            if (group != null) {
                list.add(group);
            }
        });
        return list;
    }

    @Override
    public MdmDictEngineSyncVo obtainSyncList(MdmSyncStandardDictReqVo mdmSyncStandardDictReqVo) {
        List<MdmDictTypeEntity> mdmDictTypeEntities = this.lambdaQuery()
                .in(!CollectionUtils.isEmpty(mdmSyncStandardDictReqVo.getDictTypeCodeList()), MdmDictTypeEntity::getDictTypeCode, mdmSyncStandardDictReqVo.getDictTypeCodeList())
                .list();
        List<MdmDictDataEntity> mdmDictDataEntities = mdmDictDataService.lambdaQuery()
                .in(!CollectionUtils.isEmpty(mdmSyncStandardDictReqVo.getDictTypeCodeList()), MdmDictDataEntity::getDictTypeCode, mdmSyncStandardDictReqVo.getDictTypeCodeList())
                .list();
        List<MdmDictAttrConfEntity> mdmDictAttrConfEntities = mdmDictAttrConfService.lambdaQuery()
                .in(!CollectionUtils.isEmpty(mdmSyncStandardDictReqVo.getDictTypeCodeList()), MdmDictAttrConfEntity::getDictTypeCode, mdmSyncStandardDictReqVo.getDictTypeCodeList())
                .list();
        MdmDictEngineSyncVo mdmDictEngineSyncVo = new MdmDictEngineSyncVo();
        mdmDictEngineSyncVo.setMdmDictTypeRespVoList(CrmBeanUtil.copyList(mdmDictTypeEntities,MdmDictTypeRespVo.class));
        mdmDictEngineSyncVo.setMdmDictDataRespVoList(CrmBeanUtil.copyList(mdmDictDataEntities,MdmDictDataRespVo.class));
        mdmDictEngineSyncVo.setMdmDictAttrConfRespVoList(CrmBeanUtil.copyList(mdmDictAttrConfEntities,MdmDictAttrConfRespVo.class));
        return mdmDictEngineSyncVo;
    }

    private DictGroupRedisVo getDictGroupRedis(String dictTypeCode) {
        DictGroupRedisVo respVo = null;
        respVo = new DictGroupRedisVo();

        List<MdmDictDataRespVo> respVoList = mdmDictDataService.queryByDictTypeCode(dictTypeCode);
        if (CollectionUtil.listNotEmptyNotSizeZero(respVoList)) {
            List<MdmDictDataRespVo> dataTree = generateTreeByParentDictCode(respVoList);

            List<DictDataRedisVo> dataList = new ArrayList<>();
            for (MdmDictDataRespVo item :
                    dataTree) {
                DictDataRedisVo copy = CrmBeanUtil.copy(item, DictDataRedisVo.class);
                copy.setPath(dictTypeCode + DictUtil.PATH_SPLIT + item.getDictCode());
                copy.setParentDictCode(null);
                setChildren(item.getChildren(), copy, dataList);
                dataList.add(copy);
            }
            respVo.setDictTypeCode(dictTypeCode);
            respVo.setDataList(dataList);
            List<MdmDictAttrConfRespVo> list = mdmDictAttrConfService.findList(dictTypeCode).stream().sorted((x1, x2) -> new BigDecimal(x2.getShowOrder()).compareTo(new BigDecimal(x1.getShowOrder()))).collect(Collectors.toList());
            if (list.size() > 0) {
                respVo.setConfList(CrmBeanUtil.copyList(list, DictAttrConfRedisVo.class));
            }
        }
        return respVo;
    }

    private void setChildren(List<MdmDictDataRespVo> children, DictDataRedisVo node, List<DictDataRedisVo> list) {
        if (CollectionUtil.listNotEmptyNotSizeZero(children)) {
            for (MdmDictDataRespVo child :
                    children) {
                DictDataRedisVo copy = CrmBeanUtil.copy(child, DictDataRedisVo.class);
                copy.setParentDictCode(node.getDictCode());
                copy.setPath(node.getPath() + DictUtil.PATH_SPLIT + copy.getDictCode());
                if (CollectionUtil.listNotEmptyNotSizeZero(child.getChildren())) {
                    setChildren(child.getChildren(), copy, list);
                }
                list.add(copy);
            }
        }
    }

    private List<MdmDictDataRespVo> generateTreeByParentDictCode(List<MdmDictDataRespVo> totalList) {

        totalList = totalList.stream().sorted(Comparator.comparing(MdmDictDataRespVo::getDictSort)).collect(Collectors.toList());

        //构建树list
        List<MdmDictDataRespVo> treeList = new ArrayList<>();
        //当前操作层级数据
        List<MdmDictDataRespVo> curLevelList = new ArrayList<>();
        //未操作数据
        List<MdmDictDataRespVo> restList = new ArrayList<>();

        //key:id
        Map<String, MdmDictDataRespVo> totalMap = totalList.stream().collect(Collectors.toMap(MdmDictDataRespVo::getDictCode, v -> v));

        //查找第一层
        for (MdmDictDataRespVo item : totalList) {
            if (StringUtils.isEmpty(item.getParentDictCode()) || !totalMap.containsKey(item.getParentDictCode())) {
                treeList.add(item);
                curLevelList.add(item);
            } else {
                restList.add(item);
            }
        }

        //构建数据，从第二层开始
        while (curLevelList.size() > 0 && restList.size() > 0) {
            List<MdmDictDataRespVo> restTempList = new ArrayList<>();
            List<MdmDictDataRespVo> curLevelTempList = new ArrayList<>();
            Map<String, String> curLevelMap = curLevelList.stream().collect(Collectors.toMap(MdmDictDataRespVo::getDictCode, MdmDictDataRespVo::getDictCode));
            Map<String, List<MdmDictDataRespVo>> curLevelChildrenMap = new LinkedHashMap<>(16);

            for (MdmDictDataRespVo item : restList) {
                if (curLevelMap.containsKey(item.getParentDictCode())) {
                    curLevelTempList.add(item);

                    List<MdmDictDataRespVo> childrenList = new ArrayList<>();
                    if (curLevelChildrenMap.containsKey(item.getParentDictCode())) {
                        childrenList.addAll(curLevelChildrenMap.get(item.getParentDictCode()));
                    }
                    childrenList.add(item);
                    curLevelChildrenMap.put(item.getParentDictCode(), childrenList);
                } else {
                    restTempList.add(item);
                }
            }

            for (MdmDictDataRespVo item : curLevelList) {
                if (curLevelChildrenMap.containsKey(item.getDictCode())) {
                    item.setChildren(curLevelChildrenMap.get(item.getDictCode()));
                }
            }

            curLevelList.clear();
            curLevelList.addAll(curLevelTempList);
            restList.clear();
            restList.addAll(restTempList);
        }

        return treeList;
    }
}
