package com.biz.crm.employeedirectory.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.employeedirectory.mapper.MdmEmployeeDirectoryMapper;
import com.biz.crm.employeedirectory.model.MdmEmployeeDirectoryEntity;
import com.biz.crm.employeedirectory.service.MdmEmployeeDirectoryService;
import com.biz.crm.nebular.mdm.employeedirectory.MdmEmployeeDirectoryListRespVo;
import com.biz.crm.nebular.mdm.employeedirectory.MdmEmployeeDirectoryReqVo;
import com.biz.crm.nebular.mdm.employeedirectory.MdmEmployeeDirectoryRespVo;
import com.biz.crm.nebular.mdm.permission.MdmCurrentPermissionRespVo;
import com.biz.crm.nebular.mdm.permission.MdmCurrentUserPermissionRespVo;
import com.biz.crm.service.RedisService;
import com.biz.crm.util.*;
import com.biz.crm.utils.PinYinUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;
import java.util.zip.GZIPOutputStream;
/**
 * 员工通讯录信息实现
 * @author yangshan
 * @date 2021/3/5 14:53
 */
@Service
@Slf4j
@Transactional
public class MdmEmployeeDirectoryServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmEmployeeDirectoryMapper, MdmEmployeeDirectoryEntity> implements MdmEmployeeDirectoryService {

    @Autowired
    private MdmEmployeeDirectoryMapper mdmEmployeeDirectoryMapper;
    @Autowired
    private RedisService redisService;

    /**
     * 列表
     * 根据数据权限拿出不同权限的用户权限list，拼接成字符串当做redis的key值，再将key封装成List存到MDM_EMPLOYEE_DIRECTORY中
     * 取出时，如果MDM_EMPLOYEE_DIRECTORY不为空，遍历匹配当前用户的权限key，如果匹配则返回，匹配不上查询数据库。
     * 如果MDM_EMPLOYEE_DIRECTORY为空，所有数据从数据库查询
     * @return
     */
    @Override
    public List<MdmEmployeeDirectoryListRespVo> findList() {
        MdmCurrentPermissionRespVo permissionList = PermissionUtil.currentUserPermissionList();
        //admin key为""，取出用户权限
        List<MdmCurrentUserPermissionRespVo> userPermissionList = permissionList.getUserPermissionList();
        HashSet<String> pocCodes = new HashSet<>();
        userPermissionList.forEach(u -> {
            List<String> permissionValueList = u.getPermissionValueList();
            pocCodes.addAll(new HashSet<>(permissionValueList));
        });
        String userRedisKeyLong = pocCodes.stream().map(String::toString).collect(Collectors.joining("-"));
        String userRedisKey = this.compress(userRedisKeyLong);

        //先读取缓存，没有数据再读取数据库
        //遍历MDM_EMPLOYEE_DIRECTORY，取出key
        HashSet<String> employeeKey = (HashSet<String>) redisService.get(RedisParam.MDM_EMPLOYEE_DIRECTORY);
        if (ObjectUtils.isEmpty(employeeKey)){
            HashSet<String> objects = new HashSet<>();
            objects.add(userRedisKey);
            redisService.set(RedisParam.MDM_EMPLOYEE_DIRECTORY, objects);
        }else {
            //数据存在
            if (employeeKey.contains(userRedisKey)){
                List<MdmEmployeeDirectoryListRespVo> result = (List<MdmEmployeeDirectoryListRespVo>) redisService.get(userRedisKey);
                return result;
            }
            //新的权限，数据不存在
            employeeKey.add(userRedisKey);
            redisService.set(RedisParam.MDM_EMPLOYEE_DIRECTORY, employeeKey);
        }

        //查询所有数据，进行分组，再转成List
        List<MdmEmployeeDirectoryRespVo> respVos = this.mdmEmployeeDirectoryMapper.findList();
        Map<String, List<MdmEmployeeDirectoryRespVo>> collect = respVos.stream().collect(Collectors.groupingBy(MdmEmployeeDirectoryRespVo::getFirstNameInitials));
        List<MdmEmployeeDirectoryListRespVo> result = collect.entrySet().stream().sorted(Map.Entry.comparingByKey())
                .map(e -> new MdmEmployeeDirectoryListRespVo(e.getKey(), e.getValue())).collect(Collectors.toList());

        redisService.set(userRedisKey, result);
        return result;
    }

    /**
     * redis清除
     */
    public void cleanRedis(){
        HashSet<String> o = (HashSet<String>) redisService.get(RedisParam.MDM_EMPLOYEE_DIRECTORY);
        o.forEach(directoryKey -> {
            redisService.del(directoryKey);
        });
        redisService.del(RedisParam.MDM_EMPLOYEE_DIRECTORY);
    }

    /**
     * 新增
     *
     * @param reqVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmEmployeeDirectoryReqVo reqVo) {
        this.doCheck(reqVo);
        //姓氏首字母
        String nameInitials = PinYinUtil.getPinYin(reqVo.getEmployeeName());
        char chars = nameInitials.charAt(0);
        String firstNameInitials = Character.toString(chars);
        reqVo.setFirstNameInitials(firstNameInitials);
        MdmEmployeeDirectoryEntity entity = CrmBeanUtil.copy(reqVo, MdmEmployeeDirectoryEntity.class);
        this.save(entity);
        this.cleanRedis();
    }

    /**
     * 删除(人员编码)
     *
     * @param employeeCodes
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteEmployeeCode(List<String> employeeCodes) {
        Assert.notNull(employeeCodes, "缺失参数");
        List<MdmEmployeeDirectoryRespVo> list = mdmEmployeeDirectoryMapper.selectByEmployeeCodes(employeeCodes);

        if (CollectionUtils.isNotEmpty(list)){
            List<String> idList = new ArrayList<>();
            list.forEach(o -> {
                idList.add(o.getId());
            });
            this.removeByIds(idList);
        }
        this.cleanRedis();
    }

    /**
     * 删除(职位编码)
     *
     * @param posCodes
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByPosCode(List<String> posCodes) {
        Assert.notNull(posCodes, "缺失参数");
        List<MdmEmployeeDirectoryRespVo> list = mdmEmployeeDirectoryMapper.selectByPosCodes(posCodes);

        if (CollectionUtils.isNotEmpty(list)){
            List<String> idList = new ArrayList<>();
            list.forEach(o -> {
                idList.add(o.getId());
            });
            this.removeByIds(idList);
        }
        this.cleanRedis();
    }

    /**
     * 新增校验
     *
     * @param reqVo
     */
    public void doCheck(MdmEmployeeDirectoryReqVo reqVo){
        AssertUtils.isNotEmpty(reqVo.getUserCode(), "userCode不能为空");
        AssertUtils.isNotEmpty(reqVo.getEmployeeCode(), "员工编码不能为空");
        AssertUtils.isNotEmpty(reqVo.getEmployeeName(), "员工名字不能为空");
        AssertUtils.isNotEmpty(reqVo.getOrgCode(), "员工组织编码不能为空");
        AssertUtils.isNotEmpty(reqVo.getOrgName(), "员工组织名称不能为空");
        AssertUtils.isNotEmpty(reqVo.getPosCode(), "员工职位编码不能为空");
        AssertUtils.isNotEmpty(reqVo.getPosName(), "员工职位名称不能为空");
        AssertUtils.isNotEmpty(reqVo.getPhone(), "员工电话不能为空");
    }

    /**
     * 压缩字符串
     * @param str
     * @return
     */
    public String  compress(String str) {
        if (str == null || str.length() == 0) {
            return str;
        }
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        GZIPOutputStream gzip = null;
        try {
            gzip = new GZIPOutputStream(out);
            gzip.write(str.getBytes());
            gzip.close();
            return out.toString("ISO-8859-1");
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
    }
}
