package com.biz.crm.function.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.mdm.AdminEnum;
import com.biz.crm.function.mapper.MdmFunctionMapper;
import com.biz.crm.function.model.MdmFunctionEntity;
import com.biz.crm.function.service.MdmFunctionRoleService;
import com.biz.crm.function.service.MdmFunctionService;
import com.biz.crm.icon.service.IMdmIconService;
import com.biz.crm.nebular.mdm.constant.FunctionConfigConstant;
import com.biz.crm.nebular.mdm.function.*;
import com.biz.crm.nebular.mdm.icon.MdmIconReqVo;
import com.biz.crm.nebular.mdm.icon.MdmIconRespVo;
import com.biz.crm.nebular.mdm.pageconfig.MdmFunctionSubButtonRespVo;
import com.biz.crm.nebular.mdm.pageconfig.MdmFunctionSubPermissionVo;
import com.biz.crm.nebular.mdm.pageconfig.MdmFunctionSubReqVo;
import com.biz.crm.nebular.mdm.pageconfig.MdmFunctionSubRespVo;
import com.biz.crm.nebular.mdm.permission.MdmDataPermissionReqVo;
import com.biz.crm.nebular.mdm.role.req.MdmRoleFunctionReqVo;
import com.biz.crm.nebular.mdm.role.req.MdmRoleFunctionTreeVo;
import com.biz.crm.nebular.mdm.sync.MdmFunctionEngineSyncVo;
import com.biz.crm.nebular.mdm.sync.MdmSyncStandardFunctionReqVo;
import com.biz.crm.permission.service.IMdmDataPermissionService;
import com.biz.crm.role.service.MdmRoleService;
import com.biz.crm.tableconfig.manager.MdmColumnConfigConverter;
import com.biz.crm.tableconfig.model.MdmColumnConfigEntity;
import com.biz.crm.tableconfig.model.MdmFunctionSubButtonEntity;
import com.biz.crm.tableconfig.model.MdmFunctionSubEntity;
import com.biz.crm.tableconfig.service.IMdmColumnConfigService;
import com.biz.crm.tableconfig.service.IMdmFunctionSubButtonService;
import com.biz.crm.tableconfig.service.IMdmFunctionSubService;
import com.biz.crm.util.*;
import com.biz.crm.util.cache.TableConfigUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 菜单表接口实现
 *
 * @author zxw
 * @date 2020-11-18 10:26:31
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmFunctionServiceExpandImpl")
public class MdmFunctionServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmFunctionMapper, MdmFunctionEntity> implements MdmFunctionService {

    @Autowired
    private MdmFunctionMapper mdmFunctionMapper;
    @Autowired
    private MdmFunctionRoleService mdmFunctionRoleService;
    @Autowired
    private IMdmIconService mdmIconService;
    @Value("${whiteFunctionCodeList:}")
    private String whiteFunctionCodeList;
    @Autowired
    private IMdmFunctionSubService mdmFunctionSubService;
    @Autowired
    private IMdmDataPermissionService mdmDataPermissionService;
    @Autowired
    private MdmRoleService mdmRoleService;
    @Autowired
    private IMdmFunctionSubButtonService mdmFunctionSubButtonService;
    @Autowired
    private IMdmColumnConfigService mdmColumnConfigService;

    @Override
    public MdmFunctionRespVo query(MdmFunctionReqVo mdmFunctionReqVo) {
        MdmFunctionEntity one = this.lambdaQuery()
                .eq(!StringUtils.isEmpty(mdmFunctionReqVo.getId()), MdmFunctionEntity::getId, mdmFunctionReqVo.getId())
                .eq(!StringUtils.isEmpty(mdmFunctionReqVo.getFunctionCode()), MdmFunctionEntity::getFunctionCode, mdmFunctionReqVo.getFunctionCode())
                .in(!CollectionUtils.isEmpty(mdmFunctionReqVo.getFunctionCodeList()), MdmFunctionEntity::getFunctionCode, mdmFunctionReqVo.getFunctionCodeList())
                .eq(!StringUtils.isEmpty(mdmFunctionReqVo.getParentCode()), MdmFunctionEntity::getFunctionCode, mdmFunctionReqVo.getParentCode())
                .eq(!StringUtils.isEmpty(mdmFunctionReqVo.getFunctionModule()), MdmFunctionEntity::getFunctionModule, mdmFunctionReqVo.getFunctionModule())
                .one();
        return CrmBeanUtil.copy(one, MdmFunctionRespVo.class);
    }

    @Override
    public List<String> conditionCodeList(MdmFunctionReqVo mdmFunctionReqVo) {
        return this.lambdaQuery()
                .eq(MdmFunctionEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .like(!StringUtils.isEmpty(mdmFunctionReqVo.getFunctionCode()), MdmFunctionEntity::getFunctionCode, mdmFunctionReqVo.getFunctionCode())
                .like(!StringUtils.isEmpty(mdmFunctionReqVo.getFunctionName()), MdmFunctionEntity::getFunctionName, mdmFunctionReqVo.getFunctionName())
                .eq(!StringUtils.isEmpty(mdmFunctionReqVo.getParentCode()), MdmFunctionEntity::getParentCode, mdmFunctionReqVo.getParentCode())
                .eq(!StringUtils.isEmpty(mdmFunctionReqVo.getFunctionModule()), MdmFunctionEntity::getFunctionModule, mdmFunctionReqVo.getFunctionModule())
                .eq(!StringUtils.isEmpty(mdmFunctionReqVo.getFunctionConfig()), MdmFunctionEntity::getFunctionConfig, mdmFunctionReqVo.getFunctionConfig())
                .eq(!StringUtils.isEmpty(mdmFunctionReqVo.getPageConfig()), MdmFunctionEntity::getPageConfig, mdmFunctionReqVo.getPageConfig())
                .eq(!StringUtils.isEmpty(mdmFunctionReqVo.getMenuConfig()), MdmFunctionEntity::getMenuConfig, mdmFunctionReqVo.getMenuConfig())
                .select(MdmFunctionEntity::getFunctionCode)
                .list().stream()
                .map(MdmFunctionEntity::getFunctionCode)
                .filter(functionCode -> !StringUtils.isEmpty(functionCode))
                .distinct()
                .collect(Collectors.toList());
    }

    @Override
    public List<MdmFunctionRespVo> listCondition(MdmFunctionReqVo mdmFunctionReqVo) {
        List<MdmFunctionEntity> list = this.lambdaQuery()
                .eq(MdmFunctionEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .in(!CollectionUtils.isEmpty(mdmFunctionReqVo.getFunctionCodeList()),MdmFunctionEntity::getFunctionCode,mdmFunctionReqVo.getFunctionCodeList())
                .like(!StringUtils.isEmpty(mdmFunctionReqVo.getFunctionCode()), MdmFunctionEntity::getFunctionCode, mdmFunctionReqVo.getFunctionCode())
                .like(!StringUtils.isEmpty(mdmFunctionReqVo.getFunctionName()), MdmFunctionEntity::getFunctionName, mdmFunctionReqVo.getFunctionName())
                .eq(!StringUtils.isEmpty(mdmFunctionReqVo.getParentCode()), MdmFunctionEntity::getParentCode, mdmFunctionReqVo.getParentCode())
                .eq(!StringUtils.isEmpty(mdmFunctionReqVo.getFunctionModule()), MdmFunctionEntity::getFunctionModule, mdmFunctionReqVo.getFunctionModule())
                .eq(!StringUtils.isEmpty(mdmFunctionReqVo.getFunctionConfig()), MdmFunctionEntity::getFunctionConfig, mdmFunctionReqVo.getFunctionConfig())
                .eq(!StringUtils.isEmpty(mdmFunctionReqVo.getPageConfig()), MdmFunctionEntity::getPageConfig, mdmFunctionReqVo.getPageConfig())
                .eq(!StringUtils.isEmpty(mdmFunctionReqVo.getMenuConfig()), MdmFunctionEntity::getMenuConfig, mdmFunctionReqVo.getMenuConfig())
                .and(x -> x.eq(MdmFunctionEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode()).or().isNull(MdmFunctionEntity::getDelFlag))
                .orderByAsc(MdmFunctionEntity::getFunctionOrder)
                .list();
        if (CollectionUtils.isEmpty(list)) {
            return Collections.emptyList();
        }
        List<MdmFunctionRespVo> mdmFunctionRespVos = CrmBeanUtil.copyList(list, MdmFunctionRespVo.class);
        setUpFunctionIcon(mdmFunctionRespVos);
        return mdmFunctionRespVos;
    }

    /**
     * 查询当前权限拥有菜单
     *
     * @param mdmFunctionReqVo vo
     * @return list
     */
    protected List<MdmFunctionRespVo> findCurrentUserFunctionList(MdmFunctionReqVo mdmFunctionReqVo) {
        List<MdmFunctionRespVo> mdmFunctionRespVos = listCondition(mdmFunctionReqVo);
        if (CollectionUtils.isEmpty(mdmFunctionRespVos)) {
            return Collections.emptyList();
        }
        MdmCheckAdminVo mdmCheckAdminVo = findCurrentUserFunctionCodeList();
        if (!mdmCheckAdminVo.isAuthorized()) {
            List<String> functionCodeList = mdmCheckAdminVo.getFunctionCodeList();
            mdmFunctionRespVos = mdmFunctionRespVos.stream()
                    .filter(x -> functionCodeList.contains(x.getFunctionCode()))
                    .collect(Collectors.toList());
        }
        return mdmFunctionRespVos;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmFunctionReqVo reqVo) {
        if (StringUtils.isEmpty(reqVo.getFunctionCode())) {
            reqVo.setFunctionCode(CodeUtil.generateCode());
        }
        valid(reqVo);
        this.save(CrmBeanUtil.copy(reqVo, MdmFunctionEntity.class));
    }

    private void valid(MdmFunctionReqVo reqVo) {
        if (StringUtils.isEmpty(reqVo.getFunctionOrder())) {
            reqVo.setFunctionOrder("1");
        }
        Assert.hasText(reqVo.getFunctionCode(), "菜单编码不能为空");
        Assert.hasText(reqVo.getFunctionName(), "菜单名称不能为空");
        Assert.hasText(reqVo.getFunctionModule(), "菜单模块不能为空");
        Assert.hasText(reqVo.getFunctionConfig(), "菜单类型不能为空");
        String pageConfig = reqVo.getPageConfig();
        if (!StringUtils.isEmpty(pageConfig)) {
            reqVo.setPageConfig(pageConfig.toUpperCase());
        }
        String menuConfig = reqVo.getMenuConfig();
        if (!StringUtils.isEmpty(menuConfig)) {
            reqVo.setMenuConfig(menuConfig.toUpperCase());
        }
        if (!YesNoEnum.yesNoEnum.Y.getValue().equalsIgnoreCase(reqVo.getMenuConfig())) {
            reqVo.setRelationBillCode(null);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmFunctionReqVo reqVo) {
        valid(reqVo);
        Assert.hasText(reqVo.getId(), "主键不能为空");
        MdmFunctionEntity entity = this.getById(reqVo.getId());
        Assert.notNull(entity, "菜单不存在");
        this.updateById(CrmBeanUtil.copy(reqVo, MdmFunctionEntity.class));
        String functionCode = entity.getFunctionCode();
        if (!StringUtils.isEmpty(functionCode)) {
            //清除页面配置缓存
            TableConfigUtil.removeFunction(functionCode);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(String functionCode) {
        Assert.hasText(functionCode, "菜单编码不能为空");
        List<String> allChildFunctionCodeList = findAllChildFunctionCodeList(Collections.singletonList(functionCode));
        if (CollectionUtils.isEmpty(allChildFunctionCodeList)) {
            return;
        }
        this.lambdaUpdate()
                .in(MdmFunctionEntity::getFunctionCode, allChildFunctionCodeList)
                .set(MdmFunctionEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
                .set(MdmFunctionEntity::getDelFlag,CrmEnableStatusEnum.DISABLE.getCode())
                .update();
        //清除页面配置缓存
        TableConfigUtil.removeAll();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(String functionCode) {
        Assert.hasText(functionCode, "菜单编码不能为空");
        List<String> allChildFunctionCodeList = findAllParentFunctionCodeList(Collections.singletonList(functionCode),null);
        if(!CollectionUtils.isEmpty(allChildFunctionCodeList)){
            this.lambdaUpdate()
                    .in(MdmFunctionEntity::getFunctionCode, allChildFunctionCodeList)
                    .set(MdmFunctionEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                    .update();
        }
        //清除页面配置缓存
        TableConfigUtil.removeAll();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(String functionCode) {
        Assert.hasText(functionCode, "菜单编码不能为空");
        List<String> allChildFunctionCodeList = findAllChildFunctionCodeList(Collections.singletonList(functionCode));
        if (CollectionUtils.isEmpty(allChildFunctionCodeList)) {
            return;
        }
        this.lambdaUpdate()
                .in(MdmFunctionEntity::getFunctionCode, allChildFunctionCodeList)
                .set(MdmFunctionEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
                .update();
        //清除页面配置缓存
        TableConfigUtil.removeAll();
    }

    @Override
    public List<MdmFunctionRespVo> systemList(MdmFunctionReqVo mdmFunctionReqVo) {
        mdmFunctionReqVo.setFunctionConfig(FunctionConfigConstant.SYSTEM);
        return findPermissionFunctionList(mdmFunctionReqVo);
    }

    @Override
    public List<MdmFunctionRespVo> findPermissionFunctionList(MdmFunctionReqVo mdmFunctionReqVo) {
        return findCurrentUserFunctionList(mdmFunctionReqVo);
    }

    @Override
    public List<MdmFunctionRespVo> systemTree(MdmFunctionReqVo mdmFunctionReqVo) {
        mdmFunctionReqVo.setFunctionConfig(FunctionConfigConstant.SYSTEM);
        return functionTree(mdmFunctionReqVo);
    }

    @Override
    public List<String> findUserFunctionCodeList() {
        UserRedis user = UserUtils.getUser();
        Assert.notNull(user, "用户不存在，请重新登录");
        String username = user.getUsername();
        if (!AdminEnum.ADMIN.name().equalsIgnoreCase(username)) {
            List<String> roleCodeList = mdmRoleService.getRoleByUserName(username,user.getPoscode());
            Assert.notEmpty(roleCodeList, "用户:" + username + "未分配角色");
            List<String> functionCodeList = mdmFunctionRoleService.findFunctionCodeByRole(roleCodeList);
            if (CollectionUtils.isEmpty(functionCodeList)) {
                functionCodeList = new ArrayList<>();
            }
            functionCodeList.addAll(functionWhiteList());
            return functionCodeList.stream().distinct().collect(Collectors.toList());
        }
        return this.lambdaQuery()
                .select(MdmFunctionEntity::getFunctionCode)
                .list()
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getFunctionCode()))
                .map(MdmFunctionEntity::getFunctionCode)
                .distinct()
                .collect(Collectors.toList());
    }

    @Override
    public MdmCheckAdminVo findCurrentUserFunctionCodeList() {
        MdmCheckAdminVo mdmCheckAdminVo = new MdmCheckAdminVo();
        UserRedis user = UserUtils.getUser();
        Assert.notNull(user, "用户不存在，请重新登录");
        String username = user.getUsername();
        if (AdminEnum.ADMIN.name().equalsIgnoreCase(username)) {
            //拥有超级管理员帐号
            mdmCheckAdminVo.setAuthorized(true);
            return mdmCheckAdminVo;
        }

        List<String> roleCodeList = mdmRoleService.getRoleByUserName(username,user.getPoscode());
        Assert.notEmpty(roleCodeList, "当前用户" + username + "未分配角色");

        if (roleCodeList.stream().anyMatch(x -> AdminEnum.ADMIN.name().equalsIgnoreCase(x))) {
            //拥有超级管理员角色
            mdmCheckAdminVo.setAuthorized(true);
            return mdmCheckAdminVo;
        }
        List<String> functionCodeList = mdmFunctionRoleService.findFunctionCodeByRole(roleCodeList);
        //菜单白名单
        functionCodeList.addAll(functionWhiteList());
        functionCodeList = functionCodeList.stream().distinct().collect(Collectors.toList());
        mdmCheckAdminVo.setFunctionCodeList(functionCodeList);
        return mdmCheckAdminVo;
    }

    /**
     * 设置菜单图标
     *
     * @param list 菜单集合
     */
    protected void setUpFunctionIcon(List<MdmFunctionRespVo> list) {
        List<String> iconCodeList = list.stream()
                .map(MdmFunctionRespVo::getIconCode)
                .filter(iconCode -> !StringUtils.isEmpty(iconCode))
                .distinct()
                .collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(iconCodeList)) {
            MdmIconReqVo mdmIconReqVo = new MdmIconReqVo();
            mdmIconReqVo.setIconCodeList(iconCodeList);
            List<MdmIconRespVo> mdmIconRespVos = mdmIconService.listCondition(mdmIconReqVo);
            if (!CollectionUtils.isEmpty(mdmIconRespVos)) {
                Map<String, String> iconMap = mdmIconRespVos.stream()
                        .collect(Collectors.toMap(MdmIconRespVo::getIconCode, MdmIconRespVo::getIconStyle));
                for (MdmFunctionRespVo mdmFunctionRespVo : list) {
                    if (StringUtils.isEmpty(mdmFunctionRespVo.getFunctionIcon())) {
                        mdmFunctionRespVo.setFunctionIcon(iconMap.get(mdmFunctionRespVo.getIconCode()));
                    }
                }
            }
        }
    }


    private void setUpChildren(MdmFunctionRespVo mdmFunctionRespVo, List<MdmFunctionRespVo> list, List<MdmFunctionRespVo> result, List<MdmFunctionRespVo> subList) {
        if (StringUtils.isEmpty(mdmFunctionRespVo.getParentCode()) || "".equals(mdmFunctionRespVo.getParentCode().trim())) {
            if (!result.contains(mdmFunctionRespVo)) {
                result.add(mdmFunctionRespVo);
            }
            if (!subList.contains(mdmFunctionRespVo)) {
                subList.add(mdmFunctionRespVo);
            }
            return;
        }
        if (subList.contains(mdmFunctionRespVo)) {
            return;
        }
        Optional<MdmFunctionRespVo> any = list.stream().filter(x -> mdmFunctionRespVo.getParentCode().equals(x.getFunctionCode())).findAny();
        if (!any.isPresent()) {
            return;
        }
        MdmFunctionRespVo parentTreeRespVo = any.get();
        List<MdmFunctionRespVo> children = parentTreeRespVo.getChildren();
        if (CollectionUtils.isEmpty(children)) {
            children = new ArrayList<>();
        }
        children.add(mdmFunctionRespVo);
        children.sort(Comparator.comparing(x -> Optional.ofNullable(x.getFunctionOrder()).map(Integer::parseInt).orElse(100)));
        parentTreeRespVo.setChildren(children);
        subList.add(mdmFunctionRespVo);
        setUpChildren(parentTreeRespVo, list, result, subList);
    }


    @Override
    public List<MdmFunctionRespVo> listMdmFunctionByMenuConfig(MdmFunctionReqVo mdmFunctionReqVo) {
        List<MdmFunctionEntity> entities = mdmFunctionMapper.selectList(Wrappers.lambdaQuery(MdmFunctionEntity.class).eq(MdmFunctionEntity::getMenuConfig, YesNoEnum.yesNoEnum.Y.getValue()).like(!StringUtils.isEmpty(mdmFunctionReqVo.getFunctionName()),MdmFunctionEntity::getFunctionName,mdmFunctionReqVo.getFunctionName()));
        List<MdmFunctionRespVo> mdmFunctionRespVos = CrmBeanUtil.copyList(entities, MdmFunctionRespVo.class);
        return mdmFunctionRespVos;
    }

    @Override
    public PageResult<MdmFunctionRespVo> functionPage(MdmFunctionReqVo mdmFunctionReqVo) {
        Page<MdmFunctionRespVo> page = PageUtil.buildPage(mdmFunctionReqVo.getPageNum(), mdmFunctionReqVo.getPageSize());
        List<MdmFunctionRespVo> list = mdmFunctionMapper.findList(page, mdmFunctionReqVo);
        return PageResult.<MdmFunctionRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public List<MdmRoleFunctionTreeVo> roleFunctionTree(MdmRoleFunctionReqVo mdmRoleFunctionReqVo) {
        String roleCode = mdmRoleFunctionReqVo.getRoleCode();
        Assert.hasText(roleCode, "角色编码不能为空");

        MdmFunctionReqVo mdmFunctionReqVo = new MdmFunctionReqVo();
        mdmFunctionReqVo.setFunctionConfig(mdmFunctionReqVo.getFunctionConfig());
        List<MdmRoleFunctionTreeVo> functionList = mdmFunctionMapper.roleFunctionTree(mdmFunctionReqVo);
        if (CollectionUtils.isEmpty(functionList)) {
            return Collections.emptyList();
        }

        List<String> functionCodeList = mdmFunctionRoleService.findFunctionCodeByRole(roleCode);
        if (!CollectionUtils.isEmpty(functionCodeList)) {
            functionList = functionList.stream().peek(x -> {
                if (functionCodeList.contains(x.getFunctionCode())) {
                    x.setFunctionSelect(YesNoEnum.yesNoEnum.ONE.getValue());
                }
            }).collect(Collectors.toList());
        }
        List<MdmRoleFunctionTreeVo> rootList = new ArrayList<>();
        for (MdmRoleFunctionTreeVo mdmRoleFunctionTreeVo : functionList) {
            if (StringUtils.isEmpty(mdmRoleFunctionTreeVo.getParentCode())) {
                rootList.add(mdmRoleFunctionTreeVo);
            }
            List<MdmRoleFunctionTreeVo> children = mdmRoleFunctionTreeVo.getChildren();
            if (children == null) {
                children = new ArrayList<>();
            }
            for (MdmRoleFunctionTreeVo roleFunctionTreeVo : functionList) {
                String parentCode = roleFunctionTreeVo.getParentCode();
                if (!StringUtils.isEmpty(parentCode) && parentCode.equals(mdmRoleFunctionTreeVo.getFunctionCode())) {
                    children.add(roleFunctionTreeVo);
                }
            }
            children.sort(Comparator.comparing(x -> Optional.ofNullable(x.getFunctionOrder()).orElse(100)));
            mdmRoleFunctionTreeVo.setChildren(children);
        }
        rootList.sort(Comparator.comparing(x -> Optional.ofNullable(x.getFunctionOrder()).orElse(100)));
        return rootList;
    }

    @Override
    public List<MdmFunctionRespVo> functionTree(MdmFunctionReqVo mdmFunctionReqVo) {
        MdmCheckAdminVo mdmCheckAdminVo = findCurrentUserFunctionCodeList();
        List<String> functionCodeList = mdmCheckAdminVo.getFunctionCodeList();
        if (!mdmCheckAdminVo.isAuthorized() && CollectionUtils.isEmpty(functionCodeList)) {
            return Collections.emptyList();
        }
        MdmFunctionReqVo condition = new MdmFunctionReqVo();
        condition.setFunctionConfig(mdmFunctionReqVo.getFunctionConfig());
//        condition.setFunctionCodeList(functionCodeList);
        condition.setParentCode(mdmFunctionReqVo.getParentCode());
        List<MdmFunctionRespVo> allFunction = listCondition(condition);
        if (CollectionUtils.isEmpty(allFunction)) {
            return Collections.emptyList();
        }
        mdmFunctionReqVo.setFunctionCodeList(functionCodeList);
        allFunction = convert(allFunction);
        allFunction = convert(allFunction, mdmFunctionReqVo);
        excludeCodeAndChildren(allFunction, mdmFunctionReqVo.getExcludeFunctionCode());
        return allFunction;
    }


    /**
     * 树形构建
     */
    protected List<MdmFunctionRespVo> convert(List<MdmFunctionRespVo> list) {
        ArrayList<MdmFunctionRespVo> result = new ArrayList<>();
        if (!CollectionUtils.isEmpty(list)) {
            for (MdmFunctionRespVo mdmFunctionRespVo : list) {
                if (StringUtils.isEmpty(mdmFunctionRespVo.getParentCode())) {
                    result.add(mdmFunctionRespVo);
                }
                List<MdmFunctionRespVo> children = mdmFunctionRespVo.getChildren();
                if (children == null) {
                    children = new ArrayList<>();
                }
                for (MdmFunctionRespVo functionRespVo : list) {
                    if (mdmFunctionRespVo.getFunctionCode().equals(functionRespVo.getParentCode())) {
                        children.add(functionRespVo);
                    }
                }
                children.sort(Comparator.comparing(x -> {
                    String functionOrder = x.getFunctionOrder();
                    if (StringUtils.isEmpty(functionOrder)) {
                        return 1000;
                    }
                    return Integer.parseInt(functionOrder);
                }));
                mdmFunctionRespVo.setChildren(children);
            }
            result.sort(Comparator.comparing(x -> {
                String functionOrder = x.getFunctionOrder();
                if (StringUtils.isEmpty(functionOrder)) {
                    return 1000;
                }
                return Integer.parseInt(functionOrder);
            }));
        }
        return result;
    }

    protected List<MdmFunctionRespVo> convertPermission(List<MdmFunctionRespVo> list) {
        for (MdmFunctionRespVo mdmFunctionRespVo : list) {
            mdmFunctionRespVo.setChildren(convertPermission(mdmFunctionRespVo.getChildren()));
        }
        for (MdmFunctionRespVo mdmFunctionRespVo : list) {
            List<MdmFunctionRespVo> children = mdmFunctionRespVo.getChildren();
            if (YesNoEnum.yesNoEnum.ZERO.getValue().equals(mdmFunctionRespVo.getFunctionPermission())) {
                if (!org.springframework.util.CollectionUtils.isEmpty(children)) {
                    if (children.stream().anyMatch(x-> YesNoEnum.yesNoEnum.ONE.getValue().equals(x.getFunctionPermission()))) {
                        mdmFunctionRespVo.setFunctionPermission(YesNoEnum.yesNoEnum.ONE.getValue());
                    }
                }
            }
        }
        return list;
    }

    /**
     * 树形过滤
     */
    protected List<MdmFunctionRespVo> convert(List<MdmFunctionRespVo> collect, MdmFunctionReqVo mdmFunctionReqVo) {
        if (CollectionUtils.isEmpty(collect)) {
            return collect;
        }
        for (MdmFunctionRespVo mdmFunctionRespVo : collect) {
            mdmFunctionRespVo.setChildren(convert(mdmFunctionRespVo.getChildren(), mdmFunctionReqVo));
        }
        return collect.stream()
                .filter(x -> !CollectionUtils.isEmpty(x.getChildren())
                        || treeFilter(mdmFunctionReqVo, x))
                .collect(Collectors.toList());
    }

    protected boolean treeFilter(MdmFunctionReqVo mdmFunctionReqVo, MdmFunctionRespVo x) {
        return (CollectionUtils.isEmpty(mdmFunctionReqVo.getFunctionCodeList()) || mdmFunctionReqVo.getFunctionCodeList().contains(x.getFunctionCode()))
                && (StringUtils.isEmpty(mdmFunctionReqVo.getFunctionName()) || x.getFunctionName().contains(mdmFunctionReqVo.getFunctionName()))
                && (StringUtils.isEmpty(mdmFunctionReqVo.getFunctionCode()) || x.getFunctionCode().contains(mdmFunctionReqVo.getFunctionCode()))
                && (StringUtils.isEmpty(mdmFunctionReqVo.getPageConfig()) || mdmFunctionReqVo.getPageConfig().equals(x.getPageConfig()));
    }

    @Override
    public List<MdmFunctionRespVo> constructFunctionTree(String functionName, List<MdmFunctionRespVo> mdmFunctionRespVos, List<String> selectFunctionCodeList) {
        List<MdmFunctionRespVo> result = new ArrayList<>();
        List<MdmFunctionRespVo> subList = new ArrayList<>();
        List<String> finalFunctionCodeList = findUserFunctionCodeList();
        mdmFunctionRespVos.stream()
                .filter(x -> StringUtils.isEmpty(functionName) || x.getFunctionName().contains(functionName))
                .filter(x -> CollectionUtils.isEmpty(selectFunctionCodeList) || selectFunctionCodeList.contains(x.getFunctionCode()))
                .filter(x -> StringUtils.isEmpty(x.getParentCode()) || finalFunctionCodeList.contains(x.getFunctionCode()))
                .forEach(x -> setUpChildren(x, mdmFunctionRespVos, result, subList));
        return result.stream()
                .filter(x -> !CollectionUtils.isEmpty(x.getChildren()) || finalFunctionCodeList.contains(x.getFunctionCode()))
                .sorted(Comparator.comparing(x -> Optional.ofNullable(x.getFunctionOrder()).map(Integer::parseInt).orElse(100)))
                .collect(Collectors.toList());
    }

    /**
     * 过滤下级节点
     *
     * @param tree                集合
     * @param excludeFunctionCode 当前菜单编码
     */
    protected void excludeCodeAndChildren(List<MdmFunctionRespVo> tree, String excludeFunctionCode) {
        if (!CollectionUtils.isEmpty(tree) && !StringUtils.isEmpty(excludeFunctionCode)) {

            Iterator<MdmFunctionRespVo> iterator = tree.iterator();
            while (iterator.hasNext()) {
                MdmFunctionRespVo item = iterator.next();
                if (item.getFunctionCode().equals(excludeFunctionCode)) {
                    iterator.remove();
                    return;
                }
                excludeCodeAndChildren(item.getChildren(), excludeFunctionCode);
            }
        }
    }

    @Override
    public List<String> findCurrentAndSubFunctionCodeList(String functionCode) {
        List<String> currentFunctionCodeList = this.lambdaQuery()
                .eq(MdmFunctionEntity::getFunctionCode, functionCode)
                .select(MdmFunctionEntity::getFunctionCode)
                .list()
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getFunctionCode()))
                .map(MdmFunctionEntity::getFunctionCode)
                .distinct()
                .collect(Collectors.toList());
        ArrayList<String> result = new ArrayList<>();
        while (!CollectionUtils.isEmpty(currentFunctionCodeList)) {
            result.addAll(currentFunctionCodeList);
            currentFunctionCodeList = this.lambdaQuery()
                    .in(MdmFunctionEntity::getParentCode, currentFunctionCodeList)
                    .select(MdmFunctionEntity::getFunctionCode)
                    .list()
                    .stream()
                    .filter(x -> x != null && !StringUtils.isEmpty(x.getFunctionCode()))
                    .map(MdmFunctionEntity::getFunctionCode)
                    .distinct()
                    .collect(Collectors.toList());
        }
        return result;
    }

    @Override
    public List<MdmFunctionRespVo> functionSearchTree(MdmFunctionSearchReqVo mdmFunctionSearchReqVo) {
        MdmCheckAdminVo mdmCheckAdminVo = findCurrentUserFunctionCodeList();
        log.info("查询下级指定菜单权限:{}",mdmCheckAdminVo);
        List<String> functionCodeList = mdmCheckAdminVo.getFunctionCodeList();
        if (!mdmCheckAdminVo.isAuthorized() && CollectionUtils.isEmpty(functionCodeList)) {
            return Collections.emptyList();
        }
        MdmFunctionReqVo mdmFunctionReqVo = new MdmFunctionReqVo();
        mdmFunctionReqVo.setFunctionCodeList(functionCodeList);
        mdmFunctionReqVo.setFunctionConfig(mdmFunctionSearchReqVo.getFunctionConfig());
        List<MdmFunctionRespVo> mdmFunctionRespVos = listCondition(mdmFunctionReqVo);
        log.info("查询下级指针权限菜单:{}",mdmFunctionRespVos);
        List<MdmFunctionRespVo> root = new ArrayList<>();
        String specificRootCode = mdmFunctionSearchReqVo.getSpecificRootCode();
        for (MdmFunctionRespVo mdmFunctionRespVo : mdmFunctionRespVos) {
            if (!StringUtils.isEmpty(specificRootCode)) {
                if (specificRootCode.equals(mdmFunctionRespVo.getFunctionCode())) {
                    root.add(mdmFunctionRespVo);
                }
            } else if (StringUtils.isEmpty(mdmFunctionRespVo.getParentCode())) {
                root.add(mdmFunctionRespVo);
            }
            List<MdmFunctionRespVo> children = mdmFunctionRespVo.getChildren();
            if (children == null) {
                children = new ArrayList<>();
            }
            for (MdmFunctionRespVo functionRespVo : mdmFunctionRespVos) {
                if (mdmFunctionRespVo.getFunctionCode().equals(functionRespVo.getParentCode())) {
                    children.add(functionRespVo);
                }
            }
            children.sort(Comparator.comparing(x -> Optional.ofNullable(x.getFunctionOrder()).map(Integer::parseInt).orElse(100)));
            mdmFunctionRespVo.setChildren(children);
        }
        excludeCodeAndChildren(root, mdmFunctionSearchReqVo.getExcludeFunctionCode());
        return root;
    }

    /**
     * 需要过滤的菜单
     *
     * @return list
     */
    protected List<String> functionWhiteList() {

        ArrayList<String> list = new ArrayList<>();
        list.add("CRM20201128000000156");
        if (!StringUtils.isEmpty(whiteFunctionCodeList)) {
            list.addAll(Arrays.asList(whiteFunctionCodeList.split(",")));
        }
        return list;
    }

    @Override
    public Map<String, String> flattenTree(List<MdmFunctionRespVo> list) {
        if (CollectionUtils.isEmpty(list)) {
            return Collections.emptyMap();
        }
        Map<String, String> map = new HashMap<>(list.size());
        flattenTree("", list, map);
        return map;
    }

    protected void flattenTree(String path, List<MdmFunctionRespVo> functionRespVos, Map<String, String> map) {
        for (MdmFunctionRespVo mdmFunctionRespVo : functionRespVos) {
            String newPath = path + mdmFunctionRespVo.getFunctionName();
            List<MdmFunctionRespVo> children = mdmFunctionRespVo.getChildren();
            if (CollectionUtils.isEmpty(children)) {
                map.put(mdmFunctionRespVo.getFunctionCode(), newPath);
                continue;
            }
            newPath += "-";
            flattenTree(newPath, children, map);
        }
    }

    @Override
    public Integer functionDepth() {
        List<MdmFunctionRespVo> respVos = functionTree(new MdmFunctionReqVo());
        return functionHight(respVos);
    }

    @Override
    public List<MdmFunctionPermissionVo> findDataPermissionFunctionTree(MdmFunctionReqVo mdmFunctionReqVo) {
        List<MdmFunctionRespVo> list = mdmFunctionMapper.findSimpleList(new MdmFunctionReqVo());
        if (CollectionUtils.isEmpty(list)) {
            return Collections.emptyList();
        }
        for (MdmFunctionRespVo mdmFunctionPermissionVo : list) {
            mdmFunctionPermissionVo.setFunctionAuthorized(YesNoEnum.yesNoEnum.ZERO.getValue());
            mdmFunctionPermissionVo.setFunctionPermission(YesNoEnum.yesNoEnum.ZERO.getValue());
        }
        //如果已配置权限对象，高亮显示
        List<MdmFunctionSubPermissionVo> mdmFunctionSubPermissionVos = mdmFunctionSubService.findMdmFunctionSubPermissionVos(new MdmFunctionSubReqVo());
        if (!CollectionUtils.isEmpty(mdmFunctionSubPermissionVos)) {
            Set<String> functionCodeSet = mdmFunctionSubPermissionVos.stream()
                    .map(MdmFunctionSubPermissionVo::getParentCode)
                    .filter(parentCode -> !StringUtils.isEmpty(parentCode))
                    .collect(Collectors.toSet());
            if (!CollectionUtils.isEmpty(functionCodeSet)) {
                for (MdmFunctionRespVo mdmFunctionPermissionVo : list) {
                    if (functionCodeSet.contains(mdmFunctionPermissionVo.getFunctionCode())) {
                        mdmFunctionPermissionVo.setFunctionPermission(YesNoEnum.yesNoEnum.ONE.getValue());
                    }
                }
            }
        }
        //已经配置数据权限的回显
        MdmDataPermissionReqVo mdmDataPermissionReqVo = new MdmDataPermissionReqVo();
        Map<String, List<String>> dataPermissionMap = mdmDataPermissionService.findDataPermissionMap(mdmDataPermissionReqVo);
        if (dataPermissionMap != null && !dataPermissionMap.isEmpty()) {
            for (MdmFunctionRespVo mdmFunctionPermissionVo : list) {
                if (dataPermissionMap.containsKey(mdmFunctionPermissionVo.getFunctionCode())) {
                    mdmFunctionPermissionVo.setFunctionAuthorized(YesNoEnum.yesNoEnum.ONE.getValue());
                }
            }
        }
        MdmFunctionReqVo functionReqVo = new MdmFunctionReqVo();
        List<MdmFunctionRespVo> filterList = mdmFunctionMapper.findSimpleList(mdmFunctionReqVo);
        if (!CollectionUtils.isEmpty(filterList)) {
            List<String> filterFunctionCodeList = filterList.stream()
                    .filter(functionCode -> !StringUtils.isEmpty(functionCode))
                    .map(MdmFunctionRespVo::getFunctionCode)
                    .distinct()
                    .collect(Collectors.toList());
            functionReqVo.setFunctionCodeList(filterFunctionCodeList);
        }
        return CrmBeanUtil.copyList(convertPermission(convert(convert(list), functionReqVo)), MdmFunctionPermissionVo.class);
//        return MdmFunctionMap.INSTANCE.function2PermissionVo(convert(convert(list), functionReqVo));
    }

    @Override
    public List<MdmFunctionPermissionVo> pageEngineTree(MdmFunctionReqVo mdmFunctionReqVo) {
        return CrmBeanUtil.copyList(allFunctionTree(mdmFunctionReqVo), MdmFunctionPermissionVo.class);
    }

    @Override
    public List<MdmFunctionRespVo> allFunctionTree(MdmFunctionReqVo mdmFunctionReqVo) {
        MdmCheckAdminVo mdmCheckAdminVo = findCurrentUserFunctionCodeList();
        List<String> functionCodeList = mdmCheckAdminVo.getFunctionCodeList();
        if (!mdmCheckAdminVo.isAuthorized() && CollectionUtils.isEmpty(functionCodeList)) {
            return new ArrayList<>();
        }
        MdmFunctionReqVo condition = new MdmFunctionReqVo();
        condition.setFunctionConfig(mdmFunctionReqVo.getFunctionConfig());
//        condition.setFunctionCodeList(functionCodeList);
        List<MdmFunctionRespVo> allFunction = mdmFunctionMapper.findList(null, condition);
        if (CollectionUtils.isEmpty(allFunction)) {
            return new ArrayList<>();
        }
        if (functionCodeList == null) {
            functionCodeList = new ArrayList<>();
        }
        mdmFunctionReqVo.setFunctionCodeList(functionCodeList);
        return convert(convert(allFunction), mdmFunctionReqVo);
    }

    @Override
    public MdmFunctionEngineSyncVo syncList(MdmSyncStandardFunctionReqVo reqVo) {
        //判断是否查询当前及下级
        List<String> currentAndSubFunctionCodeList = reqVo.getCurrentAndSubFunctionCodeList();
        if(!CollectionUtils.isEmpty(currentAndSubFunctionCodeList)){
            List<String> result = new ArrayList<>();
            List<String> list = currentAndSubFunctionCodeList;
            while (!org.springframework.util.CollectionUtils.isEmpty(list)) {
                result.addAll(list);
                list = this.lambdaQuery()
                        .in(MdmFunctionEntity::getParentCode, list)
                        .select(MdmFunctionEntity::getFunctionCode)
                        .list()
                        .stream()
                        .map(MdmFunctionEntity::getFunctionCode)
                        .filter(functionCode ->!StringUtils.isEmpty(functionCode))
                        .collect(Collectors.toList());
            }
            if(!CollectionUtils.isEmpty(result)){
                reqVo.setFunctionCodeList(result);
            }
        }
        //查询菜单，列表，按钮，字段配置
        List<MdmFunctionEntity> functionEntityList = this.lambdaQuery()
                .in(!CollectionUtils.isEmpty(reqVo.getFunctionCodeList()), MdmFunctionEntity::getFunctionCode, reqVo.getFunctionCodeList())
                .list();
        List<MdmFunctionSubEntity> mdmFunctionSubEntityList = mdmFunctionSubService.lambdaQuery()
                .in(!CollectionUtils.isEmpty(reqVo.getFunctionCodeList()), MdmFunctionSubEntity::getParentCode, reqVo.getFunctionCodeList())
                .list();
        List<MdmFunctionSubButtonEntity> subButtonEntityList = mdmFunctionSubButtonService.lambdaQuery()
                .in(!CollectionUtils.isEmpty(reqVo.getFunctionCodeList()), MdmFunctionSubButtonEntity::getParentCode, reqVo.getFunctionCodeList())
                .list();
        List<MdmColumnConfigEntity> configEntityList = mdmColumnConfigService.lambdaQuery()
                .in(!CollectionUtils.isEmpty(reqVo.getFunctionCodeList()), MdmColumnConfigEntity::getParentCode, reqVo.getFunctionCodeList())
                .list();
        MdmFunctionEngineSyncVo respVo = new MdmFunctionEngineSyncVo();
        respVo.setMdmFunctionRespVoList(CrmBeanUtil.copyList(functionEntityList,MdmFunctionRespVo.class));
        respVo.setMdmFunctionSubRespVoList(CrmBeanUtil.copyList(mdmFunctionSubEntityList, MdmFunctionSubRespVo.class));
        respVo.setMdmFunctionSubButtonRespVoList(CrmBeanUtil.copyList(subButtonEntityList, MdmFunctionSubButtonRespVo.class));
        respVo.setMdmColumnConfigRespVoList(MdmColumnConfigConverter.INSTANCE.entity2RespVo(configEntityList));
        return respVo;
    }

    /**
     * 查询指定菜单树的最大深度
     *
     * @param respVos list
     * @return int
     */
    protected int functionHight(List<MdmFunctionRespVo> respVos) {
        if (CollectionUtils.isEmpty(respVos)) {
            return 0;
        }
        int hight = 0;
        for (MdmFunctionRespVo mdmFunctionRespVo : respVos) {
            hight = Math.max(functionHight(mdmFunctionRespVo.getChildren()), hight);
        }
        return hight + 1;
    }

    protected List<String> findAllChildFunctionCodeList(List<String> functionCodeList) {
        if (CollectionUtils.isEmpty(functionCodeList)) {
            return new ArrayList<>();
        }
        List<String> realFunctionCodeList = functionCodeList.stream()
                .filter(x -> !StringUtils.isEmpty(x))
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(realFunctionCodeList)) {
            return new ArrayList<>();
        }
        List<MdmFunctionPermissionVo> list = mdmFunctionMapper.findFunctionCodeList(new MdmFunctionReqVo());
        if (CollectionUtils.isEmpty(list)) {
            return new ArrayList<>();
        }
        List<String> result = new ArrayList<>();
        while (!CollectionUtils.isEmpty(realFunctionCodeList)) {
            result.addAll(realFunctionCodeList);
            List<String> finalRealFunctionCodeList = realFunctionCodeList;
            realFunctionCodeList = list.stream()
                    .filter(x -> finalRealFunctionCodeList.contains(x.getParentCode()))
                    .map(MdmFunctionPermissionVo::getFunctionCode)
                    .collect(Collectors.toList());
        }
        return result;
    }

    protected List<String> findAllParentFunctionCodeList(List<String> functionCodeList,String enableStatus) {
        if (CollectionUtils.isEmpty(functionCodeList)) {
            return new ArrayList<>();
        }
        List<String> realFunctionCodeList = functionCodeList.stream()
                .filter(x -> !StringUtils.isEmpty(x))
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(realFunctionCodeList)) {
            return new ArrayList<>();
        }
        MdmFunctionReqVo mdmFunctionReqVo = new MdmFunctionReqVo();
        mdmFunctionReqVo.setEnableStatus(enableStatus);
        List<MdmFunctionPermissionVo> list = mdmFunctionMapper.findFunctionCodeList(mdmFunctionReqVo);
        if (CollectionUtils.isEmpty(list)) {
            return new ArrayList<>();
        }
        List<String> result = new ArrayList<>();

        while (!CollectionUtils.isEmpty(realFunctionCodeList)) {
            result.addAll(realFunctionCodeList);
            List<String> finalRealFunctionCodeList = realFunctionCodeList;
            realFunctionCodeList = list.stream()
                    .filter(x -> finalRealFunctionCodeList.contains(x.getFunctionCode()))
                    .map(MdmFunctionPermissionVo::getParentCode)
                    .filter(x -> !StringUtils.isEmpty(x))
                    .collect(Collectors.toList());
        }
        return result;
    }
}


