package com.biz.crm.icon.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.CacheManagerConstant;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.icon.mapper.MdmIconMapper;
import com.biz.crm.icon.model.MdmIconEntity;
import com.biz.crm.icon.service.IMdmIconService;
import com.biz.crm.nebular.mdm.icon.MdmIconReqVo;
import com.biz.crm.nebular.mdm.icon.MdmIconRespVo;
import com.biz.crm.util.CodeUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.cache.TableConfigUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.List;

/**
 * 接口实现
 *
 * @author zxw
 * @date 2020-12-05 11:10:24
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmIconServiceExpandImpl")
@CacheConfig(cacheManager = CacheManagerConstant.CACHE_MANAGER)
public class MdmIconServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmIconMapper, MdmIconEntity> implements IMdmIconService {

    @Resource
    private MdmIconMapper mdmIconMapper;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<MdmIconRespVo> findList(MdmIconReqVo reqVo) {
        Page<MdmIconRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmIconRespVo> list = mdmIconMapper.findList(page, reqVo);
        return PageResult.<MdmIconRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param id       id
     * @param iconCode 图标编码
     * @return mdmIconRespVo
     */
    @Override
    public MdmIconRespVo query(String id, String iconCode) {
        if (StringUtils.isEmpty(id) && StringUtils.isEmpty(iconCode)) {
            return null;
        }
        MdmIconEntity one = this.lambdaQuery()
                .eq(!StringUtils.isEmpty(id), MdmIconEntity::getId, id)
                .eq(!StringUtils.isEmpty(iconCode), MdmIconEntity::getIconCode, iconCode)
                .one();
        return CrmBeanUtil.copy(one, MdmIconRespVo.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(cacheNames = CacheManagerConstant.MDM_ICON,allEntries = true)
    public void save(MdmIconReqVo reqVo) {
        if (StringUtils.isEmpty(reqVo.getIconCode())) {
            reqVo.setIconCode(CodeUtil.generateCode());
        } else {
            Integer count = this.lambdaQuery()
                    .eq(MdmIconEntity::getIconCode, reqVo.getIconCode())
                    .count();
            Assert.isTrue(count == null || count < 1, "图标编码已存在");
        }
        reqVo.setId(null);
        this.save(CrmBeanUtil.copy(reqVo, MdmIconEntity.class));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(cacheNames = CacheManagerConstant.MDM_ICON,allEntries = true)
    public void update(MdmIconReqVo reqVo) {
        Assert.hasText(reqVo.getId(), "数据主键不能为空");
        Assert.hasText(reqVo.getIconCode(), "图标编码不能为空");
        Integer count = this.lambdaQuery()
                .eq(MdmIconEntity::getId, reqVo.getId())
                .count();
        Assert.isTrue(count != null && count > 0, "图标不存在");
        long codeCount = this.lambdaQuery()
                .eq(MdmIconEntity::getIconCode, reqVo.getIconCode())
                .select(MdmIconEntity::getId)
                .list()
                .stream().filter(x -> !reqVo.getId().equals(x.getId()))
                .count();
        Assert.isTrue(codeCount < 1, "图标编码重复");
        this.updateById(CrmBeanUtil.copy(reqVo, MdmIconEntity.class));
        //清除页面配置缓存
        TableConfigUtil.removeAll();
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(cacheNames = CacheManagerConstant.MDM_ICON,allEntries = true)
    public void deleteBatch(List<String> ids) {
        Assert.notEmpty(ids, "id集合不能为空");
        this.removeByIds(ids);
        //清除页面配置缓存
        TableConfigUtil.removeAll();
    }

    /**
     * 启用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(cacheNames = CacheManagerConstant.MDM_ICON,allEntries = true)
    public void enableBatch(List<String> ids) {
        Assert.notEmpty(ids, "id集合不能为空");
        this.lambdaUpdate()
                .in(MdmIconEntity::getId, ids)
                .set(MdmIconEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .update();
        //清除页面配置缓存
        TableConfigUtil.removeAll();
    }

    /**
     * 禁用
     *
     * @param ids
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(cacheNames = CacheManagerConstant.MDM_ICON,allEntries = true)
    public void disableBatch(List<String> ids) {
        Assert.notEmpty(ids, "id集合不能为空");
        this.lambdaUpdate()
                .in(MdmIconEntity::getId, ids)
                .set(MdmIconEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
                .update();
        //清除页面配置缓存
        TableConfigUtil.removeAll();
    }

    @Override
    @Cacheable(sync = true,cacheNames = CacheManagerConstant.MDM_ICON)
    public List<MdmIconRespVo> listCondition(MdmIconReqVo mdmIconReqVo) {
        List<MdmIconEntity> list = this.lambdaQuery()
                .eq(MdmIconEntity::getEnableStatus,CrmEnableStatusEnum.ENABLE.getCode())
                .in(!CollectionUtils.isEmpty(mdmIconReqVo.getIconCodeList()),MdmIconEntity::getIconCode,mdmIconReqVo.getIconCodeList())
                .eq(!StringUtils.isEmpty(mdmIconReqVo.getIconType()),MdmIconEntity::getIconType,mdmIconReqVo.getIconType())
                .eq(!StringUtils.isEmpty(mdmIconReqVo.getIconCode()),MdmIconEntity::getIconCode, mdmIconReqVo.getIconCode())
                .like(!StringUtils.isEmpty(mdmIconReqVo.getIconName()),MdmIconEntity::getIconName, mdmIconReqVo.getIconName())
                .list();
        return CrmBeanUtil.copyList(list, MdmIconRespVo.class);
    }
}
