package com.biz.crm.kms.service.impl;

import com.biz.crm.base.BusinessException;
import com.biz.crm.common.service.BaseTreeService;
import com.biz.crm.config.resttemplate.RestTemplateUtils;
import com.biz.crm.kms.config.KmsProperties;
import com.biz.crm.kms.config.MapperConfig;
import com.biz.crm.kms.service.MdmCategoryService;
import com.biz.crm.nebular.mdm.common.BaseTreeReqVo;
import com.biz.crm.nebular.mdm.common.BaseTreeRespVo;
import com.biz.crm.nebular.mdm.kms.api.KmsCategoryVo;
import com.biz.crm.nebular.mdm.productlevel.req.MdmProductLevelReqVo;
import com.biz.crm.productlevel.model.MdmProductLevelEntity;
import com.biz.crm.productlevel.service.MdmProductLevelService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author zxw
 * @date 2021-03-08 15:48
 **/
@Service
@ConditionalOnMissingBean(name = "MdmCategoryExpandImpl")
@Slf4j
public class MdmCategoryServiceImpl implements MdmCategoryService {

    @Autowired
    private KmsProperties kmsProperties;

    @Resource
    private BaseTreeService baseTreeService;

    @Resource
    private MdmProductLevelService mdmProductLevelService;

    @Override
    public void add(KmsCategoryVo kmsCategoryVo) {
        Assert.hasText(kmsCategoryVo.getId(), "id不能为空");
        kmsCategoryVo.setUserId(kmsProperties.getUserId());
        kmsCategoryVo.setUsername(kmsProperties.getUsername());
        kmsCategoryVo.setTenantryId(kmsProperties.getTenantryId());

        String url = kmsProperties.getUrl() + "/KmsProCategoryController/addProCategory?access_token=" + kmsProperties.getAccessToken();

        ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsCategoryVo, String.class);
        log.info("[产品分类][新增]result:{}", entity);
        if (!HttpStatus.OK.equals(entity.getStatusCode())) {
            throw new BusinessException(MapperConfig.ERROR_MSG);
        }
    }

    @Override
    public void add(MdmProductLevelReqVo mdmProductLevelReqVo) {
        log.info("[产品分类][同步产品层级]reqVo:{}", mdmProductLevelReqVo);
        if (!MapperConfig.validProduct(kmsProperties)) {
            return;
        }
        Assert.hasText(mdmProductLevelReqVo.getProductLevelCode(), "产品层级编码不能为空");
        KmsCategoryVo kmsCategoryVo = convertCategory(mdmProductLevelReqVo);
        add(kmsCategoryVo);
    }

    protected KmsCategoryVo convertCategory(MdmProductLevelReqVo mdmProductLevelReqVo) {
        KmsCategoryVo kmsCategoryVo = new KmsCategoryVo();
        kmsCategoryVo.setId(mdmProductLevelReqVo.getProductLevelCode());
        kmsCategoryVo.setPid(mdmProductLevelReqVo.getParentCode());
        kmsCategoryVo.setProCategoryDescription(mdmProductLevelReqVo.getRemarks());
        kmsCategoryVo.setProCategoryName(mdmProductLevelReqVo.getProductLevelName());
        return kmsCategoryVo;
    }

    @Override
    public void update(MdmProductLevelReqVo mdmProductLevelReqVo) {
        if (!MapperConfig.validProduct(kmsProperties)) {
            return;
        }
        Assert.hasText(mdmProductLevelReqVo.getProductLevelCode(), "产品层级编码不能为空");
        KmsCategoryVo kmsCategoryVo = convertCategory(mdmProductLevelReqVo);

        kmsCategoryVo.setUserId(kmsProperties.getUserId());
        kmsCategoryVo.setUsername(kmsProperties.getUsername());
        kmsCategoryVo.setTenantryId(kmsProperties.getTenantryId());

        String url = kmsProperties.getUrl() + "/KmsProCategoryController/updateProCategory?access_token=" + kmsProperties.getAccessToken();

        ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsCategoryVo, String.class);
        log.info("[kms产品分类][update]entity:{}", entity);
    }

    @Override
    public void remove(String productLevelCode) {
        if (!MapperConfig.validProduct(kmsProperties)) {
            return;
        }
        if (StringUtils.isEmpty(productLevelCode)) {
            return;
        }
        KmsCategoryVo kmsCategoryVo = new KmsCategoryVo();
        kmsCategoryVo.setId(productLevelCode);
        kmsCategoryVo.setUserId(kmsProperties.getUserId());
        kmsCategoryVo.setUsername(kmsProperties.getUsername());
        kmsCategoryVo.setTenantryId(kmsProperties.getTenantryId());

        String url = kmsProperties.getUrl() + "/KmsProCategoryController/deleteById?access_token=" + kmsProperties.getAccessToken();
        ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsCategoryVo, String.class);
        log.info("[产品分类][删除]result:{}", entity);
        if (!HttpStatus.OK.equals(entity.getStatusCode())) {
            throw new BusinessException("kms删除失败");
        }
    }

    @Override
    public void remove(List<String> productLevelCodeList) {
        if (!MapperConfig.validProduct(kmsProperties)) {
            return;
        }
        if (!CollectionUtils.isEmpty(productLevelCodeList)) {
            for (String s : productLevelCodeList) {
                remove(s);
            }
        }
    }

    @Override
    public void productLevelSync() {
        List<BaseTreeRespVo> baseTreeRespVos = baseTreeService.productLevelTree(new BaseTreeReqVo());
        if (CollectionUtils.isEmpty(baseTreeRespVos)) {
            return;
        }
        try {
            while (!CollectionUtils.isEmpty(baseTreeRespVos)) {
                for (BaseTreeRespVo baseTreeRespVo : baseTreeRespVos) {
                    KmsCategoryVo kmsCategoryVo = new KmsCategoryVo();
                    kmsCategoryVo.setId(baseTreeRespVo.getCode());
                    kmsCategoryVo.setPid(baseTreeRespVo.getParentCode());
                    kmsCategoryVo.setProCategoryName(baseTreeRespVo.getName());
                    add(kmsCategoryVo);
                }
                baseTreeRespVos = baseTreeRespVos.stream()
                        .filter(x -> !CollectionUtils.isEmpty(x.getChildren()))
                        .flatMap(x -> x.getChildren().stream())
                        .collect(Collectors.toList());
            }
        } catch (Exception e) {
            log.error("kms产品层级同步失败:{}", e);
        }
    }

    @Override
    public void productLevelClear() {
        List<String> productLevelCodeList = mdmProductLevelService.lambdaQuery()
                .select(MdmProductLevelEntity::getProductLevelCode)
                .list()
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getProductLevelCode()))
                .map(MdmProductLevelEntity::getProductLevelCode)
                .collect(Collectors.toList());

        if (CollectionUtils.isEmpty(productLevelCodeList)) {
            return;
        }

        remove(productLevelCodeList);
    }

    @Override
    public void addBatch(List<MdmProductLevelReqVo> list) {
        if (!MapperConfig.validProduct(kmsProperties)) {
            return;
        }
        if(!CollectionUtils.isEmpty(list)){
            try {
                list.parallelStream().forEach(this::add);
            } catch (Exception e) {
                log.error("批量新增产品层级失败:{}", e);
            }
        }
    }

}
