package com.biz.crm.kms.service.impl;

import com.biz.crm.base.BusinessException;
import com.biz.crm.config.resttemplate.RestTemplateUtils;
import com.biz.crm.kms.config.KmsProperties;
import com.biz.crm.kms.manager.KmsConverter;
import com.biz.crm.kms.service.MdmDirectSystemService;
import com.biz.crm.nebular.mdm.kms.MdmDirectSystemVo;
import com.biz.crm.nebular.mdm.kms.api.KmsDirectSystemVo;
import com.biz.crm.nebular.mdm.kms.api.KmsResultVo;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * 直营体系业务层
 *
 * @author zxw
 * @date 2021-02-19 09:17
 **/
@Service
@ConditionalOnMissingBean(name = "MdmDirectSystemServiceExpandImpl")
@Slf4j
public class MdmDirectSystemServiceImpl implements MdmDirectSystemService {

    @Autowired
    private KmsProperties kmsProperties;

    @Override
    public List<MdmDirectSystemVo> directSystemSelect(MdmDirectSystemVo mdmDirectSystemVo) {
        if (!valid()) {
            return new ArrayList<>();
        }
        String url = kmsProperties.getUrl() + "/directSystemController/findDirectSystemListWithNum";
        ParameterizedTypeReference<KmsResultVo<List<KmsDirectSystemVo>>> reference = new ParameterizedTypeReference<KmsResultVo<List<KmsDirectSystemVo>>>() {};
        ResponseEntity<KmsResultVo<List<KmsDirectSystemVo>>> exchange = RestTemplateUtils.exchange(url, HttpMethod.POST, null, kmsProperties, reference);
        List<KmsDirectSystemVo> kmsDirectSystemVos = Optional.of(exchange)
                .map(HttpEntity::getBody)
                .map(KmsResultVo::getData)
                .orElse(null);
        if(CollectionUtils.isEmpty(kmsDirectSystemVos)){
            return new ArrayList<>();
        }
        String directSystemName = mdmDirectSystemVo.getDirectSystemName();
        String enableStatus = mdmDirectSystemVo.getEnableStatus();
        List<MdmDirectSystemVo> mdmDirectSystemVos = KmsConverter.INSTANCE.directSystem2Mdm(kmsDirectSystemVos);
        mdmDirectSystemVos=mdmDirectSystemVos.stream()
                .filter(x -> StringUtils.isEmpty(directSystemName) || x.getDirectSystemName().contains(directSystemName))
                .filter(x->StringUtils.isEmpty(enableStatus)||enableStatus.equals(x.getEnableStatus()))
                .collect(Collectors.toList());
        return mdmDirectSystemVos;
    }

    protected boolean valid() {
        return !StringUtils.isEmpty(kmsProperties.getAccessToken());
    }

    @Override
    public void enable(List<String> ids) {
        Assert.notEmpty(ids, "直营体系id不能为空");
        String url = kmsProperties.getUrl() + "/directSystemController/stopOrUse?access_token="+kmsProperties.getAccessToken();
        for (String id : ids) {
            KmsDirectSystemVo kmsDirectSystemVo = new KmsDirectSystemVo();
            kmsDirectSystemVo.setDirectSystemId(id);
            kmsDirectSystemVo.setComomStatus("ENABLE");
            kmsDirectSystemVo.setTenantryId(kmsProperties.getTenantryId());
            kmsDirectSystemVo.setUserId(kmsProperties.getUserId());
            ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsDirectSystemVo, String.class);
            if (!HttpStatus.OK.equals(entity.getStatusCode())) {
                throw new BusinessException("启用失败");
            }
        }
    }

    @Override
    public void disable(List<String> ids) {
        Assert.notEmpty(ids, "直营体系id不能为空");
        String url = kmsProperties.getUrl() + "/directSystemController/stopOrUse?access_token="+kmsProperties.getAccessToken();
        for (String id : ids) {
            KmsDirectSystemVo kmsDirectSystemVo = new KmsDirectSystemVo();
            kmsDirectSystemVo.setDirectSystemId(id);
            kmsDirectSystemVo.setComomStatus("DISABLE");
            kmsDirectSystemVo.setTenantryId(kmsProperties.getTenantryId());
            kmsDirectSystemVo.setUserId(kmsProperties.getUserId());
            ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsDirectSystemVo, String.class);
            if (!HttpStatus.OK.equals(entity.getStatusCode())) {
                throw new BusinessException("禁用失败");
            }
        }
    }

    @Override
    public List<MdmDirectSystemVo> unAddSelect(MdmDirectSystemVo mdmDirectSystemVo) {
        String url = kmsProperties.getUrl() + "/directSystemController/findDirectSystemSelect";
        ParameterizedTypeReference<KmsResultVo<List<KmsDirectSystemVo>>> reference = new ParameterizedTypeReference<KmsResultVo<List<KmsDirectSystemVo>>>() {};
        ResponseEntity<KmsResultVo<List<KmsDirectSystemVo>>> exchange = RestTemplateUtils.exchange(url, HttpMethod.POST, null, kmsProperties, reference);
        List<KmsDirectSystemVo> kmsDirectSystemVos = Optional.ofNullable(exchange)
                .map(HttpEntity::getBody)
                .map(KmsResultVo::getData)
                .orElse(null);
        if(CollectionUtils.isEmpty(kmsDirectSystemVos)){
            return new ArrayList<>();
        }
        String directSystemName = mdmDirectSystemVo.getDirectSystemName();
        String enableStatus = mdmDirectSystemVo.getEnableStatus();
        List<MdmDirectSystemVo> mdmDirectSystemVos = KmsConverter.INSTANCE.directSystem2Mdm(kmsDirectSystemVos);
        mdmDirectSystemVos=mdmDirectSystemVos.stream()
                .filter(x -> StringUtils.isEmpty(directSystemName) || x.getDirectSystemName().contains(directSystemName))
                .filter(x->StringUtils.isEmpty(enableStatus)||enableStatus.equals(x.getEnableStatus()))
                .collect(Collectors.toList());
        return mdmDirectSystemVos;
    }

    @Override
    public void add(MdmDirectSystemVo mdmDirectSystemVo) {
        Assert.hasText(mdmDirectSystemVo.getBsDirectId(), "后台直营体系id不能为空");
        Assert.hasText(mdmDirectSystemVo.getDirectSystemCode(), "直营体系编码不能为空");
        Assert.hasText(mdmDirectSystemVo.getDirectSystemName(), "直营体系名称不能为空");

        List<MdmDirectSystemVo> mdmDirectSystemVos = directSystemSelect(mdmDirectSystemVo);
        if(!CollectionUtils.isEmpty(mdmDirectSystemVos)){
            for (MdmDirectSystemVo directSystemVo : mdmDirectSystemVos) {
                if (mdmDirectSystemVo.getDirectSystemCode().equals(directSystemVo.getDirectSystemCode())) {
                    throw new BusinessException("直营体系编码已存在");
                }
            }
        }

        KmsDirectSystemVo kmsDirectSystemVo = new KmsDirectSystemVo();
        kmsDirectSystemVo.setBsDirectId(mdmDirectSystemVo.getBsDirectId());
        kmsDirectSystemVo.setDirectSystemCode(mdmDirectSystemVo.getDirectSystemCode());
        kmsDirectSystemVo.setDirectSystemName(mdmDirectSystemVo.getDirectSystemName());
        kmsDirectSystemVo.setUsername(kmsProperties.getUsername());
        kmsDirectSystemVo.setUserId(kmsProperties.getUserId());
        kmsDirectSystemVo.setTenantryId(kmsProperties.getTenantryId());

        String url = kmsProperties.getUrl() + "/directSystemController/saveDirectSystem?access_token=" + kmsProperties.getAccessToken();

        ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsDirectSystemVo, String.class);
        log.info("[直营体系][新增]result:{}",entity);
        if (!HttpStatus.OK.equals(entity.getStatusCode())) {
            throw new BusinessException("新增失败");
        }
    }
}
