package com.biz.crm.kms.service.impl;

import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.resttemplate.RestTemplateUtils;
import com.biz.crm.kms.config.KmsProperties;
import com.biz.crm.kms.config.MapperConfig;
import com.biz.crm.kms.config.ParamConstant;
import com.biz.crm.kms.manager.KmsConverter;
import com.biz.crm.kms.service.MdmGrabRuleService;
import com.biz.crm.nebular.mdm.kms.*;
import com.biz.crm.nebular.mdm.kms.api.*;
import com.biz.crm.util.DateUtil;
import com.fasterxml.jackson.databind.node.ObjectNode;
import io.swagger.annotations.ApiModelProperty;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.net.URL;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * @author zxw
 * @date 2021-02-23 15:01
 **/
@Service
@ConditionalOnMissingBean(name = "MdmGrabRuleServiceExpandImpl")
@Slf4j
public class MdmGrabRuleServiceImpl implements MdmGrabRuleService {

    @Autowired
    private KmsProperties kmsProperties;

    private static final String KMS_ID = "id";

    private static final String MDM_ID = "kms_id";

    public String convert(String id) {
        if (id == null || "".equals(id.trim())) {
            return id;
        }
        if (!id.equals(KMS_ID) && !id.equals(MDM_ID)) {
            return id;
        }
        return KMS_ID.equals(id) ? MDM_ID : KMS_ID;
    }

    @Override
    public List<MdmGrabOrderTypeVo> orderTypeList(MdmDirectSystemVo mdmDirectSystemVo) {
        String directSystemId = mdmDirectSystemVo.getDirectSystemId();
        Assert.hasText(directSystemId, "直营体系id不能为空");
        String bsDirectId = mdmDirectSystemVo.getBsDirectId();
        Assert.hasText(bsDirectId, "后台直营体系id不能为空");

        saveBatchBsInvoice(directSystemId, bsDirectId);

        ObjectNode instance = MapperConfig.instance(kmsProperties);
        instance.put(ParamConstant.DIRECT_SYSTEM_ID, directSystemId);

        String url = kmsProperties.getUrl() + "/orderTypeController/findOrderTypeList";

        ParameterizedTypeReference<KmsResultVo<List<MdmGrabOrderTypeVo>>> reference = new ParameterizedTypeReference<KmsResultVo<List<MdmGrabOrderTypeVo>>>() {
        };

        return Optional.of(RestTemplateUtils.exchange(url, HttpMethod.POST, null, instance, reference))
                .map(HttpEntity::getBody)
                .map(KmsResultVo::getData)
                .map(x -> {
                    for (MdmGrabOrderTypeVo mdmGrabOrderTypeVo : x) {
                        mdmGrabOrderTypeVo.setInvoiceId(mdmGrabOrderTypeVo.getId());
                    }
                    return x;
                })
                .orElse(new ArrayList<>());
    }

    /**
     * 批量添加后台单据类型
     * @param directSystemId
     * @param bsDirectId
     */
    protected void saveBatchBsInvoice(String directSystemId, String bsDirectId) {
        ObjectNode bsParam = MapperConfig.instance(kmsProperties);
        bsParam.put(ParamConstant.BS_DIRECT_ID, bsDirectId);
        String url = kmsProperties.getUrl() + "/orderTypeController/findBsOrderTypeList";
        ParameterizedTypeReference<KmsResultVo<List<MdmGrabOrderTypeVo>>> reference = new ParameterizedTypeReference<KmsResultVo<List<MdmGrabOrderTypeVo>>>() {
        };
        Optional.of(RestTemplateUtils.exchange(url, HttpMethod.POST, null, bsParam, reference))
                .map(HttpEntity::getBody)
                .filter(x->MapperConfig.SUCCESS.equals(x.getCode()))
                .map(KmsResultVo::getData)
                .ifPresent(x->{
                    if(!CollectionUtils.isEmpty(x)){
                        String orderTypeUrl = kmsProperties.getUrl() + "/orderTypeController/saveOrderType?access_token=" + kmsProperties.getAccessToken();
                        ParameterizedTypeReference<KmsResultVo<Object>> orderTypeReference = new ParameterizedTypeReference<KmsResultVo<Object>>() {
                        };
                        x.parallelStream().forEach(k->{
                            ObjectNode orderTypeParam = MapperConfig.instance(kmsProperties);
                            orderTypeParam.put(ParamConstant.DIRECT_SYSTEM_ID, directSystemId);
                            orderTypeParam.put(ParamConstant.BS_INVOICE_ID, k.getBsInvoiceId());
                            orderTypeParam.put(ParamConstant.INVOICE_NAME, k.getInvoiceName());

                            Optional.of(RestTemplateUtils.exchange(orderTypeUrl,HttpMethod.POST,null,orderTypeParam,orderTypeReference))
                                    .map(HttpEntity::getBody)
                                    .filter(v->MapperConfig.ERROR.equals(v.getCode()))
                                    .ifPresent(v->{
                                        throw new BusinessException(v.getMessage());
                                    });
                        });
                    }
                });
    }

    @Override
    public PageResult<MdmGrabRuleVo> page(MdmGrabRuleVo mdmGrabRuleVo) {
        String directSystemId = mdmGrabRuleVo.getDirectSystemId();
        Assert.hasText(directSystemId, "直营体系id不能为空");
//        String invoiceId = mdmGrabRuleVo.getInvoiceId();
//        Assert.hasText(invoiceId, "单据类型id不能为空");
        String bsDirectId = mdmGrabRuleVo.getBsDirectId();
        Assert.hasText(bsDirectId, "后台直营体系id不能为空");
//        String bsInvoiceId = mdmGrabRuleVo.getBsInvoiceId();
//        Assert.hasText(bsInvoiceId, "后台单据类型id不能为空");

        Integer current = Optional.ofNullable(mdmGrabRuleVo.getPageNum()).orElse(1);
        Integer size = Optional.ofNullable(mdmGrabRuleVo.getPageSize()).orElse(25);
        KmsPageVo kmsPageVo = new KmsPageVo(current, size);

        KmsGrabRuleVo kmsGrabRuleVo = new KmsGrabRuleVo();
        kmsGrabRuleVo.setInvoiceId(mdmGrabRuleVo.getInvoiceId());
        kmsGrabRuleVo.setDirectSystemId(directSystemId);
        kmsGrabRuleVo.setName(mdmGrabRuleVo.getName());

        KmsGrabRulePageVo kmsGrabRulePageVo = new KmsGrabRulePageVo();
        kmsGrabRulePageVo.setUserId(kmsProperties.getUserId());
        kmsGrabRulePageVo.setUsername(kmsProperties.getUsername());
        kmsGrabRulePageVo.setTenantryId(kmsProperties.getTenantryId());
        kmsGrabRulePageVo.setReqVo(kmsGrabRuleVo);
        kmsGrabRulePageVo.setPage(kmsPageVo);

        String url = kmsProperties.getUrl() + "/grabRuleController/findGrabRuleListByPage";

        ParameterizedTypeReference<KmsResultVo<KmsPageResultVo<KmsGrabRuleVo>>> reference = new ParameterizedTypeReference<KmsResultVo<KmsPageResultVo<KmsGrabRuleVo>>>() {
        };

        ResponseEntity<KmsResultVo<KmsPageResultVo<KmsGrabRuleVo>>> entity = RestTemplateUtils.exchange(url, HttpMethod.POST, null, kmsGrabRulePageVo, reference);

        KmsPageResultVo<KmsGrabRuleVo> kmsPageResultVo = Optional.ofNullable(entity)
                .map(HttpEntity::getBody)
                .map(KmsResultVo::getData)
                .orElse(null);

        if (kmsPageResultVo == null || kmsPageResultVo.getRecords() == null || kmsPageResultVo.getRecords() == 0) {
            PageResult<MdmGrabRuleVo> emptyPage = new PageResult<>();
            emptyPage.setCount(0L);
            emptyPage.setData(new ArrayList<>());
            return emptyPage;
        }

        PageResult<MdmGrabRuleVo> mdmGrabRuleVoPageResult = KmsConverter.INSTANCE.grabRule2Mdm(kmsPageResultVo);
        List<MdmGrabRuleVo> data = mdmGrabRuleVoPageResult.getData();
        for (MdmGrabRuleVo grabRuleVo : data) {
            grabRuleVo.setBsDirectId(bsDirectId);
            grabRuleVo.setBsInvoiceId(mdmGrabRuleVo.getBsInvoiceId());
            grabRuleVo.setDirectSystemId(directSystemId);
            grabRuleVo.setInvoiceId(mdmGrabRuleVo.getInvoiceId());
        }
        return mdmGrabRuleVoPageResult;
    }

    @Override
    public void enable(List<String> ids) {
        Assert.notEmpty(ids, "id集合不能为空");

        KmsGrabRuleVo kmsGrabRuleVo = new KmsGrabRuleVo();
        kmsGrabRuleVo.setIds(ids);
        kmsGrabRuleVo.setComomStatus(ParamConstant.ENABLE);
        kmsGrabRuleVo.setTenantryId(kmsProperties.getTenantryId());
        kmsGrabRuleVo.setUsername(kmsGrabRuleVo.getUsername());
        kmsGrabRuleVo.setUserId(kmsGrabRuleVo.getUserId());

        String url = kmsProperties.getUrl() + "/grabRuleController/modifyGrabRuleStatus?access_token=" + kmsProperties.getAccessToken();

        ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsGrabRuleVo, String.class);
        if (!HttpStatus.OK.equals(entity.getStatusCode())) {
            throw new BusinessException("启用失败");
        }
    }

    @Override
    public void disable(List<String> ids) {
        Assert.notEmpty(ids, "id集合不能为空");

        KmsGrabRuleVo kmsGrabRuleVo = new KmsGrabRuleVo();
        kmsGrabRuleVo.setIds(ids);
        kmsGrabRuleVo.setComomStatus(ParamConstant.DISABLE);
        kmsGrabRuleVo.setTenantryId(kmsProperties.getTenantryId());
        kmsGrabRuleVo.setUsername(kmsGrabRuleVo.getUsername());
        kmsGrabRuleVo.setUserId(kmsGrabRuleVo.getUserId());

        String url = kmsProperties.getUrl() + "/grabRuleController/modifyGrabRuleStatus?access_token=" + kmsProperties.getAccessToken();

        ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsGrabRuleVo, String.class);
        if (!HttpStatus.OK.equals(entity.getStatusCode())) {
            throw new BusinessException("禁用失败");
        }
    }

    @Override
    public void remove(List<String> ids) {
        Assert.notEmpty(ids, "id集合不能为空");

        KmsGrabRuleVo kmsGrabRuleVo = new KmsGrabRuleVo();
        kmsGrabRuleVo.setIds(ids);
        kmsGrabRuleVo.setTenantryId(kmsProperties.getTenantryId());
        kmsGrabRuleVo.setUsername(kmsGrabRuleVo.getUsername());
        kmsGrabRuleVo.setUserId(kmsGrabRuleVo.getUserId());

        String url = kmsProperties.getUrl() + "/grabRuleController/deleteGrabRuleByIds?access_token=" + kmsProperties.getAccessToken();

        ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsGrabRuleVo, String.class);
        if (!HttpStatus.OK.equals(entity.getStatusCode())) {
            throw new BusinessException("删除失败");
        }
    }

    @Override
    public List<MdmRuleParamVo> paramList(MdmGrabRuleVo mdmGrabRuleVo) {
        String bsDirectId = mdmGrabRuleVo.getBsDirectId();
        Assert.hasText(bsDirectId, "后台体系id不能为空");
        String bsInvoiceId = mdmGrabRuleVo.getBsInvoiceId();
        Assert.hasText(bsInvoiceId, "后台单据类型id不能为空");

        ObjectNode paramMap = MapperConfig.instance(kmsProperties);
        paramMap.put(ParamConstant.BS_DIRECT_ID, bsDirectId);
        paramMap.put(ParamConstant.BS_INVOICE_ID, bsInvoiceId);

        String url = kmsProperties.getUrl() + "/ruleParamController/findRuleParamList";

        ParameterizedTypeReference<KmsResultVo<List<MdmRuleParamVo>>> reference = new ParameterizedTypeReference<KmsResultVo<List<MdmRuleParamVo>>>() {
        };

        List<MdmRuleParamVo> mdmRuleParamVos = Optional.of(RestTemplateUtils.exchange(url, HttpMethod.POST, null, paramMap, reference))
                .map(HttpEntity::getBody)
                .map(KmsResultVo::getData)
                .orElse(new ArrayList<>());
        if (CollectionUtils.isEmpty(mdmRuleParamVos)) {
            return new ArrayList<>();
        }
        return mdmRuleParamVos.stream().peek(x -> {
            String isRequired = x.getIsRequired();
            x.setBsFieldId(x.getId());
            if (!StringUtils.isEmpty(isRequired) && ParamConstant.YES.equals(isRequired)) {
                x.setIsRequired(ParamConstant.ONE);
            } else {
                x.setIsRequired(ParamConstant.ZERO);
            }
            x.setFieldEn(convert(x.getFieldEn()));
        }).collect(Collectors.toList());
    }

    @Override
    public List<KmsGrabStoreVo> findGrabStoreList(MdmGrabRuleVo mdmGrabRuleVo) {
        String directSystemId = mdmGrabRuleVo.getDirectSystemId();
        Assert.hasText(directSystemId, "直营体系id不能为空");
        String invoiceId = mdmGrabRuleVo.getInvoiceId();
        Assert.hasText(invoiceId, "单据类型id不能为空");

        ObjectNode map = MapperConfig.instance(kmsProperties);
        map.put(ParamConstant.DIRECT_SYSTEM_ID, directSystemId);
        map.put(ParamConstant.INVOICE_ID, invoiceId);

        String url = kmsProperties.getUrl() + "/ruleStoreController/findStoreGroupListForSave";

        ParameterizedTypeReference<KmsResultVo<List<KmsGrabStoreVo>>> reference = new ParameterizedTypeReference<KmsResultVo<List<KmsGrabStoreVo>>>() {
        };
        List<KmsGrabStoreVo> kmsGrabStoreVos = Optional.of(RestTemplateUtils.exchange(url, HttpMethod.POST, null, map, reference))
                .map(HttpEntity::getBody)
                .map(KmsResultVo::getData)
                .orElse(new ArrayList<>());

        List<KmsGrabStoreVo> collect = kmsGrabStoreVos.stream().filter(distinctByKey(KmsGrabStoreVo::getId)).collect(Collectors.toList());
        for (KmsGrabStoreVo kmsGrabStoreVo : collect) {
            kmsGrabStoreVo.setChildList(distinctId(kmsGrabStoreVo.getChildList()));
        }
//        if (!CollectionUtils.isEmpty(kmsGrabStoreVos)) {
//            for (KmsGrabStoreVo kmsGrabStoreVo : kmsGrabStoreVos) {
//                kmsGrabStoreVo.setChildList(filterSelect(kmsGrabStoreVo.getChildList()));
//            }
//            kmsGrabStoreVos=kmsGrabStoreVos.stream().filter(x-> !x.isForbidden()&&!CollectionUtils.isEmpty(x.getChildList())).collect(Collectors.toList());
//        }
        return collect;
    }

    private List<KmsGrabStoreVo> distinctId(List<KmsGrabStoreVo> childList) {
        if(CollectionUtils.isEmpty(childList)){
            return new ArrayList<>();
        }
        for (KmsGrabStoreVo kmsGrabStoreVo : childList) {
            kmsGrabStoreVo.setChildList(distinctId(kmsGrabStoreVo.getChildList()));
        }
        return childList.stream().filter(distinctByKey(KmsGrabStoreVo::getId)).collect(Collectors.toList());
    }

    public static <T> Predicate<T> distinctByKey(Function<? super T, ?> keyExtractor) {
        Map<Object,Boolean> seen = new ConcurrentHashMap<>();
        return t -> seen.putIfAbsent(keyExtractor.apply(t), Boolean.TRUE) == null;
    }



    private List<KmsGrabStoreVo> filterSelect(List<KmsGrabStoreVo> list) {
        if (CollectionUtils.isEmpty(list)) {
            return new ArrayList<>();
        }
        for (KmsGrabStoreVo kmsGrabStoreVo : list) {
            kmsGrabStoreVo.setChildList(filterSelect(kmsGrabStoreVo.getChildList()));
        }
        return list.stream().filter(x->!x.isForbidden()&&StringUtils.isEmpty(x.getComomStatus())).collect(Collectors.toList());
    }

    @Override
    public void save(MdmGrabRuleVo mdmGrabRuleVo) {
        Assert.hasText(mdmGrabRuleVo.getAccountId(), "客户id不能为空");
        Assert.hasText(mdmGrabRuleVo.getBsDirectId(), "后台直营体系id不能为空");
        Assert.hasText(mdmGrabRuleVo.getBsInvoiceId(), "后台单据类型id不能为空");
        Assert.hasText(mdmGrabRuleVo.getDirectSystemId(), "直营体系id不能为空");
        Assert.hasText(mdmGrabRuleVo.getDirectSystemName(), "直营体系名称不能为空");
        Assert.hasText(mdmGrabRuleVo.getInvoiceId(), "单据类型id不能为空");
        Assert.hasText(mdmGrabRuleVo.getInvoiceName(), "单据类型名称不能为空");

        Assert.hasText(mdmGrabRuleVo.getName(), "规则名称不能为空");

        Assert.isTrue(verifyDateFormat(mdmGrabRuleVo.getStartDate(), DateUtil.DEFAULT_TIME_PATTERN), "开始时间格式不正确");
        Assert.isTrue(verifyDateFormat(mdmGrabRuleVo.getEndDate(), DateUtil.DEFAULT_TIME_PATTERN), "结束时间格式不正确");
        Assert.isTrue(mdmGrabRuleVo.getEndDate().compareTo(mdmGrabRuleVo.getStartDate()) > 0, "开始时间要小于结束时间");

        Assert.hasText(mdmGrabRuleVo.getRequency(), "执行间隔时间不能为空");

        SimpleDateFormat sdf = new SimpleDateFormat(DateUtil.DEFAULT_TIME_PATTERN);
        try {
            int endMinutes = 60*sdf.parse(mdmGrabRuleVo.getEndDate()).getHours()+sdf.parse(mdmGrabRuleVo.getEndDate()).getMinutes();
            int startMinutes = 60*sdf.parse(mdmGrabRuleVo.getStartDate()).getHours()+sdf.parse(mdmGrabRuleVo.getStartDate()).getMinutes();
            Assert.isTrue((endMinutes - startMinutes) > Integer.parseInt(mdmGrabRuleVo.getRequency()), "时间间隔要小于开始时间与结束时间之差");
        } catch (ParseException e) {
            log.error("[单据规则][save]时间转换:{}",e);
        }
        KmsGrabRuleVo kmsGrabRuleVo = KmsConverter.INSTANCE.grabRule2Kms(mdmGrabRuleVo);
        kmsGrabRuleVo.setUserId(kmsProperties.getUserId());
        kmsGrabRuleVo.setUsername(kmsProperties.getUsername());
        kmsGrabRuleVo.setTenantryId(kmsProperties.getTenantryId());
        List<MdmGrabParamVo> params = kmsGrabRuleVo.getParams();
        if(!CollectionUtils.isEmpty(params)){
            params= params.stream().peek(x -> {
                x.setFieldEn(convert(x.getFieldEn()));
                if (ParamConstant.INPUT.equals(x.getFieldClassify())&&x.getOptionName()==null) {
                    x.setOptionName("");
                }
            }).collect(Collectors.toList());
            kmsGrabRuleVo.setParams(params);
        }
        String url = kmsProperties.getUrl() + "/grabRuleController/saveGrabRule?access_token=" + kmsProperties.getAccessToken();

        ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsGrabRuleVo, String.class);

        if (!HttpStatus.OK.equals(entity.getStatusCode())) {
            throw new BusinessException("新增失败");
        }
    }

    @Override
    public void update(MdmGrabRuleVo mdmGrabRuleVo) {
        Assert.hasText(mdmGrabRuleVo.getId(), "id不能为空");
        remove(Collections.singletonList(mdmGrabRuleVo.getId()));
        save(mdmGrabRuleVo);
//        Assert.hasText(mdmGrabRuleVo.getAccountId(), "客户id不能为空");
//        Assert.hasText(mdmGrabRuleVo.getBsDirectId(), "后台直营体系id不能为空");
//        Assert.hasText(mdmGrabRuleVo.getBsInvoiceId(), "后台单据类型id不能为空");
//        Assert.hasText(mdmGrabRuleVo.getDirectSystemId(), "直营体系id不能为空");
//        Assert.hasText(mdmGrabRuleVo.getDirectSystemName(), "直营体系名称不能为空");
//        Assert.hasText(mdmGrabRuleVo.getInvoiceId(), "单据类型id不能为空");
//        Assert.hasText(mdmGrabRuleVo.getInvoiceName(), "单据类型名称不能为空");
//
//        Assert.hasText(mdmGrabRuleVo.getName(), "规则名称不能为空");
//
//        Assert.isTrue(verifyDateFormat(mdmGrabRuleVo.getStartDate(), DateUtil.DEFAULT_TIME_PATTERN), "开始时间格式不正确");
//        Assert.isTrue(verifyDateFormat(mdmGrabRuleVo.getEndDate(), DateUtil.DEFAULT_TIME_PATTERN), "结束时间格式不正确");
//        Assert.isTrue(mdmGrabRuleVo.getEndDate().compareTo(mdmGrabRuleVo.getStartDate()) > 0, "开始时间要小于结束时间");
//
//        Assert.hasText(mdmGrabRuleVo.getRequency(), "执行间隔时间不能为空");
//
//        SimpleDateFormat sdf = new SimpleDateFormat(DateUtil.DEFAULT_TIME_PATTERN);
//        try {
//            int endMinutes = 60*sdf.parse(mdmGrabRuleVo.getEndDate()).getHours()+sdf.parse(mdmGrabRuleVo.getEndDate()).getMinutes();
//            int startMinutes = 60*sdf.parse(mdmGrabRuleVo.getStartDate()).getHours()+sdf.parse(mdmGrabRuleVo.getStartDate()).getMinutes();
//            Assert.isTrue((endMinutes - startMinutes) > Integer.parseInt(mdmGrabRuleVo.getRequency()), "时间间隔要小于开始时间与结束时间之差");
//        } catch (ParseException e) {
//            e.printStackTrace();
//        }
//        KmsGrabRuleVo kmsGrabRuleVo = KmsConverter.INSTANCE.grabRule2Kms(mdmGrabRuleVo);
//        kmsGrabRuleVo.setUserId(kmsProperties.getUserId());
//        kmsGrabRuleVo.setUsername(kmsProperties.getUsername());
//        kmsGrabRuleVo.setTenantryId(kmsProperties.getTenantryId());
//
//        String url = kmsProperties.getUrl() + "/grabRuleController/updateGrabRule?access_token=" + kmsProperties.getAccessToken();
//
//        ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsGrabRuleVo, String.class);
//
//        if (!HttpStatus.OK.equals(entity.getStatusCode())) {
//            throw new BusinessException("修改失败");
//        }
    }

    @Override
    public MdmGrabRuleVo query(MdmGrabRuleVo mdmGrabRuleVo) {
        String id = mdmGrabRuleVo.getId();
        Assert.hasText(id, "规则id不能为空");
        String directSystemId = mdmGrabRuleVo.getDirectSystemId();
        Assert.hasText(directSystemId, "直营体系id不能为空");
        String invoiceId = mdmGrabRuleVo.getInvoiceId();
        Assert.hasText(invoiceId, "单据类型id不能为空");
        String bsDirectId = mdmGrabRuleVo.getBsDirectId();
        Assert.hasText(bsDirectId, "后台直营体系id不能为空");
        String bsInvoiceId = mdmGrabRuleVo.getBsInvoiceId();
        Assert.hasText(bsInvoiceId, "后台单据类型id不能为空");

        ObjectNode paramMap = MapperConfig.instance(kmsProperties);
        paramMap.put(ParamConstant.ID, id);
        paramMap.put(ParamConstant.DIRECT_SYSTEM_ID, directSystemId);
        paramMap.put(ParamConstant.INVOICE_ID, invoiceId);
        paramMap.put(ParamConstant.BS_DIRECT_ID, bsDirectId);
        paramMap.put(ParamConstant.BS_INVOICE_ID, bsInvoiceId);
        paramMap.put(ParamConstant.RULE_ID, id);

        String ruleUrl = kmsProperties.getUrl() + "/grabRuleController/getGrabRuleById";
        ParameterizedTypeReference<KmsResultVo<MdmGrabRuleVo>> ruleReference = new ParameterizedTypeReference<KmsResultVo<MdmGrabRuleVo>>() {
        };


        MdmGrabRuleVo result = Optional.of(RestTemplateUtils.exchange(ruleUrl, HttpMethod.POST, null, paramMap, ruleReference))
                .map(HttpEntity::getBody)
                .map(KmsResultVo::getData)
                .orElse(null);
        if (result == null) {
            return null;
        }
        List<KmsStoreVo> storeIdList = new ArrayList<>();
        List<KmsGrabStoreVo> grabStoreList = findGrabStoreList(mdmGrabRuleVo);
        if(!CollectionUtils.isEmpty(grabStoreList)){
            String storeUrl = kmsProperties.getUrl() + "/ruleStoreController/findRuleStoreList";
            ParameterizedTypeReference<KmsResultVo<KmsPageResultVo<KmsStoreRespVo>>> storeReference = new ParameterizedTypeReference<KmsResultVo<KmsPageResultVo<KmsStoreRespVo>>>() {
            };
            KmsGrabRulePageVo kmsGrabRulePageVo = new KmsGrabRulePageVo();
            kmsGrabRulePageVo.setPage(new KmsPageVo(1, -1));
            KmsGrabRuleVo kmsGrabRuleVo = new KmsGrabRuleVo();
            kmsGrabRuleVo.setBsDirectId(bsDirectId);
            kmsGrabRuleVo.setBsInvoiceId(bsInvoiceId);
            kmsGrabRuleVo.setTenantryId(kmsProperties.getTenantryId());
            kmsGrabRuleVo.setRuleId(id);
            kmsGrabRulePageVo.setReqVo(kmsGrabRuleVo);
            kmsGrabRulePageVo.setUsername(kmsProperties.getUsername());
            kmsGrabRulePageVo.setUserId(kmsProperties.getUserId());
            kmsGrabRulePageVo.setTenantryId(kmsProperties.getTenantryId());
            List<KmsStoreRespVo> kmsStoreRespVos = Optional.of(RestTemplateUtils.exchange(storeUrl, HttpMethod.POST, null, kmsGrabRulePageVo, storeReference))
                    .map(HttpEntity::getBody)
                    .map(KmsResultVo::getData)
                    .map(KmsPageResultVo::getRows)
                    .orElse(new ArrayList<>());
            List<String> collect = kmsStoreRespVos.stream()
                    .filter(Objects::nonNull)
                    .filter(x -> !StringUtils.isEmpty(x.getStoreName()))
                    .map(KmsStoreRespVo::getStoreName)
                    .collect(Collectors.toList());

            if(!CollectionUtils.isEmpty(collect)){
                while (!CollectionUtils.isEmpty(grabStoreList)) {
                    List<String> idList = grabStoreList.stream()
                            .filter(x -> !StringUtils.isEmpty(x.getName()))
                            .filter(x -> collect.contains(x.getName()))
                            .map(KmsGrabStoreVo::getId)
                            .collect(Collectors.toList());
                    if(!CollectionUtils.isEmpty(idList)){

                        storeIdList.addAll(idList.stream().map(KmsStoreVo::new).collect(Collectors.toList()));
                    }
                    grabStoreList = grabStoreList.stream()
                            .filter(x -> !CollectionUtils.isEmpty(x.getChildList()))
                            .flatMap(x -> x.getChildList().stream())
                            .collect(Collectors.toList());
                }
            }
        }
        result.setStores(storeIdList);

        List<MdmRuleParamVo> mdmRuleParamVos = paramList(mdmGrabRuleVo);
        List<MdmGrabParamVo> params = new ArrayList<>();
        if(!CollectionUtils.isEmpty(mdmRuleParamVos)){
            String ruleParamUrl = kmsProperties.getUrl() + "/ruleParamController/findRuleParamDetailList";

            ParameterizedTypeReference<KmsResultVo<List<KmsRuleParamRespVo>>> paramReference=new ParameterizedTypeReference<KmsResultVo<List<KmsRuleParamRespVo>>>() {
            };

            ResponseEntity<KmsResultVo<List<KmsRuleParamRespVo>>> paramEntity = RestTemplateUtils.exchange(ruleParamUrl, HttpMethod.POST, null, paramMap, paramReference);
            List<KmsRuleParamRespVo> kmsRuleParamRespVos = Optional.of(paramEntity)
                    .map(HttpEntity::getBody)
                    .map(KmsResultVo::getData)
                    .orElse(new ArrayList<>());
            if(!CollectionUtils.isEmpty(kmsRuleParamRespVos)){
                Map<String, String> fieldMap = kmsRuleParamRespVos.stream().collect(Collectors.toMap(KmsRuleParamRespVo::getOptionName, KmsRuleParamRespVo::getFieldCn));

                for (MdmRuleParamVo mdmRuleParamVo : mdmRuleParamVos) {
                    String fieldCn = mdmRuleParamVo.getFieldCn();
                    List<MdmRuleParamFieldVo> options = mdmRuleParamVo.getOptions();
                    if(!CollectionUtils.isEmpty(options)){
                        for (MdmRuleParamFieldVo option : options) {
                            String optionName = option.getOptionName();
                            if(!StringUtils.isEmpty(optionName)){
                                String s = fieldMap.get(optionName);
                                if (!StringUtils.isEmpty(s) && s.equals(fieldCn)) {
                                    MdmGrabParamVo mdmGrabParamVo = new MdmGrabParamVo();
                                    mdmGrabParamVo.setFieldClassify(mdmGrabParamVo.getFieldClassify());
                                    mdmGrabParamVo.setFieldCn(mdmGrabParamVo.getFieldCn());
                                    mdmGrabParamVo.setFieldEn(mdmGrabParamVo.getFieldEn());
                                    mdmGrabParamVo.setOptionName(option.getOptionName());
                                    mdmGrabParamVo.setBsFieldId(option.getFieldId());

                                    mdmRuleParamVo.setValue(option.getOptionValue());
                                    params.add(mdmGrabParamVo);
                                }
                            }
                        }
                    }else{
                        for (KmsRuleParamRespVo kmsRuleParamRespVo : kmsRuleParamRespVos) {
                            if (kmsRuleParamRespVo.getFieldCn().equals(mdmRuleParamVo.getFieldCn())) {
                                mdmRuleParamVo.setValue(kmsRuleParamRespVo.getOptionName());
                            }
                        }
                    }
                }
            }
        }
        result.setParams(params);
        result.setMdmRuleParamVos(mdmRuleParamVos);
        result.setDirectSystemId(mdmGrabRuleVo.getDirectSystemId());
        result.setBsDirectId(mdmGrabRuleVo.getBsDirectId());
        result.setInvoiceId(mdmGrabRuleVo.getInvoiceId());
        result.setBsInvoiceId(mdmGrabRuleVo.getBsInvoiceId());
        return result;
    }

    private boolean verifyDateFormat(String dataStr, String formatStr) {
        boolean flag = false;
        if (com.biz.crm.util.StringUtils.isNotEmpty(dataStr) && com.biz.crm.util.StringUtils.isNotEmpty(formatStr)) {
            try {
                SimpleDateFormat sdf = new SimpleDateFormat(formatStr);
                Date parse = sdf.parse(dataStr);
                String dateFormatStr = sdf.format(parse);
                if (dataStr.equals(dateFormatStr)) {
                    flag = true;
                }
            } catch (Exception e) {

            }
        }
        return flag;
    }

}
