package com.biz.crm.kms.service.impl;

import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.resttemplate.RestTemplateUtils;
import com.biz.crm.kms.config.KmsProperties;
import com.biz.crm.kms.manager.KmsConverter;
import com.biz.crm.kms.service.MdmShelfProductService;
import com.biz.crm.nebular.mdm.kms.MdmProductUnitVo;
import com.biz.crm.nebular.mdm.kms.MdmShelfProductVo;
import com.biz.crm.nebular.mdm.kms.api.*;
import org.redisson.misc.Hash;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author zxw
 * @date 2021-02-22 18:25
 **/
@Service
@ConditionalOnMissingBean(name = "MdmShelfProductServiceExpandImpl")
public class MdmShelfProductServiceImpl implements MdmShelfProductService {

    @Resource
    private KmsProperties kmsProperties;

    @Override
    public List<MdmProductUnitVo> findProductUnit(MdmShelfProductVo mdmShelfProductVo) {
        String proDetailsNumber = mdmShelfProductVo.getProDetailsNumber();
        Assert.hasText(proDetailsNumber, "企业商品编码不能为空");
        KmsProductVo kmsProductVo = new KmsProductVo();
        kmsProductVo.setProDetailsNumber(proDetailsNumber);
        kmsProductVo.setTenantryId(kmsProperties.getTenantryId());
        kmsProductVo.setUserId(kmsProperties.getUserId());
        kmsProductVo.setUsername(kmsProperties.getUsername());

        String url = kmsProperties.getUrl() + "/shelfProductController/getProductUnitList";

        ParameterizedTypeReference<KmsResultVo<List<KmsProductUnitVo>>> reference = new ParameterizedTypeReference<KmsResultVo<List<KmsProductUnitVo>>>() {
        };

        ResponseEntity<KmsResultVo<List<KmsProductUnitVo>>> entity = RestTemplateUtils.exchange(url, HttpMethod.POST, null, kmsProductVo, reference);
        List<KmsProductUnitVo> kmsProductUnitVos = Optional.ofNullable(entity)
                .map(HttpEntity::getBody)
                .map(KmsResultVo::getData)
                .orElse(new ArrayList<>());
        if (CollectionUtils.isEmpty(kmsProductUnitVos)) {
            return new ArrayList<>();
        }
        return KmsConverter.INSTANCE.productUnit2Mdm(kmsProductUnitVos);
    }

    @Override
    public MdmShelfProductVo query(MdmShelfProductVo mdmShelfProductVo) {
        String id = mdmShelfProductVo.getId();
        Assert.hasText(id, "不能为空");
        Map<String, String> paramMap = new HashMap<String, String>();
        paramMap.put("id", id);
        paramMap.put("tenantryId", kmsProperties.getTenantryId());
        paramMap.put("userId", kmsProperties.getUserId());
        paramMap.put("username", kmsProperties.getUsername());

        String url = kmsProperties.getUrl() + "/shelfProductController/getShelfProductById";

        ParameterizedTypeReference<KmsResultVo<KmsProductVo>> reference=new ParameterizedTypeReference<KmsResultVo<KmsProductVo>>() {
        };

        ResponseEntity<KmsResultVo<KmsProductVo>> entity = RestTemplateUtils.exchange(url, HttpMethod.POST, null, paramMap, reference);
        MdmShelfProductVo result = Optional.ofNullable(entity)
                .map(HttpEntity::getBody)
                .map(KmsResultVo::getData)
                .map(KmsConverter.INSTANCE::product2Mdm)
                .orElse(new MdmShelfProductVo());
        result.setId(id);
        result.setSellPartyCode(mdmShelfProductVo.getSellPartyCode());
        result.setDirectSystemId(mdmShelfProductVo.getDirectSystemId());
        result.setDirectSystemName(mdmShelfProductVo.getDirectSystemName());
        return result;
    }

    @Override
    public List<MdmShelfProductVo> findProductList(MdmShelfProductVo mdmShelfProductVo) {
        String sellPartyId = mdmShelfProductVo.getSellPartyId();
        Assert.hasText(sellPartyId, "售达方id不能为空");
        String directSystemId = mdmShelfProductVo.getDirectSystemId();
        Assert.hasText(directSystemId, "体系id不能为空");

        KmsProductVo kmsProductVo = new KmsProductVo();
        kmsProductVo.setUsername(kmsProperties.getUsername());
        kmsProductVo.setUserId(kmsProperties.getUserId());
        kmsProductVo.setTenantryId(kmsProperties.getTenantryId());
        kmsProductVo.setDirectSystemId(directSystemId);
        kmsProductVo.setSellPartyId(sellPartyId);

        String url = kmsProperties.getUrl() + "/shelfProductController/getProductList";
        ParameterizedTypeReference<KmsResultVo<List<KmsProductVo>>> reference = new ParameterizedTypeReference<KmsResultVo<List<KmsProductVo>>>() {
        };
        ResponseEntity<KmsResultVo<List<KmsProductVo>>> entity = RestTemplateUtils.exchange(url, HttpMethod.POST, null, kmsProductVo, reference);
        List<KmsProductVo> kmsProductVos = Optional.ofNullable(entity)
                .map(HttpEntity::getBody)
                .map(KmsResultVo::getData)
                .orElse(null);
        if (CollectionUtils.isEmpty(kmsProductVos)) {
            return new ArrayList<>();
        }
        String proDetailsName = mdmShelfProductVo.getProDetailsName();
        List<MdmShelfProductVo> mdmShelfProductVos = KmsConverter.INSTANCE.product2Mdm(kmsProductVos);
        return mdmShelfProductVos.stream()
                .filter(x -> StringUtils.isEmpty(proDetailsName) || x.getProDetailsName().contains(proDetailsName))
                .collect(Collectors.toList());
    }

    @Override
    public void save(MdmShelfProductVo mdmShelfProductVo) {
        Assert.hasText(mdmShelfProductVo.getDirectSystemId(), "直营体系id不能为空");
        Assert.hasText(mdmShelfProductVo.getDirectSystemName(), "直营体系名称不能为空");
        Assert.hasText(mdmShelfProductVo.getOrderUnitId(), "订货单位id不能为空");
        Assert.hasText(mdmShelfProductVo.getOrderUnit(), "订货单位名称不能为空");
        Assert.hasText(mdmShelfProductVo.getProDetailsNumber(), "商品编号不能为空");
        Assert.hasText(mdmShelfProductVo.getProDetailsName(), "商品名称不能为空");
        Assert.hasText(mdmShelfProductVo.getSellPartyId(), "售达方id不能为空");
        Assert.hasText(mdmShelfProductVo.getStoreProductCode(), "门店商品编码不能为空");
        KmsProductVo kmsProductVo = KmsConverter.INSTANCE.product2Kms(mdmShelfProductVo);
        kmsProductVo.setUsername(kmsProperties.getUsername());
        kmsProductVo.setUserId(kmsProperties.getUserId());
        kmsProductVo.setTenantryId(kmsProperties.getTenantryId());
        kmsProductVo.setDirectSystemName(mdmShelfProductVo.getDirectSystemName());

        String url = kmsProperties.getUrl() + "/shelfProductController/saveShelfProduct?access_token=" + kmsProperties.getAccessToken();

        ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsProductVo, String.class);
        if (!HttpStatus.OK.equals(entity.getStatusCode())) {
            throw new BusinessException("新增失败");
        }
    }

    @Override
    public void remove(List<String> ids) {
        Assert.notEmpty(ids, "id集合不能为空");
        KmsProductVo kmsProductVo = new KmsProductVo();
        kmsProductVo.setUsername(kmsProperties.getUsername());
        kmsProductVo.setUserId(kmsProperties.getUserId());
        kmsProductVo.setTenantryId(kmsProperties.getTenantryId());
        kmsProductVo.setIds(ids);

        String url = kmsProperties.getUrl() + "/shelfProductController/deleteShelfProductByIds?access_token=" + kmsProperties.getAccessToken();

        ResponseEntity<String> entity = RestTemplateUtils.postForEntity(url, null, kmsProductVo, String.class);
        if (!HttpStatus.OK.equals(entity.getStatusCode())) {
            throw new BusinessException("删除失败");
        }
    }

    @Override
    public PageResult<MdmShelfProductVo> page(MdmShelfProductVo mdmShelfProductVo) {
        String directSystemId = mdmShelfProductVo.getDirectSystemId();
        Assert.hasText(directSystemId, "体系id不能为空");

        KmsProductVo kmsProductVo = new KmsProductVo();
        kmsProductVo.setDirectSystemId(mdmShelfProductVo.getDirectSystemId());
        kmsProductVo.setTenantryId(kmsProperties.getTenantryId());
        kmsProductVo.setSearchValue(mdmShelfProductVo.getProDetailsName());

        KmsProductPageReqVo kmsProductPageReqVo = new KmsProductPageReqVo();
        kmsProductPageReqVo.setUserId(kmsProperties.getUserId());
        kmsProductPageReqVo.setUsername(kmsProperties.getUsername());
        kmsProductPageReqVo.setTenantryId(kmsProperties.getTenantryId());

        String url = kmsProperties.getUrl() + "/shelfProductController/findShelfProductListByPage";

        ParameterizedTypeReference<KmsResultVo<KmsResultBodyVo<KmsPageResultVo<KmsProductVo>>>> reference =
                new ParameterizedTypeReference<KmsResultVo<KmsResultBodyVo<KmsPageResultVo<KmsProductVo>>>>() {
                };

        Integer current = Optional.ofNullable(mdmShelfProductVo.getPageNum()).orElse(1);
        Integer size = Optional.ofNullable(mdmShelfProductVo.getPageSize()).orElse(25);
        KmsPageVo kmsPageVo = new KmsPageVo(current, size);
        kmsProductPageReqVo.setPage(kmsPageVo);
        kmsProductPageReqVo.setReqVo(kmsProductVo);


        ResponseEntity<KmsResultVo<KmsResultBodyVo<KmsPageResultVo<KmsProductVo>>>> entity = RestTemplateUtils.exchange(url, HttpMethod.POST, null, kmsProductPageReqVo, reference);

        KmsPageResultVo<KmsProductVo> pageResultVo = Optional.ofNullable(entity)
                .map(HttpEntity::getBody)
                .map(KmsResultVo::getData)
                .map(KmsResultBodyVo::getBody)
                .orElse(null);

        if (pageResultVo == null || pageResultVo.getRecords() == null || pageResultVo.getRecords() == 0) {

            PageResult<MdmShelfProductVo> emptyPage = new PageResult<>();
            emptyPage.setCount(0L);
            emptyPage.setData(new ArrayList<>());
            return emptyPage;
        }

        PageResult<MdmShelfProductVo> voPageResult = KmsConverter.INSTANCE.product2Mdm(pageResultVo);

        if (voPageResult != null) {
            List<MdmShelfProductVo> data = voPageResult.getData();
            if (!CollectionUtils.isEmpty(data)) {
                data.forEach(x -> {
                    x.setDirectSystemId(mdmShelfProductVo.getDirectSystemId());
                    x.setDirectSystemName(mdmShelfProductVo.getDirectSystemName());
                });
            }
        }
        return voPageResult;
    }
}
