package com.biz.crm.login.controller;

import com.biz.crm.aop.CrmGlobalLog;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.mdm.LoginFromTypeEnum;
import com.biz.crm.login.service.MdmSfaAppletLoginService;
import com.biz.crm.nebular.mdm.login.*;
import com.biz.crm.util.Result;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.weixin.OpenIdUtil;
import com.biz.crm.util.weixin.OpenIdVo;
import com.biz.crm.utils.LoginAesUtil;
import com.biz.crm.utils.LoginEncryptUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;

@RestController
@RequestMapping("/mdmSfaAppletLoginController")
@Api(tags = "MDM-登录登出-SFA微信小程序")
@CrmGlobalLog
@Slf4j
public class MdmSfaAppletLoginController {

    @Autowired
    private MdmSfaAppletLoginService mdmSfaAppletLoginService;

    @ApiOperation("登录（账号+密码+openId）")
    @PostMapping("/login")
    public Result<MdmLoginRespVo> login(@RequestBody MdmAppletLoginEncryptReqVo reqVo) {
        return Result.ok(mdmSfaAppletLoginService.login(revert(reqVo)));
    }

    @ApiOperation("登录（手机号+密码+openId）")
    @PostMapping("/loginByPhone")
    public Result<MdmLoginRespVo> loginByPhone(@RequestBody MdmAppletLoginEncryptReqVo reqVo) {
        return Result.ok(mdmSfaAppletLoginService.loginByPhone(revert(reqVo)));
    }

    @ApiOperation("登录（邮箱+密码+openId）")
    @PostMapping("/loginByEmail")
    public Result<MdmLoginRespVo> loginByEmail(@RequestBody MdmAppletLoginEncryptReqVo reqVo) {
        return Result.ok(mdmSfaAppletLoginService.loginByEmail(revert(reqVo)));
    }

    @ApiOperation("登录（手机号+验证码+openId）")
    @PostMapping("/loginByPhoneVerification")
    public Result<MdmLoginRespVo> loginByPhoneVerification(@RequestBody MdmAppletLoginEncryptReqVo reqVo) {
        return Result.ok(mdmSfaAppletLoginService.loginByPhoneVerification(revert(reqVo)));
    }

    @ApiOperation("登录（邮箱+验证码+openId）")
    @PostMapping("/loginByEmailVerification")
    public Result<MdmLoginRespVo> loginByEmailVerification(@RequestBody MdmAppletLoginEncryptReqVo reqVo) {
        return Result.ok(mdmSfaAppletLoginService.loginByEmailVerification(revert(reqVo)));
    }

    @ApiOperation("发送手机号登录验证码")
    @PostMapping("/sendVerificationForLoginByMessage")
    public Result sendVerificationForLoginByMessage(@RequestBody MdmAppletLoginEncryptReqVo reqVo) {
        mdmSfaAppletLoginService.sendVerificationForLoginByMessage(revert(reqVo));
        return Result.ok();
    }

    @ApiOperation("发送邮箱登录验证码")
    @PostMapping("/sendVerificationForLoginByEmail")
    public Result sendVerificationForLoginByEmail(@RequestBody MdmAppletLoginEncryptReqVo reqVo) {
        mdmSfaAppletLoginService.sendVerificationForLoginByEmail(revert(reqVo));
        return Result.ok();
    }

    @ApiOperation("免密登录（通过微信登录凭证）并且获得openId")
    @PostMapping("/loginByCode")
    public Result<MdmLoginRespVo> loginByCode(@RequestBody MdmAppletLoginEncryptReqVo reqVo) {
        MdmLoginRespVo mdmLoginRespVo = new MdmLoginRespVo();
        mdmLoginRespVo.setAppletOpenIdLoginSuccessFlag(YesNoEnum.yesNoEnum.ONE.getValue());
        try {
            String code = reqVo.getCxfyzzpcxaqcwmai();
            Assert.hasText(code, "缺失微信登录凭证");
            OpenIdVo openIdVo = OpenIdUtil.getSfaAppletOpenId(code);
            Assert.notNull(openIdVo, "获取openId失败");
            Assert.hasText(openIdVo.getOpenid(), "获取openId失败");
            mdmLoginRespVo.setOpenId(openIdVo.getOpenid());
            MdmAppletLoginReqVo revert = revert(reqVo);
            revert.setOpenId(openIdVo.getOpenid());
            mdmLoginRespVo = mdmSfaAppletLoginService.loginByOpenId(revert);
        } catch (Exception e) {
            e.printStackTrace();
            log.error("免密登录（通过微信登录凭证）失败：{}", e);
            mdmLoginRespVo.setAppletOpenIdLoginSuccessFlag(YesNoEnum.yesNoEnum.ZERO.getValue());
        }
        return Result.ok(mdmLoginRespVo);
    }

    @ApiOperation("（不安全）免密登录（通过openId）")
    @PostMapping("/loginByOpenId")
    public Result<MdmLoginRespVo> loginByOpenId(@RequestBody MdmAppletLoginEncryptReqVo reqVo) {
        MdmLoginRespVo mdmLoginRespVo = null;
        try {
            mdmLoginRespVo = mdmSfaAppletLoginService.loginByOpenId(revert(reqVo));
            mdmLoginRespVo.setAppletOpenIdLoginSuccessFlag(YesNoEnum.yesNoEnum.ONE.getValue());
        } catch (Exception e) {
            e.printStackTrace();
            log.error("免密登录（通过openId）失败：{}", e);
            mdmLoginRespVo = new MdmLoginRespVo();
            mdmLoginRespVo.setAppletOpenIdLoginSuccessFlag(YesNoEnum.yesNoEnum.ZERO.getValue());
        }
        return Result.ok(mdmLoginRespVo);
    }

    @ApiOperation("发送重置密码验证码（手机号）")
    @PostMapping("/sendVerificationForLoginAndResetByMessage")
    public Result sendVerificationForLoginAndResetByMessage(@RequestBody MdmAppletLoginEncryptReqVo reqVo) {
        mdmSfaAppletLoginService.sendVerificationForLoginAndResetByMessage(revert(reqVo));
        return Result.ok();
    }

    @ApiOperation("发送重置密码验证码（邮箱）")
    @PostMapping("/sendVerificationForLoginAndResetByEmail")
    public Result sendVerificationForLoginAndResetByEmail(@RequestBody MdmAppletLoginEncryptReqVo reqVo) {
        mdmSfaAppletLoginService.sendVerificationForLoginAndResetByEmail(revert(reqVo));
        return Result.ok();
    }

    @ApiOperation("校验手机号是否存在且可用")
    @PostMapping("/checkPhoneExistAndUsable")
    public Result checkPhoneExistAndUsable(@RequestBody MdmAppletLoginEncryptReqVo reqVo) {
        mdmSfaAppletLoginService.checkPhoneExistAndUsable(revert(reqVo));
        return Result.ok();
    }

    @ApiOperation("校验邮箱是否存在且可用")
    @PostMapping("/checkEmailExistAndUsable")
    public Result checkEmailExistAndUsable(@RequestBody MdmAppletLoginEncryptReqVo reqVo) {
        mdmSfaAppletLoginService.checkEmailExistAndUsable(revert(reqVo));
        return Result.ok();
    }

    @ApiOperation("验证重置密码验证码并登录（手机号+验证码）")
    @PostMapping("/loginAndResetByPhoneVerification")
    public Result<MdmLoginRespVo> loginAndResetByPhoneVerification(@RequestBody MdmAppletLoginEncryptReqVo reqVo) {
        return Result.ok(mdmSfaAppletLoginService.loginAndResetByPhoneVerification(revert(reqVo)));
    }

    @ApiOperation("验证重置密码验证码并登录（邮箱+验证码）")
    @PostMapping("/loginAndResetByEmailVerification")
    public Result<MdmLoginRespVo> loginAndResetByEmailVerification(@RequestBody MdmAppletLoginEncryptReqVo reqVo) {
        return Result.ok(mdmSfaAppletLoginService.loginAndResetByEmailVerification(revert(reqVo)));
    }

    @ApiOperation("获取登录方式的控制参数")
    @PostMapping("/getLoginTypeControlConfig")
    public Result<MdmLoginTypeControlVo> getLoginTypeControlConfig() {
        return Result.ok(mdmSfaAppletLoginService.getLoginTypeControlConfig());
    }

    @ApiOperation("退出登录（退出当前会话）")
    @PostMapping("/logout")
    public Result logout() {
        mdmSfaAppletLoginService.logout();
        return Result.ok();
    }

    @ApiOperation("退出登录（退出当前会话账号在SFA微信小程序的全部会话）")
    @PostMapping("/logoutFromType")
    public Result logoutFromType() {
        mdmSfaAppletLoginService.logoutFromType();
        return Result.ok();
    }

    @ApiOperation("退出登录（退出当前会话账号在所有平台的全部会话）")
    @PostMapping("/logoutAll")
    public Result logoutAll() {
        mdmSfaAppletLoginService.logoutAll();
        return Result.ok();
    }

    @ApiOperation(value = "根据微信登录凭证获取sfa小程序openId", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "code", value = "微信登录凭证", required = true, dataType = "String", paramType = "query")
    })
    @GetMapping("/getSfaAppletOpenIdByCode")
    public Result<MdmAppletLoginParamRespVo> getSfaAppletOpenIdByCode(@RequestParam(value = "code", required = true) String code) {
        OpenIdVo openIdVo = OpenIdUtil.getSfaAppletOpenId(code);
        Assert.notNull(openIdVo, "获取openId失败");
        Assert.hasText(openIdVo.getOpenid(), "获取openId失败");
        MdmAppletLoginParamRespVo respVo = new MdmAppletLoginParamRespVo();
        respVo.setOpenId(openIdVo.getOpenid());
        return Result.ok(respVo);
    }

    /**
     * 把加密参数转换成已有参数
     *
     * @param reqVo
     * @return
     */
    private MdmAppletLoginReqVo revert(MdmAppletLoginEncryptReqVo reqVo) {
        ThreadLocalUtil.delObj();
        MdmAppletLoginReqVo loginReqVo = new MdmAppletLoginReqVo();
        if (StringUtils.isNotEmpty(reqVo.getUjlrwebjruzddjnu())) {
            loginReqVo.setUserName(reqVo.getUjlrwebjruzddjnu());
        }
        if (StringUtils.isNotEmpty(reqVo.getPthxivgijazekrss())) {
            loginReqVo.setPassword(LoginAesUtil.decrypt(reqVo.getPthxivgijazekrss()));
        } else if (StringUtils.isNotEmpty(reqVo.getPazjofxkyvyoosot())) {
            loginReqVo.setPassword(LoginEncryptUtil.decrypt(reqVo.getPazjofxkyvyoosot()));
        } else if (StringUtils.isNotEmpty(reqVo.getPenijmgsbybkfjvr())) {
            loginReqVo.setPassword(reqVo.getPenijmgsbybkfjvr());
        } else {

        }
        if (StringUtils.isNotEmpty(reqVo.getTwpbmjhfxodboggf())) {
            loginReqVo.setUserPhone(reqVo.getTwpbmjhfxodboggf());
        }
        if (StringUtils.isNotEmpty(reqVo.getExzvsxwhvcvcctyk())) {
            loginReqVo.setEmail(reqVo.getExzvsxwhvcvcctyk());
        }
        if (StringUtils.isNotEmpty(reqVo.getVfymxygsefoeapvk())) {
            loginReqVo.setVerificationCode(reqVo.getVfymxygsefoeapvk());
        }
        if (StringUtils.isNotEmpty(reqVo.getOictwizfeumvvukh())) {
            loginReqVo.setOpenId(reqVo.getOictwizfeumvvukh());
        }
        if (StringUtils.isNotEmpty(reqVo.getWhspxyaumrkpgppm())) {
            loginReqVo.setHeadImgUrl(reqVo.getWhspxyaumrkpgppm());
        }
        if (StringUtils.isNotEmpty(reqVo.getWntyjilgzehqkbmy())) {
            loginReqVo.setNickName(reqVo.getWntyjilgzehqkbmy());
        }
        loginReqVo.setFromType(LoginFromTypeEnum.APPLET_SFA.getValue());
        return loginReqVo;
    }

}
