package com.biz.crm.material.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.material.mapper.MdmMaterialGroupMapper;
import com.biz.crm.material.mapper.MdmMaterialGroupRelMapper;
import com.biz.crm.material.mapper.MdmMaterialMapper;
import com.biz.crm.material.model.MdmMaterialEntity;
import com.biz.crm.material.model.MdmMaterialGroupEntity;
import com.biz.crm.material.model.MdmMaterialGroupRelEntity;
import com.biz.crm.material.model.MdmMaterialMediaEntity;
import com.biz.crm.material.service.MdmMaterialGroupService;
import com.biz.crm.material.service.MdmMaterialMediaService;
import com.biz.crm.material.service.MdmMaterialService;
import com.biz.crm.nebular.mdm.material.MdmMaterialMediaReqVo;
import com.biz.crm.nebular.mdm.material.MdmMaterialMediaRespVo;
import com.biz.crm.nebular.mdm.material.MdmMaterialReqVo;
import com.biz.crm.nebular.mdm.material.MdmMaterialRespVo;
import com.biz.crm.nebular.mdm.materialGroup.MdmMaterialGroupReqVo;
import com.biz.crm.nebular.mdm.materialGroup.MdmMaterialGroupRespVo;
import com.biz.crm.nebular.mdm.productlevel.resp.MdmProductLevelRespVo;
import com.biz.crm.nebular.mdm.productlevel.resp.MdmProductLevelVo;
import com.biz.crm.product.model.MdmProductEntity;
import com.biz.crm.product.model.MdmProductMaterialEntity;
import com.biz.crm.productlevel.service.MdmProductLevelService;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.PageUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.ValidateUtils;
import io.jsonwebtoken.lang.Assert;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * @ClassName: MdmMaterialGroupServiceImpl
 * @Author: yangfan
 * @Description: 物料组实现
 * @Date: 2021/8/19 11:34
 * @Version: 1.0
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmMaterialGroupServiceExpandImpl")
public class MdmMaterialGroupServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmMaterialGroupMapper, MdmMaterialGroupEntity> implements MdmMaterialGroupService {

  @Autowired
  private MdmMaterialGroupMapper materialGroupMapper;
  @Autowired
  private MdmMaterialService mdmMaterialService;
  @Autowired
  private MdmMaterialGroupRelMapper groupRelMapper;
  @Autowired
  private CrmLogSendUtil crmLogSendUtil;
  @Autowired
  private MdmProductLevelService mdmProductLevelService;
  @Autowired
  private MdmMaterialMediaService mdmMaterialMediaService;




  @Override
  public PageResult<MdmMaterialGroupRespVo> findList(MdmMaterialGroupReqVo groupReqVo) {
    Page<MdmMaterialGroupRespVo> page = PageUtil.buildPage(groupReqVo.getPageNum(), groupReqVo.getPageSize());
    List<MdmMaterialGroupRespVo> list = materialGroupMapper.findList(page, groupReqVo);
    return PageResult.<MdmMaterialGroupRespVo>builder()
        .data(list)
        .count(page.getTotal())
        .build();
  }

  @Override
  public MdmMaterialGroupRespVo detail(String id, String materialGroupCode) {
    MdmMaterialGroupRespVo respVo = new MdmMaterialGroupRespVo();
    if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(materialGroupCode)) {
      MdmMaterialGroupEntity one = this.lambdaQuery()
          .eq(StringUtils.isNotEmpty(id), MdmMaterialGroupEntity::getId, id)
          .eq(StringUtils.isNotEmpty(materialGroupCode), MdmMaterialGroupEntity::getMaterialGroupCode, materialGroupCode)
          .one();
      if (one != null) {

        // 对象转换
        respVo = CrmBeanUtil.copy(one, MdmMaterialGroupRespVo.class);
        // 获取物料信息
        List<MdmMaterialRespVo> materialList = mdmMaterialService.findMaterialByMaterialGroupCode(one.getMaterialGroupCode());
        respVo.setMaterialList(materialList);
      }
    }
    return respVo;
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void save(MdmMaterialGroupReqVo groupReqVo) {

    // 校验字段
    this.FieldValidation(groupReqVo);

    // 查询物料信息
    MdmMaterialGroupEntity materialGroupEntity = this.getOne(
        Wrappers.lambdaQuery(MdmMaterialGroupEntity.class).eq(MdmMaterialGroupEntity::getMaterialGroupCode, groupReqVo.getMaterialGroupCode()).eq(MdmMaterialGroupEntity::getEnableStatus, "009"));
    if(materialGroupEntity!=null){
      throw new BusinessException("此物料组编码"+groupReqVo.getMaterialGroupCode()+"已存在");
    }

    // 新增
    MdmMaterialGroupEntity entity = CrmBeanUtil.copy(groupReqVo, MdmMaterialGroupEntity.class);
    List<MdmMaterialReqVo> materialList = groupReqVo.getMaterialList();
    entity.setMaterialNum(materialList==null?0:materialList.size());
    this.save(entity);
    // 保存与物料的关联信息
    if (CollectionUtil.listNotEmpty(materialList)) {
      materialList.forEach(item -> {
        MdmMaterialGroupRelEntity relEntity = CrmBeanUtil.copy(item, MdmMaterialGroupRelEntity.class);
        relEntity.setMaterialGroupCode(entity.getMaterialGroupCode());
        groupRelMapper.insert(relEntity);
      });
    }
    Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
    if (menuObject != null) {
      crmLogSendUtil.sendForAdd(menuObject.toString(), entity.getId(), entity.getMaterialGroupCode(), CrmBeanUtil.copy(entity, MdmMaterialGroupReqVo.class));
    }
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void update(MdmMaterialGroupReqVo groupReqVo) {
    // 校验字段
    this.FieldValidation(groupReqVo);

    MdmMaterialGroupEntity entity = this.getById(groupReqVo.getId());
    Assert.notNull(entity, "无效的id");
    MdmMaterialReqVo oldObject = CrmBeanUtil.copy(entity, MdmMaterialReqVo.class);
    CrmBeanUtil.copyProperties(groupReqVo, entity);
    List<MdmMaterialReqVo> materialList = groupReqVo.getMaterialList();
    entity.setMaterialNum(materialList==null?0:materialList.size());
    this.updateById(entity);

    // 删除物料关系信息
    groupRelMapper.delete(Wrappers.lambdaQuery(MdmMaterialGroupRelEntity.class).eq(MdmMaterialGroupRelEntity::getMaterialGroupCode, entity.getMaterialGroupCode()));

    // 保存与物料的关联信息
    if (CollectionUtil.listNotEmpty(materialList)) {
      materialList.forEach(item -> {
        MdmMaterialGroupRelEntity relEntity = CrmBeanUtil.copy(item, MdmMaterialGroupRelEntity.class);
        relEntity.setMaterialGroupCode(entity.getMaterialGroupCode());
        groupRelMapper.insert(relEntity);
      });
    }

    Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
    if (menuObject != null) {
      MdmMaterialGroupReqVo newObject = CrmBeanUtil.copy(entity, MdmMaterialGroupReqVo.class);
      crmLogSendUtil.sendForUpdate(menuObject.toString(),
          entity.getId(), entity.getMaterialGroupCode(), oldObject, newObject);
    }

  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void deleteBatch(List<String> ids) {
    ValidateUtils.validate(ids, "ids不能为空");
    Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
    for (String id : ids) {
      Assert.hasText(id, "不能传空值");
      MdmMaterialGroupEntity material = this.getById(id);
      Assert.notNull(material, "物料组不存在");

      // 删除关联表信息
      groupRelMapper.delete(Wrappers.lambdaQuery(MdmMaterialGroupRelEntity.class).eq(MdmMaterialGroupRelEntity::getMaterialGroupCode, material.getMaterialGroupCode()));
      if (menuObject != null) {
        crmLogSendUtil.sendForDel(menuObject.toString(),
            id, material.getMaterialGroupCode(), CrmBeanUtil.copy(material, MdmMaterialGroupReqVo.class));
      }
    }
    this.removeByIds(ids);
  }

  @Override
  public void enableBatch(List<String> ids) {
    Assert.notEmpty(ids, "ids不能为空");
    this.lambdaUpdate()
        .in(MdmMaterialGroupEntity::getId, ids)
        .set(MdmMaterialGroupEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
        .update();
  }

  @Override
  public void disableBatch(List<String> ids) {
    Assert.notEmpty(ids, "ids不能为空");
    this.lambdaUpdate()
        .in(MdmMaterialGroupEntity::getId, ids)
        .set(MdmMaterialGroupEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
        .update();
  }

  @Override
  public PageResult<MdmMaterialGroupRespVo> pageGroupCpsList(MdmMaterialGroupReqVo groupReqVo) {
    Page<MdmMaterialGroupRespVo> page = PageUtil.buildPage(groupReqVo.getPageNum(), groupReqVo.getPageSize());

    List<String> materialGroupCodes = groupReqVo.getMaterialGroupCodes();
    /**
     * 组装物料组信息
     * 如果物料编码或者物料编码集合不为空 整合物料编码信息，根据物料编码信息 查询物料组信息
     */
    if(StringUtils.isNotEmpty(groupReqVo.getMaterialCode()) || CollectionUtil.listNotEmpty(groupReqVo.getMaterialCodes())){
      List<String> materialCodes = groupReqVo.getMaterialCodes();
      if(StringUtils.isNotEmpty(groupReqVo.getMaterialCode())){
        materialCodes.add(groupReqVo.getMaterialCode());
      }

      // 查找物料组信息
      List<MdmMaterialGroupRelEntity> materialGroupRelList = groupRelMapper.selectList(Wrappers.lambdaQuery(MdmMaterialGroupRelEntity.class).in(MdmMaterialGroupRelEntity::getMaterialCode, materialCodes));
      if(CollectionUtil.listNotEmpty(materialGroupRelList)){
        List<String> materialGroupList = materialGroupRelList.stream().map(MdmMaterialGroupRelEntity::getMaterialGroupCode).distinct().collect(Collectors.toList());
        materialGroupCodes.addAll(materialGroupList);
        groupReqVo.setMaterialGroupCodes(materialGroupCodes);
      }
    }

    List<MdmMaterialGroupRespVo> list = materialGroupMapper.findCpsList(page, groupReqVo);
    if(CollectionUtil.listNotEmpty(list)){
      list.forEach(item->{

        // 添加物料组信息
        List<MdmMaterialRespVo> materialList = mdmMaterialService.findMaterialByMaterialGroupCode(item.getMaterialGroupCode());

          if(CollectionUtil.listNotEmpty(materialList)){
            materialList.forEach(materialItem->{

              // 查询所有产品层级信息
              List<MdmProductLevelVo> productLevelList = mdmProductLevelService.getParentCpsList(null, materialItem.getProductLevelCode());
              materialItem.setProductLevels(productLevelList);

              // 查询照片信息
              List<MdmMaterialMediaRespVo> mediaPicList = mdmMaterialMediaService.findMediaListByCondition(materialItem.getId(), materialItem.getMaterialCode());
              materialItem.setPicList(mediaPicList);
            });
          }

        item.setMaterialList(materialList);
      });
    }
    return PageResult.<MdmMaterialGroupRespVo>builder()
        .data(list)
        .count(page.getTotal())
        .build();
  }


  private void FieldValidation(MdmMaterialGroupReqVo groupReqVo) {
    AssertUtils.isNotEmpty(groupReqVo.getMaterialGroupCode(), "物料组编码不能为空");
    AssertUtils.isNotEmpty(groupReqVo.getMaterialGroupName(), "物料组名称不能为空");
    AssertUtils.isNotEmpty(groupReqVo.getDescription(), "描述不能为空");
    Validate.isTrue(groupReqVo.getMaterialGroupCode().length() < 64, "物料组编码，在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(groupReqVo.getMaterialGroupName().length() < 64, "物料组编码，在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(groupReqVo.getDescription().length() < 255, "物料组描述，在进行添加时填入值超过了限定长度(255)，请检查!");
  }

}
