package com.biz.crm.org.controller;

import com.biz.crm.aop.CrmGlobalLog;
import com.biz.crm.aop.CrmLog;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.nebular.mdm.org.req.MdmOrgPageSelectReqVo;
import com.biz.crm.nebular.mdm.org.req.MdmOrgReqVo;
import com.biz.crm.nebular.mdm.org.req.MdmOrgSearchReqVo;
import com.biz.crm.nebular.mdm.org.req.MdmOrgSelectReqVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgSelectRespVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgTreeRespVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgWithPositionRespVo;
import com.biz.crm.org.service.MdmOrgService;
import com.biz.crm.util.OrgUtil;
import com.biz.crm.util.Result;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 组织表
 *
 * @author Tao.Chen
 * @date 2020-11-17 20:23:13
 */
@Slf4j
@RestController
@RequestMapping("/mdmOrgController")
@Api(tags = "MDM-组织管理")
@CrmGlobalLog
public class MdmOrgController {

    @Autowired
    private MdmOrgService mdmOrgService;

    @ApiOperation(value = "查询分页列表")
    @PostMapping("/pageList")
    @CrmDictMethod
    public Result<PageResult<MdmOrgRespVo>> pageList(@RequestBody MdmOrgReqVo mdmOrgReqVo) {
        PageResult<MdmOrgRespVo> result = mdmOrgService.findList(mdmOrgReqVo);
        return Result.ok(result);
    }

    @ApiOperation(value = "查询分页列表")
    @PostMapping("/findAllPage")
    @CrmDictMethod
    public Result<PageResult<MdmOrgRespVo>> findAllPage(@RequestBody MdmOrgReqVo mdmOrgReqVo) {
        PageResult<MdmOrgRespVo> result = mdmOrgService.findAllPage(mdmOrgReqVo);
        return Result.ok(result);
    }

    @ApiOperation(value = "组织分页选择列表")
    @PostMapping("/pageSelectList")
    @CrmDictMethod
    public Result<PageResult<MdmOrgRespVo>> pageSelectList(@RequestBody MdmOrgPageSelectReqVo reqVo) {
        PageResult<MdmOrgRespVo> result = mdmOrgService.pageSelectList(reqVo);
        return Result.ok(result);
    }

    @ApiOperation(value = "详情", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "ID", required = false, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "orgCode", value = "组织编码", required = false, dataType = "String", paramType = "query")
    })
    @GetMapping("/detail")
    @CrmDictMethod
    public Result<MdmOrgRespVo> detail(@RequestParam(value = "id", required = false) String id, @RequestParam(value = "orgCode", required = false) String orgCode) {
        return Result.ok(mdmOrgService.detail(id, orgCode));
    }

    @ApiOperation("根据id集合批量查询组织主表信息")
    @PostMapping("/detailBatchByIds")
    @CrmDictMethod
    public Result<List<MdmOrgRespVo>> detailBatchByIds(@RequestBody List<String> ids) {
        return Result.ok(mdmOrgService.detailBatch(ids, null));
    }

    @ApiOperation("根据组织编码集合批量查询组织主表信息")
    @PostMapping("/detailBatchByOrgCodeList")
    @CrmDictMethod
    public Result<List<MdmOrgRespVo>> detailBatchByOrgCodeList(@RequestBody List<String> orgCodeList) {
        return Result.ok(mdmOrgService.detailBatch(null, orgCodeList));
    }

    @ApiOperation(value = "新增")
    @PostMapping("/save")
    @CrmLog
    public Result save(@RequestBody MdmOrgReqVo mdmOrgReqVo) {
        mdmOrgService.save(mdmOrgReqVo);
        return Result.ok();
    }

    @ApiOperation(value = "更新")
    @PostMapping("/update")
    @CrmLog
    public Result update(@RequestBody MdmOrgReqVo mdmOrgReqVo) {
        mdmOrgService.update(mdmOrgReqVo);
        return Result.ok("修改成功");
    }

    @ApiOperation(value = "删除", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/delete")
    @CrmLog
    public Result delete(@RequestBody List<String> ids) {
        mdmOrgService.deleteBatch(ids);
        return Result.ok("删除成功");
    }

    @ApiOperation(value = "启用", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/enable")
    @CrmLog
    public Result enable(@RequestBody List<String> ids) {
        mdmOrgService.enableBatch(ids);
        return Result.ok("启用成功");
    }

    @ApiOperation(value = "禁用", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/disable")
    @CrmLog
    public Result disable(@RequestBody List<String> ids) {
        mdmOrgService.disableBatch(ids);
        return Result.ok("禁用成功");
    }

    @ApiOperation("组织下拉框")
    @PostMapping("/findOrgSelectList")
    @CrmDictMethod
    public Result<List<MdmOrgSelectRespVo>> findOrgSelectList(@RequestBody MdmOrgSelectReqVo reqVo) {
        return Result.ok(mdmOrgService.findOrgSelectList(reqVo));
    }

    @ApiOperation("组织下拉框(只查启用的)")
    @PostMapping("/select")
    @CrmDictMethod
    public Result<List<MdmOrgSelectRespVo>> select(@RequestBody MdmOrgSelectReqVo reqVo) {
        reqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        return Result.ok(mdmOrgService.findOrgSelectList(reqVo));
    }

    @ApiOperation("根据组织id或者编码查询全部下级（含当前）组织列表")
    @PostMapping("/findAllChildrenOrgList")
    @CrmDictMethod
    public Result<List<MdmOrgRespVo>> findAllChildrenOrgList(@RequestBody MdmOrgReqVo reqVo) {
        return Result.ok(mdmOrgService.findAllChildrenOrgList(reqVo.getId(), reqVo.getOrgCode()));
    }

    @ApiOperation("根据组织id或者编码查询当前组织的直接下级（不含当前）组织列表")
    @PostMapping("/findSingleChildrenOrgList")
    @CrmDictMethod
    public Result<List<MdmOrgRespVo>> findSingleChildrenOrgList(@RequestBody MdmOrgReqVo reqVo) {
        return Result.ok(mdmOrgService.findSingleChildrenOrgList(reqVo.getId(), reqVo.getOrgCode()));
    }

    @ApiOperation("根据组织id或者编码查询全部上级（含当前）组织列表")
    @PostMapping("/findAllParentOrgList")
    @CrmDictMethod
    public Result<List<MdmOrgRespVo>> findAllParentOrgList(@RequestBody MdmOrgReqVo reqVo) {
        return Result.ok(mdmOrgService.findAllParentOrgList(reqVo.getId(), reqVo.getOrgCode()));
    }

    @ApiOperation("根据组织id或者编码查询当前组织的直接上级组织")
    @PostMapping("/findSingleParentOrg")
    @CrmDictMethod
    public Result<MdmOrgRespVo> findSingleParentOrg(@RequestBody MdmOrgReqVo reqVo) {
        return Result.ok(mdmOrgService.findSingleParentOrg(reqVo.getId(), reqVo.getOrgCode()));
    }

    @ApiOperation(value = "根据组织id或者编码查询当前组织的直接上级组织", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "ID", required = false, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "orgCode", value = "组织编码", required = false, dataType = "String", paramType = "query")
    })
    @GetMapping("/getParentOrg")
    @CrmDictMethod
    public Result<MdmOrgRespVo> getParentOrg(@RequestParam(value = "id", required = false) String id, @RequestParam(value = "orgCode", required = false) String orgCode) {
        return Result.ok(mdmOrgService.findSingleParentOrg(id, orgCode));
    }

    @ApiOperation("重置组织树降维编码")
    @PostMapping("/resetRuleCode")
    public Result resetRuleCode() {
        mdmOrgService.resetRuleCode();
        OrgUtil.reloadAllCache();
        return Result.ok();
    }

    @ApiOperation("清除全部缓存")
    @PostMapping("/deleteAllCache")
    public Result deleteAllCache() {
        OrgUtil.deleteAllCache();
        return Result.ok();
    }

    @ApiOperation("重新加载全部缓存")
    @PostMapping("/reloadAllCache")
    public Result reloadAllCache() {
        OrgUtil.reloadAllCache();
        return Result.ok();
    }

    @ApiOperation("查询这些组织编码对应的组织，每个组织上附带这个组织（含下级组织）下的全部职位")
    @PostMapping("/findOrgWithAllPositionList")
    public Result<List<MdmOrgWithPositionRespVo>> findOrgWithAllPositionList(@RequestBody List<String> orgCodeList) {
        return Result.ok(mdmOrgService.findOrgWithAllPositionList(orgCodeList));
    }

    @ApiOperation("查询这些组织编码对应的组织，每个组织上附带这个组织（不含下级组织）下的全部职位")
    @PostMapping("/findOrgWithSinglePositionList")
    public Result<List<MdmOrgWithPositionRespVo>> findOrgWithSinglePositionList(@RequestBody List<String> orgCodeList) {
        return Result.ok(mdmOrgService.findOrgWithSinglePositionList(orgCodeList));
    }

    @ApiOperation("批量查询组织及下级组织,返回（pageSize）对应条数")
    @PostMapping("/findOrgAndChildrenList")
    public Result<List<MdmOrgRespVo>> findOrgAndChildrenList(@RequestBody MdmOrgSearchReqVo reqVo) {
        return Result.ok(mdmOrgService.findOrgAndChildrenList(reqVo));
    }

    @ApiOperation("根据组织编码集合查询组织列表")
    @PostMapping("/listCondition")
    @CrmDictMethod
    public Result<List<MdmOrgRespVo>> listCondition(@RequestBody MdmOrgReqVo reqVo) {
        return Result.ok(mdmOrgService.listCondition(reqVo));
    }

    @ApiOperation("查询所有组织（不分页，数据量大，给后台用）(启用状态、组织类型、上级编码精确查询，组织编码、名称模糊查询)")
    @PostMapping("/findOrgList")
    @CrmDictMethod
    public Result<List<MdmOrgRespVo>> findOrgList(@RequestBody MdmOrgReqVo reqVo) {
        return Result.ok(mdmOrgService.findOrgList(reqVo));
    }

    @ApiOperation("查询所有组织树形结构（给后台使用），无筛选条件，全部返回（包含停用）")
    @GetMapping("/getOrgTree")
    public Result<List<MdmOrgTreeRespVo>> getOrgTree() {
        return Result.ok(mdmOrgService.getOrgTree());
    }

    @ApiOperation("根据组织编码集合查询这些组织（不含自己）的上级组织编码，如果参数本身有上下级关系，则会返回处于上级的组织编码")
    @PostMapping("/findAllParentOrgCodeExcludeSelf")
    public Result<List<String>> findAllParentOrgCodeExcludeSelf(@RequestBody List<String> orgCodeList){
        return Result.ok(mdmOrgService.findAllParentOrgCodeExcludeSelf(orgCodeList));
    }

    @ApiOperation("组织搜索树")
    @PostMapping("/orgSearchTree")
    public Result<List<MdmOrgTreeRespVo>> orgSearchTree(@RequestBody MdmOrgReqVo mdmOrgReqVo) {
        List<MdmOrgTreeRespVo> list = mdmOrgService.orgSearchTree(mdmOrgReqVo);
        return Result.ok(list);
    }
    @ApiOperation("查询全部组织")
    @GetMapping("/getAll")
    public Result<List<MdmOrgRespVo>> getAll(){
        Result<List<MdmOrgRespVo>> result = new Result<>();
        List<MdmOrgRespVo> mdmOrgRespVos = mdmOrgService.getAll() ;
        result.setResult(mdmOrgRespVos);
        return result;
    }

}
