package com.biz.crm.parametermanage.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.PageResult;
import com.biz.crm.nebular.mdm.parametermanage.req.MdmParameterManageReqVo;
import com.biz.crm.nebular.mdm.parametermanage.resp.MdmParameterManageRespVo;
import com.biz.crm.parameter.vo.ParameterVo;
import com.biz.crm.parametermanage.mapper.MdmParameterManageMapper;
import com.biz.crm.parametermanage.model.MdmParameterManageEntity;
import com.biz.crm.parametermanage.service.MdmParameterManageService;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 参数配置接口实现
 *
 * @author Tao.Chen
 * @date 2020-11-24 10:47:10
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmParameterManageServiceExpandImpl")
public class MdmParameterManageServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmParameterManageMapper, MdmParameterManageEntity> implements MdmParameterManageService {

    @Resource
    private MdmParameterManageMapper mdmParameterManageMapper;

    @Override
    public PageResult<MdmParameterManageRespVo> findList(MdmParameterManageReqVo reqVo) {
        Page<MdmParameterManageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmParameterManageRespVo> list = mdmParameterManageMapper.findList(page, reqVo);
        return PageResult.<MdmParameterManageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public MdmParameterManageRespVo query(MdmParameterManageReqVo reqVo) {
        if (StringUtils.isNotEmpty(reqVo.getId()) || StringUtils.isNotEmpty(reqVo.getParameterCode())) {
            MdmParameterManageEntity one = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(reqVo.getId()), MdmParameterManageEntity::getId, reqVo.getId())
                    .eq(StringUtils.isNotEmpty(reqVo.getParameterCode()), MdmParameterManageEntity::getParameterCode, reqVo.getParameterCode())
                    .one();
            if (one != null) {
                return CrmBeanUtil.copy(one, MdmParameterManageRespVo.class);
            }
        }
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmParameterManageReqVo reqVo) {
        verify(reqVo);
        MdmParameterManageEntity entity = CrmBeanUtil.copy(reqVo, MdmParameterManageEntity.class);
        this.save(entity);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmParameterManageReqVo reqVo) {
        Assert.hasText(reqVo.getId(), "缺失id");
        verify(reqVo);
        MdmParameterManageEntity entity = this.getById(reqVo.getId());
        Assert.notNull(entity, "无效的id");
        this.updateById(CrmBeanUtil.copy(reqVo, MdmParameterManageEntity.class));
        ParamUtil.deleteRedisCache(entity.getParameterCode());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        Assert.notEmpty(ids, "缺失参数");
        List<MdmParameterManageEntity> list = this.lambdaQuery()
                .in(MdmParameterManageEntity::getId, ids)
                .select(MdmParameterManageEntity::getParameterCode)
                .list();
        if (list != null) {
            list.stream().map(MdmParameterManageEntity::getParameterCode).forEach(ParamUtil::deleteRedisCache);
        }
        removeByIds(ids);
    }

    @Override
    public void deleteAllCache() {
        List<MdmParameterManageEntity> list = this.lambdaQuery()
                .select(MdmParameterManageEntity::getParameterCode)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            this.deleteCacheByCodeList(list.stream().map(MdmParameterManageEntity::getParameterCode).collect(Collectors.toList()));
        }
    }

    @Override
    public void deleteCacheByIds(List<String> ids) {
        if (CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            List<MdmParameterManageEntity> list = this.lambdaQuery()
                    .in(MdmParameterManageEntity::getId, ids)
                    .select(MdmParameterManageEntity::getParameterCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                this.deleteCacheByCodeList(list.stream().map(MdmParameterManageEntity::getParameterCode).collect(Collectors.toList()));
            }
        }
    }

    @Override
    public void deleteCacheByCodeList(List<String> codes) {
        if (CollectionUtil.listNotEmptyNotSizeZero(codes)) {
            codes.forEach(ParamUtil::deleteRedisCache);
        }
    }

    @Override
    public List<ParameterVo> getParameterRedisList(List<String> parameterCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(parameterCodeList)) {
            List<MdmParameterManageEntity> list = this.lambdaQuery()
                    .in(MdmParameterManageEntity::getParameterCode, parameterCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return CrmBeanUtil.copyList(list, ParameterVo.class);
            }
        }
        return new ArrayList<>();
    }

    /**
     * 校验参数
     *
     * @param reqVo
     */
    private void verify(MdmParameterManageReqVo reqVo) {
        Assert.hasText(reqVo.getParameterCode(), "参数不能为空");
        Assert.hasText(reqVo.getParameterName(), "参数名称不能为空");
        Assert.hasText(reqVo.getDefaultValue(), "默认值不能为空");
        Assert.hasText(reqVo.getParameterValue(), "参数值不能为空");
        List<MdmParameterManageEntity> list = this.lambdaQuery()
                .eq(MdmParameterManageEntity::getParameterCode, reqVo.getParameterCode())
                .select(MdmParameterManageEntity::getId)
                .list()
                .stream().filter(x -> StringUtils.isEmpty(reqVo.getId()) || !reqVo.getId().equals(x.getId())).collect(Collectors.toList());
        Assert.isTrue(CollectionUtil.listEmpty(list), "参数重复");
    }
}
