package com.biz.crm.poi.service.impl;

import com.biz.crm.base.BusinessException;
import com.biz.crm.customer.service.MdmCustomerMsgService;
import com.biz.crm.nebular.mdm.constant.RegionLevelEnum;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgReqVo;
import com.biz.crm.nebular.mdm.poi.resp.MdmAmapDistrictRespVo;
import com.biz.crm.nebular.mdm.poi.resp.MdmAmapDistrictStatisticianRespVo;
import com.biz.crm.nebular.mdm.poi.resp.MdmTerminalStatisticianTypeTotalVo;
import com.biz.crm.nebular.mdm.poi.resp.MdmTerminalStatisticianTypeVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalVo;
import com.biz.crm.nebular.mdm.terminal.req.MdmAmapDistrictStatisticianReqVo;
import com.biz.crm.poi.model.MdmAmapPoiCityEntity;
import com.biz.crm.poi.model.MdmAmapPoiEntity;
import com.biz.crm.poi.model.MdmAmapPoiTypeEntity;
import com.biz.crm.poi.service.MdmAmapPoiCityService;
import com.biz.crm.poi.service.MdmAmapPoiService;
import com.biz.crm.poi.service.MdmAmapPoiTypeService;
import com.biz.crm.poi.service.MdmAmapStatisticianService;
import com.biz.crm.terminal.model.MdmTerminalCustomerElasticsearchEntity;
import com.biz.crm.terminal.service.MdmTerminalElasticsearchService;
import com.biz.crm.terminal.service.MdmTerminalService;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.PinyinUtils;
import com.biz.crm.utils.EsBoolQueryBuilder;
import com.biz.crm.utils.EsEmptyAggregationPage;
import lombok.extern.slf4j.Slf4j;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.tophits.TopHits;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;
import org.springframework.data.elasticsearch.core.aggregation.AggregatedPage;
import org.springframework.data.elasticsearch.core.query.NativeSearchQuery;
import org.springframework.data.elasticsearch.core.query.NativeSearchQueryBuilder;
import org.springframework.data.elasticsearch.core.query.SearchQuery;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author zxw
 * @date 2021-05-14 15:11
 **/
@Service
@ConditionalOnMissingBean(name = "MdmAmapStatisticianServiceExpandImpl")
@Slf4j
public class MdmAmapStatisticianServiceImpl implements MdmAmapStatisticianService {

    @Autowired
    private MdmCustomerMsgService mdmCustomerMsgService;
    @Autowired
    private MdmTerminalService mdmTerminalService;
    @Autowired
    private MdmAmapPoiCityService mdmAmapPoiCityService;
    @Autowired
    private MdmAmapPoiTypeService mdmAmapPoiTypeService;
    @Autowired
    private MdmAmapPoiService mdmAmapPoiService;
    @Autowired
    private MdmTerminalElasticsearchService mdmTerminalElasticsearchService;
    @Autowired
    private ElasticsearchRestTemplate elasticsearchRestTemplate;


    @Override
    public void statisticalType(MdmTerminalStatisticianTypeTotalVo result) {
        List<MdmTerminalStatisticianTypeVo> mdmTerminalStatisticianTypeVos = mdmTerminalService.statisticalType(new MdmTerminalVo());
        List<MdmTerminalStatisticianTypeVo> mdmCustomerStatisticianTypeVos = mdmCustomerMsgService.statisticalType(new MdmCustomerMsgReqVo());
        MdmTerminalStatisticianTypeVo terminal = new MdmTerminalStatisticianTypeVo();
        terminal.setStatisticianType("terminal");
        terminal.setStatisticianName("终端");
        Integer terminalCount = mdmTerminalStatisticianTypeVos.stream()
                .map(x -> Optional.ofNullable(x.getStatisticianCount()).orElse(0))
                .mapToInt(x -> x)
                .sum();
        terminal.setStatisticianCount(terminalCount);
        terminal.setChildren(mdmTerminalStatisticianTypeVos);

        MdmTerminalStatisticianTypeVo customer = new MdmTerminalStatisticianTypeVo();
        customer.setStatisticianType("customer");
        customer.setStatisticianName("客户");
        Integer customerCount = mdmCustomerStatisticianTypeVos.stream()
                .map(x -> Optional.ofNullable(x.getStatisticianCount()).orElse(0))
                .mapToInt(x -> x)
                .sum();
        customer.setStatisticianCount(customerCount);
        customer.setChildren(mdmCustomerStatisticianTypeVos);

        result.setCooperationTypeCount(customerCount + terminalCount);
        result.setCooperationTypeList(Arrays.asList(customer, terminal));
    }

    @Override
    public List<MdmAmapDistrictStatisticianRespVo> statisticalDistrict(MdmAmapDistrictStatisticianReqVo mdmAmapDistrictStatisticianReqVo) {
        List<String> statisticianTypeList = mdmAmapDistrictStatisticianReqVo.getStatisticianTypeList();
        if (CollectionUtils.isEmpty(statisticianTypeList)) {
            return Collections.emptyList();
        }
        //查询字典范围内客户与终端
        List<String> typeList = new ArrayList<>();
        Map<String, String> customerTypeMap = DictUtil.dictMap("customer_type");
        Map<String, String> terminalTypeMap = DictUtil.dictMap("terminal_type");
        typeList.addAll(customerTypeMap.keySet());
        typeList.addAll(terminalTypeMap.keySet());
        if (CollectionUtils.isEmpty(typeList)) {
            return Collections.emptyList();
        }

        List<MdmAmapDistrictStatisticianRespVo> districtStatistician = new ArrayList<>();
        String regionLevel = null;
        //换成crm区域编码
        String amapDistrictCode = mdmAmapDistrictStatisticianReqVo.getAmapDistrictCode();
        if (!StringUtils.isEmpty(amapDistrictCode)) {
            MdmAmapPoiCityEntity amapPoiCityEntity = mdmAmapPoiCityService.lambdaQuery()
                    .eq(MdmAmapPoiCityEntity::getAmapCode, mdmAmapDistrictStatisticianReqVo.getAmapDistrictCode())
                    .select(MdmAmapPoiCityEntity::getRegionCode, MdmAmapPoiCityEntity::getRegionLevel)
                    .one();
            log.info("高德城市:{}", amapPoiCityEntity);
            if (amapPoiCityEntity == null) {
                return Collections.emptyList();
            }
            regionLevel = amapPoiCityEntity.getRegionLevel();
            String regionCode = amapPoiCityEntity.getRegionCode();
            if (regionCode == null || regionLevel == null) {
                return Collections.emptyList();
            }
            mdmAmapDistrictStatisticianReqVo.setAmapDistrictCode(regionCode);
            amapDistrictCode = regionCode;
        }
        //es省市区聚合
        String name = mdmAmapDistrictStatisticianReqVo.getCustomerOrTerminalName();
        String supplyFullName = mdmAmapDistrictStatisticianReqVo.getSupplyFullName();
        String finalAmapDistrictCode = amapDistrictCode;
        BoolQueryBuilder boolQueryBuilder = EsBoolQueryBuilder.lambdaQuery()
                .filter(QueryBuilders.termsQuery("clientSubclass.keyword", typeList))
                .filter(!StringUtils.isEmpty(name), () -> QueryBuilders.matchPhraseQuery("name", name))
                .filter(!StringUtils.isEmpty(amapDistrictCode),()->QueryBuilders.multiMatchQuery(finalAmapDistrictCode,"amapProvinceCode","amapCityCode","amapDistrictCode"))
                .filter(!StringUtils.isEmpty(supplyFullName), () -> QueryBuilders.matchPhraseQuery("supplyNameList", supplyFullName))
                .build();

        TermsAggregationBuilder termsAggregationBuilder = null;
        if (!StringUtils.isEmpty(amapDistrictCode) && regionLevel != null) {
            if (RegionLevelEnum.PROVINCE.getCode().equals(regionLevel)) {
                termsAggregationBuilder = AggregationBuilders
                        .terms("groupArea")
                        .field("cityCode.keyword").size(35);
            } else if (RegionLevelEnum.CITY.getCode().equals(regionLevel)) {
                termsAggregationBuilder = AggregationBuilders
                        .terms("groupArea")
                        .field("districtCode.keyword").size(100);
            }else{
                return Collections.emptyList();
            }
        } else {
            termsAggregationBuilder = AggregationBuilders
                    .terms("groupArea")
                    .field("provinceCode.keyword").size(100);
        }

        NativeSearchQuery build = new NativeSearchQueryBuilder()
                .withQuery(boolQueryBuilder)
                .addAggregation(termsAggregationBuilder)
                .withPageable(EsEmptyAggregationPage.getInstance())
                .build();
        AggregatedPage<MdmTerminalCustomerElasticsearchEntity> entities = elasticsearchRestTemplate
                .queryForPage(build, MdmTerminalCustomerElasticsearchEntity.class);
        Terms terms = entities.getAggregations().get("groupArea");

        districtStatistician=terms.getBuckets().stream()
                .map(x -> {
                    MdmAmapDistrictStatisticianRespVo respVo = new MdmAmapDistrictStatisticianRespVo();
                    respVo.setAmapCode(x.getKeyAsString());
                    respVo.setAmapStatistician(((int) x.getDocCount()));
                    return respVo;
                }).collect(Collectors.toList());

        if (CollectionUtils.isEmpty(districtStatistician)) {
            return Collections.emptyList();
        }
        //转换成高德地图编码与名称
        List<String> amapCodeList = districtStatistician
                .stream()
                .map(MdmAmapDistrictStatisticianRespVo::getAmapCode)
                .filter(amapCode -> !StringUtils.isEmpty(amapCode))
                .collect(Collectors.toList());
        Map<String, MdmAmapPoiCityEntity> collect = mdmAmapPoiCityService.lambdaQuery()
                .in(MdmAmapPoiCityEntity::getRegionCode, amapCodeList)
                .select(MdmAmapPoiCityEntity::getAmapCode, MdmAmapPoiCityEntity::getAmapName, MdmAmapPoiCityEntity::getRegionCode)
                .list()
                .stream()
                .filter(x -> !StringUtils.isEmpty(x.getRegionCode()))
                .collect(Collectors.toMap(MdmAmapPoiCityEntity::getRegionCode, Function.identity()));
        Map<String, MdmAmapDistrictStatisticianRespVo> map = districtStatistician.stream()
                .filter(x -> collect.containsKey(x.getAmapCode()))
                .peek(x -> {
                    MdmAmapPoiCityEntity mdmAmapPoiCityEntity = collect.get(x.getAmapCode());
                    x.setAmapCode(mdmAmapPoiCityEntity.getAmapCode());
                    String amapName = mdmAmapPoiCityEntity.getAmapName();
                    x.setAmapName(amapName);
                    if (!StringUtils.isEmpty(amapName)) {
                        if (amapName.endsWith("省")) {
                            amapName = amapName.substring(0, amapName.length() - 1);
                        }
                        x.setAmapPinyin(PinyinUtils.toPinLower(amapName, ""));
                    }
                })
                .collect(Collectors.toMap(MdmAmapDistrictStatisticianRespVo::getAmapCode, Function.identity(), (x1, x2) -> {
                    Integer i1 = Optional.ofNullable(x1.getAmapStatistician()).orElse(0);
                    Integer i2 = Optional.ofNullable(x2.getAmapStatistician()).orElse(0);
                    x1.setAmapStatistician(i1 + i2);
                    return x1;
                }));
        return new ArrayList<>(map.values());
    }

    @Override
    public void unCooperationStatisticalType(MdmTerminalStatisticianTypeTotalVo vo) {
        //查询所有类型
        List<MdmAmapPoiTypeEntity> list = mdmAmapPoiTypeService.lambdaQuery()
                .select(MdmAmapPoiTypeEntity::getTypeCode, MdmAmapPoiTypeEntity::getParentCode, MdmAmapPoiTypeEntity::getTypeName)
                .list();
        if (CollectionUtils.isEmpty(list)) {
            return;
        }

        //查询poi数据聚合
        TermsAggregationBuilder termsAggregationBuilder = AggregationBuilders
                .terms("groupType")
                .field("typeCode").size(5000);
        SearchQuery build = new NativeSearchQueryBuilder()
                .withPageable(EsEmptyAggregationPage.getInstance())
                .addAggregation(termsAggregationBuilder)
                .build();
        AggregatedPage<MdmAmapPoiEntity> mdmAmapPoiEntities = elasticsearchRestTemplate.queryForPage(build, MdmAmapPoiEntity.class);
        Terms terms = mdmAmapPoiEntities.getAggregations().get("groupType");
        List<? extends Terms.Bucket> buckets = terms.getBuckets();
        Map<String, Integer> poiMap = new HashMap<>(buckets.size());
        for (Terms.Bucket bucket : buckets) {
            poiMap.put(bucket.getKeyAsString(), (int) bucket.getDocCount());
        }

        //构建树形，封装数据
        List<MdmTerminalStatisticianTypeVo> result = new ArrayList<>();
        for (MdmAmapPoiTypeEntity mdmAmapPoiTypeEntity : list) {
            MdmTerminalStatisticianTypeVo mdmTerminalStatisticianTypeVo = new MdmTerminalStatisticianTypeVo();
            mdmTerminalStatisticianTypeVo.setStatisticianType(mdmAmapPoiTypeEntity.getTypeCode());
            mdmTerminalStatisticianTypeVo.setStatisticianName(mdmAmapPoiTypeEntity.getTypeName());
            mdmTerminalStatisticianTypeVo.setStatisticianCount(Optional.ofNullable(poiMap.get(mdmAmapPoiTypeEntity.getTypeCode())).orElse(0));
            List<MdmTerminalStatisticianTypeVo> children = new ArrayList<>();
            //封装根节点
            String parentCode = mdmAmapPoiTypeEntity.getParentCode();
            if (StringUtils.isEmpty(parentCode)) {
                result.add(mdmTerminalStatisticianTypeVo);
            }
            for (MdmAmapPoiTypeEntity amapPoiTypeEntity : list) {
                if (mdmAmapPoiTypeEntity.getTypeCode().equals(amapPoiTypeEntity.getParentCode())) {
                    MdmTerminalStatisticianTypeVo statisticianTypeVo = new MdmTerminalStatisticianTypeVo();
                    statisticianTypeVo.setStatisticianType(amapPoiTypeEntity.getTypeCode());
                    statisticianTypeVo.setStatisticianName(amapPoiTypeEntity.getTypeName());
                    statisticianTypeVo.setStatisticianCount(Optional.ofNullable(poiMap.get(mdmAmapPoiTypeEntity.getTypeCode())).orElse(0));
                    children.add(statisticianTypeVo);
                }
            }
            mdmTerminalStatisticianTypeVo.setChildren(children);
        }
        int sum = 0;
        for (MdmTerminalStatisticianTypeVo typeVo : result) {
            typeVo.setStatisticianCount(postCardinality(typeVo));
            sum += typeVo.getStatisticianCount();
        }
        vo.setUnCooperationTypeCount(sum);
        vo.setUnUooperationTypeList(result);
    }

    protected Integer postCardinality(MdmTerminalStatisticianTypeVo typeVo) {
        if (typeVo != null) {
            List<MdmTerminalStatisticianTypeVo> children = typeVo.getChildren();
            if (!CollectionUtils.isEmpty(children)) {
                for (MdmTerminalStatisticianTypeVo child : children) {
                    child.setStatisticianCount(postCardinality(child));
                }
                return children.stream()
                        .mapToInt(MdmTerminalStatisticianTypeVo::getStatisticianCount)
                        .sum() + typeVo.getStatisticianCount();
            }
            return typeVo.getStatisticianCount();
        }
        return 0;
    }

    @Override
    public List<MdmAmapDistrictStatisticianRespVo> unCooperationStatisticalDistrict(MdmAmapDistrictStatisticianReqVo mdmAmapDistrictStatisticianReqVo) {
        String amapDistrictCode = mdmAmapDistrictStatisticianReqVo.getAmapDistrictCode();
        List<MdmAmapDistrictStatisticianRespVo> result = new ArrayList<>();
        //客户与终端类型
        List<String> statisticianTypeList = mdmAmapDistrictStatisticianReqVo.getStatisticianTypeList();
        if (CollectionUtils.isEmpty(statisticianTypeList)) {
            return result;
        }
        //根据不同传参，分别对省市区聚合
        String area = null;
        String areaName = null;
        if (StringUtils.isEmpty(amapDistrictCode)) {
            area = "amapProvinceCode.keyword";
            areaName = "amapProvinceName";
        } else {
            String regionLevel = obtainRegionLevel(amapDistrictCode);
            if (RegionLevelEnum.PROVINCE.getCode().equals(regionLevel)) {
                area = "amapCityCode.keyword";
                areaName = "amapCityName";
            }else if (RegionLevelEnum.CITY.getCode().equals(regionLevel)) {
                area = "amapDistrictCode.keyword";
                areaName = "amapDistrictName";
            }else{
                return Collections.emptyList();
            }
        }
        String name = mdmAmapDistrictStatisticianReqVo.getCustomerOrTerminalName();
        BoolQueryBuilder boolQueryBuilder = EsBoolQueryBuilder.lambdaQuery()
                .filter(!StringUtils.isEmpty(amapDistrictCode),()->QueryBuilders.multiMatchQuery(amapDistrictCode,"amapProvinceCode","amapCityCode","amapDistrictCode"))
                .filter(!StringUtils.isEmpty(name), () -> QueryBuilders.matchPhraseQuery("name", name))
                .build();
        if(!CollectionUtils.isEmpty(statisticianTypeList)){
            for(String type:statisticianTypeList){
                boolQueryBuilder.should(QueryBuilders.matchPhraseQuery("typeCode", type));
            }
            boolQueryBuilder.minimumShouldMatch(1);
        }
        TermsAggregationBuilder groupArea = AggregationBuilders
                .terms("groupArea")
                .field(area).size(100)
                .subAggregation(AggregationBuilders.topHits("top").size(1).fetchSource(areaName, null));
        NativeSearchQuery searchQuery = new NativeSearchQueryBuilder()
                .withPageable(EsEmptyAggregationPage.getInstance())
                .withQuery(boolQueryBuilder)
                .addAggregation(groupArea)
                .build();
        AggregatedPage<MdmAmapPoiEntity> mdmAmapPoiEntities = elasticsearchRestTemplate.queryForPage(searchQuery, MdmAmapPoiEntity.class);
        Terms terms = mdmAmapPoiEntities.getAggregations().get("groupArea");
        for (Terms.Bucket bucket : terms.getBuckets()) {
            MdmAmapDistrictStatisticianRespVo respVo = new MdmAmapDistrictStatisticianRespVo();
            respVo.setAmapCode(bucket.getKeyAsString());
            respVo.setAmapStatistician(((int) bucket.getDocCount()));

            TopHits topHits = bucket.getAggregations().get("top");
            SearchHit[] hits = topHits.getHits().getHits();
            for (SearchHit hit : hits) {
                Map<String, Object> sourceAsMap = hit.getSourceAsMap();
                Object o = sourceAsMap.get(areaName);
                if (o != null) {
                    String amapName = (String) o;
                    if (amapName.endsWith("省")) {
                        amapName = amapName.substring(0, amapName.length() - 1);
                    }
                    respVo.setAmapName(amapName);
                    respVo.setAmapPinyin(PinyinUtils.toPinLower(amapName, ""));
                }
            }
            result.add(respVo);
        }
        return result;
    }

    protected String obtainRegionLevel(String amapDistrictCode) {
        return Optional.ofNullable(mdmAmapPoiCityService.lambdaQuery()
                .eq(MdmAmapPoiCityEntity::getAmapCode, amapDistrictCode)
                .select(MdmAmapPoiCityEntity::getRegionLevel)
                .one())
                .filter(x -> !StringUtils.isEmpty(x.getRegionLevel()))
                .map(MdmAmapPoiCityEntity::getRegionLevel)
                .orElseThrow(() -> new BusinessException("区域编码不存在:" + amapDistrictCode));
    }

    @Override
    public List<MdmAmapDistrictRespVo> district(MdmAmapDistrictStatisticianReqVo mdmAmapDistrictStatisticianReqVo) {
        List<String> statisticianTypeList = mdmAmapDistrictStatisticianReqVo.getStatisticianTypeList();
        if (CollectionUtils.isEmpty(statisticianTypeList)) {
            return Collections.emptyList();
        }
        String amapDistrictCode = mdmAmapDistrictStatisticianReqVo.getAmapDistrictCode();
        if (org.apache.commons.lang3.StringUtils.isEmpty(amapDistrictCode)) {
            return Collections.emptyList();
        }
        MdmAmapPoiCityEntity amapPoiCityEntity = mdmAmapPoiCityService.lambdaQuery()
                .eq(MdmAmapPoiCityEntity::getAmapCode, mdmAmapDistrictStatisticianReqVo.getAmapDistrictCode())
                .select(MdmAmapPoiCityEntity::getRegionCode, MdmAmapPoiCityEntity::getAmapCode, MdmAmapPoiCityEntity::getAmapCode)
                .one();
        log.info("高德地图区域查询:{}", amapDistrictCode);
        String regionCode = Optional.ofNullable(amapPoiCityEntity)
                .map(MdmAmapPoiCityEntity::getRegionCode)
                .filter(x -> !StringUtils.isEmpty(x))
                .orElseThrow(() -> new BusinessException("区域不存在:" + amapDistrictCode));
        mdmAmapDistrictStatisticianReqVo.setAmapDistrictCode(regionCode);
        return mdmTerminalElasticsearchService.listCondition(mdmAmapDistrictStatisticianReqVo);
    }

    @Override
    public List<MdmAmapDistrictRespVo> unCooperationDistrict(MdmAmapDistrictStatisticianReqVo mdmAmapDistrictStatisticianReqVo) {
        String amapDistrictCode = mdmAmapDistrictStatisticianReqVo.getAmapDistrictCode();
        if (StringUtils.isEmpty(amapDistrictCode)) {
            return Collections.emptyList();
        }
        List<String> statisticianTypeList = mdmAmapDistrictStatisticianReqVo.getStatisticianTypeList();
        if (CollectionUtils.isEmpty(statisticianTypeList)) {
            return Collections.emptyList();
        }
        String customerOrTerminalName = mdmAmapDistrictStatisticianReqVo.getCustomerOrTerminalName();
        List<MdmAmapPoiEntity> list = mdmAmapPoiService.lambdaQuery()
                .eq(MdmAmapPoiEntity::getAmapDistrictCode, amapDistrictCode)
                .in(MdmAmapPoiEntity::getTypeCode, statisticianTypeList)
                .like(!StringUtils.isEmpty(customerOrTerminalName), MdmAmapPoiEntity::getPoiName, customerOrTerminalName)
                .list();
        if (CollectionUtils.isEmpty(list)) {
            return Collections.emptyList();
        }
        return list.stream()
                .map(x -> {
                    MdmAmapDistrictRespVo respVo = new MdmAmapDistrictRespVo();
                    respVo.setAmapCode(x.getAmapDistrictCode());
                    respVo.setAmapName(x.getAmapDistrictName());
                    respVo.setStatisticianCode(x.getId());
                    respVo.setStatisticianName(x.getPoiName());
                    respVo.setStatisticianName(x.getTypeCode());
                    respVo.setStatisticianName(x.getAddress());
                    respVo.setStatisticianContactPhone(x.getTel());
                    respVo.setStatisticianContactName(x.getPoiName());
                    respVo.setStatisticianFullName(x.getPoiName());
                    respVo.setLongitude(x.getLongitude());
                    respVo.setLatitude(x.getLatitude());
                    return respVo;
                }).collect(Collectors.toList());
    }

    @Override
    public MdmTerminalStatisticianTypeTotalVo statisticalTotal() {
        MdmTerminalStatisticianTypeTotalVo result = new MdmTerminalStatisticianTypeTotalVo();
        statisticalType(result);
        unCooperationStatisticalType(result);
        return result;
    }

}