package com.biz.crm.position.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.nebular.mdm.role.resp.MdmRoleRespVo;
import com.biz.crm.position.mapper.MdmPositionRoleMapper;
import com.biz.crm.position.model.MdmPositionRoleEntity;
import com.biz.crm.position.service.MdmPositionRoleService;
import com.biz.crm.role.service.MdmRoleService;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.StringUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 一个拥有权限信息的业务系统用户定义接口实现
 *
 * @author zxw
 * @date 2020-09-18 14:06:30
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmPositionRoleServiceExpandImpl")
@Transactional(rollbackFor = Exception.class)
public class MdmPositionRoleServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmPositionRoleMapper, MdmPositionRoleEntity> implements MdmPositionRoleService {

    @Resource
    private MdmRoleService mdmRoleService;

    @Override
    public List<String> findRoleCodeByPosition(List<String> positionCodeList) {
        if (CollectionUtils.isEmpty(positionCodeList)) {
            return new ArrayList<>();
        }
        return this.lambdaQuery()
                .in(MdmPositionRoleEntity::getPositionCode, positionCodeList)
                .select(MdmPositionRoleEntity::getRoleCode)
                .list()
                .stream()
                .map(MdmPositionRoleEntity::getRoleCode)
                .collect(Collectors.toList());
    }

    @Override
    public List<String> findPositionCodeCodeByRole(List<String> roleCodeList) {
        if (CollectionUtils.isEmpty(roleCodeList)) {
            return new ArrayList<>();
        }
        return this.lambdaQuery()
                .in(MdmPositionRoleEntity::getRoleCode, roleCodeList)
                .select(MdmPositionRoleEntity::getPositionCode)
                .list()
                .stream()
                .filter(x->x!=null&&!StringUtils.isEmpty(x.getPositionCode()))
                .map(MdmPositionRoleEntity::getPositionCode)
                .collect(Collectors.toList());
    }

    @Override
    public void addRole(List<String> roleCodeList, String positionCode) {
        Assert.notEmpty(roleCodeList, "角色编码集合不能为空");
        Assert.hasText(positionCode, "职位编码不能为空");
        List<MdmPositionRoleEntity> collect = roleCodeList.stream().map(x -> {
            MdmPositionRoleEntity mdmPositionRoleEntity = new MdmPositionRoleEntity();
            mdmPositionRoleEntity.setPositionCode(positionCode);
            mdmPositionRoleEntity.setRoleCode(x);
            mdmPositionRoleEntity.setId(x + positionCode);
            return mdmPositionRoleEntity;
        }).collect(Collectors.toList());
        this.saveBatch(collect);
    }

    @Override
    public void addPosition(List<String> positionCodeList, String roleCode) {
        Assert.notEmpty(positionCodeList, "职位编码集合不能为空");
        Assert.hasText(roleCode, "角色编码不能为空");
        List<MdmPositionRoleEntity> collect = positionCodeList.stream().map(x -> {
            MdmPositionRoleEntity mdmPositionRoleEntity = new MdmPositionRoleEntity();
            mdmPositionRoleEntity.setPositionCode(x);
            mdmPositionRoleEntity.setRoleCode(roleCode);
            mdmPositionRoleEntity.setId(roleCode + x);
            return mdmPositionRoleEntity;
        }).collect(Collectors.toList());
        this.saveBatch(collect);
    }

    @Override
    public void removePositionCode(String positionCode) {
        if (StringUtils.isNotEmpty(positionCode)) {
            this.removePositionCodes(Collections.singletonList(positionCode));
        }
    }

    @Override
    public void removePositionCodes(List<String> positionCodes) {
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodes)) {
            this.lambdaUpdate()
                    .in(MdmPositionRoleEntity::getPositionCode, positionCodes)
                    .remove();
        }
    }

    @Override
    public void removeRoleCode(String roleCode) {
        if (StringUtils.isNotEmpty(roleCode)) {
            this.removeRoleCodes(Collections.singletonList(roleCode));
        }
    }

    @Override
    public void removeRoleCodes(List<String> roleCodes) {
        if (CollectionUtil.listNotEmptyNotSizeZero(roleCodes)) {
            this.lambdaUpdate()
                    .in(MdmPositionRoleEntity::getRoleCode, roleCodes)
                    .remove();
        }
    }

    @Override
    public void removeRelation(List<String> ids) {
        if (CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            this.lambdaUpdate()
                    .in(MdmPositionRoleEntity::getId, ids)
                    .remove();
        }
    }

    @Override
    public void removeRolePositions(String roleCode, List<String> positionCodeList) {
        if (StringUtils.isNotEmpty(roleCode) && CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            this.lambdaUpdate()
                    .eq(MdmPositionRoleEntity::getRoleCode, roleCode)
                    .in(MdmPositionRoleEntity::getPositionCode, positionCodeList)
                    .remove();
        }
    }

    @Override
    public void replacePositionsRole(List<String> positionCodeList, String originRoleCode, String roleCode) {
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList) && StringUtils.isNotEmpty(originRoleCode) && StringUtils.isNotEmpty(roleCode)) {
            this.lambdaUpdate()
                    .in(MdmPositionRoleEntity::getPositionCode, positionCodeList)
                    .eq(MdmPositionRoleEntity::getRoleCode, originRoleCode)
                    .remove();
            this.lambdaUpdate()
                    .in(MdmPositionRoleEntity::getPositionCode, positionCodeList)
                    .eq(MdmPositionRoleEntity::getRoleCode, roleCode)
                    .remove();
            this.addPosition(positionCodeList, roleCode);
        }
    }

    @Override
    public List<String> getRoleRelationPositionCodeList(List<String> roleCodes) {
        if (CollectionUtil.listNotEmptyNotSizeZero(roleCodes)) {
            List<MdmPositionRoleEntity> list = this.lambdaQuery()
                    .in(MdmPositionRoleEntity::getRoleCode, roleCodes)
                    .select(MdmPositionRoleEntity::getPositionCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return list.stream().map(MdmPositionRoleEntity::getPositionCode).distinct().collect(Collectors.toList());
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmRoleRespVo> getPositionRoleList(String positionCode) {
        if (StringUtils.isNotEmpty(positionCode)) {
            List<MdmPositionRoleEntity> positionRoleEntityList = this.lambdaQuery()
                    .eq(MdmPositionRoleEntity::getPositionCode, positionCode)
                    .select(MdmPositionRoleEntity::getRoleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(positionRoleEntityList)) {
                return mdmRoleService.queryBatchByRoleCodeList(positionRoleEntityList.stream().map(MdmPositionRoleEntity::getRoleCode).collect(Collectors.toList()));
            }
        }
        return new ArrayList<>();
    }

    @Override
    public Map<String, List<String>> getPositionRoleCodeGroupByPositionCodeList(List<String> positionCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            List<MdmPositionRoleEntity> list = this.lambdaQuery()
                    .in(MdmPositionRoleEntity::getPositionCode, positionCodeList)
                    .select(MdmPositionRoleEntity::getPositionCode, MdmPositionRoleEntity::getRoleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return list.stream().collect(Collectors.groupingBy(MdmPositionRoleEntity::getPositionCode, Collectors.mapping(MdmPositionRoleEntity::getRoleCode, Collectors.toList())));
            }
        }
        return new HashMap<>(16);
    }

    @Override
    public Map<String, List<MdmRoleRespVo>> getPositionRoleGroupByPositionCodeList(List<String> positionCodeList) {
        Map<String, List<MdmRoleRespVo>> map = new HashMap<>(16);
        if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
            List<List<String>> partition = Lists.partition(positionCodeList, 500);
            for (List<String> positionCodeListItem :
                    partition) {
                List<MdmPositionRoleEntity> list = this.lambdaQuery()
                        .in(MdmPositionRoleEntity::getPositionCode, positionCodeListItem)
                        .select(MdmPositionRoleEntity::getPositionCode, MdmPositionRoleEntity::getRoleCode)
                        .list();
                if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                    final Map<String, MdmRoleRespVo> roleMap = mdmRoleService.queryBatchByRoleCodeList(list.stream().map(MdmPositionRoleEntity::getRoleCode).collect(Collectors.toList()))
                            .stream().collect(Collectors.toMap(MdmRoleRespVo::getRoleCode, v -> v));
                    Map<String, List<String>> groupMap = list.stream().collect(Collectors.groupingBy(MdmPositionRoleEntity::getPositionCode, Collectors.mapping(MdmPositionRoleEntity::getRoleCode, Collectors.toList())));
                    for (Map.Entry<String, List<String>> entry :
                            groupMap.entrySet()) {
                        List<MdmRoleRespVo> roleList = new ArrayList<>();
                        for (String item :
                                entry.getValue()) {
                            if (roleMap.containsKey(item)) {
                                roleList.add(roleMap.get(item));
                            }
                        }
                        map.put(entry.getKey(), roleList);
                    }
                }
            }
        }
        return map;
    }
}
