package com.biz.crm.positionlevel.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.annotation.Klock;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CodeRuleEnum;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.nebular.mdm.positionlevel.req.MdmPositionLevelReqVo;
import com.biz.crm.nebular.mdm.positionlevel.req.MdmPositionLevelSelectReqVo;
import com.biz.crm.nebular.mdm.positionlevel.resp.MdmPositionLevelRespVo;
import com.biz.crm.nebular.mdm.positionlevel.resp.MdmPositionLevelSelectRespVo;
import com.biz.crm.nebular.mdm.role.resp.MdmRoleRespVo;
import com.biz.crm.position.model.MdmPositionEntity;
import com.biz.crm.position.service.MdmPositionService;
import com.biz.crm.positionlevel.mapper.MdmPositionLevelMapper;
import com.biz.crm.positionlevel.model.MdmPositionLevelEntity;
import com.biz.crm.positionlevel.model.MdmPositionLevelRoleEntity;
import com.biz.crm.positionlevel.service.MdmPositionLevelRoleService;
import com.biz.crm.positionlevel.service.MdmPositionLevelService;
import com.biz.crm.role.mode.MdmRoleEntity;
import com.biz.crm.role.service.MdmRoleService;
import com.biz.crm.util.*;
import com.biz.crm.utils.MdmConstant;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 职位级别 接口实现
 *
 * @author Tao.Chen
 * @date 2020-10-22 10:56:51
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmPositionLevelServiceExpandImpl")
public class MdmPositionLevelServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmPositionLevelMapper, MdmPositionLevelEntity> implements MdmPositionLevelService {

    @Resource
    private MdmPositionLevelMapper mdmPositionLevelMapper;
    @Resource
    private MdmPositionLevelRoleService mdmPositionLevelRoleService;
    @Resource
    private MdmRoleService mdmRoleService;
    @Resource
    private MdmPositionService mdmPositionService;
    @Autowired
    private CrmLogSendUtil crmLogSendUtil;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<MdmPositionLevelRespVo> findList(MdmPositionLevelReqVo reqVo) {


        Page<MdmPositionLevelRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmPositionLevelRespVo> list = mdmPositionLevelMapper.findList(page, reqVo);
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            List<String> positionLevelCodeList = list.stream().map(MdmPositionLevelRespVo::getPositionLevelCode).collect(Collectors.toList());

            List<MdmPositionLevelRoleEntity> positionLevelRoleEntityList = mdmPositionLevelRoleService.lambdaQuery()
                    .in(MdmPositionLevelRoleEntity::getPositionLevelCode, positionLevelCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(positionLevelRoleEntityList)) {
                List<MdmRoleEntity> roleEntityList = mdmRoleService.lambdaQuery()
                        .in(MdmRoleEntity::getRoleCode, positionLevelRoleEntityList.stream().map(MdmPositionLevelRoleEntity::getRoleCode).collect(Collectors.toSet()))
                        .select(MdmRoleEntity::getRoleCode, MdmRoleEntity::getRoleName)
                        .list();
                if (CollectionUtil.listNotEmptyNotSizeZero(roleEntityList)) {
                    final Map<String, MdmRoleEntity> roleMap = roleEntityList.stream().collect(Collectors.toMap(MdmRoleEntity::getRoleCode, v->v));
                    final Map<String, List<MdmPositionLevelRoleEntity>> roleGroupMap = positionLevelRoleEntityList.stream().collect(Collectors.groupingBy(MdmPositionLevelRoleEntity::getPositionLevelCode));
                    list.forEach(item -> {
                        if (roleGroupMap.containsKey(item.getPositionLevelCode())) {
                            List<MdmPositionLevelRoleEntity> roleItemList = roleGroupMap.get(item.getPositionLevelCode());
                            List<MdmRoleEntity> roleList = new ArrayList<>();
                            roleItemList.forEach(roleItem -> {
                                if (roleMap.containsKey(roleItem.getRoleCode())) {
                                    roleList.add(roleMap.get(roleItem.getRoleCode()));
                                }
                            });
                            item.setRoleCode(roleList.stream().map(MdmRoleEntity::getRoleCode).collect(Collectors.joining(",")));
                            item.setRoleName(roleList.stream().map(MdmRoleEntity::getRoleName).collect(Collectors.joining(",")));
                        }
                    });
                }
            }
        }
        return PageResult.<MdmPositionLevelRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public MdmPositionLevelRespVo queryForUpdate(String id) {
        if (StringUtils.isNotEmpty(id)) {
            MdmPositionLevelEntity entity = this.getById(id);
            Assert.notNull(entity, "职位级别不存在");
            MdmPositionLevelRespVo respVo = CrmBeanUtil.copy(entity, MdmPositionLevelRespVo.class);
            List<MdmPositionLevelRoleEntity> positionLevelRoleEntityList = mdmPositionLevelRoleService.lambdaQuery()
                    .eq(MdmPositionLevelRoleEntity::getPositionLevelCode, entity.getPositionLevelCode())
                    .select(MdmPositionLevelRoleEntity::getRoleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(positionLevelRoleEntityList)) {
                List<MdmRoleEntity> roleEntityList = mdmRoleService.lambdaQuery()
                        .in(MdmRoleEntity::getRoleCode, positionLevelRoleEntityList.stream().map(MdmPositionLevelRoleEntity::getRoleCode).collect(Collectors.toList()))
                        .list();
                List<MdmRoleRespVo> roleList = CrmBeanUtil.copyList(roleEntityList, MdmRoleRespVo.class);
                respVo.setRoleCode(roleList.stream().map(MdmRoleRespVo::getRoleCode).collect(Collectors.joining(",")));
                respVo.setRoleName(roleList.stream().map(MdmRoleRespVo::getRoleName).collect(Collectors.joining(",")));
                respVo.setRoleList(roleList);
            }
            return respVo;
        }
        return null;
    }

    @Override
    public MdmPositionLevelRespVo detail(String id, String positionLevelCode) {
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(positionLevelCode)) {
            MdmPositionLevelEntity entity = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(id), MdmPositionLevelEntity::getId, id)
                    .eq(StringUtils.isNotEmpty(positionLevelCode), MdmPositionLevelEntity::getPositionLevelCode, positionLevelCode)
                    .one();
            Assert.notNull(entity, "职位级别不存在");
            MdmPositionLevelRespVo respVo = CrmBeanUtil.copy(entity, MdmPositionLevelRespVo.class);
            List<MdmPositionLevelRoleEntity> positionLevelRoleEntityList = mdmPositionLevelRoleService.lambdaQuery()
                    .eq(MdmPositionLevelRoleEntity::getPositionLevelCode, entity.getPositionLevelCode())
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(positionLevelRoleEntityList)) {
                List<MdmRoleEntity> roleEntityList = mdmRoleService.lambdaQuery()
                        .eq(MdmRoleEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                        .in(MdmRoleEntity::getRoleCode, positionLevelRoleEntityList.stream().map(MdmPositionLevelRoleEntity::getRoleCode).collect(Collectors.toList()))
                        .list();
                List<MdmRoleRespVo> roleList = CrmBeanUtil.copyList(roleEntityList, MdmRoleRespVo.class);
                respVo.setRoleCode(roleList.stream().map(MdmRoleRespVo::getRoleCode).collect(Collectors.joining(",")));
                respVo.setRoleName(roleList.stream().map(MdmRoleRespVo::getRoleName).collect(Collectors.joining(",")));
                respVo.setRoleList(roleList);
            }
            return respVo;
        }
        return null;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmPositionLevelReqVo reqVo) {
            reqVo.setId(null);
        Assert.hasText(reqVo.getPositionLevelName(), "缺失职位级别名称");
//        Assert.hasText(reqVo.getRoleCode(), "缺失角色编码");
        String nameNum = this.mdmPositionLevelMapper.queryPositionLevelNameOnlyOne(reqVo.getPositionLevelName());
        AssertUtils.isTrue(StringUtils.getZero().equals(nameNum),"当前职位级别名称已存在，请重新输入");

        MdmPositionLevelEntity entity = CrmBeanUtil.copy(reqVo, MdmPositionLevelEntity.class);
        if (StringUtils.isEmpty(entity.getPositionLevelCode())) {
            entity.setPositionLevelCode(CodeUtil.generateCode(CodeRuleEnum.MDM_POSITION_LEVEL_CODE.getCode()));
        } else {
            List<MdmPositionLevelEntity> list = this.lambdaQuery()
                    .eq(MdmPositionLevelEntity::getPositionLevelCode, entity.getPositionLevelCode())
                    .select(MdmPositionLevelEntity::getPositionLevelCode)
                    .list();
            Assert.isTrue(CollectionUtil.listEmpty(list), "职位级别编码已经存在");
            Integer count = this.lambdaQuery()
                    .eq(MdmPositionLevelEntity::getPositionLevelName, reqVo.getPositionLevelName())
                    .count();
            Assert.isTrue(count == null || count < 1, "职位级别名称已存在");
        }
        entity.setSuffixSequence(0);
        this.save(entity);
        reqVo.setId(entity.getId());
        if (StringUtils.isNotEmpty(reqVo.getRoleCode())) {
            mdmPositionLevelRoleService.addPositionLevelRole(entity.getPositionLevelCode(), Arrays.asList(reqVo.getRoleCode().split(",")));
        }

        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            //记录日志
            crmLogSendUtil.sendForAdd(menuObject.toString(), entity.getId(), entity.getPositionLevelCode(), CrmBeanUtil.copy(entity, MdmPositionLevelReqVo.class));
        }
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmPositionLevelReqVo reqVo) {
        Assert.hasText(reqVo.getId(), "缺失id");
        MdmPositionLevelEntity entity = this.getById(reqVo.getId());
        MdmPositionLevelReqVo oldObject = CrmBeanUtil.copy(entity, MdmPositionLevelReqVo.class);
        Assert.notNull(entity, "职位级别不存在");
        Assert.hasText(reqVo.getPositionLevelCode(), "缺失职位级别编码");
        Assert.hasText(reqVo.getPositionLevelName(), "缺失职位级别名称");
        Assert.isTrue(entity.getPositionLevelCode().equals(reqVo.getPositionLevelCode()), "职位级别编码不能修改");
        CrmBeanUtil.copyProperties(reqVo, entity);

        List<MdmPositionLevelEntity> collect = this.lambdaQuery()
                .eq(MdmPositionLevelEntity::getPositionLevelName, reqVo.getPositionLevelName())
                .select(MdmPositionLevelEntity::getId)
                .list()
                .stream().filter(x -> !reqVo.getId().equals(x.getId())).collect(Collectors.toList());
        Assert.isTrue(CollectionUtil.listEmpty(collect), "职位级别名称已存在");

        this.updateById(entity);
        mdmPositionLevelRoleService.removePositionLevel(entity.getPositionLevelCode());
        if (StringUtils.isNotEmpty(reqVo.getRoleCode())) {
            mdmPositionLevelRoleService.addPositionLevelRole(entity.getPositionLevelCode(), Arrays.asList(reqVo.getRoleCode().split(",")));
        }

        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            //记录日志
            crmLogSendUtil.sendForUpdate(menuObject.toString(),
                    entity.getId(), oldObject.getPositionLevelCode(), oldObject, CrmBeanUtil.copy(entity, MdmPositionLevelReqVo.class));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        Assert.notEmpty(ids, "请传入参数");
        List<MdmPositionLevelEntity> list = mdmPositionLevelMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(list)) {
            Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
            list.forEach(o -> {
                //TODO 需要优化，不要循环
                List<MdmPositionEntity> positionList = mdmPositionService.lambdaQuery().eq(MdmPositionEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode()).eq(MdmPositionEntity::getPositionLevelCode, o.getPositionLevelCode()).list();
                Assert.isTrue(CollectionUtil.listEmpty(positionList), "职位级别" + o.getPositionLevelName() + "已经关联了职位，不能删除");
                mdmPositionLevelRoleService.removePositionLevel(o.getPositionLevelCode());
                this.removeById(o.getId());
                if (menuObject != null) {
                    //记录日志
                    crmLogSendUtil.sendForDel(menuObject.toString(),
                            o.getId(), o.getPositionLevelCode(), o);
                }
            });
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        Assert.notEmpty(ids, "ids不能为空");
        this.lambdaUpdate()
                .in(MdmPositionLevelEntity::getId, ids)
                .set(MdmPositionLevelEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .update();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        Assert.notEmpty(ids, "ids不能为空");
        this.lambdaUpdate()
                .in(MdmPositionLevelEntity::getId, ids)
                .set(MdmPositionLevelEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
                .update();
    }

    @Override
    public List<MdmPositionLevelSelectRespVo> selectList(MdmPositionLevelSelectReqVo reqVo) {
        Integer pageSize = reqVo.getPageSize();
        if (pageSize == null || pageSize == 0) {
            pageSize = Integer.valueOf(ParamUtil.getParameterValue(ParameterParam.SELECT_BOX_OPTION_SIZE));
        }
        List<MdmPositionLevelSelectRespVo> list = new ArrayList<>();
        Set<String> codeList = new HashSet<>(16);
        if (org.apache.commons.lang3.StringUtils.isNotEmpty(reqVo.getSelectedCode())) {
            codeList.add(reqVo.getSelectedCode());
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSelectedCodeList())) {
            codeList.addAll(reqVo.getSelectedCodeList());
        }
        if (!codeList.isEmpty()) {
            reqVo.setSelectedCodeList(new ArrayList<>(codeList));
        }
        if (pageSize > 0) {
            list.addAll(mdmPositionLevelMapper.positionLevelSelectList(new Page<>(1L, pageSize, false), reqVo));
        }
        return list;
    }

    @Override
    @Klock(keys = {MdmConstant.POSITION_LEVEL_GENERATE_POSITION_NAME, "#positionLevelCode"}, waitTime = 5, leaseTime = 5)
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRES_NEW)
    public String generatePositionNameByPositionLevelCode(String positionLevelCode) {
        Assert.hasText(positionLevelCode, "缺失职位级别编码");
        MdmPositionLevelEntity positionLevel = this.lambdaQuery()
                .eq(MdmPositionLevelEntity::getPositionLevelCode, positionLevelCode)
                .select(MdmPositionLevelEntity::getPositionLevelCode, MdmPositionLevelEntity::getPositionLevelName, MdmPositionLevelEntity::getSuffixSequence, MdmPositionLevelEntity::getEnableStatus)
                .one();
        Assert.notNull(positionLevel, "无效的职位级别");
        Assert.isTrue(CrmEnableStatusEnum.ENABLE.getCode().equals(positionLevel.getEnableStatus()), "该职位级别已停用");

        Integer suffixSequence = positionLevel.getSuffixSequence();
        if (suffixSequence == null) {
            suffixSequence = 0;
        }
        Set<String> positionNameSet = new HashSet<>(16);
        List<MdmPositionEntity> positionList = mdmPositionService.lambdaQuery()
                .like(MdmPositionEntity::getPositionName, positionLevel.getPositionLevelName())
                .select(MdmPositionEntity::getPositionName)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(positionList)) {
            positionNameSet.addAll(positionList.stream().map(MdmPositionEntity::getPositionName).collect(Collectors.toSet()));
        }

        String positionName = positionLevel.getPositionLevelName() + "_" + (++suffixSequence);
        while (true) {
            if (positionNameSet.isEmpty() || !positionNameSet.contains(positionName)) {
                break;
            }
            positionName = positionLevel.getPositionLevelName() + "_" + (++suffixSequence);
        }
        this.lambdaUpdate()
                .eq(MdmPositionLevelEntity::getPositionLevelCode, positionLevelCode)
                .set(MdmPositionLevelEntity::getSuffixSequence, suffixSequence)
                .update();
        return positionName;
    }

}
