package com.biz.crm.pricesetting.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BaseIdEntity;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.customer.service.MdmCustomerMsgService;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.mdm.PriceSettingFieldEnum;
import com.biz.crm.eunm.mdm.PriceSettingStatusEnum;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgReqVo;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgRespVo;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgSelectRespVo;
import com.biz.crm.nebular.mdm.pricesetting.req.MdmPriceSearchQueryVo;
import com.biz.crm.nebular.mdm.pricesetting.req.MdmPriceSearchReqVo;
import com.biz.crm.nebular.mdm.pricesetting.req.MdmPriceSettingReqVo;
import com.biz.crm.nebular.mdm.pricesetting.resp.MdmPriceConditionGroupRelFieldRespVo;
import com.biz.crm.nebular.mdm.pricesetting.resp.MdmPriceResp;
import com.biz.crm.nebular.mdm.pricesetting.resp.MdmPriceSettingRespVo;
import com.biz.crm.nebular.mdm.product.req.MdmProductReqVo;
import com.biz.crm.nebular.mdm.product.resp.MdmProductRespVo;
import com.biz.crm.pricesetting.mapper.MdmPriceSettingMapper;
import com.biz.crm.pricesetting.model.MdmPriceConditionGroupRelFieldEntity;
import com.biz.crm.pricesetting.model.MdmPriceSettingEntity;
import com.biz.crm.pricesetting.service.IMdmPriceSettingService;
import com.biz.crm.pricesetting.service.MdmPriceConditionGroupRelFieldService;
import com.biz.crm.pricesetting.service.MdmPriceConditionTypeRelGroupService;
import com.biz.crm.product.service.MdmProductService;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 价格维护主表 接口实现
 *
 * @author huanglong
 * @date 2020-10-20 13:49:03
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmPriceSettingServiceExpandImpl")
public class MdmPriceSettingServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmPriceSettingMapper, MdmPriceSettingEntity> implements IMdmPriceSettingService {

    @Resource
    private MdmPriceSettingMapper mdmPriceSettingMapper;

    @Resource
    private MdmPriceSettingServiceHelper mdmPriceSettingServiceHelper;

    @Resource
    private MdmProductService mdmProductService;

    @Resource
    private MdmCustomerMsgService mdmCustomerMsgService;

    @Resource
    private MdmPriceConditionGroupRelFieldService mdmPriceConditionGroupRelFieldService;

    private final String DATE_REGEX = "^((([0-9]{3}[1-9]|[0-9]{2}[1-9][0-9]{1}|[0-9]{1}[1-9][0-9]{2}|[1-9][0-9]{3})-(((0[13578]|1[02])-(0[1-9]|[12][0-9]|3[01]))|((0[469]|11)-(0[1-9]|[12][0-9]|30))|(02-(0[1-9]|[1][0-9]|2[0-8]))))|((([0-9]{2})(0[48]|[2468][048]|[13579][26])|((0[48]|[2468][048]|[3579][26])00))-02-29))$";

    @Resource
    private MdmPriceConditionTypeRelGroupService mdmPriceConditionTypeRelGroupService;

    @Resource
    private IMdmPriceSettingService mdmPriceSettingService;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;


    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    @CrmDictMethod
    public PageResult<MdmPriceSettingRespVo> findList(MdmPriceSettingReqVo reqVo) {
        Page<MdmPriceSettingRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmPriceSettingRespVo> list = mdmPriceSettingMapper.findList(page, reqVo);
        if (!CollectionUtils.isEmpty(list)) {
            String now = LocalDate.now().format(DateTimeFormatter.ofPattern(DateUtil.DEFAULT_DAY_PATTERN));
            for (MdmPriceSettingRespVo mdmPriceSettingRespVo : list) {
                String beginDate = mdmPriceSettingRespVo.getBeginDate();
                String endDate = mdmPriceSettingRespVo.getEndDate();
                if (!StringUtils.isEmpty(beginDate) && !StringUtils.isEmpty(endDate)) {
                    if (endDate.compareTo(now) < 0 && beginDate.compareTo(now) < 0) {
                        mdmPriceSettingRespVo.setPriceStateName(PriceSettingStatusEnum.EXPIRED.getDes());
                    }
                    if (endDate.compareTo(now) > 0 && endDate.compareTo(now) > 0) {
                        mdmPriceSettingRespVo.setPriceStateName(PriceSettingStatusEnum.TO_BE_EFFECTIVE.getDes());
                    }
                    if (endDate.compareTo(now) >= 0 && beginDate.compareTo(now) <= 0) {
                        mdmPriceSettingRespVo.setPriceStateName(PriceSettingStatusEnum.IN_EFFECT.getDes());
                    }
                }
            }
        }
        return PageResult.<MdmPriceSettingRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public Boolean checkTimeTruncation(MdmPriceSettingReqVo reqVo) {
        String priceSettingCode = reqVo.getPriceSettingCode();
        if(!StringUtils.isEmpty(priceSettingCode)){
            Integer count = this.lambdaQuery()
                    .eq(MdmPriceSettingEntity::getPriceSettingCode, reqVo.getPriceSettingCode())
                    .count();
            if (count != null && count > 0) {
                return true;
            }
        }
        return mdmPriceSettingServiceHelper.checkTimeTruncation(reqVo);
    }

    /**
     * 查询
     *
     * @param reqVo
     * @return mdmPriceSettingRespVo
     */
    @Override
    @CrmDictMethod
    public MdmPriceSettingRespVo query(MdmPriceSettingReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getId(), "查询编码不能为空");
        MdmPriceSettingEntity mdmPriceSettingEntity = mdmPriceSettingMapper.selectById(reqVo.getId());
        if (ObjectUtils.isEmpty(mdmPriceSettingEntity)) {
            return null;
        }
        MdmPriceSettingRespVo mdmPriceSettingRespVo = CrmBeanUtil.copy(mdmPriceSettingEntity, MdmPriceSettingRespVo.class);
        if (StringUtils.isNotEmpty(mdmPriceSettingRespVo.getProductCode())) {
            MdmProductRespVo mdmProductRespVo = mdmProductService.detail(null, mdmPriceSettingRespVo.getProductCode());
            if (!ObjectUtils.isEmpty(mdmProductRespVo)) {
                mdmPriceSettingRespVo.setMdmProductRespVo(mdmProductRespVo);
            }
        }

        List<MdmPriceConditionGroupRelFieldRespVo> mdmPriceConditionGroupRelFieldRespVoList = new ArrayList<>();
        if (StringUtils.isNotEmpty(mdmPriceSettingEntity.getConditionGroupCode())) {
            List<MdmPriceConditionGroupRelFieldEntity> list = mdmPriceConditionGroupRelFieldService.lambdaQuery()
                    .eq(MdmPriceConditionGroupRelFieldEntity::getConditionGroupCode, mdmPriceSettingEntity.getConditionGroupCode())
                    .select(MdmPriceConditionGroupRelFieldEntity::getConditionGroupCode, MdmPriceConditionGroupRelFieldEntity::getFieldCode, MdmPriceConditionGroupRelFieldEntity::getFieldName)
                    .list();

            if (list != null) {
                mdmPriceConditionGroupRelFieldRespVoList.addAll(CrmBeanUtil.copyList(list, MdmPriceConditionGroupRelFieldRespVo.class));
            }
        }
        mdmPriceSettingRespVo.setMdmPriceConditionGroupRelFieldRespVos(mdmPriceConditionGroupRelFieldRespVoList);
        return mdmPriceSettingRespVo;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmPriceSettingReqVo reqVo) {
        mdmPriceSettingServiceHelper.saveCheck(reqVo);
        //保存数据
        mdmPriceSettingServiceHelper.dataUniqueSaveOrUpdate(reqVo);
    }

    /**
     * 新增时进行时间交叉校验，如果有时间交叉则返回true
     *
     * @param mdmPriceSettingReqVo 请求VO
     * @return 是否与之前的数据的时间有交叉
     */
    @Override
    public Boolean checkDate(MdmPriceSettingReqVo mdmPriceSettingReqVo) {
        //1.参数校验
        mdmPriceSettingServiceHelper.saveCheck(mdmPriceSettingReqVo);
        //2.查询同类产品的价格设定
        QueryWrapper<MdmPriceSettingEntity> wrapper = new QueryWrapper<>();
        wrapper.eq("condition_type_code", mdmPriceSettingReqVo.getConditionTypeCode());
        wrapper.eq("condition_group_code", mdmPriceSettingReqVo.getConditionGroupCode());
        mdmPriceSettingServiceHelper.setFieldCondition(wrapper, mdmPriceSettingReqVo);
        List<MdmPriceSettingEntity> mdmPriceSettingEntities = mdmPriceSettingMapper.selectList(wrapper);
        boolean flag = false;
        if (CollectionUtil.listNotEmpty(mdmPriceSettingEntities)) {
            //循环进行时间交叉比较
            long dateStart = DateUtil.str2Date(mdmPriceSettingReqVo.getBeginDate().trim().concat(" ").concat(mdmPriceSettingReqVo.getBeginDateSecond()), DateUtil.datetimeFormat).getTime();
            long dateEnd = DateUtil.str2Date(mdmPriceSettingReqVo.getEndDate().trim().concat(" ").concat(mdmPriceSettingReqVo.getEndDateSecond()), DateUtil.datetimeFormat).getTime();
            for (MdmPriceSettingEntity entity : mdmPriceSettingEntities) {
                //如果传递的数据带有ID，那么结果集中有该ID的旧数据将不再进行交叉筛选
                if (StringUtils.isNotEmpty(mdmPriceSettingReqVo.getId()) && mdmPriceSettingReqVo.getId().equalsIgnoreCase(entity.getId())) {
                    continue;
                }
                //新增价格结束时间<旧数据价格起始时间或新增价格起始时间>旧数据价格结束时间
                long oldStart = DateUtil.str2Date(entity.getBeginDate().trim().concat(" ").concat(entity.getBeginDateSecond()), DateUtil.datetimeFormat).getTime();
                long oldEnd = DateUtil.str2Date(entity.getEndDate().trim().concat(" ").concat(entity.getEndDateSecond()), DateUtil.datetimeFormat).getTime();
                if (!(dateEnd < oldStart || dateStart > oldEnd)) {
                    flag = true;
                    break;
                }
            }
        }
        return flag;
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmPriceSettingReqVo reqVo) {
        mdmPriceSettingServiceHelper.saveCheck(reqVo);
        //修改数据
        mdmPriceSettingServiceHelper.dataUniqueSaveOrUpdate(reqVo);
    }

    /**
     * 删除
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(MdmPriceSettingReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getIds(), "删除编码集合不能为空");
        List<MdmPriceSettingEntity> mdmPriceSettingEntities = mdmPriceSettingMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(mdmPriceSettingEntities)) {
            mdmPriceSettingMapper.deleteBatchIds(reqVo.getIds());
        }
    }

    /**
     * 启用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(MdmPriceSettingReqVo reqVo) {
        List<String> ids = reqVo.getIds();
        Assert.notEmpty(ids, "id集合不能为空");
        List<MdmPriceSettingEntity> list = this.lambdaQuery()
                .in(MdmPriceSettingEntity::getId, ids)
                .list();
        if(CollectionUtils.isEmpty(list)){
            return;
        }
        List<String> collect = list.stream()
                .filter(x -> !CrmEnableStatusEnum.ENABLE.getCode().equals(x.getEnableStatus()))
                .map(BaseIdEntity::getId)
                .collect(Collectors.toList());
        List<MdmPriceSettingEntity> entities = list.stream()
                .filter(x -> !CrmEnableStatusEnum.ENABLE.getCode().equals(x.getEnableStatus()))
                .collect(Collectors.toList());
        if(CollectionUtils.isEmpty(collect)){
            return;
        }
        this.lambdaUpdate()
                .in(MdmPriceSettingEntity::getId, collect)
                .set(MdmPriceSettingEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .update();
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuCodeObj != null) {
            for (MdmPriceSettingEntity entity : entities) {
                MdmPriceSettingEntity newEntity = CrmBeanUtil.copy(entities, MdmPriceSettingEntity.class);
                newEntity.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
                crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),
                        entity.getId(), entity.getId(), entity, newEntity);
            }
        }
    }

    /**
     * 禁用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(MdmPriceSettingReqVo reqVo) {
        List<String> ids = reqVo.getIds();
        Assert.notEmpty(ids, "id集合不能为空");
        List<MdmPriceSettingEntity> list = this.lambdaQuery()
                .in(MdmPriceSettingEntity::getId, ids)
                .list();
        if(CollectionUtils.isEmpty(list)){
            return;
        }
        List<String> collect = list.stream()
                .filter(x -> !CrmEnableStatusEnum.DISABLE.getCode().equals(x.getEnableStatus()))
                .map(BaseIdEntity::getId)
                .collect(Collectors.toList());
        List<MdmPriceSettingEntity> entities = list.stream()
                .filter(x -> !CrmEnableStatusEnum.DISABLE.getCode().equals(x.getEnableStatus()))
                .collect(Collectors.toList());
        if(CollectionUtils.isEmpty(collect)){
            return;
        }
        this.lambdaUpdate()
                .in(MdmPriceSettingEntity::getId, collect)
                .set(MdmPriceSettingEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
                .update();
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuCodeObj != null) {
            for (MdmPriceSettingEntity entity : entities) {
                MdmPriceSettingEntity newEntity = CrmBeanUtil.copy(entities, MdmPriceSettingEntity.class);
                newEntity.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
                crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),
                        entity.getId(), entity.getId(), entity, newEntity);
            }
        }
    }

    @Override
    public Map<String, BigDecimal> findPriceByGoodsAndCusCode(List<String> goodsCodes, String cusCodes) {
        Map<String, BigDecimal> reMap = new HashMap<>();
        if (org.springframework.util.CollectionUtils.isEmpty(goodsCodes)) {
            return reMap;
        }
        for (String goodsCode : goodsCodes) {
            reMap.put(goodsCode, BigDecimal.ONE);
        }

        //TODO 从常量来
        final String salePrice = "sale_price";
        MdmPriceSearchReqVo searchReqVo = new MdmPriceSearchReqVo();
        searchReqVo.setCustomerCode(cusCodes);
        searchReqVo.setProductCodeList(goodsCodes);
        searchReqVo.setSearchTime(DateUtil.dateNow2Str());
        searchReqVo.setConditionTypeCodeList(Collections.singletonList(salePrice));
        Map<String, Map<String, MdmPriceResp>> productMap = mdmPriceSettingService.search(searchReqVo);

        for (Map.Entry<String, Map<String, MdmPriceResp>> entry :
                productMap.entrySet()) {
            BigDecimal price = BigDecimal.ZERO;
            Map<String, MdmPriceResp> priceTypeMap = entry.getValue();
            if (priceTypeMap != null && priceTypeMap.get(salePrice) != null) {
                price = priceTypeMap.get(salePrice).getPrice();
            }
            reMap.put(entry.getKey(), price);
        }
        return reMap;

    }

    @Override
    public List<MdmCustomerMsgSelectRespVo> findCustomerSelectList(MdmCustomerMsgReqVo mdmCustomerMsgReqVo) {
        mdmCustomerMsgReqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        return mdmCustomerMsgService.list(mdmCustomerMsgReqVo);
    }

    @Override
    public List<MdmProductRespVo> findProductSelectList(MdmProductReqVo mdmProductReqVo) {
        return CrmBeanUtil.copyList(mdmProductService.list(), MdmProductRespVo.class);
    }

    @Override
    public Map<String, Map<String, MdmPriceResp>> search(MdmPriceSearchReqVo mdmPriceSearchReqVo) {
        String customerCode = mdmPriceSearchReqVo.getCustomerCode();
        Assert.hasText(customerCode, "客户编码不能为空");
        MdmCustomerMsgRespVo mdmCustomerMsgRespVo = mdmCustomerMsgService.query(null, customerCode);
        if (mdmCustomerMsgRespVo == null) {
            return Collections.emptyMap();
        }
        String orgCode = mdmCustomerMsgRespVo.getOrgCode();
        if(!StringUtils.isEmpty(orgCode)){
            mdmPriceSearchReqVo.setOrgCode(orgCode.split(",")[0]);
        }
        mdmPriceSearchReqVo.setChannel(mdmCustomerMsgRespVo.getChannel());
        mdmPriceSearchReqVo.setPriceGroup(mdmCustomerMsgRespVo.getPriceGroup());
        return inquiry(mdmPriceSearchReqVo);
    }

    @Override
    public MdmPriceSettingRespVo queryUnique(MdmPriceSettingReqVo mdmPriceSettingReqVo) {
        MdmPriceSettingRespVo mdmPriceSettingRespVo = new MdmPriceSettingRespVo();
        QueryWrapper<MdmPriceSettingEntity> wrapper = mdmPriceSettingServiceHelper.wrapperCondition(mdmPriceSettingReqVo);
        if (wrapper.isEmptyOfWhere()) {
            mdmPriceSettingRespVo.setErrorMsg("当前维度包含多条价格数据");
            return mdmPriceSettingRespVo;
        }
        List<MdmPriceSettingEntity> mdmPriceSettingEntities = mdmPriceSettingMapper.selectList(wrapper);
        if (CollectionUtils.isEmpty(mdmPriceSettingEntities)) {
            mdmPriceSettingRespVo.setErrorMsg("当前维度不存在价格数据");
            return mdmPriceSettingRespVo;
        }
        if (mdmPriceSettingEntities.size() > 1) {
            mdmPriceSettingRespVo.setErrorMsg("当前维度包含多条价格数据");
            return mdmPriceSettingRespVo;
        }
        return CrmBeanUtil.copy(mdmPriceSettingEntities.get(0), MdmPriceSettingRespVo.class);
    }


    /**
     * 分类匹配: （当前客户信息数据与价格数据一致时，返回true）
     *
     * @param mdmCustomerMsgRespVo 客户信息
     * @param priceResp            价格
     * @param field                分类编码
     * @return boolean
     */
    protected boolean conditionTypeMatch(MdmCustomerMsgRespVo mdmCustomerMsgRespVo, MdmPriceResp priceResp, String field) {
        switch (PriceSettingFieldEnum.match(field)) {
            case CUSTOMER_CODE:
                String priceCustomerCode = priceResp.getCustomerCode();
                return !StringUtils.isEmpty(priceCustomerCode) && priceCustomerCode.equals(mdmCustomerMsgRespVo.getCustomerCode());
            case ORG_CODE:
                String priceOrgCode = priceResp.getOrgCode();
                return !StringUtils.isEmpty(priceOrgCode) && priceOrgCode.equals(mdmCustomerMsgRespVo.getOrgCode());
            case CHANNEL:
                String priceChannel = priceResp.getChannel();
                return !StringUtils.isEmpty(priceChannel) && priceChannel.equals(mdmCustomerMsgRespVo.getChannel());
            case PRODUCT_CODE:
                return true;
            default:
                return false;
        }
    }

    protected boolean conditionTypeMatch(MdmPriceSearchReqVo mdmPriceSearchReqVo, MdmPriceResp priceResp, String field) {
        switch (PriceSettingFieldEnum.match(field)) {
            case CUSTOMER_CODE:
                return !StringUtils.isEmpty(priceResp.getCustomerCode()) && priceResp.getCustomerCode().equals(mdmPriceSearchReqVo.getCustomerCode());
            case ORG_CODE:
                return !StringUtils.isEmpty(priceResp.getOrgCode()) && priceResp.getOrgCode().equals(mdmPriceSearchReqVo.getOrgCode());
            case CHANNEL:
                return !StringUtils.isEmpty(priceResp.getChannel()) && priceResp.getChannel().equals(mdmPriceSearchReqVo.getChannel());
            case PRODUCT_CODE:
                return true;
            case TERMINAL:
                return !StringUtils.isEmpty(priceResp.getTerminalCode()) && priceResp.getTerminalCode().equals(mdmPriceSearchReqVo.getTerminalCode());
            case PRICE_GROUP:
                return !StringUtils.isEmpty(priceResp.getPriceGroup()) && priceResp.getPriceGroup().equals(mdmPriceSearchReqVo.getPriceGroup());
            case TERMINAL_TYPE:
                return !StringUtils.isEmpty(priceResp.getTerminalType()) && priceResp.getTerminalType().equals(mdmPriceSearchReqVo.getTerminalType());
            case CUSTOMER_TYPE:
                return !StringUtils.isEmpty(priceResp.getCustomerType()) && priceResp.getCustomerType().equals(mdmPriceSearchReqVo.getCustomerType());
            case EXT1:
                return !StringUtils.isEmpty(priceResp.getExt1()) && priceResp.getExt1().equals(mdmPriceSearchReqVo.getExt1());
            case EXT2:
                return !StringUtils.isEmpty(priceResp.getExt2()) && priceResp.getExt2().equals(mdmPriceSearchReqVo.getExt2());
            default:
                return false;
        }
    }


    @Override
    public Map<String, Map<String, MdmPriceResp>> inquiry(MdmPriceSearchReqVo mdmPriceSearchReqVo) {
        List<String> productCodeList = mdmPriceSearchReqVo.getProductCodeList();
        if (CollectionUtils.isEmpty(productCodeList)) {
            return Collections.emptyMap();
        }
        List<String> conditionTypeCodeList = mdmPriceSearchReqVo.getConditionTypeCodeList();
        if (CollectionUtils.isEmpty(conditionTypeCodeList)) {
            return Collections.emptyMap();
        }
        String searchTime = mdmPriceSearchReqVo.getSearchTime();
        if (StringUtils.isEmpty(searchTime)) {
            mdmPriceSearchReqVo.setSearchTime(LocalDate.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
        } else {
            Assert.isTrue(searchTime.matches(DATE_REGEX), "时间格式不正确");
        }
        Map<String, Map<String, MdmPriceResp>> priceMap = productCodeList.stream()
                .filter(x -> !StringUtils.isEmpty(x))
                .distinct()
                .collect(Collectors.toMap(x -> x, x -> new HashMap<>()));

        List<MdmPriceResp> list = searchProductList(mdmPriceSearchReqVo);
        log.info("[价格维护][价格数据]:{}", list);
        if (CollectionUtils.isEmpty(list)) {
            return priceMap;
        }

        Map<String, List<List<String>>> fieldListMap = mdmPriceConditionTypeRelGroupService.groupList(conditionTypeCodeList);
        log.info("[价格维护][高优先级分类编码]:{}", fieldListMap);
        if (ObjectUtils.isEmpty(fieldListMap)) {
            return priceMap;
        }

        return list.stream().collect(Collectors.groupingBy(MdmPriceResp::getProductCode, Collectors.collectingAndThen(Collectors.toList(),
                x -> {
                    if (CollectionUtils.isEmpty(x)) {
                        return null;
                    }
                    return x.stream()
                            .filter(k -> k != null && !StringUtils.isEmpty(k.getConditionTypeCode()))
                            .filter(k -> fieldListMap.containsKey(k.getConditionTypeCode()))
                            .collect(Collectors.groupingBy(MdmPriceResp::getConditionTypeCode
                                    , Collectors.collectingAndThen(Collectors.toList(), conditionTypePriceList -> {
                                                if (CollectionUtils.isEmpty(conditionTypePriceList)) {
                                                    return null;
                                                }
                                        return conditionTypePriceList.stream()
                                                .filter(priceResp -> {
                                                    String conditionTypeCode = priceResp.getConditionTypeCode();
                                                    List<List<String>> typeFieldList = fieldListMap.get(conditionTypeCode);
                                                    if(!CollectionUtils.isEmpty(typeFieldList)){
                                                        for (int i = 0; i < typeFieldList.size(); i++) {
                                                            List<String> fieldList = typeFieldList.get(i);
                                                            if(!CollectionUtils.isEmpty(fieldList)){
                                                                boolean b = fieldList.stream()
                                                                        .allMatch(field -> conditionTypeMatch(mdmPriceSearchReqVo, priceResp, field));
                                                                if (b) {
                                                                    priceResp.setFilterSort(i);
                                                                    return true;
                                                                }
                                                            }
                                                        }
                                                    }
                                                    return false;
                                                })
                                                .min(Comparator.comparing(p -> Optional.ofNullable(p.getFilterSort()).orElse(100)))
                                                .orElse(null);
                                            }
                                    )));
                })));
    }

    protected List<MdmPriceResp> searchProductList(MdmPriceSearchReqVo mdmPriceSearchReqVo) {
        MdmPriceSearchQueryVo mdmPriceSearchQueryVo = new MdmPriceSearchQueryVo();
        mdmPriceSearchQueryVo.setProductCodeList(mdmPriceSearchReqVo.getProductCodeList());
        mdmPriceSearchQueryVo.setConditionTypeCodeList(mdmPriceSearchReqVo.getConditionTypeCodeList());
        mdmPriceSearchQueryVo.setSearchTime(mdmPriceSearchReqVo.getSearchTime());
        mdmPriceSearchQueryVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        return mdmPriceSettingMapper.search(mdmPriceSearchQueryVo);
    }


}
