package com.biz.crm.product.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.availablelist.service.CusAvailablelistService;
import com.biz.crm.availablelist.service.TerminalAvailablelistService;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CodeRuleEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.FileTypeEnum;
import com.biz.crm.eunm.IsShelfEnum;
import com.biz.crm.kms.service.MdmProService;
import com.biz.crm.mq.RocketMQConstant;
import com.biz.crm.mq.RocketMQMessageBody;
import com.biz.crm.mq.RocketMQProducer;
import com.biz.crm.nebular.mdm.availablelist.AvalibleListProductLevelChangeDto;
import com.biz.crm.nebular.mdm.availablelist.CusAvailablelistVo;
import com.biz.crm.nebular.mdm.constant.DictConstant;
import com.biz.crm.nebular.mdm.pricesetting.req.MdmPriceSearchReqVo;
import com.biz.crm.nebular.mdm.pricesetting.resp.MdmPriceProductUnitRespVo;
import com.biz.crm.nebular.mdm.pricesetting.resp.MdmPriceResp;
import com.biz.crm.nebular.mdm.product.req.*;
import com.biz.crm.nebular.mdm.product.resp.*;
import com.biz.crm.nebular.mdm.productlevel.resp.MdmProductLevelRespVo;
import com.biz.crm.pricesetting.service.IMdmPriceSettingService;
import com.biz.crm.product.event.ProductChangeEvent;
import com.biz.crm.product.mapper.MdmProductMapper;
import com.biz.crm.product.model.MdmProductEntity;
import com.biz.crm.product.model.MdmProductIntroductionEntity;
import com.biz.crm.product.model.MdmProductMaterialEntity;
import com.biz.crm.product.model.MdmProductMediaEntity;
import com.biz.crm.product.service.IMdmProductIntroductionService;
import com.biz.crm.product.service.MdmProductMaterialService;
import com.biz.crm.product.service.MdmProductMediaService;
import com.biz.crm.product.service.MdmProductService;
import com.biz.crm.productlevel.model.MdmProductLevelEntity;
import com.biz.crm.productlevel.service.MdmProductLevelService;
import com.biz.crm.util.*;
import com.bizunited.platform.common.util.JsonUtils;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.assertj.core.util.Lists;
import org.assertj.core.util.Sets;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 商品主表接口实现
 *
 * @author crm
 * @date 2020-08-27 17:34:16
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmProductServiceExpandImpl")
public class MdmProductServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmProductMapper, MdmProductEntity> implements MdmProductService {

    @Resource
    private MdmProductMapper mdmProductMapper;
    @Autowired
    private MdmProductServiceHelper productServiceHelper;
    @Autowired
    private MdmProductMediaService mdmProductMediaService;
    @Autowired
    private MdmProductMaterialService mdmProductMaterialService;
    @Resource
    private IMdmProductIntroductionService mdmProductIntroductionService;
    @Resource
    private CusAvailablelistService cusAvailablelistService;
    @Resource
    private TerminalAvailablelistService terminalAvailablelistService;
    @Resource
    private MdmProductLevelService mdmProductLevelService;
    @Resource
    private RocketMQProducer rocketMQProducer;
    @Autowired
    private CrmLogSendUtil crmLogSendUtil;
    @Autowired
    private IMdmPriceSettingService mdmPriceSettingService;

    @Autowired
    private MdmProService mdmProService;


    @Value("${rocketmq.topic}" + "${rocketmq.environment-variable}")
    private String topic;

    @Resource
    private ApplicationContext applicationContext;

    /**
     * 列表
     *
     * @param mdmProductReqVo
     * @return
     */
    @Override
    public PageResult<MdmProductRespVo> findList(MdmProductReqVo mdmProductReqVo) {
        Page<MdmProductRespVo> page = PageUtil.buildPage(mdmProductReqVo.getPageNum(), mdmProductReqVo.getPageSize());
        if (StringUtils.isNotEmpty(mdmProductReqVo.getCode())) {
            MdmProductLevelRespVo productLevel = mdmProductLevelService.detail(null, mdmProductReqVo.getCode());
            if (productLevel != null) {
                mdmProductReqVo.setRuleCode(productLevel.getRuleCode());
            }
        }
        List<MdmProductRespVo> list = mdmProductMapper.findList(page, mdmProductReqVo);
        productServiceHelper.convertListDate(list);
        return PageResult.<MdmProductRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param id
     * @param productCode
     * @return mdmProductRespVo
     */
    @Override
    public MdmProductRespVo detail(String id, String productCode) {
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(productCode)) {
            MdmProductEntity one = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(id), MdmProductEntity::getId, id)
                    .eq(StringUtils.isNotEmpty(productCode), MdmProductEntity::getProductCode, productCode)
                    .one();

            if (one != null) {
                MdmProductRespVo respVo = CrmBeanUtil.copy(one, MdmProductRespVo.class);

                //获取产品层级名称
                String productLevelCode = one.getProductLevelCode();
                if (!StringUtils.isEmpty(productLevelCode)) {
                    MdmProductLevelEntity productLevel = mdmProductLevelService.lambdaQuery()
                            .eq(MdmProductLevelEntity::getProductLevelCode, one.getProductLevelCode())
                            .one();
                    if (productLevel != null) {
                        respVo.setProductLevelName(productLevel.getProductLevelName());
                    }
                }
                //查询商品物料
                List<MdmProductMaterialRespVo> materialRespVos = productServiceHelper.findMaterialByProductCode(one.getProductCode());
                respVo.setMaterialList(materialRespVos);

                //查询商品图片,视频
                List<MdmProductMediaRespVo> medias = mdmProductMediaService.list(new MdmProductMediaReqVo().setProductCode(one.getProductCode()));
                if (CollectionUtils.isNotEmpty(medias)) {
                    Map<String, List<MdmProductMediaRespVo>> mediaMap = medias.stream().collect(Collectors.groupingBy(MdmProductMediaRespVo::getType));
                    List<MdmProductMediaRespVo> pictureList = mediaMap.get(FileTypeEnum.PICTURE.getCode());
                    respVo.setPictureList(pictureList);
                    respVo.setVideoList(mediaMap.get(FileTypeEnum.VIDEO.getCode()));
                }
                //查询商品介绍
                MdmProductIntroductionRespVo introduction = productServiceHelper.findIntroductionByProductCode(one.getProductCode());
                respVo.setIntroductionVo(introduction);
                return respVo;
            }
        }
        return null;
    }

    @Override
    public List<MdmProductRespVo> queryBatch(List<String> ids, List<String> productCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(ids) || CollectionUtil.listNotEmptyNotSizeZero(productCodeList)) {
            List<MdmProductEntity> list = this.lambdaQuery()
                    .in(CollectionUtil.listNotEmptyNotSizeZero(ids), MdmProductEntity::getId, ids)
                    .or()
                    .in(CollectionUtil.listNotEmptyNotSizeZero(productCodeList), MdmProductEntity::getProductCode, productCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return CrmBeanUtil.copyList(list, MdmProductRespVo.class);
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmProductRespVo> queryBatchByProductLevelCodeList(List<String> productLevelCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(productLevelCodeList)) {
            List<String> childrenProductLevelCodeListIncludeSelf = ProductLevelUtil.getChildrenProductLevelCodeListIncludeSelf(productLevelCodeList);
            if (CollectionUtil.listNotEmptyNotSizeZero(childrenProductLevelCodeListIncludeSelf)) {
                List<MdmProductEntity> list = this.lambdaQuery()
                        .in(MdmProductEntity::getProductLevelCode, childrenProductLevelCodeListIncludeSelf)
                        .list();
                if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                    return CrmBeanUtil.copyList(list, MdmProductRespVo.class);
                }
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmProductQueryRespVo> queryProductList(MdmProductQueryReqVo reqVo) {
        List<String> allUnderProductLevelCodeList = new ArrayList<>();
        if (StringUtils.isNotEmpty(reqVo.getUnderProductLevelCode())) {
            List<String> childrenProductLevelCodeListIncludeSelfEnable = ProductLevelUtil.getChildrenProductLevelCodeListIncludeSelfEnable(reqVo.getUnderProductLevelCode());
            if (CollectionUtil.listEmpty(childrenProductLevelCodeListIncludeSelfEnable)) {
                return new ArrayList<>();
            }
            allUnderProductLevelCodeList.addAll(childrenProductLevelCodeListIncludeSelfEnable);
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getUnderProductLevelCodeList())) {
            List<String> childrenProductLevelCodeListIncludeSelfEnable = ProductLevelUtil.getChildrenProductLevelCodeListIncludeSelfEnable(reqVo.getUnderProductLevelCodeList());
            if (CollectionUtil.listEmpty(childrenProductLevelCodeListIncludeSelfEnable)) {
                return new ArrayList<>();
            }
            if (CollectionUtil.listNotEmptyNotSizeZero(allUnderProductLevelCodeList)) {
                allUnderProductLevelCodeList.retainAll(childrenProductLevelCodeListIncludeSelfEnable);
                if (CollectionUtil.listEmpty(allUnderProductLevelCodeList)) {
                    return new ArrayList<>();
                }
            } else {
                allUnderProductLevelCodeList.addAll(childrenProductLevelCodeListIncludeSelfEnable);
            }
        }
        List<MdmProductEntity> list = this.lambdaQuery()
                .eq(StringUtils.isNotEmpty(reqVo.getProductLevelCode()), MdmProductEntity::getProductLevelCode, reqVo.getProductLevelCode())
                .eq(StringUtils.isNotEmpty(reqVo.getProductType()), MdmProductEntity::getProductType, reqVo.getProductType())
                .eq(StringUtils.isNotEmpty(reqVo.getSaleUnit()), MdmProductEntity::getSaleUnit, reqVo.getSaleUnit())
                .eq(StringUtils.isNotEmpty(reqVo.getBaseUnit()), MdmProductEntity::getBaseUnit, reqVo.getBaseUnit())
                .eq(StringUtils.isNotEmpty(reqVo.getIsShelf()), MdmProductEntity::getIsShelf, reqVo.getIsShelf())
                .eq(StringUtils.isNotEmpty(reqVo.getBarCode()), MdmProductEntity::getBarCode, reqVo.getBarCode())
                .eq(StringUtils.isNotEmpty(reqVo.getSaleCompany()), MdmProductEntity::getSaleCompany, reqVo.getSaleCompany())
                .eq(StringUtils.isNotEmpty(reqVo.getMaintenancePicture()), MdmProductEntity::getMaintenancePicture, reqVo.getMaintenancePicture())
                .eq(StringUtils.isNotEmpty(reqVo.getMaintenanceIntroduction()), MdmProductEntity::getMaintenanceIntroduction, reqVo.getMaintenanceIntroduction())
                .eq(StringUtils.isNotEmpty(reqVo.getEnableStatus()), MdmProductEntity::getEnableStatus, reqVo.getEnableStatus())
                .in(CollectionUtil.listNotEmptyNotSizeZero(reqVo.getProductCodeList()), MdmProductEntity::getProductCode, reqVo.getProductCodeList())
                .in(CollectionUtil.listNotEmptyNotSizeZero(allUnderProductLevelCodeList), MdmProductEntity::getProductLevelCode, allUnderProductLevelCodeList)
                .in(CollectionUtil.listNotEmptyNotSizeZero(reqVo.getProductLevelCodeList()), MdmProductEntity::getProductLevelCode, reqVo.getProductLevelCodeList())
                .like(StringUtils.isNotEmpty(reqVo.getProductCode()), MdmProductEntity::getProductCode, reqVo.getProductCode())
                .like(StringUtils.isNotEmpty(reqVo.getProductName()), MdmProductEntity::getProductName, reqVo.getProductName())
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            return CrmBeanUtil.copyList(list, MdmProductQueryRespVo.class);
        }
        return new ArrayList<>();
    }

    @SneakyThrows
    @Override
    public List<MdmProductAdviseRespVo> adviseQuery(List<String> productCodeList) {
        if (CollectionUtils.isEmpty(productCodeList)) {
            return new ArrayList<>();
        }
        List<MdmProductAdviseRespVo> list = new ArrayList<>(productCodeList.size());
        for (String productCode : productCodeList) {
            MdmProductRespVo mdmProductRespVo = detail(null, productCode);
            if (mdmProductRespVo != null) {
                MdmProductAdviseRespVo mdmProductAdviseRespVo = CrmBeanUtil.copy(mdmProductRespVo, MdmProductAdviseRespVo.class);

                mdmProductAdviseRespVo.setMaterialList(JsonUtils.obj2JsonString(mdmProductRespVo.getMaterialList()));
                mdmProductAdviseRespVo.setPictureList(JsonUtils.obj2JsonString(mdmProductRespVo.getPictureList()));
                mdmProductAdviseRespVo.setVideoList(JsonUtils.obj2JsonString(mdmProductRespVo.getVideoList()));
                mdmProductAdviseRespVo.setIntroductionVo(JsonUtils.obj2JsonString(mdmProductRespVo.getIntroductionVo()));
                mdmProductAdviseRespVo.setImgUrl(mdmProductRespVo.getPrimaryPictureUrl());
                list.add(mdmProductAdviseRespVo);
            }
        }
        return list;
    }

    @Override
    public Map<String, MdmProductAdviseAndPriceRespVo> queryByProductCodeAndCus(MdmProductAdviseAndPriceReqVo mdmProductAdviseAndPriceReqVo) {
        // 构造价格查询条件
        MdmPriceSearchReqVo mdmPriceSearchReqVo = CrmBeanUtil.copy(mdmProductAdviseAndPriceReqVo, MdmPriceSearchReqVo.class);
        // 产品条件编码变量名不同，所以手动赋值
        List<String> productCodes = mdmProductAdviseAndPriceReqVo.getProductCodes();
        if (CollectionUtils.isEmpty(productCodes)) {
            return Collections.emptyMap();
        }
        mdmPriceSearchReqVo.setProductCodeList(productCodes);
//        Map<String, Map<String, MdmPriceResp>> search = mdmPriceSettingService.search(mdmPriceSearchReqVo);
//        if (search == null || search.size() == 0) {
//            return Collections.emptyMap();
//        }
        Map<String, MdmProductAdviseAndPriceRespVo> collect = mdmPriceSettingService.search(mdmPriceSearchReqVo)
                .entrySet()
                .stream()
                .collect(Collectors.toMap(Map.Entry::getKey,
                        x -> {
                            MdmProductRespVo detail = detail(null, x.getKey());
                            return Optional.ofNullable(CrmBeanUtil.copy(detail, MdmProductAdviseAndPriceRespVo.class))
                                    .map(k -> {
                                        k.setImgUrl(detail.getPrimaryPictureUrl());
                                        Map<String, MdmPriceResp> value = x.getValue();
                                        if (value != null) {
                                            MdmPriceResp price = value.get("sale_price");
                                            if (price != null) {
                                                k.setSalePrice(price.getPrice());
                                            }
                                        }
                                        return k;
                                    })
                                    .orElse(new MdmProductAdviseAndPriceRespVo());
                        })
                );
        for (String productCode : productCodes) {
            collect.putIfAbsent(productCode, null);
        }
        return collect;
    }


    /**
     * 新增
     *
     * @param mdmProductReqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmProductReqVo mdmProductReqVo) {
        AssertUtils.isNotEmpty(mdmProductReqVo.getProductName(), "商品名称不能为空");
        mdmProductReqVo.setId(null);
        if (StringUtils.isEmpty(mdmProductReqVo.getProductCode())) {
            mdmProductReqVo.setProductCode(CodeUtil.generateCode(CodeRuleEnum.MDM_PRODUCT_CODE.getCode()));
        } else {
            List<MdmProductEntity> list = this.lambdaQuery()
                    .eq(MdmProductEntity::getProductCode, mdmProductReqVo.getProductCode())
                    .select(MdmProductEntity::getId)
                    .list();
            Assert.isTrue(CollectionUtil.listEmpty(list), "商品编码已经存在");
        }
        MdmProductBuilder.builder(productServiceHelper, mdmProductReqVo)
                .init()
                .check()
                .convert()
                .save();
        sendChangeMsg(Collections.singletonList(mdmProductReqVo.getProductCode()));
        sendProductLevelChangeMsg(null, mdmProductReqVo);
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            MdmProductEntity one = this.lambdaQuery()
                    .eq(MdmProductEntity::getProductCode, mdmProductReqVo.getProductCode())
                    .one();
            crmLogSendUtil.sendForAdd((String) menuObject, one.getId(), one.getProductCode(), CrmBeanUtil.copy(one, MdmProductReqVo.class));
        }
        mdmProService.add(mdmProductReqVo);


        productChangeSendEvent(ProductChangeHandlerReqVo.builder()
                .productCode(mdmProductReqVo.getProductCode())
                .currentLevelCode(mdmProductReqVo.getProductLevelCode())
                .build());
    }

    /**
     * 更新
     *
     * @param mdmProductReqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmProductReqVo mdmProductReqVo) {
        Assert.hasText(mdmProductReqVo.getId(), "主键不能为空");
        MdmProductEntity one = this.getById(mdmProductReqVo.getId());
        Assert.notNull(one, "未找到商品");
        MdmProductReqVo oldObject = CrmBeanUtil.copy(one, MdmProductReqVo.class);
        Assert.isTrue(IsShelfEnum.DOWN.getCode().equals(one.getIsShelf()), "只能编辑状态为“下架”的商品");
        Assert.isTrue(one.getProductCode().equals(mdmProductReqVo.getProductCode()), "商品编码不能修改");
        MdmProductBuilder.builder(productServiceHelper, mdmProductReqVo)
                .init()
                .check()
                .convert()
                .update();
        sendChangeMsg(Collections.singletonList(mdmProductReqVo.getProductCode()));
        sendProductLevelChangeMsg(CrmBeanUtil.copy(one, MdmProductReqVo.class), mdmProductReqVo);
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            MdmProductEntity newEntity = this.getById(mdmProductReqVo.getId());
            MdmProductReqVo newObject = CrmBeanUtil.copy(newEntity, MdmProductReqVo.class);
            crmLogSendUtil.sendForUpdate((String) menuObject, one.getId(), one.getProductCode(), oldObject, newObject);
        }
        mdmProService.update(mdmProductReqVo);

        productChangeSendEvent(ProductChangeHandlerReqVo.builder()
                .productCode(mdmProductReqVo.getProductCode())
                .oldLevelCode(oldObject.getProductLevelCode())
                .currentLevelCode(mdmProductReqVo.getProductLevelCode())
                .build());
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        Assert.notEmpty(ids, "请传入参数");
        List<MdmProductEntity> list = this.lambdaQuery()
                .in(MdmProductEntity::getId, ids)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);

            list.forEach(item -> {
                Assert.isTrue(!IsShelfEnum.UP.getCode().equals(item.getIsShelf()), "上架的商品不能删除");
            });

            list.forEach(item -> {

                //删除关联视频和图片
                mdmProductMediaService.lambdaUpdate()
                        .eq(MdmProductMediaEntity::getProductCode, item.getProductCode())
                        .remove();

                //删除关联物料
                mdmProductMaterialService.lambdaUpdate()
                        .eq(MdmProductMaterialEntity::getProductCode, item.getProductCode())
                        .remove();

                //删除关联富文本
                mdmProductIntroductionService.lambdaUpdate()
                        .eq(MdmProductIntroductionEntity::getProductCode, item.getProductCode())
                        .remove();
                if (menuObject != null) {
                    crmLogSendUtil.sendForDel((String) menuObject, item.getId(), item.getProductCode(), CrmBeanUtil.copy(item, MdmProductReqVo.class));
                }
            });
            mdmProService.remove(list.stream().map(MdmProductEntity::getProductCode).collect(Collectors.toList()));
            this.removeByIds(list.stream().map(MdmProductEntity::getId).collect(Collectors.toList()));
            sendDelMsg(list.stream().map(MdmProductEntity::getProductCode).collect(Collectors.toList()));
        }
    }

    /**
     * 启用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        Assert.notEmpty(ids, "ids不能为空");
        List<String> productCodeList = new ArrayList<>();
        ids.forEach(id -> {
            Assert.hasText(id, "不能传空值");
            MdmProductEntity one = this.getById(id);
            Assert.notNull(one, "商品不存在");
            if (!CrmEnableStatusEnum.ENABLE.getCode().equals(one.getEnableStatus())) {
                one.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
                this.updateById(one);
                productCodeList.add(one.getProductCode());
            }
        });
        sendChangeMsg(productCodeList);
    }

    /**
     * 禁用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        Assert.notEmpty(ids, "ids不能为空");
        List<String> productCodeList = new ArrayList<>();
        ids.forEach(id -> {
            Assert.hasText(id, "不能传空值");
            MdmProductEntity one = this.getById(id);
            Assert.notNull(one, "商品不存在");
            if (!CrmEnableStatusEnum.DISABLE.getCode().equals(one.getEnableStatus())) {
                one.setIsShelf(IsShelfEnum.DOWN.getCode());
                one.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
                this.updateById(one);
                productCodeList.add(one.getProductCode());
            }
        });
        sendChangeMsg(productCodeList);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void upShelf(List<String> ids) {
        Assert.notEmpty(ids, "ids不能为空");
        List<String> productCodeList = new ArrayList<>();
        ids.forEach(id -> {
            Assert.hasText(id, "不能传空值");
            MdmProductEntity one = this.getById(id);
            Assert.notNull(one, "商品不存在");
            Assert.isTrue(CrmEnableStatusEnum.ENABLE.getCode().equals(one.getEnableStatus()), "只能上架状态为“启用”的商品");
            if (!IsShelfEnum.UP.getCode().equals(one.getIsShelf())) {
                one.setIsShelf(IsShelfEnum.UP.getCode());
                this.updateById(one);
                productCodeList.add(one.getProductCode());
            }
        });
        sendChangeMsg(productCodeList);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void downShelf(List<String> ids) {
        Assert.notEmpty(ids, "ids不能为空");
        List<String> productCodeList = new ArrayList<>();
        ids.forEach(id -> {
            Assert.hasText(id, "不能传空值");
            MdmProductEntity one = this.getById(id);
            Assert.notNull(one, "商品不存在");
            if (!IsShelfEnum.DOWN.getCode().equals(one.getIsShelf())) {
                one.setIsShelf(IsShelfEnum.DOWN.getCode());
                this.updateById(one);
                productCodeList.add(one.getProductCode());
            }
        });
        sendChangeMsg(productCodeList);
    }

    public void sendProductLevelChangeMsg(MdmProductReqVo original, MdmProductReqVo current) {
        if (current == null && original == null) {
            return;
        }
        try {
            String productCode = current != null ? current.getProductCode() : original.getProductCode();
            AvalibleListProductLevelChangeDto productLevelChangeDTO = new AvalibleListProductLevelChangeDto();
            productLevelChangeDTO.setGoodsCode(productCode);
            productLevelChangeDTO.setNewLevelCode(current != null ? current.getProductLevelCode() : null);
            productLevelChangeDTO.setOldLevelCode(current != null ? original.getProductLevelCode() : null);
            log.info("商品产品层级修改通知:{}", productLevelChangeDTO);
            RocketMQMessageBody rocketMQMessageBody = new RocketMQMessageBody();
            rocketMQMessageBody.setTopic(topic);
            rocketMQMessageBody.setTag(RocketMQConstant.CONSUMER_GROUP.AVAILBLELIST_REFRSH_UPDATE_PRODUCT_LEVEL);
            rocketMQMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(rocketMQMessageBody));
            rocketMQProducer.convertAndSend(rocketMQMessageBody);
        } catch (Exception e) {
            log.error("产品层级通知失败:{}", e);
        }
    }


    @Override
    public void sendChangeMsg(List<String> productCodeList) {
        if (CollectionUtils.isEmpty(productCodeList)) {
            return;
        }
        log.info("商品修改通知:{}", productCodeList);
        RocketMQMessageBody rocketMQMessageBody = new RocketMQMessageBody();
        rocketMQMessageBody.setTopic(topic);
        rocketMQMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.AVAILBLELIST_UPDATEPRODUCT);
        rocketMQMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(adviseQuery(productCodeList)));
        rocketMQProducer.convertAndSend(rocketMQMessageBody);
    }

    @Override
    public void sendDelMsg(List<String> productCodeList) {
        log.info("商品删除通知:{}", productCodeList);
        RocketMQMessageBody rocketMQMessageBody = new RocketMQMessageBody();
        rocketMQMessageBody.setTopic(topic);
        rocketMQMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.AVAILBLELIST_DELPRODUCT);
        rocketMQMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(productCodeList));
        rocketMQProducer.convertAndSend(rocketMQMessageBody);
//        cusAvailablelistService.delByGoodsCodes(productCodeList);
//        terminalAvailablelistService.delByGoodsCodes(productCodeList);
    }

    @Override
    public Map<String, List<MdmProductSaleCompanyRespVo>> findSaleCompanyByProductCodeList(List<String> productCodeList) {
        log.info("商品管理：根据商品编码集合查询物料销售公司,入参:{}", productCodeList);
        if (CollectionUtils.isEmpty(productCodeList)) {
            return Collections.emptyMap();
        }
        List<MdmProductSaleCompanyRespVo> companyList = mdmProductMapper.findSaleCompanyByProductCodeList(productCodeList);
        log.info("商品管理：根据商品编码集合查询物料销售公司,结果:{}", companyList);
        if (CollectionUtils.isEmpty(companyList)) {
            return productCodeList.stream()
                    .distinct()
                    .collect(Collectors.toMap(Function.identity(), x -> new ArrayList<>()));
        }
        Map<String, String> map = DictUtil.getDictValueMapsByCodes(DictConstant.SALE_COMPANY);
        log.info("商品管理：根据商品编码集合查询物料销售公司,字典:{}", map);
        return companyList.stream()
                .peek(x -> {
                    String saleCompanyCode = x.getSaleCompanyCode();
                    if (!StringUtils.isEmpty(saleCompanyCode) && map != null) {
                        x.setSaleCompanyName(map.get(saleCompanyCode));
                    }
                })
                .collect(Collectors.groupingBy(MdmProductSaleCompanyRespVo::getProductCode));
    }

    @Override
    public List<MdmProductAndProductLevelRespVo> findProductAndProductLevelByCodeOrName(MdmProductAndProductLevelReqVo reqVo) {
        if (reqVo.getPageSize() == null || reqVo.getPageSize() == 0) {
            reqVo.setPageSize(Integer.valueOf(ParamUtil.getParameterValue(ParameterParam.SELECT_BOX_OPTION_SIZE)));
        }
        return mdmProductMapper.findProductAndProductLevelByCodeOrName(new Page<>(1L, reqVo.getPageSize(), false), reqVo);
    }

    @Override
    public List<MdmProductSelectRespVo> productSelectList(MdmProductSelectReqVo reqVo) {
        reqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        Integer pageSize = reqVo.getPageSize();
        if (pageSize == null || pageSize == 0) {
            pageSize = Integer.valueOf(ParamUtil.getParameterValue(ParameterParam.SELECT_BOX_OPTION_SIZE));
        }
        List<MdmProductSelectRespVo> list = new ArrayList<>();
        Set<String> codeList = new HashSet<>(16);
        if (StringUtils.isNotEmpty(reqVo.getSelectedCode())) {
            codeList.add(reqVo.getSelectedCode());
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSelectedCodeList())) {
            codeList.addAll(reqVo.getSelectedCodeList());
        }
        if (!codeList.isEmpty()) {
            reqVo.setSelectedCodeList(new ArrayList<>(codeList));
        }
        if (pageSize > 0) {
            list.addAll(mdmProductMapper.productSelectList(new Page<>(1L, pageSize, false), reqVo, null, null));
        }
        Set<String> productLevelCodes = Sets.newHashSet();
        list.forEach(o -> {
            o.setProductCodeName(o.getProductCode() + "-" + o.getProductName());
            if (org.apache.commons.lang3.StringUtils.isNotEmpty(o.getProductLevelCode())) {
                productLevelCodes.add(o.getProductLevelCode());
            }
        });
        if (CollectionUtils.isNotEmpty(productLevelCodes)) {
            Map<String, String> map = ProductLevelUtil.getProductLevelByCodeList(Lists.newArrayList(productLevelCodes)).stream().collect(Collectors.toMap(MdmProductLevelRespVo::getProductLevelCode, MdmProductLevelRespVo::getProductLevelName));
            list.forEach(o -> {
                if (StringUtils.isNotEmpty(o.getProductLevelCode())) {
                    o.setProductLevelName(map.get(o.getProductLevelCode()));
                }
            });
        }
        return list;
    }

    @Override
    public List<MdmPriceProductUnitRespVo> findProductUnit(String productCode) {
        List<MdmPriceProductUnitRespVo> list = new ArrayList<>();
        if (StringUtils.isNotEmpty(productCode)) {
            MdmProductEntity one = this.lambdaQuery()
                    .eq(MdmProductEntity::getProductCode, productCode)
                    .select(MdmProductEntity::getProductCode, MdmProductEntity::getBaseUnit, MdmProductEntity::getSaleUnit)
                    .one();
            if (one != null) {
                if (StringUtils.isNotEmpty(one.getBaseUnit())) {
                    String baseUnit = one.getBaseUnit();
                    MdmPriceProductUnitRespVo baseUnitVo = new MdmPriceProductUnitRespVo();
                    baseUnitVo.setUnitType("baseUnit");
                    baseUnitVo.setUnitTypeName("基本单位");
                    baseUnitVo.setPriceUnit(baseUnit);
                    baseUnitVo.setPriceName(DictUtil.dictValue(DictConstant.PRODUCT_BASE_UNIT, baseUnit));
                    list.add(baseUnitVo);
                }
                if (StringUtils.isNotEmpty(one.getSaleUnit())) {
                    String saleUnit = one.getSaleUnit();
                    MdmPriceProductUnitRespVo saleUnitVo = new MdmPriceProductUnitRespVo();
                    saleUnitVo.setUnitType("saleUnit");
                    saleUnitVo.setUnitTypeName("销售单位");
                    saleUnitVo.setPriceUnit(saleUnit);
                    saleUnitVo.setPriceName(DictUtil.dictValue(DictConstant.PRODUCT_SALE_UNIT, saleUnit));
                    list.add(saleUnitVo);
                }
            }
        }
        return list;
    }

    @Override
    public List<MdmProductRespVo> listCondition(MdmProductReqVo mdmProductReqVo) {
        List<MdmProductEntity> list = this.lambdaQuery()
                .like(!StringUtils.isEmpty(mdmProductReqVo.getProductCode()), MdmProductEntity::getProductCode, mdmProductReqVo.getProductCode())
                .like(!StringUtils.isEmpty(mdmProductReqVo.getProductName()), MdmProductEntity::getProductName, mdmProductReqVo.getProductName())
                .eq(!StringUtils.isEmpty(mdmProductReqVo.getProductLevelCode()), MdmProductEntity::getProductLevelCode, mdmProductReqVo.getProductLevelCode())
                .gt(!StringUtils.isEmpty(mdmProductReqVo.getBeginDate()), MdmProductEntity::getBeginDate, mdmProductReqVo.getBeginDate())
                .lt(!StringUtils.isEmpty(mdmProductReqVo.getEndDate()), MdmProductEntity::getEndDate, mdmProductReqVo.getEndDate())
                .eq(MdmProductEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .list();
        return CrmBeanUtil.copyList(list, MdmProductRespVo.class);
    }

    @Override
    public List<CusAvailablelistVo> queryShelfProductList(MdmProductShelfPriceReqVo mdmProductShelfPriceReqVo) {
        String conditionTypeCode = mdmProductShelfPriceReqVo.getConditionTypeCode();
        Assert.hasText(conditionTypeCode, "条件类型编码不能为空");
        String customerCode = mdmProductShelfPriceReqVo.getCustomerCode();
        Assert.hasText(customerCode, "客户编码不能为空");
        List<String> upProductCodeList = this.lambdaQuery()
                .eq(MdmProductEntity::getIsShelf, IsShelfEnum.UP.getCode())
                .select(MdmProductEntity::getProductCode)
                .list()
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getProductCode()))
                .map(MdmProductEntity::getProductCode)
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(upProductCodeList)) {
            return new ArrayList<>();
        }

        List<MdmProductAdviseRespVo> mdmProductAdviseRespVos = adviseQuery(upProductCodeList);
        if (CollectionUtils.isEmpty(mdmProductAdviseRespVos)) {
            return new ArrayList<>();
        }
        List<CusAvailablelistVo> cusAvailablelistVos = CrmBeanUtil.copyList(mdmProductAdviseRespVos, CusAvailablelistVo.class);
        MdmPriceSearchReqVo mdmPriceSearchReqVo = new MdmPriceSearchReqVo();
        mdmPriceSearchReqVo.setProductCodeList(upProductCodeList);
        mdmPriceSearchReqVo.setConditionTypeCodeList(Collections.singletonList(conditionTypeCode));
        mdmPriceSearchReqVo.setCustomerCode(customerCode);
        Map<String, Map<String, MdmPriceResp>> priceResult = mdmPriceSettingService.search(mdmPriceSearchReqVo);
        if (priceResult == null) {
            return cusAvailablelistVos;
        }
        Map<String, MdmPriceResp> mdmPriceRespMap = priceResult.get(conditionTypeCode);
        if (mdmPriceRespMap == null) {
            return cusAvailablelistVos;
        }
        cusAvailablelistVos = cusAvailablelistVos.stream().peek(x -> {
            String productCode = x.getProductCode();
            BigDecimal price = Optional.ofNullable(mdmPriceRespMap.get(productCode))
                    .map(MdmPriceResp::getPrice)
                    .orElse(null);
            x.setPrice(price);
        }).collect(Collectors.toList());
        return cusAvailablelistVos;
    }

    /**
     * sfa查询商品信息列表专用接口
     *
     * @param reqVo
     * @return
     */
    @Override
    @CrmDictMethod
    public PageResult<MdmProductBySfaRespVo> findProductListSourceSfa(MdmProductBySfaReqVo reqVo) {
        Page<MdmProductBySfaRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        if (StringUtils.isNotEmpty(reqVo.getCode())) {
            MdmProductLevelRespVo productLevel = mdmProductLevelService.detail(null, reqVo.getCode());
            if (productLevel != null) {
                reqVo.setRuleCode(productLevel.getRuleCode());
            }
        }
        List<MdmProductBySfaRespVo> list = mdmProductMapper.findProductListSourceSfa(page, reqVo);
        return PageResult.<MdmProductBySfaRespVo>builder()
                .count(page.getTotal())
                .data(list)
                .build();
    }

    /**
     * sfa tpm活动查询商品基础信息
     *
     * @param productCodeList
     * @return
     */
    @Override
    @CrmDictMethod
    public List<MdmProductBaseDataRespVo> findProductBaseSourceSfaTpmAct(List<String> productCodeList) {
        Assert.notNull(productCodeList, "商品编码不能为空");
        String now = LocalDateTime.now().format(CrmDateUtils.yyyyMMdd);
        List<MdmProductBaseDataRespVo> list = mdmProductMapper.findProductBaseSourceSfaTpmAct(productCodeList, now);
        return list;
    }

    /**
     * 商品信息变更发送事件
     *
     * @param vo
     */
    private void productChangeSendEvent(ProductChangeHandlerReqVo vo) {
        applicationContext.publishEvent(new ProductChangeEvent(vo));
    }

    @Override
    public PageResult<MdmProductRespVo> pageListWithCondition(MdmProductReqVo mdmProductReqVo) {
        Page<MdmProductRespVo> page = PageUtil.buildPage(mdmProductReqVo.getPageNum(), mdmProductReqVo.getPageSize());
        if (StringUtils.isNotEmpty(mdmProductReqVo.getCode())) {
            MdmProductLevelRespVo productLevel = mdmProductLevelService.detail(null, mdmProductReqVo.getCode());
            if (productLevel != null) {
                mdmProductReqVo.setRuleCode(productLevel.getRuleCode());
            }
        }
        List<MdmProductRespVo> list = mdmProductMapper.pageListWithCondition(page, mdmProductReqVo);
        if (CollectionUtil.listNotEmptyNotSizeZero(list)){
            for (MdmProductRespVo mdmProductRespVo : list) {
                List<MdmProductMaterialEntity> materialList = mdmProductMaterialService.lambdaQuery()
                        .eq(!StringUtils.isEmpty(mdmProductRespVo.getProductCode()), MdmProductMaterialEntity::getProductCode, mdmProductRespVo.getProductCode()).list();
                if (CollectionUtil.listNotEmptyNotSizeZero(materialList)){
                    mdmProductRespVo.setMaterialList(CrmBeanUtil.copyList(materialList, MdmProductMaterialRespVo.class));
                }
            }
        }
        productServiceHelper.convertListDate(list);
        return PageResult.<MdmProductRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }
}
