package com.biz.crm.productlevel.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.annotation.Klock;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CodeRuleEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.kms.service.MdmCategoryService;
import com.biz.crm.material.model.MdmMaterialEntity;
import com.biz.crm.material.service.MdmMaterialService;
import com.biz.crm.nebular.mdm.dict.resp.DictDataVo;
import com.biz.crm.nebular.mdm.productlevel.req.MdmProductLevelReqVo;
import com.biz.crm.nebular.mdm.productlevel.req.MdmProductLevelSelectReqVo;
import com.biz.crm.nebular.mdm.productlevel.resp.*;
import com.biz.crm.productlevel.mapper.MdmProductLevelMapper;
import com.biz.crm.productlevel.model.MdmProductLevelEntity;
import com.biz.crm.productlevel.service.MdmProductLevelService;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 接口实现
 *
 * @author Liao.Hua
 * @date 2020-09-01 17:47:01
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmProductLevelServiceExpandImpl")
public class MdmProductLevelServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmProductLevelMapper, MdmProductLevelEntity> implements MdmProductLevelService {

    @Resource
    private MdmProductLevelMapper mdmProductLevelMapper;

    @Resource
    private MdmMaterialService mdmMaterialService;
    @Autowired
    private CrmLogSendUtil crmLogSendUtil;

    @Autowired
    private MdmCategoryService mdmCategoryService;


    @Override
    public PageResult<MdmProductLevelRespVo> findList(MdmProductLevelReqVo mdmProductLevelReqVo) {
        Page<MdmProductLevelRespVo> page = PageUtil.buildPage(mdmProductLevelReqVo.getPageNum(), mdmProductLevelReqVo.getPageSize());
        if (StringUtils.isNotEmpty(mdmProductLevelReqVo.getCode())) {
            MdmProductLevelEntity one = this.lambdaQuery()
                    .eq(MdmProductLevelEntity::getProductLevelCode, mdmProductLevelReqVo.getCode())
                    .one();
            if (one != null) {
                mdmProductLevelReqVo.setRuleCode(one.getRuleCode());
            }
        }

        List<String> productLevelTypeSort = new ArrayList<>();
        List<DictDataVo> productLevelTypeList = DictUtil.list("mdm_product_level_type");
        if (CollectionUtil.listNotEmptyNotSizeZero(productLevelTypeList)) {
            productLevelTypeSort.addAll(productLevelTypeList.stream().map(DictDataVo::getDictCode).collect(Collectors.toList()));
        }
        List<MdmProductLevelRespVo> list = mdmProductLevelMapper.findList(page, mdmProductLevelReqVo, productLevelTypeSort);
        return PageResult.<MdmProductLevelRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public MdmProductLevelRespVo detail(String id, String productLevelCode) {
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(productLevelCode)) {
            MdmProductLevelEntity one = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(id), MdmProductLevelEntity::getId, id)
                    .eq(StringUtils.isNotEmpty(productLevelCode), MdmProductLevelEntity::getProductLevelCode, productLevelCode)
                    .one();
            if (one != null) {
                MdmProductLevelRespVo respVo = CrmBeanUtil.copy(one, MdmProductLevelRespVo.class);
                if (StringUtils.isNotEmpty(one.getParentCode())) {
                    MdmProductLevelEntity parent = this.lambdaQuery()
                            .eq(MdmProductLevelEntity::getProductLevelCode, one.getParentCode())
                            .select(MdmProductLevelEntity::getProductLevelName)
                            .one();
                    if (parent != null) {
                        respVo.setParentName(parent.getProductLevelName());
                    }
                }
                return respVo;
            }
        }
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmProductLevelReqVo mdmProductLevelReqVo) {
        Assert.notNull(mdmProductLevelReqVo, "空参数");
        Assert.hasText(mdmProductLevelReqVo.getProductLevelName(), "产品层级名称不能为空");
        if (StringUtils.isNotEmpty(mdmProductLevelReqVo.getProductLevelCode())) {
            List<MdmProductLevelEntity> list = this.lambdaQuery()
                    .eq(MdmProductLevelEntity::getProductLevelCode, mdmProductLevelReqVo.getProductLevelCode())
                    .select(MdmProductLevelEntity::getId)
                    .list();
            Assert.isTrue(CollectionUtil.listEmpty(list), "编码[" + mdmProductLevelReqVo.getProductLevelCode() + "]已存在");
        } else {
            mdmProductLevelReqVo.setProductLevelCode(CodeUtil.generateCode(CodeRuleEnum.MDM_PRODUCT_LEVEL_CODE.getCode()));
        }
        int levelNum = 1;
        if (StringUtils.isNotEmpty(mdmProductLevelReqVo.getParentCode())) {
            MdmProductLevelRespVo parent = this.detail(null, mdmProductLevelReqVo.getParentCode());
            Assert.notNull(parent, "上级编码无效");
            levelNum = parent.getLevelNum() + 1;
        }
        MdmProductLevelEntity entity = CrmBeanUtil.copy(mdmProductLevelReqVo, MdmProductLevelEntity.class);
        entity.setRuleCode(generateRuleCodeByParentCode(entity.getParentCode()));
        entity.setLevelNum(levelNum);
        this.save(entity);
        ProductLevelUtil.deleteAllCache();
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            crmLogSendUtil.sendForAdd((String) menuObject, entity.getId(), entity.getProductLevelCode(), mdmProductLevelReqVo);
        }
        mdmCategoryService.add(mdmProductLevelReqVo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmProductLevelReqVo mdmProductLevelReqVo) {
        Assert.notNull(mdmProductLevelReqVo, "空参数");
        Assert.hasText(mdmProductLevelReqVo.getId(), "id不能为空");
        Assert.hasText(mdmProductLevelReqVo.getProductLevelCode(), "产品层级编码不能为空");
        Assert.hasText(mdmProductLevelReqVo.getProductLevelName(), "产品层级名称不能为空");
        MdmProductLevelEntity entity = this.getById(mdmProductLevelReqVo.getId());
        Assert.notNull(entity, "id无效");
        Assert.isTrue(entity.getProductLevelCode().equals(mdmProductLevelReqVo.getProductLevelCode()), "产品层级编码不能修改");

        int levelNum = 1;
        if (StringUtils.isNotEmpty(mdmProductLevelReqVo.getParentCode())) {
            MdmProductLevelEntity parent = this.lambdaQuery()
                    .eq(MdmProductLevelEntity::getProductLevelCode, mdmProductLevelReqVo.getParentCode())
                    .select(MdmProductLevelEntity::getId, MdmProductLevelEntity::getProductLevelCode, MdmProductLevelEntity::getRuleCode, MdmProductLevelEntity::getLevelNum)
                    .one();
            Assert.notNull(parent, "上级产品层级不存在");
            Assert.isTrue(!parent.getId().equals(entity.getId()), "上级产品层级不能是自己");
            Assert.isTrue(!parent.getRuleCode().startsWith(entity.getRuleCode()),"上级产品层级不能是当前层级的下级");
            levelNum = parent.getLevelNum() + 1;
        }

        //是否更新降维编码
        boolean updateRuleCode = false;
        String enableStatusChangeTo = "";
        if (!(entity.getParentCode() == null ? "" : entity.getParentCode()).equals((mdmProductLevelReqVo.getParentCode() == null ? "" : mdmProductLevelReqVo.getParentCode()))) {
            //上级编码发生变化，重置降维编码
            updateRuleCode = true;
        }
        if (!(entity.getEnableStatus()).equals(mdmProductLevelReqVo.getEnableStatus())) {
            //启用状态发生变化
            enableStatusChangeTo = mdmProductLevelReqVo.getEnableStatus();
        }
        MdmProductLevelReqVo oldObj = CrmBeanUtil.copy(entity, MdmProductLevelReqVo.class);
        CrmBeanUtil.copyProperties(mdmProductLevelReqVo, entity);
        this.updateById(entity);
        if (updateRuleCode) {
            //更新降维编码
            String ruleCode = generateRuleCodeByParentCode(entity.getParentCode());
            updateCurAndChildren(entity.getProductLevelCode(), ruleCode, levelNum);
        }
        if (StringUtils.isNotEmpty(enableStatusChangeTo)) {
            if (CrmEnableStatusEnum.ENABLE.getCode().equals(enableStatusChangeTo)) {
                this.enableBatch(Collections.singletonList(entity.getId()));
            } else if (CrmEnableStatusEnum.DISABLE.getCode().equals(enableStatusChangeTo)) {
                this.disableBatch(Collections.singletonList(entity.getId()));
            } else {
                throw new BusinessException("启用状态错误");
            }
        }
        ProductLevelUtil.deleteAllCache();
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            MdmProductLevelReqVo newObject = CrmBeanUtil.copy(entity, MdmProductLevelReqVo.class);
            crmLogSendUtil.sendForUpdate((String) menuObject, entity.getId(), entity.getProductLevelCode(), oldObj, newObject);
        }
        mdmCategoryService.update(mdmProductLevelReqVo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        Assert.notEmpty(ids, "参数为空");
        List<MdmProductLevelEntity> list = this.lambdaQuery()
                .in(MdmProductLevelEntity::getId, ids)
                .list();
        Assert.notEmpty(list, "无效的参数");

        List<String> productLevelCodeList = list.stream().map(MdmProductLevelEntity::getProductLevelCode).collect(Collectors.toList());

        List<MdmProductLevelEntity> childrenList = this.lambdaQuery()
                .in(MdmProductLevelEntity::getParentCode, productLevelCodeList)
                .notIn(MdmProductLevelEntity::getId, ids)
                .list();
        Assert.isTrue(CollectionUtil.listEmpty(childrenList), "当前产品层级包含子层级，无法删除，若需要删除请先删除子层级");

        List<MdmMaterialEntity> materialList = mdmMaterialService.lambdaQuery()
                .in(MdmMaterialEntity::getProductLevelCode, productLevelCodeList)
                .select(MdmMaterialEntity::getId)
                .list();
        Assert.isTrue(CollectionUtil.listEmpty(materialList), "当前产品层级已关联物料，无法删除，若需要删除请先删除关联物料");

        this.removeByIds(ids);
        ProductLevelUtil.deleteAllCache();
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            list.forEach(item -> {
                crmLogSendUtil.sendForDel((String) menuObject, item.getId(), item.getProductLevelCode(), CrmBeanUtil.copy(item, MdmProductLevelReqVo.class));
            });
        }
        mdmCategoryService.remove(productLevelCodeList);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        Assert.notEmpty(ids, "缺失参数");
        List<MdmProductLevelEntity> list = this.lambdaQuery()
                .in(MdmProductLevelEntity::getId, ids)
                .select(MdmProductLevelEntity::getId, MdmProductLevelEntity::getRuleCode, MdmProductLevelEntity::getEnableStatus)
                .list()
                .stream().filter(x -> !CrmEnableStatusEnum.ENABLE.getCode().equals(x.getEnableStatus()))
                .collect(Collectors.toList());
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            List<String> collect = list.stream().map(MdmProductLevelEntity::getRuleCode).collect(Collectors.toList());
            Set<String> parentRuleCodesExcludeSelf = TreeRuleCodeUtil.splitParentRuleCodesExcludeAnySelf(collect);
            if (!parentRuleCodesExcludeSelf.isEmpty()) {
                List<MdmProductLevelEntity> parentList = this.lambdaQuery()
                        .in(MdmProductLevelEntity::getRuleCode, parentRuleCodesExcludeSelf)
                        .select(MdmProductLevelEntity::getId, MdmProductLevelEntity::getEnableStatus)
                        .list()
                        .stream().filter(x -> !CrmEnableStatusEnum.ENABLE.getCode().equals(x.getEnableStatus()))
                        .collect(Collectors.toList());
                Assert.isTrue(CollectionUtil.listEmpty(parentList), "存在未启用的上级层级，不能启用当前层级");
            }
            this.lambdaUpdate()
                    .in(MdmProductLevelEntity::getId, list.stream().map(MdmProductLevelEntity::getId).collect(Collectors.toSet()))
                    .set(MdmProductLevelEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                    .update();
            ProductLevelUtil.deleteAllCache();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        Assert.notEmpty(ids, "缺失参数");
        List<MdmProductLevelEntity> list = this.lambdaQuery()
                .in(MdmProductLevelEntity::getId, ids)
                .select(MdmProductLevelEntity::getRuleCode)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            List<String> collect = list.stream().map(item -> item.getRuleCode() + "%").collect(Collectors.toList());
            List<MdmProductLevelRespVo> childrenByRuleCodeList = mdmProductLevelMapper.findChildrenByRuleCodeList(collect, CrmEnableStatusEnum.ENABLE.getCode());
            if (CollectionUtil.listNotEmptyNotSizeZero(childrenByRuleCodeList)) {
                this.lambdaUpdate()
                        .in(MdmProductLevelEntity::getProductLevelCode, childrenByRuleCodeList.stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList()))
                        .set(MdmProductLevelEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
                        .update();
                ProductLevelUtil.deleteAllCache();
            }
        }
    }

    @Override
    public List<String> getCurrentAndSubLevelCode(String productLevelCode) {
        List<String> list = new ArrayList<>();
        if (StringUtils.isNotEmpty(productLevelCode)) {
            List<MdmProductLevelRespVo> childrenList = this.getChildrenList(null, productLevelCode);
            if (CollectionUtil.listNotEmptyNotSizeZero(childrenList)) {
                list.addAll(childrenList.stream().map(MdmProductLevelRespVo::getProductLevelCode).collect(Collectors.toList()));
            }
        }
        return list;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @Klock
    public void resetRuleCode() {
        long l = System.currentTimeMillis();
        log.info("-----------重置产品层级降维编码_" + l + "_开始-----------");
        mdmProductLevelMapper.copyIdToRuleCode();
        log.info("-----------重置产品层级降维编码_" + l + "_编码重置成id成功-----------");
        mdmProductLevelMapper.setNullNotExistParentCode();
        log.info("-----------重置产品层级降维编码_" + l + "_清除无效上级编码成功-----------");
        List<MdmProductLevelEntity> topList = this.lambdaQuery()
                .isNull(MdmProductLevelEntity::getParentCode)
                .or()
                .eq(MdmProductLevelEntity::getParentCode, "")
                .list();

        for (int i = 0; i < topList.size(); i++) {
            updateCurAndChildren(topList.get(i).getProductLevelCode(), TreeRuleCodeUtil.numToSingleCode(i + 1), 1);
        }
        log.info("-----------重置产品层级降维编码_" + l + "_编码重新生成成功-----------");
        ProductLevelUtil.deleteAllCache();
        log.info("-----------重置产品层级降维编码_" + l + "_缓存清除成功-----------");
        log.info("-----------重置产品层级降维编码_" + l + "_结束-----------");
    }

    @Override
    public List<MdmProductLevelRespVo> getParentList(String id, String productLevelCode) {
        List<MdmProductLevelRespVo> list = new ArrayList<>();
        if (StringUtils.isEmpty(id) && StringUtils.isEmpty(productLevelCode)) {
            MdmProductLevelEntity one = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(id), MdmProductLevelEntity::getId, id)
                    .eq(StringUtils.isNotEmpty(productLevelCode), MdmProductLevelEntity::getProductLevelCode, productLevelCode)
                    .one();
            if (one != null) {
                Set<String> parentRuleCodes = TreeRuleCodeUtil.splitParentRuleCodes(one.getRuleCode());
                if (!parentRuleCodes.isEmpty()) {
                    List<MdmProductLevelEntity> parentList = this.lambdaQuery()
                            .in(MdmProductLevelEntity::getRuleCode, parentRuleCodes)
                            .list();
                    if (CollectionUtil.listNotEmptyNotSizeZero(parentList)) {
                        list.addAll(CrmBeanUtil.copyList(parentList, MdmProductLevelRespVo.class));
                    }
                }
            }
        }
        return list;
    }

    @Override
    public List<MdmProductLevelVo> getParentCpsList(String id, String productLevelCode) {
        List<MdmProductLevelVo> list = new ArrayList<>();
        if (StringUtils.isEmpty(id) || StringUtils.isEmpty(productLevelCode)) {
            MdmProductLevelEntity one = this.lambdaQuery()
                .eq(StringUtils.isNotEmpty(id), MdmProductLevelEntity::getId, id)
                .eq(StringUtils.isNotEmpty(productLevelCode), MdmProductLevelEntity::getProductLevelCode, productLevelCode)
                .one();
            if (one != null) {
                Set<String> parentRuleCodes = TreeRuleCodeUtil.splitParentRuleCodes(one.getRuleCode());
                if (!parentRuleCodes.isEmpty()) {
                    List<MdmProductLevelEntity> parentList = this.lambdaQuery()
                        .in(MdmProductLevelEntity::getRuleCode, parentRuleCodes)
                        .list();
                    if (CollectionUtil.listNotEmptyNotSizeZero(parentList)) {
                        list.addAll(CrmBeanUtil.copyList(parentList, MdmProductLevelVo.class));
                    }
                }
            }
        }
        return list;
    }

    @Override
    public List<MdmProductLevelRespVo> getChildrenList(String id, String productLevelCode) {
        List<MdmProductLevelRespVo> list = new ArrayList<>();
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(productLevelCode)) {
            MdmProductLevelEntity one = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(id), MdmProductLevelEntity::getId, id)
                    .eq(StringUtils.isNotEmpty(productLevelCode), MdmProductLevelEntity::getProductLevelCode, productLevelCode)
                    .one();
            if (one != null) {
                List<MdmProductLevelEntity> children = this.lambdaQuery()
                        .likeRight(MdmProductLevelEntity::getRuleCode, one.getRuleCode())
                        .list();
                if (CollectionUtil.listNotEmptyNotSizeZero(children)) {
                    list.addAll(CrmBeanUtil.copyList(children, MdmProductLevelRespVo.class));
                }
            }
        }
        return list;
    }

    @Override
    public List<String> getAllChildrenProductLevelCodeList(String id, String productLevelCode) {
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(productLevelCode)) {
            MdmProductLevelEntity one = this.lambdaQuery()
                    .eq(StringUtils.isNotEmpty(id), MdmProductLevelEntity::getId, id)
                    .eq(StringUtils.isNotEmpty(productLevelCode), MdmProductLevelEntity::getProductLevelCode, productLevelCode)
                    .select(MdmProductLevelEntity::getRuleCode)
                    .one();
            if (one != null) {
                List<MdmProductLevelEntity> list = this.lambdaQuery()
                        .likeRight(MdmProductLevelEntity::getRuleCode, one.getRuleCode())
                        .select(MdmProductLevelEntity::getProductLevelCode)
                        .list();
                if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                    return list.stream().map(MdmProductLevelEntity::getProductLevelCode).collect(Collectors.toList());
                }
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmProductLevelSelectRespVo> productLevelSelectList(MdmProductLevelSelectReqVo reqVo) {
        List<MdmProductLevelSelectRespVo> list = new ArrayList<>();
        Integer pageSize = reqVo.getPageSize();
        if (pageSize == null || pageSize == 0) {
            pageSize = Integer.valueOf(ParamUtil.getParameterValue(ParameterParam.SELECT_BOX_OPTION_SIZE));
        }
        String underThisRuleCode = null;
        String notUnderThisRuleCode = null;
        if (StringUtils.isNotEmpty(reqVo.getUnderProductLevelCode())) {
            MdmProductLevelEntity one = this.lambdaQuery()
                    .eq(MdmProductLevelEntity::getProductLevelCode, reqVo.getUnderProductLevelCode())
                    .select(MdmProductLevelEntity::getRuleCode)
                    .one();
            if (one != null) {
                underThisRuleCode = one.getRuleCode();
            }
        }
        if (StringUtils.isNotEmpty(reqVo.getNotUnderProductLevelCode())) {
            MdmProductLevelEntity one = this.lambdaQuery()
                    .eq(MdmProductLevelEntity::getProductLevelCode, reqVo.getNotUnderProductLevelCode())
                    .select(MdmProductLevelEntity::getRuleCode)
                    .one();
            if (one != null) {
                notUnderThisRuleCode = one.getRuleCode();
            }
        }
        Set<String> codeList = new HashSet<>(16);
        if (org.apache.commons.lang3.StringUtils.isNotEmpty(reqVo.getSelectedCode())) {
            codeList.add(reqVo.getSelectedCode());
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSelectedCodeList())) {
            codeList.addAll(reqVo.getSelectedCodeList());
        }
        if (!codeList.isEmpty()) {
            reqVo.setSelectedCodeList(new ArrayList<>(codeList));
        }
        if (pageSize > 0) {
            list.addAll(mdmProductLevelMapper.findProductLevelSelectList(new Page<>(1L, pageSize, false), reqVo, underThisRuleCode, notUnderThisRuleCode));
        }
        return list;
    }

    @Override
    public List<MdmProductLevelRedisVo> findRedisListByProductLevelCodeList(List<String> productLevelCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(productLevelCodeList)) {
            List<MdmProductLevelEntity> list = this.lambdaQuery()
                    .in(MdmProductLevelEntity::getProductLevelCode, productLevelCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                Map<String, String> codeNameMap = new HashMap<>(16);
                Set<String> parentCodeSet = list.stream().filter(item -> StringUtils.isNotEmpty(item.getParentCode())).map(MdmProductLevelEntity::getParentCode).collect(Collectors.toSet());
                if (!parentCodeSet.isEmpty()) {
                    List<MdmProductLevelEntity> parentCodeNameList = this.lambdaQuery()
                            .in(MdmProductLevelEntity::getProductLevelCode, parentCodeSet)
                            .select(MdmProductLevelEntity::getProductLevelCode, MdmProductLevelEntity::getProductLevelName)
                            .list();
                    if (CollectionUtil.listNotEmptyNotSizeZero(parentCodeNameList)) {
                        codeNameMap.putAll(parentCodeNameList.stream().collect(Collectors.toMap(MdmProductLevelEntity::getProductLevelCode, MdmProductLevelEntity::getProductLevelName)));
                    }
                }
                return list.stream().map(item -> {
                    MdmProductLevelRedisVo copy = CrmBeanUtil.copy(item, MdmProductLevelRedisVo.class);
                    if (codeNameMap.containsKey(copy.getProductLevelCode())) {
                        copy.setProductLevelName(codeNameMap.get(copy.getProductLevelCode()));
                    }
                    return copy;
                }).collect(Collectors.toList());
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmProductLevelParentChildrenRedisVo> findRedisProductLevelParentListByCodeList(List<String> productLevelCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(productLevelCodeList)) {
            List<MdmProductLevelEntity> list = this.lambdaQuery()
                    .in(MdmProductLevelEntity::getProductLevelCode, productLevelCodeList)
                    .select(MdmProductLevelEntity::getProductLevelCode, MdmProductLevelEntity::getRuleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                Set<String> parentRuleCodes = TreeRuleCodeUtil.splitParentRuleCodes(list.stream().map(MdmProductLevelEntity::getRuleCode).collect(Collectors.toList()));
                List<MdmProductLevelEntity> parentList = this.lambdaQuery()
                        .in(MdmProductLevelEntity::getRuleCode, parentRuleCodes)
                        .list();
                Set<String> collect = parentList.stream().filter(item -> StringUtils.isNotEmpty(item.getParentCode())).map(MdmProductLevelEntity::getParentCode).collect(Collectors.toSet());
                Map<String, String> parentProductLevelCodeNameMap = new HashMap<>(16);
                if (!collect.isEmpty()) {
                    parentProductLevelCodeNameMap.putAll(this.lambdaQuery()
                            .in(MdmProductLevelEntity::getProductLevelCode, collect)
                            .select(MdmProductLevelEntity::getProductLevelCode, MdmProductLevelEntity::getProductLevelName)
                            .list()
                            .stream()
                            .collect(Collectors.toMap(MdmProductLevelEntity::getProductLevelCode, MdmProductLevelEntity::getProductLevelName)));
                }
                List<MdmProductLevelRedisVo> parentRedisVoList = parentList.stream().map(item -> {
                    MdmProductLevelRedisVo copy = CrmBeanUtil.copy(item, MdmProductLevelRedisVo.class);
                    if (StringUtils.isNotEmpty(copy.getParentCode())) {
                        if (parentProductLevelCodeNameMap.containsKey(copy.getParentCode())) {
                            copy.setParentName(parentProductLevelCodeNameMap.get(copy.getParentCode()));
                        }
                    }
                    return copy;
                }).collect(Collectors.toList());
                List<MdmProductLevelParentChildrenRedisVo> result = new ArrayList<>();
                for (MdmProductLevelEntity item :
                        parentList) {
                    MdmProductLevelParentChildrenRedisVo redisVo = new MdmProductLevelParentChildrenRedisVo();
                    redisVo.setProductLevelCode(item.getProductLevelCode());
                    redisVo.setList(parentRedisVoList.stream().filter(x -> item.getRuleCode().startsWith(x.getRuleCode())).collect(Collectors.toList()));
                    result.add(redisVo);
                }
                return result;
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmProductLevelParentChildrenRedisVo> findRedisProductLevelChildrenListByCodeList(List<String> productLevelCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(productLevelCodeList)) {
            List<MdmProductLevelEntity> productLevelList = this.lambdaQuery()
                    .in(MdmProductLevelEntity::getProductLevelCode, productLevelCodeList)
                    .select(MdmProductLevelEntity::getProductLevelCode, MdmProductLevelEntity::getRuleCode)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(productLevelList)) {
                List<String> likeRightRuleCodeList = productLevelList.stream().map(item -> item.getRuleCode() + "%").collect(Collectors.toList());
                List<MdmProductLevelRespVo> childrenList = mdmProductLevelMapper.findChildrenByRuleCodeList(likeRightRuleCodeList, null);
                if (CollectionUtil.listNotEmptyNotSizeZero(childrenList)) {
                    List<MdmProductLevelRedisVo> redisList = CrmBeanUtil.copyList(childrenList, MdmProductLevelRedisVo.class);
                    List<MdmProductLevelParentChildrenRedisVo> resultList = new ArrayList<>();
                    for (MdmProductLevelEntity item :
                            productLevelList) {
                        MdmProductLevelParentChildrenRedisVo redisVo = new MdmProductLevelParentChildrenRedisVo();
                        redisVo.setProductLevelCode(item.getProductLevelCode());
                        redisVo.setList(redisList.stream().filter(x -> x.getRuleCode().startsWith(item.getRuleCode())).collect(Collectors.toList()));
                        resultList.add(redisVo);
                    }
                    return resultList;
                }
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmProductLevelReloadRedisVo> findRedisReloadList() {
        final List<MdmProductLevelEntity> all = this.lambdaQuery().list();
        if (CollectionUtil.listNotEmptyNotSizeZero(all)) {
            final Map<String, String> productLevelCodeNameMap = all.stream().collect(Collectors.toMap(MdmProductLevelEntity::getProductLevelCode, MdmProductLevelEntity::getProductLevelName));
            final List<MdmProductLevelRedisVo> redisList = all.stream().map(item -> {
                MdmProductLevelRedisVo copy = CrmBeanUtil.copy(item, MdmProductLevelRedisVo.class);
                if (StringUtils.isNotEmpty(copy.getParentCode())) {
                    copy.setParentName(productLevelCodeNameMap.get(copy.getParentCode()));
                }
                return copy;
            }).collect(Collectors.toList());
            return redisList.stream().map(item -> {
                MdmProductLevelReloadRedisVo redisVo = new MdmProductLevelReloadRedisVo();
                redisVo.setProductLevelCode(item.getProductLevelCode());
                redisVo.setSelf(item);
                redisVo.setParentList(redisList.stream().filter(x -> item.getRuleCode().startsWith(x.getRuleCode())).collect(Collectors.toList()));
                redisVo.setChildrenList(redisList.stream().filter(x -> x.getRuleCode().startsWith(item.getRuleCode())).collect(Collectors.toList()));
                return redisVo;
            }).collect(Collectors.toList());
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmProductLevelRespVo> getAllProductLevelList() {
        List<MdmProductLevelEntity> list = this.lambdaQuery().list();
        if (list.size() > 0) {
            return CrmBeanUtil.copyList(list, MdmProductLevelRespVo.class);
        }
        return new ArrayList<>();
    }

    /**
     * 更新当前及下级降维编码
     *
     * @param productLevelCode 当前id
     * @param ruleCode         当前降维编码
     * @param levelNum         当前层级
     */
    private void updateCurAndChildren(String productLevelCode, String ruleCode, int levelNum) {

        //更新当前
        this.lambdaUpdate()
                .eq(MdmProductLevelEntity::getProductLevelCode, productLevelCode)
                .set(MdmProductLevelEntity::getRuleCode, ruleCode)
                .set(MdmProductLevelEntity::getLevelNum, levelNum)
                .update();

        //查询下一层
        List<MdmProductLevelEntity> list = this.lambdaQuery()
                .eq(MdmProductLevelEntity::getParentCode, productLevelCode)
                .select(MdmProductLevelEntity::getProductLevelCode)
                .list();

        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            //遍历下级
            for (int i = 0; i < list.size(); i++) {
                //递归调用
                updateCurAndChildren(list.get(i).getProductLevelCode(), ruleCode + TreeRuleCodeUtil.numToSingleCode(i + 1), levelNum + 1);
            }
        }
    }

    /**
     * 根据parentCode生成当前降维编码
     *
     * @param parentCode
     * @return
     */
    private String generateRuleCodeByParentCode(String parentCode) {
        String ruleCode = "";
        boolean top = true;
        String prefix = "";
        if (StringUtils.isNotEmpty(parentCode)) {
            MdmProductLevelEntity parent = this.lambdaQuery().eq(MdmProductLevelEntity::getProductLevelCode, parentCode).one();
            if (parent != null) {
                top = false;
                prefix = parent.getRuleCode();
            }
        }
        List<MdmProductLevelEntity> list = null;
        if (top) {
            list = this.lambdaQuery().isNull(MdmProductLevelEntity::getParentCode).or().eq(MdmProductLevelEntity::getParentCode, "").list();
        } else {
            list = this.lambdaQuery().eq(MdmProductLevelEntity::getParentCode, parentCode).list();
        }
        if (CollectionUtil.listEmpty(list)) {
            ruleCode = prefix + TreeRuleCodeUtil.numToSingleCode(1);
        } else {
            Set<Integer> set = new LinkedHashSet<>();
            for (MdmProductLevelEntity item :
                    list) {
                if (StringUtils.isNotEmpty(item.getRuleCode())) {
                    try {
                        Integer integer = TreeRuleCodeUtil.getCurLevelCodeValue(item.getRuleCode());
                        set.add(integer);
                    } catch (BusinessException e) {
                        throw new BusinessException("产品层级" + e.getMsg());
                    }
                }
            }
            int maxRuleCodeValueInLevel = TreeRuleCodeUtil.getMaxRuleCodeValueInLevel();
            for (int i = 1; i <= maxRuleCodeValueInLevel; i++) {
                if (i == maxRuleCodeValueInLevel) {
                    throw new BusinessException("降维编码越界，请联系管理员处理");
                }
                if (!set.contains(i)) {
                    ruleCode = prefix + TreeRuleCodeUtil.numToSingleCode(i);
                    break;
                }
            }
        }
        return ruleCode;
    }


}
