package com.biz.crm.region.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.nebular.mdm.constant.RegionLevelEnum;
import com.biz.crm.nebular.mdm.region.*;
import com.biz.crm.region.mapper.MdmRegionMapper;
import com.biz.crm.region.model.MdmRegionEntity;
import com.biz.crm.region.service.MdmRegionService;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 接口实现
 *
 * @author zxw
 * @date 2020-11-20 14:59:37
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmRegionServiceExpandImpl")
public class MdmRegionServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmRegionMapper, MdmRegionEntity> implements MdmRegionService {

    @Resource
    private MdmRegionMapper mdmRegionMapper;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    @Override
    public PageResult<MdmRegionRespVo> pageList(MdmRegionReqVo mdmRegionReqVo) {
        if (StringUtils.isNotEmpty(mdmRegionReqVo.getParentCode())) {
            //查询下级
            List<MdmRegionRespVo> list = mdmRegionMapper.pageList(null, new MdmRegionReqVo().setParentCode(mdmRegionReqVo.getParentCode()));
            return PageResult.<MdmRegionRespVo>builder()
                    .data(list)
                    .count(list != null ? list.size() : 0L)
                    .build();
        } else if (StringUtils.isEmpty(mdmRegionReqVo.getRegionCode()) && StringUtils.isEmpty(mdmRegionReqVo.getRegionName())) {
            List<MdmRegionRespVo> list = mdmRegionMapper.findFirstLevel();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                list.forEach(item -> item.setHasChild(YesNoEnum.yesNoEnum.ONE.getValue().equalsIgnoreCase(String.valueOf(item.getHasChildFlag()))));
            }
//            //查询第一层
//            List<MdmRegionEntity> topRegion = this.lambdaQuery()
//                    .isNull(MdmRegionEntity::getParentCode)
//                    .or()
//                    .eq(MdmRegionEntity::getParentCode, "")
//                    .list();
//            List<MdmRegionRespVo> list = new ArrayList<>();
//            if (CollectionUtil.listNotEmptyNotSizeZero(topRegion)) {
//                list.addAll(CrmBeanUtil.copyList(topRegion, MdmRegionRespVo.class));
//                List<String> regionCodeList = list.stream()
//                        .filter(x -> !StringUtils.isEmpty(x.getRegionCode()))
//                        .map(MdmRegionRespVo::getRegionCode)
//                        .collect(Collectors.toList());
//                this.lambdaQuery()
//                        .in(MdmRegionEntity::getParentCode)
//            }
//            List<MdmRegionRespVo> list = mdmRegionMapper.pageList(null, new MdmRegionReqVo().setRegionLevel(RegionLevelEnum.COUNTRY.getCode()));
            return PageResult.<MdmRegionRespVo>builder()
                    .data(list)
                    .count(list != null ? list.size() : 0L)
                    .build();
        } else {
            //查询分页（平铺）
            Page<MdmRegionRespVo> page = PageUtil.buildPage(mdmRegionReqVo.getPageNum(), mdmRegionReqVo.getPageSize());
            List<MdmRegionRespVo> list = mdmRegionMapper.pageList(page, mdmRegionReqVo);
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                list.forEach(item -> {
                    item.setHasChild(false);
                });
            }
            return PageResult.<MdmRegionRespVo>builder()
                    .data(list)
                    .count(page.getTotal())
                    .build();
        }
    }

    @Override
    public PageResult<MdmRegionRespVo> findExportList(MdmRegionReqVo mdmRegionReqVo) {
        Page<MdmRegionRespVo> page = PageUtil.buildPage(mdmRegionReqVo.getPageNum(), mdmRegionReqVo.getPageSize());
        List<MdmRegionRespVo> list = mdmRegionMapper.pageList(page, mdmRegionReqVo);
        return PageResult.<MdmRegionRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public List<MdmRegionRespVo> findList(MdmRegionReqVo reqVo) {
        return mdmRegionMapper.findList(reqVo);
    }

    @Override
    public MdmRegionRespVo query(String id, String regionCode) {
        if (StringUtils.isEmpty(id) && StringUtils.isEmpty(regionCode)) {
            return null;
        }
        return mdmRegionMapper.query(id, regionCode);
    }

    @Override
    public String queryByName(String regionCode) {
        MdmRegionEntity entity = mdmRegionMapper.selectOne(new QueryWrapper<>(new MdmRegionEntity().setRegionCode(regionCode)));
        if(Objects.isNull(entity)){
            return "";
        }
        return entity.getRegionName();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmRegionReqVo reqVo) {
        String regionCode = valid(reqVo);
//        reqVo.setHasChild(false);
        MdmRegionEntity one = this.lambdaQuery()
                .eq(MdmRegionEntity::getRegionCode, regionCode)
                .last(DatabaseTypeUtil.SEGMENT)
                .select(MdmRegionEntity::getId)
                .one();
        Assert.isNull(one, "当前区域编码已存在，请重新输入");
        if (!StringUtils.isEmpty(reqVo.getParentCode())) {
            MdmRegionEntity parent = this.lambdaQuery()
                    .eq(MdmRegionEntity::getRegionCode, reqVo.getParentCode())
                    .select(MdmRegionEntity::getId, MdmRegionEntity::getHasChild)
                    .one();
            Assert.notNull(parent, "上级区域不存在");
            Boolean hasChild = parent.getHasChild();
            if (hasChild == null || !hasChild) {
                this.lambdaUpdate()
                        .eq(MdmRegionEntity::getId, parent.getId())
                        .set(MdmRegionEntity::getHasChild, true)
                        .update();
            }
        }

        MdmRegionEntity entity = CrmBeanUtil.copy(reqVo, MdmRegionEntity.class);
        this.save(entity);
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            crmLogSendUtil.sendForAdd(((String) menuObject), entity.getId(),entity.getRegionCode(),CrmBeanUtil.copy(entity,MdmRegionRespVo.class));
        }
    }

    protected String valid(MdmRegionReqVo reqVo) {
        String regionCode = reqVo.getRegionCode();
        Assert.hasText(regionCode, "区域编码不能为空");
//        Assert.isTrue(regionCode.matches("^[A-Za-z0-9]+$"), "区域编码请输入字母和数字组合");
        String regionName = reqVo.getRegionName();
        Assert.hasText(regionName, "区域名称不能为空");
//        Assert.hasText(reqVo.getRegionLevel(), "区域层级不能为空");
//        String latitude = reqVo.getLatitude();
//        String longitude = reqVo.getLongitude();
//        Assert.hasText(latitude, "纬度不能为空");
//        Assert.isTrue(latitude.matches("^[0-9]+(.[0-9]{2})?$"), "纬度仅支持输入数字，保留两位小数");
//        Assert.hasText(longitude, "经度不能为空");
//        Assert.isTrue(longitude.matches("^[0-9]+(.[0-9]{2})?$"), "经度仅支持输入数字，保留两位小数");
//        String email = reqVo.getEmail();
//        if (!StringUtils.isEmpty(email)) {
//            Assert.isTrue(email.matches("^\\d{6}$"), "邮编仅支持6位数字");
//        }
        return regionCode;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmRegionReqVo reqVo) {
        String id = reqVo.getId();
        Assert.hasText(id, "主键不能为空");
        MdmRegionEntity entity = this.getById(id);
        Assert.notNull(entity, "区域不存在");
        List<MdmRegionEntity> collect = this.lambdaQuery()
                .eq(MdmRegionEntity::getRegionCode, reqVo.getRegionCode())
                .select(MdmRegionEntity::getId)
                .list()
                .stream().filter(x -> !id.equals(x.getId())).collect(Collectors.toList());
        Assert.isTrue(CollectionUtil.listEmpty(collect), "当前区域编码已存在，请重新输入");
//        updateChildMark(reqVo, entity);
        updateSubRegionLevel(reqVo, entity);
        this.updateById(CrmBeanUtil.copy(reqVo, MdmRegionEntity.class));
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            crmLogSendUtil.sendForUpdate(((String) menuObject), entity.getId(),entity.getRegionCode(),CrmBeanUtil.copy(entity,MdmRegionRespVo.class),CrmBeanUtil.copy(reqVo,MdmRegionRespVo.class));
        }
    }

    /**
     * 递归修改下级区域层级
     *
     * @param reqVo  vo
     * @param entity entity
     */
    protected void updateSubRegionLevel(MdmRegionReqVo reqVo, MdmRegionEntity entity) {
        String regionLevel = reqVo.getRegionLevel();
        if (!regionLevel.equals(entity.getRegionLevel())) {
            //循环修改所有
            int level = Integer.parseInt(regionLevel);
            List<MdmRegionEntity> list = this.lambdaQuery()
                    .eq(MdmRegionEntity::getParentCode, reqVo.getRegionCode())
                    .select(MdmRegionEntity::getId, MdmRegionEntity::getHasChild, MdmRegionEntity::getEnableStatus)
                    .list();
            while (!CollectionUtils.isEmpty(list)) {
                ++level;
                for (MdmRegionEntity mdmRegionEntity : list) {
                    mdmRegionEntity.setRegionLevel(String.valueOf(level));
                }
                updateBatchById(list);
                list = this.lambdaQuery()
                        .in(MdmRegionEntity::getParentCode, list.stream().map(MdmRegionEntity::getRegionCode).collect(Collectors.toList()))
                        .select(MdmRegionEntity::getId, MdmRegionEntity::getHasChild, MdmRegionEntity::getEnableStatus)
                        .list();
            }
        }
    }

    /**
     * 更新节点的叶子节点标识
     *
     * @param reqVo  vo
     * @param entity entity
     */
    protected void updateChildMark(MdmRegionReqVo reqVo, MdmRegionEntity entity) {
        String id = reqVo.getId();
        String parentCode = entity.getParentCode();
        if (StringUtils.isEmpty(parentCode)) {
            if (!StringUtils.isEmpty(reqVo.getParentCode())) {
                //原来没有上级，现在有
                reqVo.setHasChild(true);
            }
        } else {
            if (!StringUtils.isEmpty(reqVo.getParentCode()) && (!reqVo.getParentCode().equals(parentCode))) {
                //原来有上级，现在变动,
                //旧上级不再有下级,
                long count = this.lambdaQuery()
                        .eq(MdmRegionEntity::getParentCode, parentCode)
                        .select(MdmRegionEntity::getId)
                        .list()
                        .stream().filter(x -> !id.equals(x.getId()))
                        .count();
                if (count == 0) {
                    this.lambdaUpdate()
                            .eq(MdmRegionEntity::getRegionCode, parentCode)
                            .set(MdmRegionEntity::getHasChild, false)
                            .update();
                }
                //新上级原来没有下级
                Integer newCount = this.lambdaQuery()
                        .eq(MdmRegionEntity::getParentCode, reqVo.getParentCode())
                        .count();
                if (newCount != null && newCount > 0) {
                    this.lambdaUpdate()
                            .eq(MdmRegionEntity::getRegionCode, reqVo.getParentCode())
                            .set(MdmRegionEntity::getHasChild, true)
                            .update();
                }
            } else {
                //原来有上级，现在没有,且旧上级不再有下级
                long count = this.lambdaQuery()
                        .eq(MdmRegionEntity::getParentCode, parentCode)
                        .select(MdmRegionEntity::getId)
                        .list()
                        .stream().filter(x -> !id.equals(x.getId()))
                        .count();
                if (count == 0) {
                    this.lambdaUpdate()
                            .eq(MdmRegionEntity::getRegionCode, parentCode)
                            .set(MdmRegionEntity::getHasChild, false)
                            .update();
                }
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(MdmRegionReqVo reqVo) {
        log.info("[行政区域][删除]:{}", reqVo.getIds());
        List<String> ids = reqVo.getIds();
        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        List<String> regionCodeList = new ArrayList<>();
        List<MdmRegionEntity> list = this.lambdaQuery()
                .in(MdmRegionEntity::getId, ids)
                .select(MdmRegionEntity::getRegionCode)
                .list();
        while (!CollectionUtils.isEmpty(list)) {
            List<String> currentRegionCodeList = list.stream()
                    .filter(x -> x != null && !StringUtils.isEmpty(x.getRegionCode()))
                    .map(MdmRegionEntity::getRegionCode)
                    .collect(Collectors.toList());
            if (CollectionUtils.isEmpty(currentRegionCodeList)) {
                break;
            }
            regionCodeList.addAll(currentRegionCodeList);
            list = this.lambdaQuery()
                    .in(MdmRegionEntity::getParentCode, currentRegionCodeList)
                    .select(MdmRegionEntity::getRegionCode)
                    .list();
        }
        if(!CollectionUtils.isEmpty(regionCodeList)){
            this.lambdaUpdate()
                    .in(MdmRegionEntity::getRegionCode, regionCodeList)
                    .remove();
        }
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            for (String id : ids) {
                MdmRegionRespVo entity = query(id, null);
                if (entity != null) {
                    crmLogSendUtil.sendForDel(((String) menuObject), entity.getId(), entity.getRegionCode(), CrmBeanUtil.copy(entity, MdmRegionRespVo.class));
                }
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(MdmRegionReqVo reqVo) {
        List<String> ids = reqVo.getIds();
        this.lambdaUpdate()
                .in(MdmRegionEntity::getId, ids)
                .set(MdmRegionEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .update();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(MdmRegionReqVo reqVo) {
        List<String> ids = reqVo.getIds();
        this.lambdaUpdate()
                .in(MdmRegionEntity::getId, ids)
                .set(MdmRegionEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
                .update();
    }

    @Override
    public List<MdmRegionSelectRespVo> select(MdmRegionReqVo mdmRegionReqVo) {
        String currentRegionLevel = mdmRegionReqVo.getCurrentRegionLevel();
        if (!StringUtils.isEmpty(currentRegionLevel)) {
            //传入currentRegionLevel表示查询上一层级
            Assert.isTrue(currentRegionLevel.matches("^[0-9]+$"), "区域层级必须是数字");
            mdmRegionReqVo.setRegionLevel(String.valueOf(Integer.parseInt(currentRegionLevel) - 1));
        }
        List<MdmRegionEntity> list = this.lambdaQuery()
                .eq(!StringUtils.isEmpty(mdmRegionReqVo.getId()), MdmRegionEntity::getId, mdmRegionReqVo.getId())
                .in(!CollectionUtils.isEmpty(mdmRegionReqVo.getIds()), MdmRegionEntity::getId, mdmRegionReqVo.getId())
                .in(!CollectionUtils.isEmpty(mdmRegionReqVo.getRegionCodeList()), MdmRegionEntity::getRegionCode, mdmRegionReqVo.getRegionCodeList())
                .like(!StringUtils.isEmpty(mdmRegionReqVo.getRegionCode()), MdmRegionEntity::getRegionCode, mdmRegionReqVo.getRegionCode())
                .like(!StringUtils.isEmpty(mdmRegionReqVo.getRegionName()), MdmRegionEntity::getRegionName, mdmRegionReqVo.getRegionName())
                .eq(!StringUtils.isEmpty(mdmRegionReqVo.getRegionLevel()), MdmRegionEntity::getRegionLevel, mdmRegionReqVo.getRegionLevel())
                .eq(!StringUtils.isEmpty(mdmRegionReqVo.getParentCode()), MdmRegionEntity::getParentCode, mdmRegionReqVo.getParentCode())
                .eq(!StringUtils.isEmpty(mdmRegionReqVo.getEnableStatus()), MdmRegionEntity::getEnableStatus, mdmRegionReqVo.getEnableStatus())
                .list();
        return CrmBeanUtil.copyList(list, MdmRegionSelectRespVo.class);
    }

    @Override
    public List<MdmRegionSelectRespVo> selectStartWithCountry(MdmRegionSelectReqVo reqVo) {
        if (StringUtils.isNotEmpty(reqVo.getParentCode())) {
            reqVo.setRegionLevel(null);
        } else {
            reqVo.setRegionLevel(RegionLevelEnum.COUNTRY.getCode());
        }
        List<MdmRegionEntity> list = this.lambdaQuery()
                .eq(StringUtils.isNotEmpty(reqVo.getRegionLevel()), MdmRegionEntity::getRegionLevel, reqVo.getRegionLevel())
                .eq(StringUtils.isNotEmpty(reqVo.getParentCode()), MdmRegionEntity::getParentCode, reqVo.getParentCode())
                .like(StringUtils.isNotEmpty(reqVo.getRegionName()), MdmRegionEntity::getRegionName, reqVo.getRegionName())
                .select(MdmRegionEntity::getRegionCode, MdmRegionEntity::getRegionName)
                .list();
        return CrmBeanUtil.copyList(list, MdmRegionSelectRespVo.class);
    }

    @Override
    public List<MdmRegionSelectRespVo> selectStartWithProvince(MdmRegionSelectReqVo reqVo) {
        if (StringUtils.isNotEmpty(reqVo.getParentCode())) {
            reqVo.setRegionLevel(null);
        } else {
            reqVo.setRegionLevel(RegionLevelEnum.PROVINCE.getCode());
        }
        List<MdmRegionEntity> list = this.lambdaQuery()
                .eq(StringUtils.isNotEmpty(reqVo.getRegionLevel()), MdmRegionEntity::getRegionLevel, reqVo.getRegionLevel())
                .eq(StringUtils.isNotEmpty(reqVo.getParentCode()), MdmRegionEntity::getParentCode, reqVo.getParentCode())
                .like(StringUtils.isNotEmpty(reqVo.getRegionName()), MdmRegionEntity::getRegionName, reqVo.getRegionName())
                .select(MdmRegionEntity::getRegionCode, MdmRegionEntity::getRegionName)
                .list();
        return CrmBeanUtil.copyList(list, MdmRegionSelectRespVo.class);
    }

    @Override
    public List<MdmRegionSelectRespVo> regionSelect(MdmRegionSelectReqVo reqVo) {
        Integer pageSize = reqVo.getPageSize();
        if (pageSize == null || pageSize == 0) {
            pageSize = Integer.valueOf(ParamUtil.getParameterValue(ParameterParam.SELECT_BOX_OPTION_SIZE));
        }
        if (StringUtils.isNotEmpty(reqVo.getChildrenLevel())) {
            String regionLevel = RegionLevelEnum.getParentRegionLevel(reqVo.getChildrenLevel());
            if (StringUtils.isEmpty(regionLevel)) {
                return new ArrayList<>();
            }
            reqVo.setRegionLevel(regionLevel);
        }
        List<MdmRegionSelectTransportRespVo> transportRespList = new ArrayList<>();

        List<String> codeList = new ArrayList<>();
        if (StringUtils.isNotEmpty(reqVo.getSelectedRegionCode())) {
            codeList.add(reqVo.getSelectedRegionCode());
        }
        if (StringUtils.isNotEmpty(reqVo.getSelectedCode())) {
            codeList.add(reqVo.getSelectedCode());
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSelectedCodeList())) {
            codeList.addAll(reqVo.getSelectedCodeList());
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(codeList)) {
            List<MdmRegionSelectTransportRespVo> selectedList = mdmRegionMapper.regionSelect(new Page<>(1L, codeList.size(), false), reqVo, codeList, null);
            if (CollectionUtil.listNotEmptyNotSizeZero(selectedList)) {
                transportRespList.addAll(selectedList);
                pageSize -= selectedList.size();
            }
        }
        if (pageSize > 0) {
            transportRespList.addAll(mdmRegionMapper.regionSelect(new Page<>(1L, pageSize, false), reqVo, null, codeList));
        }

        if (CollectionUtil.listNotEmptyNotSizeZero(transportRespList)) {
            return transportRespList.stream().map(item -> {
                MdmRegionSelectRespVo respVo = new MdmRegionSelectRespVo();
                respVo.setRegionCode(item.getRegionCode());
                respVo.setRegionName(item.getRegionName());
                String unionName = "";
                if (StringUtils.isNotEmpty(item.getThirdParentRegionName())) {
                    unionName += item.getThirdParentRegionName() + "/";
                }
                if (StringUtils.isNotEmpty(item.getSecondParentRegionName())) {
                    unionName += item.getSecondParentRegionName() + "/";
                }
                if (StringUtils.isNotEmpty(item.getFirstParentRegionName())) {
                    unionName += item.getFirstParentRegionName() + "/";
                }
                unionName += item.getRegionName();
                respVo.setUnionName(unionName);
                return respVo;
            }).collect(Collectors.toList());
        }
        return new ArrayList<>();
    }

    @Override
    public Map<String, String> getRegionCodeNameMap(List<String> regionCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(regionCodeList)) {
            return this.lambdaQuery()
                    .in(MdmRegionEntity::getRegionCode, regionCodeList)
                    .select(MdmRegionEntity::getRegionCode, MdmRegionEntity::getRegionName)
                    .list()
                    .stream().collect(Collectors.toMap(MdmRegionEntity::getRegionCode, MdmRegionEntity::getRegionName));
        }
        return Collections.emptyMap();
    }

//    @Override
//    @Transactional(rollbackFor = Exception.class)
//    public void resetRuleCode() {
//
//        long l = System.currentTimeMillis();
//        log.info("-----------重置行政区域降维编码_" + l + "_开始-----------");
////        log.info("-----------重置行政区域降维编码_" + l + "_编码重置成id成功-----------");
////        log.info("-----------重置行政区域降维编码_" + l + "_清除无效上级编码成功-----------");
//        List<MdmRegionEntity> list = this.lambdaQuery()
//                .isNull(MdmRegionEntity::getParentCode)
//                .or()
//                .eq(MdmRegionEntity::getParentCode, "")
//                .orderByAsc(MdmRegionEntity::getRegionCode)
//                .list();
//        for (int i = 0; i < list.size(); i++) {
//            updateCurAndChildren(list.get(i).getRegionCode(), TreeRuleCodeUtil.numToSingleCode(i + 1), 1);
//        }
//        log.info("-----------重置行政区域降维编码_" + l + "_编码重新生成成功-----------");
////        log.info("-----------重置行政区域降维编码_" + l + "_缓存清除成功-----------");
//        log.info("-----------重置行政区域降维编码_" + l + "_结束-----------");
//
//    }
//
//    /**
//     * 更新行政区域及行政区域下级降维编码和层级
//     *
//     * @param regionCode  当前行政区域编码
//     * @param curRuleCode  当前行政区域降维编码
//     * @param levelNum 当前层级
//     */
//    private void updateCurAndChildren(String regionCode, String curRuleCode, int levelNum) {
//
//        //更新当前
//        this.lambdaUpdate()
//                .eq(MdmRegionEntity::getRegionCode, regionCode)
//                .set(MdmRegionEntity::getRuleCode, curRuleCode)
//                .set(MdmRegionEntity::getLevelNum, levelNum)
//                .update();
//
//        //查询下一层
//        List<MdmRegionEntity> list = this.lambdaQuery()
//                .eq(MdmRegionEntity::getParentCode, regionCode)
//                .select(MdmRegionEntity::getRegionCode)
//                .orderByAsc(MdmRegionEntity::getRegionCode)
//                .list();
//
//        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
//            //遍历下级
//            for (int i = 0; i < list.size(); i++) {
//                //递归调用
//                updateCurAndChildren(list.get(i).getRegionCode(), curRuleCode + TreeRuleCodeUtil.numToSingleCode(i + 1), (levelNum + 1));
//            }
//        }
//
//    }
}
