package com.biz.crm.shoppingcart.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.annotation.Klock;
import com.biz.crm.availablelist.service.CusAvailablelistService;
import com.biz.crm.base.BusinessException;
import com.biz.crm.customer.utils.ValidateUtils;
import com.biz.crm.eunm.dms.ShoppingCartEunm;
import com.biz.crm.nebular.mdm.availablelist.CusAvailablelistVo;
import com.biz.crm.nebular.mdm.product.resp.MdmProductSaleCompanyRespVo;
import com.biz.crm.nebular.mdm.shoppingcart.ShoppingCartVo;
import com.biz.crm.product.service.MdmProductService;
import com.biz.crm.shoppingcart.entity.ShoppingCartEntity;
import com.biz.crm.shoppingcart.mapper.ShoppingCartMapper;
import com.biz.crm.shoppingcart.service.ShoppingCartService;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.UserRedis;
import com.biz.crm.util.UserUtils;
import com.biz.crm.utils.MdmConstant;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Collectors;

/**
 * @Description:
 * @Author: heyijiang
 * @Date: 2020/10/20 11:18
 **/
@ConditionalOnMissingBean(name = "shoppingCartServiceExpandImpl")
@Service(value = "shoppingCartService")
public class ShoppingCartServiceImpl<M extends BaseMapper<T>,T> extends ServiceImpl<ShoppingCartMapper, ShoppingCartEntity> implements ShoppingCartService {

    @Autowired
    private ShoppingCartMapper shoppingCartMapper;
    @Resource
    private CusAvailablelistService cusAvailablelistService;
    @Resource
    private MdmProductService mdmProductService;

    /**
     * 1、判断该商品是否存在购物车
     * 2、如果该商品不在购物车，则购物车新增该商品数据；
     * 3、如果该商品已经在购物车里面了，则数量加
     * @param shoppingCartVo
     */
    @Override
    @Klock(keys = {MdmConstant.SHOPPINGCAT_LOCK ,"#shoppingCartVo.cusCode"},waitTime = 20,leaseTime = 5)
    @Transactional
    public void add(ShoppingCartVo shoppingCartVo) {
        ValidateUtils.validate(shoppingCartVo.getGoodsCode(), "请选定商品");
        ValidateUtils.validate(shoppingCartVo.getNum(), "请指定商品数量");
        //如果传入数量非法，则默认设置为1
        if(shoppingCartVo.getNum() == null || shoppingCartVo.getNum() <= 0){
            shoppingCartVo.setNum(1);
        }

        //1
        QueryWrapper<ShoppingCartEntity> wrapper = new QueryWrapper<>();
        wrapper.eq("cus_code", shoppingCartVo.getCusCode())
                .eq("goods_code", shoppingCartVo.getGoodsCode());
        ShoppingCartEntity entity = shoppingCartMapper.selectOne(wrapper);

        //2
        if(null == entity){
            entity = new ShoppingCartEntity();
            BeanUtils.copyProperties(shoppingCartVo, entity);
            shoppingCartMapper.insert(entity);
            return;
        }

        //3
        entity.setNum(entity.getNum() + shoppingCartVo.getNum());
        shoppingCartMapper.updateById(entity);

    }

    /**
     * 1、验证
     * 2、查询更改数量
     * @param shoppingCartVo
     */
    @Override
    @Klock(keys = {MdmConstant.SHOPPINGCAT_LOCK ,"#shoppingCartVo.cusCode"},waitTime = 20,leaseTime = 5)
    @Transactional
    public void adjustNum(ShoppingCartVo shoppingCartVo) {
        //1
        ValidateUtils.validate(shoppingCartVo.getId(), "请选定商品");
        ValidateUtils.validate(shoppingCartVo.getNum(), "请指定商品数量");

        //2
        ShoppingCartEntity entity = shoppingCartMapper.selectById(shoppingCartVo.getId());
        if(null == entity){
            throw new BusinessException("该商品已被移除购物车");
        }
        entity.setNum(shoppingCartVo.getNum());
        if(entity.getNum() <= 0){
            throw new BusinessException("数量非法");
        }
        shoppingCartMapper.updateById(entity);
    }

    /**
     * 1、获取客户购物车信息
     * 2、根据购物车里的商品编码，获取商品详细信息以及商品发货公司
     * 3、按商品发货公司分袋组装购物车
     * 4、按照发货公司编码排序
     * @param shoppingCartVo
     * @return
     */
    @Override
    public Map<String, List<ShoppingCartVo>> list(ShoppingCartVo shoppingCartVo) {
        ValidateUtils.validate(shoppingCartVo.getCusCode(), "客户编码不能为空");
        //1
        LambdaQueryWrapper<ShoppingCartEntity> wrapper = Wrappers.lambdaQuery();
        wrapper.eq(ShoppingCartEntity::getCusCode, shoppingCartVo.getCusCode());
        wrapper.like(StringUtils.isNotEmpty(shoppingCartVo.getGoodsCode()),ShoppingCartEntity::getGoodsCode,shoppingCartVo.getGoodsCode());
        wrapper.like(StringUtils.isNotEmpty(shoppingCartVo.getGoodsName()),ShoppingCartEntity::getGoodsName,shoppingCartVo.getGoodsName());
        List<ShoppingCartEntity> entities = shoppingCartMapper.selectList(wrapper);
        if(CollectionUtils.isEmpty(entities)){
            return new HashMap<>();
        }
        List<String> goodsCodes = entities.stream()
                .map(ShoppingCartEntity::getGoodsCode)
                .collect(Collectors.toList());

        //2
        Map<String, CusAvailablelistVo> goodsInfos = cusAvailablelistService.findGoodsByCusAndGoodsCodes(goodsCodes, shoppingCartVo.getCusCode());
        Map<String, List<MdmProductSaleCompanyRespVo>> salesCompanis = mdmProductService.findSaleCompanyByProductCodeList(goodsCodes);

        //3（使用TreeMap就不用进行第4步了）
        Map<String, List<ShoppingCartVo>> reMap = new TreeMap<>((o , s) -> {
        	return StringUtils.compare(o, s);
        });
        Map<String, String> unitMap = DictUtil.getDictValueMapsByCodes("product_sale_unit");
        for(ShoppingCartEntity entity : entities){
            MdmProductSaleCompanyRespVo salesCompany = null;
            if(!CollectionUtils.isEmpty(salesCompanis.get(entity.getGoodsCode()))){
                salesCompany = salesCompanis.get(entity.getGoodsCode()).get(0);
            }

            ShoppingCartVo vo = new ShoppingCartVo();
            BeanUtils.copyProperties(entity, vo);
            if(null != salesCompany && !StringUtils.isAnyBlank(salesCompany.getSaleCompanyCode() , salesCompany.getSaleCompanyName())){
                vo.setSalesCompanyCode(salesCompany.getSaleCompanyCode());
                vo.setSalesCompanyName(salesCompany.getSaleCompanyName());
            }else {
                vo.setSalesCompanyCode(MdmConstant.ZREOSTR);
                vo.setSalesCompanyName("");
            }
            CusAvailablelistVo goodsInfo = goodsInfos.get(entity.getGoodsCode());
            if(null == goodsInfo){
                vo.setGoodsState(ShoppingCartEunm.goodsStateEunm.DOWNSHELF.getCode());
            }else{
                vo.setGoodsState(ShoppingCartEunm.goodsStateEunm.NORMAL.getCode());
                vo.setGoodsInfo(goodsInfos.get(entity.getGoodsCode()));
                vo.getGoodsInfo().setSaleUnitName(
                        unitMap.get(vo.getGoodsInfo().getSaleUnit())
                );
            }

            if(!reMap.containsKey(vo.getSalesCompanyCode())){
                reMap.put(vo.getSalesCompanyCode(),new ArrayList<ShoppingCartVo>());
            }
            reMap.get(vo.getSalesCompanyCode()).add(vo);
        }
        return reMap;
    }

    @Override
    @Klock(keys = {MdmConstant.SHOPPINGCAT_LOCK ,"#cusCode"},waitTime = 20,leaseTime = 5)
    @Transactional
    public void delByIds(List<String> ids, String cusCode) {
        if(CollectionUtils.isEmpty(ids)){
            return;
        }
        shoppingCartMapper.deleteBatchIds(ids);
    }

    /**
     * 批量加入购物车
     * @param shoppingCartVos
     */
    @Override
    @Transactional
    public void addBatch(List<ShoppingCartVo> shoppingCartVos) {
        ValidateUtils.validate(shoppingCartVos, "批量加入购物车时，参数列表不能为空");
        UserRedis userRedis = UserUtils.getUser();
        shoppingCartVos.forEach(shoppingCartVo -> {
            shoppingCartVo.setCusCode(userRedis.getCustcode());
            shoppingCartVo.setAccount(userRedis.getUsername());
            this.add(shoppingCartVo);
        });
    }

    /**
     * 清空客户相关商品
     * @param cusCode
     * @param productCodes
     */
    @Override
    @Transactional
    public void clearShoppingCar(String cusCode, List<String> productCodes) {
        if(StringUtils.isEmpty(cusCode) || CollectionUtil.listEmpty(productCodes)) {
            return;
        }
        List<ShoppingCartVo> vos = this.findByCusCodeAndProductCodes(cusCode, productCodes);
        if(CollectionUtil.listEmpty(vos)) {
            return;
        }
        List<String> ids = vos.stream().map(ShoppingCartVo::getId).collect(Collectors.toList());
        this.delByIds(ids, cusCode);
    }

    /**
     * 根据客户编码和商品列表查询
     * @param cusCode
     * @param productCodes
     * @return
     */
    @Override
    public List<ShoppingCartVo> findByCusCodeAndProductCodes(String cusCode, List<String> productCodes) {
        if(StringUtils.isEmpty(cusCode) || CollectionUtil.listEmpty(productCodes)) {
            return Lists.newArrayList();
        }
        LambdaQueryWrapper<ShoppingCartEntity> wrappers = Wrappers.<ShoppingCartEntity>lambdaQuery()
                .eq(ShoppingCartEntity::getCusCode, cusCode)
                .in(ShoppingCartEntity::getGoodsCode, productCodes);
        List<ShoppingCartEntity> entities = this.getBaseMapper().selectList(wrappers);
        if(CollectionUtil.listEmpty(entities)) {
            return Lists.newArrayList();
        }
        return CrmBeanUtil.copyList(entities, ShoppingCartVo.class);
    }
}
