package com.biz.crm.tableconfig.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.CacheManagerConstant;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.mdm.TrueOrFalseEnum;
import com.biz.crm.nebular.mdm.pageconfig.MdmColumnConfigBatchReqVo;
import com.biz.crm.nebular.mdm.pageconfig.MdmColumnConfigReqVo;
import com.biz.crm.nebular.mdm.pageconfig.MdmColumnConfigRespVo;
import com.biz.crm.nebular.mdm.pageconfig.MdmColumnExportRespVo;
import com.biz.crm.tableconfig.manager.MdmColumnConfigConverter;
import com.biz.crm.tableconfig.mapper.MdmColumnConfigMapper;
import com.biz.crm.tableconfig.model.MdmColumnConfigEntity;
import com.biz.crm.tableconfig.service.IMdmColumnConfigService;
import com.biz.crm.tableconfig.service.IMdmFunctionSubButtonService;
import com.biz.crm.tableconfig.service.MdmColumnConfigPersonalService;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CrmBeanUtil;
import io.swagger.annotations.ApiModelProperty;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.lang.reflect.Field;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 字段配置接口实现
 *
 * @author zxw
 * @date 2020-11-21 12:19:06
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmColumnConfigServiceExpandImpl")
@CacheConfig(cacheNames = CacheManagerConstant.MDM_COLUMN)
public class MdmColumnConfigServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmColumnConfigMapper, MdmColumnConfigEntity> implements IMdmColumnConfigService {

    @Resource
    private MdmColumnConfigMapper mdmColumnConfigMapper;

    @Resource
    private IMdmFunctionSubButtonService mdmFunctionSubButtonService;

    @Resource
    private MdmColumnConfigPersonalService mdmColumnConfigPersonalService;

    @Override
    public PageResult<MdmColumnConfigRespVo> findList(MdmColumnConfigReqVo reqVo) {
        Page<MdmColumnConfigRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmColumnConfigEntity> list = mdmColumnConfigMapper.findList(page, reqVo);
        if (CollectionUtils.isEmpty(list)) {
            return new PageResult<>(0L, new ArrayList<>());
        }
        return PageResult.<MdmColumnConfigRespVo>builder()
                .count(page.getTotal())
                .data(MdmColumnConfigConverter.INSTANCE.entity2RespVo(list))
                .build();
    }

    @Override
    public MdmColumnConfigRespVo query(MdmColumnConfigReqVo reqVo) {
        MdmColumnConfigEntity one = this.lambdaQuery()
                .eq(!StringUtils.isEmpty(reqVo.getParentCode()), MdmColumnConfigEntity::getParentCode, reqVo.getParentCode())
                .eq(!StringUtils.isEmpty(reqVo.getFunctionCode()), MdmColumnConfigEntity::getFunctionCode, reqVo.getFunctionCode())
                .eq(!StringUtils.isEmpty(reqVo.getField()), MdmColumnConfigEntity::getField, reqVo.getField())
                .one();
        return MdmColumnConfigConverter.INSTANCE.entity2RespVo(one);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(key = "#reqVo.parentCode+#reqVo.functionCode")
    public void save(MdmColumnConfigReqVo reqVo) {
        Assert.hasText(reqVo.getParentCode(), "菜单编码不能为空");
        Assert.hasText(reqVo.getFunctionCode(), "功能编码不能为空");
        Assert.hasText(reqVo.getField(), "字段名称不能为空");
        validSearch(Collections.singletonList(reqVo));
        this.lambdaUpdate()
                .eq(MdmColumnConfigEntity::getParentCode, reqVo.getParentCode())
                .eq(MdmColumnConfigEntity::getFunctionCode, reqVo.getFunctionCode())
                .eq(MdmColumnConfigEntity::getField, reqVo.getField())
                .remove();
        reqVo.setId(null);
        this.save(MdmColumnConfigConverter.INSTANCE.reqVo2Entity(reqVo));
        mdmColumnConfigPersonalService.deleteColumnConfig(reqVo.getParentCode(), reqVo.getFunctionCode());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(allEntries = true)
    public void update(MdmColumnConfigReqVo reqVo) {
        Assert.hasText(reqVo.getParentCode(), "菜单编码不能为空");
        Assert.hasText(reqVo.getFunctionCode(), "功能编码不能为空");
        Assert.hasText(reqVo.getField(), "字段名称不能为空");
        validSearch(Collections.singletonList(reqVo));
        String id = reqVo.getId();
        Assert.hasText(id, "主键不能为空");
        MdmColumnConfigEntity entity = this.getById(id);
        Assert.notNull(entity, "字段不存在");
        this.updateById(MdmColumnConfigConverter.INSTANCE.reqVo2Entity(reqVo));
        mdmColumnConfigPersonalService.deleteColumnConfig(reqVo.getParentCode(), reqVo.getFunctionCode());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(allEntries = true)
    public void deleteBatch(MdmColumnConfigReqVo reqVo) {
        List<String> ids = reqVo.getIds();
        Assert.notEmpty(ids, "id集合不能为空");
        List<MdmColumnConfigEntity> list = this.lambdaQuery()
                .in(MdmColumnConfigEntity::getId, reqVo.getIds())
                .list();
        if (!CollectionUtils.isEmpty(list)) {
            MdmColumnConfigEntity mdmColumnConfigEntity = list.get(0);
            String parentCode = mdmColumnConfigEntity.getParentCode();
            String functionCode = mdmColumnConfigEntity.getFunctionCode();
            if (!StringUtils.isEmpty(parentCode) && !StringUtils.isEmpty(functionCode)) {
                mdmColumnConfigPersonalService.deleteColumnConfig(parentCode, functionCode);
            }
            this.removeByIds(ids);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(allEntries = true)
    public void enableBatch(MdmColumnConfigReqVo reqVo) {
        //设置状态为启用
        List<String> ids = reqVo.getIds();
        Assert.notEmpty(ids,"id集合不能为空");
        this.lambdaUpdate()
                .in(MdmColumnConfigEntity::getId, ids)
                .set(MdmColumnConfigEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .update();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(allEntries = true)
    public void disableBatch(MdmColumnConfigReqVo reqVo) {
        //设置状态为禁用
        List<String> ids = reqVo.getIds();
        Assert.notEmpty(ids,"id集合不能为空");
        this.lambdaUpdate()
                .in(MdmColumnConfigEntity::getId, ids)
                .set(MdmColumnConfigEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
                .update();
    }

    @Override
    @Cacheable(sync = true,key = "#reqVo.parentCode+#reqVo.functionCode")
    public List<MdmColumnConfigRespVo> listCondition(MdmColumnConfigReqVo reqVo) {
        String parentCode = reqVo.getParentCode();
        String functionCode = reqVo.getFunctionCode();
        if (StringUtils.isEmpty(parentCode) || StringUtils.isEmpty(functionCode)) {
            return new ArrayList<>();
        }
        List<MdmColumnConfigEntity> list = this.lambdaQuery()
                .eq(MdmColumnConfigEntity::getParentCode, parentCode)
                .eq(MdmColumnConfigEntity::getFunctionCode, functionCode)
                .list();
        if (!CollectionUtils.isEmpty(list)) {
            list = list.stream().sorted(Comparator.comparing(x -> Optional.ofNullable(x.getFormorder())
                    .map(Integer::valueOf)
                    .orElse(100))).collect(Collectors.toList());
        }
        return MdmColumnConfigConverter.INSTANCE.entity2RespVo(list);
    }

    @Override
    public List<MdmColumnExportRespVo> columnSelect(MdmColumnConfigReqVo mdmColumnConfigReqVo) {
        String parentCode = mdmColumnConfigReqVo.getParentCode();
        String functionCode = mdmColumnConfigReqVo.getFunctionCode();
        Assert.hasText(parentCode, "菜单编码不能为空");
        Assert.hasText(functionCode, "功能编码不能为空");
        List<MdmColumnExportRespVo> dynamicExportList = mdmFunctionSubButtonService.dynamicExportResolve(mdmColumnConfigReqVo);
        if(!CollectionUtils.isEmpty(dynamicExportList)){
            return dynamicExportList;
        }
        List<MdmColumnConfigEntity> list = this.lambdaQuery()
                .eq(MdmColumnConfigEntity::getParentCode, parentCode)
                .eq(MdmColumnConfigEntity::getFunctionCode, functionCode)
                .eq(!StringUtils.isEmpty(mdmColumnConfigReqVo.getColumnExport()), MdmColumnConfigEntity::getColumnExport, mdmColumnConfigReqVo.getColumnExport())
                .select(MdmColumnConfigEntity::getField, MdmColumnConfigEntity::getTitle, MdmColumnConfigEntity::getFormorder, MdmColumnConfigEntity::getDictCode, MdmColumnConfigEntity::getColumnExport)
                .list();
        if (CollectionUtils.isEmpty(list)) {
            return new ArrayList<>();
        }
        list.sort(Comparator.comparing(x -> Optional.ofNullable(x.getFormorder()).map(Integer::valueOf).orElse(100)));
        return CrmBeanUtil.copyList(list, MdmColumnExportRespVo.class);
    }

    /**
     * 字段解析
     * com.biz.crm.nebular.mdm.pageconfig.MdmColumnConfigRespVo
     *
     * @param classPath 类全路径
     * @return list
     */
    @SneakyThrows
    @Override
    public List<MdmColumnConfigRespVo> columnResolve(String classPath) {
        List<MdmColumnConfigRespVo> list = new ArrayList<>();
        for (Class<?> clazz = Class.forName(classPath); clazz != Object.class; clazz = clazz.getSuperclass()) {
            Field[] declaredFields = clazz.getDeclaredFields();
            for (Field declaredField : declaredFields) {
                if (!"formInstanceId".equals(declaredField.getName()) && !"serialVersionUID".equals(declaredField.getName())) {
                    MdmColumnConfigRespVo mdmColumnConfigRespVo = new MdmColumnConfigRespVo();
                    mdmColumnConfigRespVo.setField(declaredField.getName());
                    mdmColumnConfigRespVo.setVisible(true);
                    ApiModelProperty annotation = declaredField.getAnnotation(ApiModelProperty.class);
                    if (annotation != null) {
                        mdmColumnConfigRespVo.setTitle(annotation.value());
                    }
                    list.add(mdmColumnConfigRespVo);
                }
            }
        }
        return list;
    }

    @Override
    @CacheEvict(key = "#reqVo.parentCode+#reqVo.functionCode")
    public void addBatch(MdmColumnConfigBatchReqVo reqVo) {
        String parentCode = reqVo.getParentCode();
        Assert.hasText(parentCode, "菜单编码不能为空");
        String functionCode = reqVo.getFunctionCode();
        Assert.hasText(functionCode, "功能编码不能为空");
        List<MdmColumnConfigReqVo> mdmColumnConfigVos = reqVo.getMdmColumnConfigVos();
        Assert.notEmpty(mdmColumnConfigVos, "新增列表不能为空");

        Set<String> set = new HashSet<>(mdmColumnConfigVos.size());
        for (MdmColumnConfigReqVo mdmColumnConfigVo : mdmColumnConfigVos) {
            Assert.isTrue(set.add(mdmColumnConfigVo.getField()), "存在相同的字段:" + mdmColumnConfigVo.getField());
        }
        validSearch(mdmColumnConfigVos);

        this.lambdaUpdate()
                .eq(MdmColumnConfigEntity::getParentCode, parentCode)
                .eq(MdmColumnConfigEntity::getFunctionCode, functionCode)
                .remove();
        List<MdmColumnConfigEntity> list = mdmColumnConfigVos.stream()
                .map(x -> {
                    Assert.hasText(x.getField(), "字段名称不能为空");
                    x.setParentCode(parentCode);
                    x.setFunctionCode(functionCode);
                    MdmColumnConfigEntity entity = MdmColumnConfigConverter.INSTANCE.reqVo2Entity(x);
                    entity.setId(null);
                    return entity;
                }).collect(Collectors.toList());
        this.saveBatch(list);
        mdmColumnConfigPersonalService.deleteColumnConfig(reqVo.getParentCode(), reqVo.getFunctionCode());
    }


    /**
     * 控件类型校验
     *
     * @param list list
     */
    protected void validSearch(List<MdmColumnConfigReqVo> list) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        for (MdmColumnConfigReqVo mdmColumnConfigReqVo : list) {
            if (mdmColumnConfigReqVo != null && TrueOrFalseEnum.TRUE.getCode().equalsIgnoreCase(mdmColumnConfigReqVo.getSearch())) {
                Assert.hasText(mdmColumnConfigReqVo.getType(), "字段(" + mdmColumnConfigReqVo.getTitle() + ")允许查询需要配置控件类型");
            }
            if(org.apache.commons.lang3.StringUtils.equals("1",mdmColumnConfigReqVo.getColumnExport())){
                AssertUtils.isNotEmpty(mdmColumnConfigReqVo.getTitle(),"需要导出的字段:"+mdmColumnConfigReqVo.getField()+",标题不能为空");
            }
        }
    }
}
