package com.biz.crm.tableconfig.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.button.model.MdmButtonRoleEntity;
import com.biz.crm.button.service.MdmButtonRoleService;
import com.biz.crm.button.service.MdmButtonService;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.function.service.MdmFunctionService;
import com.biz.crm.kms.config.MapperConfig;
import com.biz.crm.nebular.mdm.button.MdmButtonRespVo;
import com.biz.crm.nebular.mdm.function.MdmFunctionReqVo;
import com.biz.crm.nebular.mdm.function.MdmFunctionRespVo;
import com.biz.crm.nebular.mdm.pageconfig.*;
import com.biz.crm.nebular.mdm.role.req.MdmRoleSubButtonReqVo;
import com.biz.crm.nebular.mdm.role.req.MdmRoleSubButtonVo;
import com.biz.crm.nebular.mdm.role.req.MdmSwitchFunctionButtonReqVo;
import com.biz.crm.tableconfig.mapper.MdmFunctionSubButtonMapper;
import com.biz.crm.tableconfig.mapper.MdmFunctionSubMapper;
import com.biz.crm.tableconfig.model.MdmFunctionSubButtonEntity;
import com.biz.crm.tableconfig.model.MdmFunctionSubEntity;
import com.biz.crm.tableconfig.service.IMdmFunctionSubButtonService;
import com.biz.crm.tableconfig.service.IMdmFunctionSubService;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.NumberUtil;
import com.biz.crm.util.PageUtil;
import com.biz.crm.util.cache.TableConfigUtil;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 功能列表与按钮关联表接口实现
 *
 * @author zxw
 * @date 2020-12-05 12:56:52
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmFunctionSubButtonServiceExpandImpl")
public class MdmFunctionSubButtonServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmFunctionSubButtonMapper, MdmFunctionSubButtonEntity> implements IMdmFunctionSubButtonService {

    @Autowired
    private MdmButtonService mdmButtonService;

    @Autowired
    private IMdmFunctionSubService mdmFunctionSubService;

    @Autowired
    private MdmFunctionSubMapper mdmFunctionSubMapper;

    @Autowired
    private MdmButtonRoleService mdmButtonRoleService;

    @Autowired
    private MdmFunctionService mdmFunctionService;

    @Autowired
    private MdmFunctionSubButtonMapper mdmFunctionSubButtonMapper;

    @Override
    public PageResult<MdmFunctionSubButtonRespVo> findList(MdmFunctionSubButtonReqVo mdmFunctionSubButtonReqVo) {
        Page<MdmFunctionSubButtonRespVo> page = PageUtil.buildPage(mdmFunctionSubButtonReqVo.getPageNum(), mdmFunctionSubButtonReqVo.getPageSize());
        List<MdmFunctionSubButtonRespVo> list = mdmFunctionSubButtonMapper.findList(page, mdmFunctionSubButtonReqVo);
        return PageResult.<MdmFunctionSubButtonRespVo>builder()
                .count(page.getTotal())
                .data(list)
                .build();
    }

    @Override
    public PageResult<JsonNode> dynamicExportSearch(MdmFunctionSubButtonReqVo mdmFunctionSubButtonReqVo) {
        //关联按钮查询
        Page<MdmFunctionSubButtonRespVo> page = PageUtil.buildPage(mdmFunctionSubButtonReqVo.getPageNum(), mdmFunctionSubButtonReqVo.getPageSize());
        List<MdmFunctionSubButtonRespVo> list = mdmFunctionSubButtonMapper.dynamicExportSearch(page, mdmFunctionSubButtonReqVo);
        if (CollectionUtils.isEmpty(list)) {
            return PageResult.empty();
        }
        //菜单树查询
        List<MdmFunctionRespVo> mdmFunctionRespVos = mdmFunctionService.functionTree(new MdmFunctionReqVo());
        if (CollectionUtils.isEmpty(mdmFunctionRespVos)) {
            return PageResult.empty();
        }
        //菜单扁平化打印字符串
        Map<String, String> flattenFunctionMap = mdmFunctionService.flattenTree(mdmFunctionRespVos);
        if (flattenFunctionMap == null || flattenFunctionMap.size() == 0) {
            return PageResult.empty();
        }
        List<JsonNode> result = list.stream()
                .filter(x -> !StringUtils.isEmpty(x.getParentCode()))
                .map(x -> {
                    String treePath = flattenFunctionMap.get(x.getParentCode());
                    if (StringUtils.isEmpty(treePath)) {
                        return null;
                    }
                    String[] functionNameArray = treePath.split("-");
                    ObjectNode jsonNode = MapperConfig.instance();
                    for (int i = 0; i < functionNameArray.length; i++) {
                        jsonNode.put("functionLevel" + i, functionNameArray[i]);
                    }
                    jsonNode.put("parentCode", x.getParentCode());
                    jsonNode.put("functionName", x.getFunctionName());
                    jsonNode.put("functionCode", x.getFunctionCode());
                    jsonNode.put("buttonName", x.getButtonName());
                    jsonNode.put("buttonCode", x.getButtonCode());
                    return jsonNode;
                })
                .filter(Objects::nonNull)
                .sorted(Comparator.comparing(x -> String.valueOf(x.get("functionLevel1"))))
                .collect(Collectors.toList());
        return PageResult.<JsonNode>builder()
                .count(page.getTotal())
                .data(result)
                .build();
    }

    @Override
    public List<MdmColumnExportRespVo> dynamicExportResolve(MdmColumnConfigReqVo mdmColumnConfigReqVo) {
        if (mdmColumnConfigReqVo == null) {
            return new ArrayList<>();
        }
        String parentCode = mdmColumnConfigReqVo.getParentCode();
        String functionCode = mdmColumnConfigReqVo.getFunctionCode();
        if ("CRM20201123000000046".equals(parentCode)) {
            Integer hight = mdmFunctionService.functionDepth();
            if (hight != null && hight > 0) {
                List<MdmColumnExportRespVo> list = new ArrayList<>();
                for (int i = 1; i <= hight; i++) {
                    MdmColumnExportRespVo mdmColumnExportRespVo = new MdmColumnExportRespVo();
                    mdmColumnExportRespVo.setField("functionLevel" + i);
                    mdmColumnExportRespVo.setTitle(NumberUtil.int2chineseNum(i) + "级菜单");
                    mdmColumnExportRespVo.setColumnExport("1");
                    list.add(mdmColumnExportRespVo);
                }
                setUpExportHeader(list);
                return list;
            }
        }
        return new ArrayList<>();
    }

    /**
     * 暂时写死表头
     *
     * @param list list
     */
    protected void setUpExportHeader(List<MdmColumnExportRespVo> list) {
        MdmColumnExportRespVo m1 = new MdmColumnExportRespVo();
        m1.setField("parentCode");
        m1.setTitle("菜单编码");
        m1.setColumnExport("1");
        list.add(m1);
        MdmColumnExportRespVo m3 = new MdmColumnExportRespVo();
        m3.setField("functionName");
        m3.setTitle("列表名称");
        m3.setColumnExport("1");
        list.add(m3);
        MdmColumnExportRespVo m4 = new MdmColumnExportRespVo();
        m4.setField("functionCode");
        m4.setTitle("列表编码");
        m4.setColumnExport("1");
        list.add(m4);
        MdmColumnExportRespVo m5 = new MdmColumnExportRespVo();
        m5.setField("buttonName");
        m5.setTitle("按钮名称");
        m5.setColumnExport("1");
        list.add(m5);
        MdmColumnExportRespVo m6 = new MdmColumnExportRespVo();
        m6.setField("buttonCode");
        m6.setTitle("按钮编码");
        m6.setColumnExport("1");
        list.add(m6);
    }

    @Override
    public void setUp(MdmFunctionSubButtonReqVo mdmFunctionSubButtonReqVo) {
        valid(mdmFunctionSubButtonReqVo);
        mdmFunctionSubButtonReqVo.setId(null);
        this.save(CrmBeanUtil.copy(mdmFunctionSubButtonReqVo, MdmFunctionSubButtonEntity.class));
    }

    @Override
    public List<MdmButtonConfigRespVo> buttonList(MdmFunctionSubButtonReqVo mdmFunctionSubButtonReqVo) {
        return mdmFunctionSubMapper.buttonList(mdmFunctionSubButtonReqVo);
    }

    @Override
    public void buttonRemove(List<String> ids) {
        Assert.notEmpty(ids, "id集合不能为空");
        List<MdmFunctionSubButtonEntity> list = this.lambdaQuery()
                .in(MdmFunctionSubButtonEntity::getId, ids)
                .list();
        if (!CollectionUtils.isEmpty(list)) {
            MdmFunctionSubButtonEntity mdmFunctionSubButtonEntity = list.get(0);
            String parentCode = mdmFunctionSubButtonEntity.getParentCode();
            String functionCode = mdmFunctionSubButtonEntity.getFunctionCode();
            if (!StringUtils.isEmpty(parentCode) && !StringUtils.isEmpty(functionCode)) {
                //清除页面配置缓存
                TableConfigUtil.removeListConfig(parentCode, functionCode);
            }
            this.removeByIds(ids);
        }
    }

    @Override
    public MdmButtonConfigRespVo buttonQuery(String id) {
        return mdmFunctionSubMapper.buttonQuery(id);
    }

    @Override
    public List<MdmRoleSubButtonVo> buttonList(MdmRoleSubButtonReqVo mdmRoleSubButtonReqVo) {
        String parentCode = mdmRoleSubButtonReqVo.getParentCode();
        Assert.hasText(parentCode, "菜单编码不能为空");
        String functionCode = mdmRoleSubButtonReqVo.getFunctionCode();
        Assert.hasText(functionCode, "功能编码不能为空");
        String roleCode = mdmRoleSubButtonReqVo.getRoleCode();
        Assert.hasText(roleCode, "角色编码不能为空");
        List<MdmRoleSubButtonVo> buttonList = mdmFunctionSubMapper.buttonSelectList(mdmRoleSubButtonReqVo);
        if (CollectionUtils.isEmpty(buttonList)) {
            return buttonList;
        }
        List<String> buttonCodeList = mdmButtonRoleService.findButtonCodeByRole(roleCode, parentCode, functionCode);
        if (!CollectionUtils.isEmpty(buttonCodeList)) {
            for (MdmRoleSubButtonVo mdmRoleSubButtonVo : buttonList) {
                String buttonCode = mdmRoleSubButtonVo.getButtonCode();
                if (!StringUtils.isEmpty(buttonCode) && buttonCodeList.contains(buttonCode)) {
                    mdmRoleSubButtonVo.setButtonSelect(YesNoEnum.yesNoEnum.ONE.getValue());
                }
            }
        }
        return buttonList;
    }

    @Override
    public void switchFunctionButton(MdmSwitchFunctionButtonReqVo mdmSwitchFunctionButtonReqVo) {
        String functionCode = mdmSwitchFunctionButtonReqVo.getFunctionCode();
        Assert.hasText(functionCode, "菜单编码不能为空");
        String roleCode = mdmSwitchFunctionButtonReqVo.getRoleCode();
        Assert.hasText(roleCode, "角色编码不能为空");
        String status = mdmSwitchFunctionButtonReqVo.getStatus();
        Assert.hasText(status, "勾选状态不能为空");
        //查询当前及下级菜单编码集合
        List<String> functionCodeList = mdmFunctionService.findCurrentAndSubFunctionCodeList(functionCode);
        if (CollectionUtils.isEmpty(functionCodeList)) {
            return;
        }
        if (YesNoEnum.yesNoEnum.ZERO.getValue().equals(status)) {
            mdmButtonRoleService.lambdaUpdate()
                    .eq(MdmButtonRoleEntity::getRoleCode, roleCode)
                    .in(MdmButtonRoleEntity::getParentCode, functionCodeList)
                    .remove();
            return;
        }
        List<MdmFunctionSubButtonEntity> mdmFunctionSubButtonEntities = this.lambdaQuery()
                .in(MdmFunctionSubButtonEntity::getParentCode, functionCodeList)
                .select(MdmFunctionSubButtonEntity::getParentCode, MdmFunctionSubButtonEntity::getFunctionCode, MdmFunctionSubButtonEntity::getButtonCode)
                .list();
        if (CollectionUtils.isEmpty(mdmFunctionSubButtonEntities)) {
            return;
        }
        List<MdmRoleSubButtonVo> mdmRoleSubButtonVos = CrmBeanUtil.copyList(mdmFunctionSubButtonEntities, MdmRoleSubButtonVo.class);
        for (MdmRoleSubButtonVo mdmRoleSubButtonVo : mdmRoleSubButtonVos) {
            mdmRoleSubButtonVo.setButtonSelect(YesNoEnum.yesNoEnum.ONE.getValue());
        }
        mdmButtonRoleService.setUp(roleCode, mdmRoleSubButtonVos);
    }


    /**
     * 校验
     *
     * @param mdmFunctionSubButtonReqVo
     */
    protected void valid(MdmFunctionSubButtonReqVo mdmFunctionSubButtonReqVo) {
        String parentCode = mdmFunctionSubButtonReqVo.getParentCode();
        Assert.hasText(parentCode, "菜单编码不能为空");
        String functionCode = mdmFunctionSubButtonReqVo.getFunctionCode();
        Assert.hasText(functionCode, "功能编码不能为空");
        String buttonCode = mdmFunctionSubButtonReqVo.getButtonCode();
        Assert.hasText(buttonCode, "按钮编码不能为空");
        MdmButtonRespVo mdmButtonRespVo = mdmButtonService.query(null, buttonCode);
        Assert.notNull(mdmButtonRespVo, "按钮不存在");
        String enableStatus = mdmButtonRespVo.getEnableStatus();
        Assert.isTrue(!StringUtils.isEmpty(enableStatus) && CrmEnableStatusEnum.ENABLE.getCode().equals(enableStatus), "按钮不是启用状态");
        Integer functionCount = mdmFunctionSubService.lambdaQuery()
                .eq(MdmFunctionSubEntity::getParentCode, parentCode)
                .eq(MdmFunctionSubEntity::getFunctionCode, functionCode)
                .count();
        Assert.isTrue(functionCount != null && functionCount > 0, "功能列表不存在");
        this.lambdaUpdate()
                .eq(MdmFunctionSubButtonEntity::getParentCode, mdmFunctionSubButtonReqVo.getParentCode())
                .eq(MdmFunctionSubButtonEntity::getFunctionCode, mdmFunctionSubButtonReqVo.getFunctionCode())
                .eq(MdmFunctionSubButtonEntity::getButtonCode, mdmFunctionSubButtonReqVo.getButtonCode())
                .remove();
        //清除页面配置缓存
        TableConfigUtil.removeListConfig(parentCode, functionCode);
    }
}
