package com.biz.crm.terminal.service.impl;

import com.biz.crm.common.PageResult;
import com.biz.crm.customer.model.MdmCustomerContactEntity;
import com.biz.crm.customer.model.MdmCustomerEntity;
import com.biz.crm.customer.service.MdmCustomerContactService;
import com.biz.crm.customer.service.MdmCustomerMsgService;
import com.biz.crm.customer.service.MdmCustomerROrgService;
import com.biz.crm.customer.service.MdmCustomerSupplyService;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.mdm.MdmCustomerOrTerminalEnum;
import com.biz.crm.eunm.sfa.SfaVisitEnum;
import com.biz.crm.nebular.mdm.customer.MdmCustomerContactReqVo;
import com.biz.crm.nebular.mdm.customer.MdmCustomerDockingReqVo;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgReqVo;
import com.biz.crm.nebular.mdm.poi.resp.MdmAmapDistrictRespVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalContactVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalSupplyVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalVo;
import com.biz.crm.nebular.mdm.terminal.req.MdmAmapDistrictStatisticianReqVo;
import com.biz.crm.nebular.mdm.terminal.req.MdmTerminalCustomerElasticsearchPageReqVo;
import com.biz.crm.nebular.mdm.terminal.resp.MdmTerminalCustomerElasticsearchRespVo;
import com.biz.crm.position.model.MdmCustomerSupplyEntity;
import com.biz.crm.terminal.mapper.MdmTerminalElasticsearchRepository;
import com.biz.crm.terminal.model.MdmTerminalContactEntity;
import com.biz.crm.terminal.model.MdmTerminalCustomerElasticsearchEntity;
import com.biz.crm.terminal.model.MdmTerminalEntity;
import com.biz.crm.terminal.model.MdmTerminalSupplyEntity;
import com.biz.crm.terminal.service.*;
import com.biz.crm.user.mapper.MdmUserMapper;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.es.permission.EsDataPermission;
import com.biz.crm.utils.EsBoolQueryBuilder;
import com.biz.crm.utils.EsBuilder;
import com.biz.crm.utils.EsScriptBuilder;
import com.biz.crm.utils.MdmConstant;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.common.geo.GeoDistance;
import org.elasticsearch.common.unit.DistanceUnit;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.query.RangeQueryBuilder;
import org.elasticsearch.index.reindex.*;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.elasticsearch.annotations.Document;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;
import org.springframework.data.elasticsearch.core.ElasticsearchTemplate;
import org.springframework.data.elasticsearch.core.query.*;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author ql
 * @date 2021/3/10
 * @time 10:13
 **/
@Service
@Slf4j
public class MdmTerminalElasticsearchServiceImpl implements MdmTerminalElasticsearchService {

    @Autowired
    private ElasticsearchRestTemplate elasticsearchRestTemplate;
    @Autowired
    private MdmTerminalElasticsearchRepository elasticsearchRepository;
    @Autowired
    private MdmTerminalService mdmTerminalService;
    @Autowired
    private MdmCustomerMsgService mdmCustomerMsgService;
    @Autowired
    private MdmCustomerContactService mdmCustomerContactService;
    @Autowired
    private MdmCustomerSupplyService mdmCustomerSupplyService;
    @Autowired
    private MdmCustomerROrgService mdmCustomerROrgService;
    @Autowired
    private MdmTerminalContactService mdmTerminalContactService;
    @Autowired
    private MdmTerminalROrgService mdmTerminalROrgService;
    @Autowired
    private MdmTerminalSupplyService mdmTerminalSupplyService;
    @Autowired
    private MdmUserMapper mdmUserMapper;


    /**
     * ec分页查询
     * 根据职位编码查询，并且根据经纬度计算出距离按照距离排序分页查询
     * 没有传递参数时，会按照默认的全局参数进行查询，如有需要，按照自己的分页参数查询
     * 如果需要某个条件下的所有数据，在某个条件下，将每页显示的记录条数变大，变得一页能装下即可，调用者注意数据量哦
     *
     * @param pageReqVo 分页及查询参数
     * @return
     */
    @Override
    @EsDataPermission(position = "positionCodeList.keyword", org = "orgCodeList.keyword")
    public PageResult<MdmTerminalCustomerElasticsearchRespVo> listWithPage(MdmTerminalCustomerElasticsearchPageReqVo pageReqVo) {
        // 经纬度解析为double
        Double latitude = null;
        Double longitude = null;
        try {
            if (pageReqVo.getLatitude() != null) {
                latitude = Double.parseDouble(pageReqVo.getLatitude());
                Assert.isTrue(latitude >= 0 && latitude <= 90, "不合法的纬度:" + latitude);
            }
            if (pageReqVo.getLongitude() != null) {
                longitude = Double.parseDouble(pageReqVo.getLongitude());
            }
        } catch (NumberFormatException e) {
            log.error("客户终端查询:" + e);
        }

        double distance = pageReqVo.getDistance() == null ? 5000 : pageReqVo.getDistance();


        Double finalLatitude = latitude;
        Double finalLongitude = longitude;
        BoolQueryBuilder boolQuery = EsBoolQueryBuilder.lambdaQuery()
                .filter(QueryBuilders.termQuery("enableStatus", CrmEnableStatusEnum.ENABLE.getCode()))
                .must(!StringUtils.isEmpty(pageReqVo.getName()), () -> QueryBuilders.matchPhraseQuery("name", pageReqVo.getName()))
                .must(!StringUtils.isEmpty(pageReqVo.getContactName()), () -> QueryBuilders.matchPhraseQuery("contactName", pageReqVo.getContactName()))
                .filter(!StringUtils.isEmpty(pageReqVo.getClientSubclass()), () -> QueryBuilders.termQuery("clientSubClass.keyword", pageReqVo.getClientSubclass()))
                .filter(!StringUtils.isEmpty(pageReqVo.getClientType()), () -> QueryBuilders.termQuery("clientType.keyword", pageReqVo.getClientType()))
                .filter(!StringUtils.isEmpty(pageReqVo.getChannel()), () -> QueryBuilders.termQuery("channel.keyword", pageReqVo.getChannel()))
                .filter(latitude != null && longitude != null, () -> QueryBuilders.geoDistanceQuery("placePoint")
                        .point(finalLatitude, finalLongitude)
                        .distance(distance, DistanceUnit.METERS))
                .build();
        EsBuilder esBuilder = EsBuilder.lambdaQuery()
                .withQuery(boolQuery)
                .withPageable(pageReqVo.getPageNum() - 1, pageReqVo.getPageSize());
        if(latitude != null && longitude != null){
            esBuilder = esBuilder.withSort(latitude != null && longitude != null, () -> SortBuilders.geoDistanceSort("placePoint", finalLatitude, finalLongitude)
                    .unit(DistanceUnit.METERS)
                    .order(SortOrder.ASC));
        }

        PageResult<MdmTerminalCustomerElasticsearchEntity> pageResult = esBuilder.queryForPage(elasticsearchRestTemplate, MdmTerminalCustomerElasticsearchEntity.class);
        Map<String, String> customerTypeMap = DictUtil.dictMap("customer_type");
        Map<String, String> terminalTypeMap = DictUtil.dictMap("terminal_type");


        List<MdmTerminalCustomerElasticsearchRespVo> data = pageResult.getData().stream()
                .map(item -> {
                    MdmTerminalCustomerElasticsearchRespVo respVo = CrmBeanUtil.copy(item, MdmTerminalCustomerElasticsearchRespVo.class);
                    if (finalLatitude != null && finalLongitude != null) {
                        // 这里调用根据经纬度计算距离的方法，2021.3.31 只保留整数。四舍五入，加0.5向下取整
                        respVo.setDistance((int) (GeoDistance.PLANE.calculate(Double.parseDouble(item.getLatitude()),
                                Double.parseDouble(item.getLongitude()), finalLatitude, finalLongitude, DistanceUnit.METERS) + 0.5));
                    }
                    //类型翻译中文
                    if (!StringUtils.isEmpty(respVo.getClientSubclass())) {
                        if (SfaVisitEnum.ClientType.TERMINAL.getVal().equals(respVo.getClientType())) {
                            respVo.setClientSubclassName(terminalTypeMap.get(respVo.getClientSubclass()));
                        } else {
                            respVo.setClientSubclassName(customerTypeMap.get(respVo.getClientSubclass()));
                        }
                    }
                    respVo.setClientTypeName(SfaVisitEnum.ClientType.GETMAP.get(respVo.getClientType()));
                    return respVo;
                })
                .collect(Collectors.toList());
        return PageResult.<MdmTerminalCustomerElasticsearchRespVo>builder()
                .count(pageResult.getCount())
                .data(data)
                .build();
    }

    /**
     * 客户类型翻译
     */
    public void translationType(List<MdmTerminalCustomerElasticsearchEntity> list) {
        if (CollectionUtil.listNotEmpty(list)) {
            list.forEach(o -> {
                if (o.getClientType().equals(SfaVisitEnum.ClientType.DEALER.getVal())) {
                    //客户
                    o.setClientTypeName(SfaVisitEnum.ClientType.DEALER.getDesc());
                    if (StringUtils.isNotEmpty(o.getClientSubclass())) {
                        o.setClientSubclassName(MdmCustomerOrTerminalEnum.customerType.GETMAP.get(o.getClientSubclass()));
                    }
                } else if (o.getClientType().equals(SfaVisitEnum.ClientType.TERMINAL.getVal())) {
                    //终端
                    o.setClientTypeName(SfaVisitEnum.ClientType.TERMINAL.getDesc());
                    if (StringUtils.isNotEmpty(o.getClientSubclass())) {
                        o.setClientSubclassName(MdmCustomerOrTerminalEnum.terminalType.GETMAP.get(o.getClientSubclass()));
                    }
                }
            });
        }
    }

    /**
     * 终端和客户 新增、修改时，想es存入记录，注意修改时ID需要和原来的相同
     *
     * @param mdmTerminalVo       终端
     * @param mdmCustomerMsgReqVo 客户
     */
    @Override
    public void add(MdmTerminalVo mdmTerminalVo, MdmCustomerMsgReqVo mdmCustomerMsgReqVo) {
        // 如果es中index不存在，就创建一个
        if (!elasticsearchRestTemplate.indexExists(MdmConstant.MDM_TERMINAL_CUSTOMER_INDEX)) {
            // 会根据MdmTerminalCustomerElasticsearchEntity类的@Document注解信息来创建
            elasticsearchRestTemplate.createIndex(MdmTerminalCustomerElasticsearchEntity.class);
        }
        // 终端经纬度、供货关系不为null时才发送到es
        if (mdmTerminalVo != null
                && !StringUtils.isEmpty(mdmTerminalVo.getLongitude())
                && !StringUtils.isEmpty(mdmTerminalVo.getLatitude())
                && !CollectionUtils.isEmpty(mdmTerminalVo.getSupplys())) {
            /// 相同的直接赋值
            MdmTerminalCustomerElasticsearchEntity elasticsearchEntity = CrmBeanUtil.copy(mdmTerminalVo, MdmTerminalCustomerElasticsearchEntity.class);
            // 手动设置其他属性
            elasticsearchEntity.setName(mdmTerminalVo.getTerminalName());
            elasticsearchEntity.setCode(mdmTerminalVo.getTerminalCode());
            // 设置Location 的经度和经纬度
            elasticsearchEntity.setPlacePoint(mdmTerminalVo.getLatitude() + "," + mdmTerminalVo.getLongitude());
            // 设置主联系人，如果有主联系人
            List<MdmTerminalContactVo> contacts = mdmTerminalVo.getContacts();
            if (!CollectionUtils.isEmpty(contacts)) {
                List<MdmTerminalContactVo> mainContact = mdmTerminalVo.getContacts()
                        .stream()
                        .filter(x -> YesNoEnum.yesNoEnum.ONE.getValue().equals(x.getContactMain()))
                        .collect(Collectors.toList());
                if (!CollectionUtils.isEmpty(mainContact)) {
                    elasticsearchEntity.setContactName(mainContact
                            .get(YesNoEnum.YesNoCodeNumberEnum.NO.getCode())
                            .getContactName());
                }
            }
            List<MdmTerminalSupplyVo> supplys = mdmTerminalVo.getSupplys();
            supplys = supplys == null ? new ArrayList<>() : supplys;
            // 设置供货关系职位编码，多个按照;分割
            elasticsearchEntity.setPositionCodes(supplys
                    .stream()
                    .map(MdmTerminalSupplyVo::getPositionCode)
                    .collect(Collectors.joining(";")));
            // 设置联系方式， 多个按照;分割
            if (!CollectionUtils.isEmpty(mdmTerminalVo.getContacts())) {
                elasticsearchEntity.setContactPhone(mdmTerminalVo.getContacts()
                        .stream()
                        .map(MdmTerminalContactVo::getContactPhone)
                        .collect(Collectors.joining(";")));
            }
            // 设置地址
            elasticsearchEntity.setAddress(mdmTerminalVo.getTerminalAddress());
            elasticsearchEntity.setClientType(SfaVisitEnum.ClientType.TERMINAL.getVal());
            elasticsearchEntity.setClientSubclass(mdmTerminalVo.getTerminalType());
            /// 保存
            elasticsearchEntity.setClientType(SfaVisitEnum.ClientType.TERMINAL.getVal());
            elasticsearchEntity.setClientSubclass(mdmTerminalVo.getTerminalType());
            //保存组织
            if (!StringUtils.isEmpty(mdmTerminalVo.getOrgCode())) {
                elasticsearchEntity.setOrgCodeList(Arrays.asList(mdmTerminalVo.getOrgCode().split(",")));
            }
            //保存对接人职位
            List<String> positionCodeList = supplys.stream()
                    .map(MdmTerminalSupplyVo::getPositionCode)
                    .filter(x -> !StringUtils.isEmpty(x))
                    .distinct()
                    .collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(positionCodeList)) {
                elasticsearchEntity.setPositionCodeList(positionCodeList);
            }
            //保存对接人客户
            List<String> customerCodeList = supplys.stream()
                    .map(MdmTerminalSupplyVo::getCustomerCode)
                    .filter(x -> !StringUtils.isEmpty(x))
                    .distinct()
                    .collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(customerCodeList)) {
                elasticsearchEntity.setCustomerCodeList(customerCodeList);
            }
            elasticsearchRepository.save(elasticsearchEntity);
            return;
        }
        // 客户经纬度、对接人不为null时才发送到es
        if (mdmCustomerMsgReqVo != null
                && !StringUtils.isEmpty(mdmCustomerMsgReqVo.getLongitude())
                && !StringUtils.isEmpty(mdmCustomerMsgReqVo.getLatitude())
                && !CollectionUtils.isEmpty(mdmCustomerMsgReqVo.getMdmCustomerDockingVos())) {
            /// 相同的属性直接赋值
            MdmTerminalCustomerElasticsearchEntity elasticsearchEntity = CrmBeanUtil.copy(mdmCustomerMsgReqVo, MdmTerminalCustomerElasticsearchEntity.class);
            // 手动设置其他属性
            elasticsearchEntity.setName(mdmCustomerMsgReqVo.getUserName());
            elasticsearchEntity.setCode(mdmCustomerMsgReqVo.getCustomerCode());
            // 设置Location 的经度和经纬度
            elasticsearchEntity.setPlacePoint(mdmCustomerMsgReqVo.getLatitude() + "," + mdmCustomerMsgReqVo.getLongitude());
            // 设置主联系人，如果有主联系人
            if (!CollectionUtils.isEmpty(mdmCustomerMsgReqVo.getMdmCustomerContactVos())) {
                List<MdmCustomerContactReqVo> mainContact = mdmCustomerMsgReqVo.getMdmCustomerContactVos()
                        .stream()
                        .filter(x -> YesNoEnum.yesNoEnum.ONE.getValue().equals(x.getContactMain()))
                        .collect(Collectors.toList());
                if (!CollectionUtils.isEmpty(mainContact)) {
                    elasticsearchEntity.setContactName(mainContact
                            .get(YesNoEnum.YesNoCodeNumberEnum.NO.getCode())
                            .getContactName());
                }
            }
            // 设置对接人职位编码，多个按照;分割
            if (mdmCustomerMsgReqVo.getMdmCustomerDockingVos() != null) {
                elasticsearchEntity.setPositionCodes(mdmCustomerMsgReqVo.getMdmCustomerDockingVos()
                        .stream()
                        .map(MdmCustomerDockingReqVo::getPositionCode)
                        .collect(Collectors.joining(";")));
                List<String> positionCodeList = mdmCustomerMsgReqVo.getMdmCustomerDockingVos().stream()
                        .map(MdmCustomerDockingReqVo::getPositionCode)
                        .filter(x -> !StringUtils.isEmpty(x))
                        .distinct()
                        .collect(Collectors.toList());
                if (!CollectionUtils.isEmpty(positionCodeList)) {
                    elasticsearchEntity.setPositionCodeList(positionCodeList);
                }
            }
            // 设置联系方式
            elasticsearchEntity.setContactPhone(mdmCustomerMsgReqVo.getCustomerContact());
            // 设置地址
            elasticsearchEntity.setAddress(mdmCustomerMsgReqVo.getRegisteredAddress());
            /// 保存
            elasticsearchEntity.setClientType(SfaVisitEnum.ClientType.DEALER.getVal());
            elasticsearchEntity.setClientSubclass(mdmCustomerMsgReqVo.getCustomerType());

            //保存组织
            if (!StringUtils.isEmpty(mdmCustomerMsgReqVo.getOrgCode())) {
                elasticsearchEntity.setOrgCodeList(Arrays.asList(mdmCustomerMsgReqVo.getOrgCode().split(",")));
            }
            elasticsearchRepository.save(elasticsearchEntity);
        }
        // 其他情况都不保存

    }

    @Override
    public void add(MdmCustomerMsgReqVo mdmCustomerMsgReqVo) {
        Assert.notNull(mdmCustomerMsgReqVo, "客户信息不能为空");
        Assert.hasText(mdmCustomerMsgReqVo.getId(), "客户id不能为空");
        MdmTerminalCustomerElasticsearchEntity entity = CrmBeanUtil.copy(mdmCustomerMsgReqVo, MdmTerminalCustomerElasticsearchEntity.class);
        entity.setName(mdmCustomerMsgReqVo.getCustomerName());
        entity.setCode(mdmCustomerMsgReqVo.getCustomerCode());
        // 设置Location 的经度和经纬度
        String latitude = mdmCustomerMsgReqVo.getLatitude();
        String longitude = mdmCustomerMsgReqVo.getLongitude();
        if (!StringUtils.isEmpty(latitude) && !StringUtils.isEmpty(longitude)) {
            try {
                Double.valueOf(latitude);
                Double.valueOf(longitude);
                entity.setPlacePoint(latitude + "," + longitude);
            } catch (NumberFormatException e) {
                log.error("经纬度格式错误" + latitude + ":" + longitude);
            }
        }
        // 设置地址
        entity.setAddress(mdmCustomerMsgReqVo.getRegisteredAddress());
        /// 保存
        entity.setClientType(SfaVisitEnum.ClientType.DEALER.getVal());
        entity.setClientSubclass(mdmCustomerMsgReqVo.getCustomerType());
        //设置组织
        if (!StringUtils.isEmpty(mdmCustomerMsgReqVo.getOrgCode())) {
            entity.setOrgCodeList(Arrays.asList(mdmCustomerMsgReqVo.getOrgCode().split(",")));
        }
        // 设置对接人职位编码
        if (mdmCustomerMsgReqVo.getMdmCustomerDockingVos() != null) {
            List<String> positionCodeList = mdmCustomerMsgReqVo.getMdmCustomerDockingVos().stream()
                    .map(MdmCustomerDockingReqVo::getPositionCode)
                    .filter(x -> !StringUtils.isEmpty(x))
                    .distinct()
                    .collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(positionCodeList)) {
                entity.setPositionCodeList(positionCodeList);
                //设置对接人名称
                List<String> nameList = mdmUserMapper.findSupplyNameList(positionCodeList);
                if (!CollectionUtils.isEmpty(nameList)) {
                    entity.setSupplyNameList(nameList);
                }
            }
        }
        //设置主联系人
        List<MdmCustomerContactReqVo> mdmCustomerContactVos = mdmCustomerMsgReqVo.getMdmCustomerContactVos();
        if (!CollectionUtils.isEmpty(mdmCustomerContactVos)) {
            mdmCustomerContactVos.stream()
                    .filter(x -> String.valueOf(YesNoEnum.YesNoCodeNumberEnum.YES.getCode()).equals(x.getContactMain()))
                    .findAny()
                    .ifPresent(x -> {
                        entity.setContactName(x.getContactName());
                        entity.setContactPhone(x.getContactPhone());
                    });
        }
        entity.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        elasticsearchRepository.save(entity);
    }

    @Override
    public void add(MdmTerminalVo mdmTerminalVo) {
        Assert.notNull(mdmTerminalVo, "终端信息不能为空");
        Assert.hasText(mdmTerminalVo.getId(), "终端id不能为空");
        MdmTerminalCustomerElasticsearchEntity entity = CrmBeanUtil.copy(mdmTerminalVo, MdmTerminalCustomerElasticsearchEntity.class);
        // 手动设置其他属性
        entity.setName(mdmTerminalVo.getTerminalName());
        entity.setCode(mdmTerminalVo.getTerminalCode());
        // 设置Location 的经度和经纬度
        String latitude = mdmTerminalVo.getLatitude();
        String longitude = mdmTerminalVo.getLongitude();
        if (!StringUtils.isEmpty(latitude) && !StringUtils.isEmpty(longitude)) {
            try {
                Double.valueOf(latitude);
                Double.valueOf(longitude);
                entity.setPlacePoint(latitude + "," + longitude);
            } catch (NumberFormatException e) {
                log.error("经纬度格式错误" + latitude + ":" + longitude);
            }
        }
        // 设置主联系人，如果有主联系人
        List<MdmTerminalContactVo> contacts = mdmTerminalVo.getContacts();
        if (!CollectionUtils.isEmpty(contacts)) {
            contacts.stream()
                    .filter(x -> String.valueOf(YesNoEnum.YesNoCodeNumberEnum.YES.getCode()).equals(x.getContactMain()))
                    .findAny()
                    .ifPresent(x -> {
                        entity.setContactPhone(x.getContactPhone());
                        entity.setContactName(x.getContactName());
                    });
        }
        List<MdmTerminalSupplyVo> supplys = mdmTerminalVo.getSupplys();
        if (!CollectionUtils.isEmpty(supplys)) {
            //保存对接人职位
            List<String> positionCodeList = supplys.stream()
                    .map(MdmTerminalSupplyVo::getPositionCode)
                    .filter(x -> !StringUtils.isEmpty(x))
                    .distinct()
                    .collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(positionCodeList)) {
                entity.setPositionCodeList(positionCodeList);
            }
            //保存对接人客户
            List<String> customerCodeList = supplys.stream()
                    .map(MdmTerminalSupplyVo::getCustomerCode)
                    .filter(x -> !StringUtils.isEmpty(x))
                    .distinct()
                    .collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(customerCodeList)) {
                entity.setCustomerCodeList(customerCodeList);
            }
        }
        // 设置地址
        entity.setAddress(mdmTerminalVo.getTerminalAddress());
        entity.setClientType(SfaVisitEnum.ClientType.TERMINAL.getVal());
        entity.setClientSubclass(mdmTerminalVo.getTerminalType());
        //保存组织
        if (!StringUtils.isEmpty(mdmTerminalVo.getOrgCode())) {
            entity.setOrgCodeList(Arrays.asList(mdmTerminalVo.getOrgCode().split(",")));
        }
        entity.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        elasticsearchRepository.save(entity);
        elasticsearchRestTemplate.refresh(MdmConstant.MDM_TERMINAL_CUSTOMER_INDEX);
    }

    @Override
    public void delete(List<String> ids) throws Exception{
        Assert.notEmpty(ids, "id不能为空");
        //TODO 此处需要重写
//        DeleteByQueryAction.INSTANCE.newRequestBuilder(elasticsearchTemplate.getClient())
//                .source(MdmConstant.MDM_TERMINAL_CUSTOMER_INDEX)
//                .filter(QueryBuilders.termsQuery("id.keyword", ids))
//                .get();

//        UpdateByQueryRequest updateByQueryRequest = new UpdateByQueryRequest();
//        updateByQueryRequest.indices("exam_information");//设置索引
//        updateByQueryRequest.setDocTypes("doc");//设置文档，固定doc
//        updateByQueryRequest.setQuery(QueryBuilders.termsQuery("paperBaseId", paperBaseId));//设置查询
//        updateByQueryRequest.setScript(script);//如果有脚本，则添加
//        updateByQueryRequest.setConflicts("proceed"); // 设置版本冲突时继续
//        updateByQueryRequest.setRefresh(true);//请求结束后，对我们写入的索引进行调用刷新
//        this.elasticsearchTemplate.getClient().updateByQuery(updateByQueryRequest, RequestOptions.DEFAULT);//进行更新

        DeleteByQueryRequest deleteByQueryRequest = new DeleteByQueryRequest(MdmConstant.MDM_TERMINAL_CUSTOMER_INDEX);
        deleteByQueryRequest.setQuery(QueryBuilders.termsQuery("id.keyword", ids));
        deleteByQueryRequest.setConflicts("proceed"); // 设置版本冲突时继续
        deleteByQueryRequest.setRefresh(true);//请求结束后，对我们写入的索引进行调用刷新
        elasticsearchRestTemplate.getClient().deleteByQuery(deleteByQueryRequest,RequestOptions.DEFAULT);

    }

    @Override
    public void reset() {
        if (elasticsearchRestTemplate.indexExists(MdmConstant.MDM_TERMINAL_CUSTOMER_INDEX)) {
            elasticsearchRepository.deleteAll();
        }
        resetCustomer();
        resetTerminal();
    }

    @Override
    public List<MdmAmapDistrictRespVo> listCondition(MdmAmapDistrictStatisticianReqVo mdmAmapDistrictStatisticianReqVo) {
        List<String> statisticianTypeList = mdmAmapDistrictStatisticianReqVo.getStatisticianTypeList();
        String amapDistrictCode = mdmAmapDistrictStatisticianReqVo.getAmapDistrictCode();
        String customerOrTerminalName = mdmAmapDistrictStatisticianReqVo.getCustomerOrTerminalName();
        String supplyFullName = mdmAmapDistrictStatisticianReqVo.getSupplyFullName();
        NativeSearchQueryBuilder queryBuilder = new NativeSearchQueryBuilder();
        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        if (!CollectionUtils.isEmpty(statisticianTypeList)) {
            boolQueryBuilder.must(QueryBuilders.termsQuery("clientSubclass.keyword", statisticianTypeList));
        }
        if (!StringUtils.isEmpty(amapDistrictCode)) {
            boolQueryBuilder.must(QueryBuilders.termQuery("districtCode.keyword", amapDistrictCode));
        }
        if (!StringUtils.isEmpty(customerOrTerminalName)) {
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("name", customerOrTerminalName));
        }
        if (!StringUtils.isEmpty(supplyFullName)) {
            boolQueryBuilder.must(QueryBuilders.matchPhraseQuery("supplyNameList", supplyFullName));
        }

        Integer pageNum = Optional.ofNullable(mdmAmapDistrictStatisticianReqVo.getPageNum()).orElse(0);
        Integer pageSize = Optional.ofNullable(mdmAmapDistrictStatisticianReqVo.getPageSize()).orElse(1000);
        NativeSearchQuery query = queryBuilder
                .withFilter(boolQueryBuilder)
                .withPageable(PageRequest.of(pageNum, pageSize))
                .build();
        List<MdmTerminalCustomerElasticsearchEntity> entityList = elasticsearchRestTemplate
                .queryForList(query, MdmTerminalCustomerElasticsearchEntity.class);
        if (CollectionUtils.isEmpty(entityList)) {
            return Collections.emptyList();
        }
        return entityList.stream()
                .map(x -> {
                    MdmAmapDistrictRespVo respVo = new MdmAmapDistrictRespVo();
                    respVo.setAmapCode(amapDistrictCode);
                    respVo.setStatisticianCode(x.getCode());
                    respVo.setStatisticianName(x.getName());
                    respVo.setStatisticianAddress(x.getAddress());
                    respVo.setStatisticianType(x.getClientSubclass());
                    respVo.setStatisticianContactName(x.getContactName());
                    respVo.setStatisticianContactPhone(x.getContactPhone());
                    String placePoint = x.getPlacePoint();
                    if (!StringUtils.isEmpty(placePoint)) {
                        String[] split = placePoint.split(",");
                        if (split.length > 1) {
                            respVo.setLatitude(split[0]);
                            respVo.setLongitude(split[1]);
                        }
                    }
                    return respVo;
                })
                .collect(Collectors.toList());
    }

    @Override
    public void enable(List<String> ids) throws Exception{
        Assert.notEmpty(ids, "id集合不能为空");
        //TODO 此处需要重写
//        BulkByScrollResponse response = UpdateByQueryAction.INSTANCE.newRequestBuilder(elasticsearchRestTemplate.getClient())
//                .source(MdmConstant.MDM_TERMINAL_CUSTOMER_INDEX)
//                .filter(QueryBuilders.termsQuery("id", ids))
//                .script(EsScriptBuilder.updateScript(Collections.singletonMap("enableStatus", CrmEnableStatusEnum.ENABLE.getCode())))
//                .get();


        //        UpdateByQueryRequest updateByQueryRequest = new UpdateByQueryRequest();
//        updateByQueryRequest.indices("exam_information");//设置索引
//        updateByQueryRequest.setDocTypes("doc");//设置文档，固定doc
//        updateByQueryRequest.setQuery(QueryBuilders.termsQuery("paperBaseId", paperBaseId));//设置查询
//        updateByQueryRequest.setScript(script);//如果有脚本，则添加
//        updateByQueryRequest.setConflicts("proceed"); // 设置版本冲突时继续
//        updateByQueryRequest.setRefresh(true);//请求结束后，对我们写入的索引进行调用刷新
//        this.elasticsearchTemplate.getClient().updateByQuery(updateByQueryRequest, RequestOptions.DEFAULT);//进行更新

        UpdateByQueryRequest updateByQueryRequest = new UpdateByQueryRequest(MdmConstant.MDM_TERMINAL_CUSTOMER_INDEX);
        updateByQueryRequest.setQuery(QueryBuilders.termsQuery("id", ids));
        updateByQueryRequest.setScript(EsScriptBuilder.updateScript(Collections.singletonMap("enableStatus", CrmEnableStatusEnum.ENABLE.getCode())));
        updateByQueryRequest.setConflicts("proceed"); // 设置版本冲突时继续
        updateByQueryRequest.setRefresh(true);//请求结束后，对我们写入的索引进行调用刷新
        elasticsearchRestTemplate.getClient().updateByQuery(updateByQueryRequest,RequestOptions.DEFAULT);
    }

    @Override
    public void disable(List<String> ids)  throws Exception{
        Assert.notEmpty(ids, "id集合不能为空");
        //TODO 此处需要重写
//        BulkByScrollResponse response = UpdateByQueryAction.INSTANCE.newRequestBuilder(elasticsearchRestTemplate.getClient())
//                .source(MdmConstant.MDM_TERMINAL_CUSTOMER_INDEX)
//                .filter(QueryBuilders.termsQuery("id", ids))
//                .script(EsScriptBuilder.updateScript(Collections.singletonMap("enableStatus", CrmEnableStatusEnum.DISABLE.getCode())))
//                .get();

        UpdateByQueryRequest updateByQueryRequest = new UpdateByQueryRequest(MdmConstant.MDM_TERMINAL_CUSTOMER_INDEX);
        updateByQueryRequest.setQuery(QueryBuilders.termsQuery("id", ids));
        updateByQueryRequest.setScript(EsScriptBuilder.updateScript(Collections.singletonMap("enableStatus", CrmEnableStatusEnum.DISABLE.getCode())));
        updateByQueryRequest.setConflicts("proceed"); // 设置版本冲突时继续
        updateByQueryRequest.setRefresh(true);//请求结束后，对我们写入的索引进行调用刷新
        elasticsearchRestTemplate.getClient().updateByQuery(updateByQueryRequest,RequestOptions.DEFAULT);
    }

    @Override
    public void updateByCodeList(List<String> codeList, Map<String, Object> map) throws Exception{
        if (!CollectionUtils.isEmpty(codeList) && !ObjectUtils.isEmpty(map)) {
            //TODO 此处需要重写
//            UpdateByQueryAction.INSTANCE.newRequestBuilder(elasticsearchRestTemplate.getClient())
//                    .source(MdmConstant.MDM_TERMINAL_CUSTOMER_INDEX)
//                    .filter(QueryBuilders.termsQuery("code.keyword", codeList))
//                    .script(EsScriptBuilder.updateScript(map))
//                    .get();

            UpdateByQueryRequest updateByQueryRequest = new UpdateByQueryRequest(MdmConstant.MDM_TERMINAL_CUSTOMER_INDEX);
            updateByQueryRequest.setQuery(QueryBuilders.termsQuery("code.keyword", codeList));
            updateByQueryRequest.setScript(EsScriptBuilder.updateScript(map));
            updateByQueryRequest.setConflicts("proceed"); // 设置版本冲突时继续
            updateByQueryRequest.setRefresh(true);//请求结束后，对我们写入的索引进行调用刷新
            elasticsearchRestTemplate.getClient().updateByQuery(updateByQueryRequest,RequestOptions.DEFAULT);
        }
    }

    @Override
    public void updateById(String id, Map<String, Object> params) {
        Assert.hasText(id, "id不能为空");
        if (params == null || params.size() < 1) {
            return;
        }
        UpdateRequest updateRequest = new UpdateRequest();
        updateRequest.doc(params);
        UpdateQuery build = new UpdateQueryBuilder()
                .withUpdateRequest(updateRequest)
                .withClass(MdmTerminalCustomerElasticsearchEntity.class)
                .withId(id)
                .build();
        elasticsearchRestTemplate.update(build);
    }


    private void resetTerminal() {
        List<MdmTerminalEntity> list = mdmTerminalService.lambdaQuery().list();
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        List<String> terminalCodeList = list.stream().map(MdmTerminalEntity::getTerminalCode).collect(Collectors.toList());
        Map<String, List<String>> orgCodeMap = mdmTerminalROrgService.findOrgCodeList(terminalCodeList);
        Map<String, MdmTerminalContactEntity> contactEntityMap = mdmTerminalContactService.lambdaQuery()
                .in(MdmTerminalContactEntity::getTerminalCode, terminalCodeList)
                .eq(MdmTerminalContactEntity::getContactMain, YesNoEnum.yesNoEnum.ONE.getValue())
                .select(MdmTerminalContactEntity::getTerminalCode, MdmTerminalContactEntity::getContactName, MdmTerminalContactEntity::getContactPhone)
                .list().stream()
                .collect(Collectors.toMap(MdmTerminalContactEntity::getTerminalCode, Function.identity()));
        Map<String, List<MdmTerminalSupplyEntity>> supplyMap = mdmTerminalSupplyService.lambdaQuery()
                .in(MdmTerminalSupplyEntity::getTerminalCode, terminalCodeList)
                .select(MdmTerminalSupplyEntity::getTerminalCode, MdmTerminalSupplyEntity::getCustomerCode, MdmTerminalSupplyEntity::getPositionCode)
                .list().stream()
                .collect(Collectors.groupingBy(MdmTerminalSupplyEntity::getTerminalCode));
        list.parallelStream().forEach(x -> {
            MdmTerminalVo terminalVo = CrmBeanUtil.copy(x, MdmTerminalVo.class);
            //设置组织
            List<String> orgCodeList = orgCodeMap.get(x.getTerminalCode());
            if (!CollectionUtils.isEmpty(orgCodeList)) {
                terminalVo.setOrgCode(String.join(",", orgCodeList));
            }
            //设置联系人
            MdmTerminalContactEntity contactEntity = contactEntityMap.get(terminalVo.getTerminalCode());
            if (contactEntity != null) {
                MdmTerminalContactVo contactVo = new MdmTerminalContactVo();
                contactVo.setContactMain(YesNoEnum.yesNoEnum.ONE.getValue());
                contactVo.setContactName(contactEntity.getContactName());
                contactVo.setContactPhone(contactEntity.getContactPhone());
                contactVo.setTerminalCode(terminalVo.getTerminalCode());
                terminalVo.setContacts(Collections.singletonList(contactVo));
            }
            //设置对接人
            List<MdmTerminalSupplyEntity> supplyEntities = supplyMap.get(terminalVo.getTerminalCode());
            if (supplyEntities != null) {
                List<MdmTerminalSupplyVo> supplyVos = CrmBeanUtil.copyList(supplyEntities, MdmTerminalSupplyVo.class);
                terminalVo.setSupplys(supplyVos);
            }
            add(terminalVo);
        });
    }

    private void resetCustomer() {
        List<MdmCustomerEntity> customerEntities = mdmCustomerMsgService.lambdaQuery().list();
        if (!CollectionUtils.isEmpty(customerEntities)) {
            Lists.partition(customerEntities, 500).parallelStream()
                    .forEach(x -> {
                        List<MdmCustomerMsgReqVo> mdmCustomerMsgReqVos = CrmBeanUtil.copyList(x, MdmCustomerMsgReqVo.class);
                        List<String> customerCodeList = x.stream()
                                .map(MdmCustomerEntity::getCustomerCode)
                                .collect(Collectors.toList());
                        //设置主联系人
                        Map<String, MdmCustomerContactEntity> contactEntityMap = mdmCustomerContactService.lambdaQuery()
                                .in(MdmCustomerContactEntity::getCustomerCode, customerCodeList)
                                .eq(MdmCustomerContactEntity::getContactMain, YesNoEnum.yesNoEnum.ONE.getValue())
                                .select(MdmCustomerContactEntity::getCustomerCode, MdmCustomerContactEntity::getContactName, MdmCustomerContactEntity::getContactPhone)
                                .list()
                                .stream()
                                .collect(Collectors.toMap(MdmCustomerContactEntity::getCustomerCode, Function.identity(), (x1, x2) -> x1));
                        //设置对接人职位编码
                        Map<String, List<String>> supplyMap = mdmCustomerSupplyService.lambdaQuery()
                                .in(MdmCustomerSupplyEntity::getCustomerCode, customerCodeList)
                                .select(MdmCustomerSupplyEntity::getPositionCode, MdmCustomerSupplyEntity::getCustomerCode)
                                .list()
                                .stream()
                                .collect(Collectors.groupingBy(MdmCustomerSupplyEntity::getCustomerCode
                                        , Collectors.mapping(MdmCustomerSupplyEntity::getPositionCode, Collectors.toList())));
                        //设置组织
                        Map<String, List<String>> orgMap = mdmCustomerROrgService.findOrgCodeList(customerCodeList);

                        for (MdmCustomerMsgReqVo mdmCustomerMsgReqVo : mdmCustomerMsgReqVos) {
                            String customerCode = mdmCustomerMsgReqVo.getCustomerCode();
                            //设置联系人
                            MdmCustomerContactEntity mdmCustomerContactEntity = contactEntityMap.get(customerCode);
                            if (mdmCustomerContactEntity != null) {
                                mdmCustomerContactEntity.setContactMain(YesNoEnum.yesNoEnum.ONE.getValue());
                                mdmCustomerMsgReqVo.setMdmCustomerContactVos(Collections.singletonList(CrmBeanUtil.copy(mdmCustomerContactEntity, MdmCustomerContactReqVo.class)));
                            }
                            //设置对接人
                            List<String> positionCodeList = supplyMap.get(customerCode);
                            if (!CollectionUtils.isEmpty(positionCodeList)) {
                                List<MdmCustomerDockingReqVo> dockingReqVos = positionCodeList.stream().map(k -> {
                                    MdmCustomerDockingReqVo mdmCustomerDockingReqVo = new MdmCustomerDockingReqVo();
                                    mdmCustomerDockingReqVo.setCustomerCode(customerCode);
                                    mdmCustomerDockingReqVo.setPositionCode(k);
                                    return mdmCustomerDockingReqVo;
                                }).collect(Collectors.toList());
                                mdmCustomerMsgReqVo.setMdmCustomerDockingVos(dockingReqVos);
                            }
                            List<String> orgCodeList = orgMap.get(customerCode);
                            if (!CollectionUtils.isEmpty(orgCodeList)) {
                                mdmCustomerMsgReqVo.setOrgCode(String.join(",", orgCodeList));
                            }
                            //保存客户
                            add(mdmCustomerMsgReqVo);
                        }
                    });
        }
    }
}
