package com.biz.crm.terminal.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalVo;
import com.biz.crm.org.service.MdmOrgService;
import com.biz.crm.terminal.mapper.MdmTerminalROrgMapper;
import com.biz.crm.terminal.model.MdmTerminalROrgEntity;
import com.biz.crm.terminal.service.MdmTerminalROrgService;
import com.biz.crm.util.CollectionUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 终端与组织关联表接口实现
 *
 * @author zxw
 * @date 2020-12-21 09:50:12
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmTerminalROrgServiceExpandImpl")
public class MdmTerminalROrgServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmTerminalROrgMapper, MdmTerminalROrgEntity> implements MdmTerminalROrgService {

    @Resource
    private MdmOrgService mdmOrgService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void setUpTerminalRelationOrg(MdmTerminalVo mdmTerminalVo) {
        String terminalCode = mdmTerminalVo.getTerminalCode();
        Assert.hasText(terminalCode, "终端编码不能为空");
        String orgCode = mdmTerminalVo.getOrgCode();
        this.lambdaUpdate()
                .eq(MdmTerminalROrgEntity::getTerminalCode, terminalCode)
                .remove();
        if (!StringUtils.isEmpty(orgCode)) {
            List<String> orgCodeList = Arrays.asList(orgCode.split(","));
            if (!CollectionUtils.isEmpty(orgCodeList)) {
                List<MdmTerminalROrgEntity> list = new ArrayList<>();
                for (String currentOrgCode : orgCodeList) {
                    MdmTerminalROrgEntity mdmTerminalROrgEntity = new MdmTerminalROrgEntity();
                    mdmTerminalROrgEntity.setOrgCode(currentOrgCode);
                    mdmTerminalROrgEntity.setTerminalCode(terminalCode);
                    list.add(mdmTerminalROrgEntity);
                }
                if (!CollectionUtils.isEmpty(list)) {
                    this.saveBatch(list);
                }
            }
        }
    }

    @Override
    public List<String> findTerminalCodeByOrg(List<String> orgCodeList) {
        if (CollectionUtils.isEmpty(orgCodeList)) {
            return new ArrayList<>();
        }
        return this.lambdaQuery()
                .in(MdmTerminalROrgEntity::getOrgCode, orgCodeList)
                .select(MdmTerminalROrgEntity::getTerminalCode)
                .list()
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getTerminalCode()))
                .map(MdmTerminalROrgEntity::getTerminalCode)
                .distinct()
                .collect(Collectors.toList());
    }

    @Override
    public Map<String, String> findOrgCodeByTerminalCodeList(List<String> terminalCodeList) {
        if (CollectionUtils.isEmpty(terminalCodeList)) {
            return Collections.emptyMap();
        }
        return this.lambdaQuery()
                .in(MdmTerminalROrgEntity::getTerminalCode, terminalCodeList)
                .select(MdmTerminalROrgEntity::getTerminalCode, MdmTerminalROrgEntity::getOrgCode)
                .list()
                .stream()
                .collect(Collectors.toMap(MdmTerminalROrgEntity::getTerminalCode, MdmTerminalROrgEntity::getOrgCode, (x1, x2) -> x1));
    }

    @Override
    public Map<String, List<MdmOrgRespVo>> findOrgGroupByTerminalCodeList(List<String> terminalCodeList) {
        if (CollectionUtil.listEmpty(terminalCodeList)) {
            return Collections.emptyMap();
        }
        List<MdmTerminalROrgEntity> list = new ArrayList<>();
        List<List<String>> partition = Lists.partition(terminalCodeList, 500);
        for (List<String> part :
                partition) {
            list.addAll(this.lambdaQuery()
                    .in(MdmTerminalROrgEntity::getTerminalCode, part)
                    .select(MdmTerminalROrgEntity::getTerminalCode, MdmTerminalROrgEntity::getOrgCode)
                    .list());
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            Map<String, List<MdmOrgRespVo>> map = new HashMap<>(16);
            final Map<String, MdmOrgRespVo> orgMap = mdmOrgService.queryBatchByOrgCodeList(list.stream().map(MdmTerminalROrgEntity::getOrgCode).collect(Collectors.toList()))
                    .stream().collect(Collectors.toMap(MdmOrgRespVo::getOrgCode, v -> v));
            Map<String, List<String>> groupMap = list.stream().collect(Collectors.groupingBy(MdmTerminalROrgEntity::getTerminalCode, Collectors.mapping(MdmTerminalROrgEntity::getOrgCode, Collectors.toList())));
            for (Map.Entry<String, List<String>> entry :
                    groupMap.entrySet()) {
                List<MdmOrgRespVo> orgList = new ArrayList<>();
                for (String item :
                        entry.getValue()) {
                    if (orgMap.containsKey(item)) {
                        orgList.add(orgMap.get(item));
                    }
                }
                map.put(entry.getKey(), orgList);
            }
            return map;
        }
        return Collections.emptyMap();
    }

    @Override
    public void setUp(List<String> terminalCodeList, String orgCode) {
        if (CollectionUtils.isEmpty(terminalCodeList)) {
            return;
        }
        this.lambdaUpdate()
                .in(MdmTerminalROrgEntity::getTerminalCode, terminalCodeList)
                .remove();
        if (StringUtils.isEmpty(orgCode)) {
            return;
        }
        List<String> orgCodeList = Arrays.asList(orgCode.split(","));
        if (!CollectionUtils.isEmpty(orgCodeList)) {
            List<MdmTerminalROrgEntity> list = new ArrayList<>();
            for (String terminalCode : terminalCodeList) {
                for (String currentOrgCode : orgCodeList) {
                    MdmTerminalROrgEntity mdmTerminalROrgEntity = new MdmTerminalROrgEntity();
                    mdmTerminalROrgEntity.setTerminalCode(terminalCode);
                    mdmTerminalROrgEntity.setOrgCode(currentOrgCode);
                    list.add(mdmTerminalROrgEntity);
                }
            }
            if (!CollectionUtils.isEmpty(list)) {
                this.saveBatch(list);
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void replaceOrg(List<String> terminalCodeList, String originOrgCode, String orgCode) {
        if (CollectionUtil.listNotEmptyNotSizeZero(terminalCodeList) && !StringUtils.isEmpty(originOrgCode) && !StringUtils.isEmpty(orgCode)) {
            this.lambdaUpdate()
                    .in(MdmTerminalROrgEntity::getTerminalCode, terminalCodeList)
                    .eq(MdmTerminalROrgEntity::getOrgCode, originOrgCode)
                    .remove();
            Set<String> collect = this.lambdaQuery()
                    .in(MdmTerminalROrgEntity::getTerminalCode, terminalCodeList)
                    .eq(MdmTerminalROrgEntity::getOrgCode, orgCode)
                    .select(MdmTerminalROrgEntity::getTerminalCode, MdmTerminalROrgEntity::getOrgCode)
                    .list()
                    .stream().map(MdmTerminalROrgEntity::getTerminalCode).collect(Collectors.toSet());

            List<MdmTerminalROrgEntity> entityList = terminalCodeList.stream().filter(item -> !collect.contains(item)).map(item -> {
                MdmTerminalROrgEntity entity = new MdmTerminalROrgEntity();
                entity.setTerminalCode(item);
                entity.setOrgCode(orgCode);
                return entity;
            }).collect(Collectors.toList());
            if (CollectionUtil.listNotEmptyNotSizeZero(entityList)) {
                this.saveBatch(entityList);
            }
        }
    }

    @Override
    public Map<String, List<String>> findOrgCodeList(List<String> terminalCodeList) {
        if (CollectionUtils.isEmpty(terminalCodeList)) {
            return Collections.emptyMap();
        }
        return this.lambdaQuery()
                .in(MdmTerminalROrgEntity::getTerminalCode, terminalCodeList)
                .list()
                .stream()
                .collect(Collectors.groupingBy(MdmTerminalROrgEntity::getTerminalCode, Collectors.mapping(MdmTerminalROrgEntity::getOrgCode, Collectors.toList())));
    }

    public List<String> findOrgCodeListByCustomerCode(String terminalCode) {
        if (StringUtils.isEmpty(terminalCode)) {
            return new ArrayList<>();
        }
        return this.lambdaQuery()
                .eq(MdmTerminalROrgEntity::getTerminalCode, terminalCode)
                .select(MdmTerminalROrgEntity::getOrgCode)
                .list()
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getTerminalCode()))
                .map(MdmTerminalROrgEntity::getTerminalCode)
                .collect(Collectors.toList());
    }

}
