package com.biz.crm.user.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.eunm.CodeRuleEnum;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.nebular.mdm.constant.UserTypeEnum;
import com.biz.crm.nebular.mdm.user.req.MdmConsumerEditReqVo;
import com.biz.crm.nebular.mdm.user.req.MdmCustomerUserPageReqVo;
import com.biz.crm.nebular.mdm.user.req.MdmUserCustomerEditReqVo;
import com.biz.crm.nebular.mdm.user.req.MdmUserEditReqVo;
import com.biz.crm.nebular.mdm.user.resp.MdmConsumerRespVo;
import com.biz.crm.nebular.mdm.user.resp.MdmCustomerUserPageRespVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRespVo;
import com.biz.crm.user.mapper.MdmCustomerUserMapper;
import com.biz.crm.user.mapper.MdmUserMapper;
import com.biz.crm.user.model.MdmUserEntity;
import com.biz.crm.user.service.MdmConsumerUserService;
import com.biz.crm.user.service.MdmCustomerUserService;
import com.biz.crm.util.CodeUtil;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.Md5EncryptionAndDecryption;
import com.biz.crm.util.PageUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.utils.LoginHelpUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import javax.annotation.Resource;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @Description:消费者用户信息表接口实现
 * @project：crm-core
 * @package：com.biz.crm.user.service.impl
 * @author：longyu
 * @create：2020/12/29 10:46
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmConsumerUserServiceExpandImpl")
public class MdmConsumerUserServiceImpl <M extends BaseMapper<T>, T> extends ServiceImpl<MdmUserMapper, MdmUserEntity> implements MdmConsumerUserService {

    @Resource
    private MdmCustomerUserMapper mdmCustomerUserMapper;

    @Override
    public PageResult<MdmCustomerUserPageRespVo> findList(MdmCustomerUserPageReqVo reqVo) {
        reqVo.setUserType(UserTypeEnum.CONSUMER.getCode());
        Page<MdmCustomerUserPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmCustomerUserPageRespVo> list = mdmCustomerUserMapper.findList(page, reqVo);
        this.setLockState(list);
        return PageResult.<MdmCustomerUserPageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public MdmConsumerRespVo query(String id, String userName) {
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(userName)){
            MdmUserEntity one = this.lambdaQuery()
                    .eq(!StringUtils.isEmpty(id),MdmUserEntity::getId,id)
                    .eq(!StringUtils.isEmpty(userName),MdmUserEntity::getUserName,userName)
                    .one();
            if (one != null){
                MdmConsumerRespVo respVo = CrmBeanUtil.copy(one,MdmConsumerRespVo.class);
                respVo.setUserPassword("");
                return respVo;
            }
        }
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmConsumerEditReqVo reqVo) {
        reqVo.setId(null);
        reqVo.setUserType(UserTypeEnum.CONSUMER.getCode());
        this.verifyData(reqVo);
        MdmUserEntity entity = CrmBeanUtil.copy(reqVo, MdmUserEntity.class);
        if (StringUtils.isEmpty(entity.getUserCode())) {
            entity.setUserCode(CodeUtil.generateCode(CodeRuleEnum.MDM_USER_CODE.getCode()));
        }
        entity.setLockState(CrmEnableStatusEnum.ENABLE.getCode());
        entity.setForceChangePassword(YesNoEnum.yesNoEnum.ONE.getValue());
        entity.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        this.save(entity);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmConsumerEditReqVo reqVo) {
        this.verifyData(reqVo);
        MdmUserEntity entity = this.getById(reqVo.getId());
        CrmBeanUtil.copyProperties(reqVo, entity);
        this.updateById(entity);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        Assert.notEmpty(ids,"id集合不能为空");
        ids.forEach(id -> {
            Assert.hasText(id,"不能传空值");
            MdmUserEntity one = this.getById(id);
            Assert.notNull(one,"无效的用户");
            Assert.isTrue(UserTypeEnum.CONSUMER.getCode().equals(one.getUserType()),"用户[" + one.getFullName() + "]不是消费用户");
        });
        //删除用户
        this.removeByIds(ids);
    }

    /**
     * 校验新增编辑参数
     *
     * @param reqVo
     */
    private void verifyData(MdmConsumerEditReqVo reqVo) {
        String userName = reqVo.getUserName();
        Assert.hasText(userName, "用户帐号不能为空");
        Assert.hasText(reqVo.getUserName(), "用户姓名不能为空");
        MdmUserEntity one = this.lambdaQuery()
                .eq(MdmUserEntity::getUserName, userName)
                .one();
        if (StringUtils.isNotEmpty(reqVo.getUserPhone())) {
            List<MdmUserEntity> list = this.lambdaQuery()
                    .eq(MdmUserEntity::getUserPhone, reqVo.getUserPhone())
                    .list()
                    .stream().filter(x -> StringUtils.isEmpty(reqVo.getId()) || !reqVo.getId().equals(x.getId()))
                    .collect(Collectors.toList());
            Assert.isTrue(CollectionUtil.listEmpty(list), "手机号码已经被占用");
        }
        if (StringUtils.isNotEmpty(reqVo.getEmail())) {
            List<MdmUserEntity> list = this.lambdaQuery()
                    .eq(MdmUserEntity::getEmail, reqVo.getEmail())
                    .list()
                    .stream().filter(x -> StringUtils.isEmpty(reqVo.getId()) || !reqVo.getId().equals(x.getId()))
                    .collect(Collectors.toList());
            Assert.isTrue(CollectionUtil.listEmpty(list), "邮箱已经被占用");
        }
        if (StringUtils.isNotEmpty(reqVo.getIdentityCardNumber())) {
            List<MdmUserEntity> list = this.lambdaQuery()
                    .eq(MdmUserEntity::getIdentityCardNumber, reqVo.getIdentityCardNumber())
                    .list()
                    .stream().filter(x -> StringUtils.isEmpty(reqVo.getId()) || !reqVo.getId().equals(x.getId()))
                    .collect(Collectors.toList());
            Assert.isTrue(CollectionUtil.listEmpty(list), "身份证号码已经存在");
        }
        if (StringUtils.isEmpty(reqVo.getId())) {
            //新增
            if (one != null) {
                String delMark = "";
                if (CrmDelFlagEnum.DELETE.getCode().equals(one.getDelFlag())) {
                    delMark = "（已逻辑删除）";
                }
                throw new BusinessException("帐号[" + userName + "]已存在" + delMark);
            }
            Assert.hasText(reqVo.getUserPassword(), "密码不能为空");
        }
        if (StringUtils.isEmpty(reqVo.getUserPassword())) {
            reqVo.setUserPassword(one.getUserPassword());
        } else {
            //密码加密
            reqVo.setUserPassword(Md5EncryptionAndDecryption.encryPwd(reqVo.getUserPassword()));
        }
    }

    private void setLockState(List<MdmCustomerUserPageRespVo> list) {
        if (CollectionUtil.listEmpty(list)) {
            return;
        }
        final Set<String> lockUserNameSet = LoginHelpUtil.checkAndGetLockUserName(list.stream().map(MdmCustomerUserPageRespVo::getUserName).collect(Collectors.toSet()));
        list.forEach(item->{
            item.setLockState(!lockUserNameSet.contains(item.getUserName()) ? CrmEnableStatusEnum.ENABLE.getCode() : CrmEnableStatusEnum.DISABLE.getCode());
        });
    }
}
