package com.biz.crm.user.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.customer.service.MdmCustomerMsgService;
import com.biz.crm.customer.service.MdmCustomerROrgService;
import com.biz.crm.eunm.CodeRuleEnum;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.mdm.MdmBpmStatusEnum;
import com.biz.crm.nebular.mdm.constant.UserTypeEnum;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgRespVo;
import com.biz.crm.nebular.mdm.user.req.*;
import com.biz.crm.nebular.mdm.user.resp.*;
import com.biz.crm.org.service.MdmOrgService;
import com.biz.crm.terminal.service.MdmTerminalSupplyService;
import com.biz.crm.user.mapper.MdmCustomerUserMapper;
import com.biz.crm.user.mapper.MdmUserMapper;
import com.biz.crm.user.model.MdmUserEntity;
import com.biz.crm.user.service.MdmCustomerUserService;
import com.biz.crm.user.service.MdmUserRelCustomerService;
import com.biz.crm.user.service.MdmUserRelRoleService;
import com.biz.crm.util.*;
import com.biz.crm.utils.LoginHelpUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 用户信息表接口实现
 *
 * @author Tao.Chen
 * @date 2020-11-17 19:04:29
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmUserCustomerServiceExpandImpl")
public class MdmCustomerUserServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmUserMapper, MdmUserEntity> implements MdmCustomerUserService {

    @Resource
    private MdmCustomerUserMapper mdmCustomerUserMapper;
    @Resource
    private MdmCustomerMsgService mdmCustomerMsgService;
    @Resource
    private MdmCustomerROrgService mdmCustomerROrgService;
    @Resource
    private MdmOrgService mdmOrgService;
    @Resource
    private MdmUserRelCustomerService mdmUserRelCustomerService;
    @Resource
    private MdmUserRelRoleService mdmUserRelRoleService;
    @Resource
    private MdmUserMapper mdmUserMapper;
    @Autowired
    private CrmLogSendUtil crmLogSendUtil;
    @Resource
    private MdmTerminalSupplyService mdmTerminalSupplyService;

    @Override
    public PageResult<MdmCustomerUserPageRespVo> findList(MdmCustomerUserPageReqVo reqVo) {
        reqVo.setUserTypeList(Arrays.asList(UserTypeEnum.CUSTOMER.getCode(), UserTypeEnum.CUSTOMER_EMPLOYEE.getCode()));
        Page<MdmCustomerUserPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmCustomerUserPageRespVo> list = mdmCustomerUserMapper.findList(page, reqVo);
        this.setCustomerCodeName(list);
        this.setLockState(list);
        return PageResult.<MdmCustomerUserPageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public MdmUserRespVo query(String id, String userName) {
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(userName)) {
            MdmUserEntity one = this.lambdaQuery()
                    .eq(!StringUtils.isEmpty(id), MdmUserEntity::getId, id)
                    .eq(!StringUtils.isEmpty(userName), MdmUserEntity::getUserName, userName)
                    .one();
            if (one != null) {
                MdmUserRespVo respVo = CrmBeanUtil.copy(one, MdmUserRespVo.class);
                respVo.setUserPassword("");
                if (UserTypeEnum.CUSTOMER.getCode().equals(respVo.getUserType())) {
                    respVo.setCustomerList(mdmCustomerMsgService.getCustomerListByUserName(one.getUserName()));
                    if (YesNoEnum.yesNoEnum.N.getValue().equals(ParamUtil.getParameterValue(ParameterParam.USER_CAN_RELATION_MANY_CUSTOMER))) {
                        if (CollectionUtil.listNotEmptyNotSizeZero(respVo.getCustomerList())) {
                            respVo.setCustomerCode(respVo.getCustomerList().get(0).getCustomerCode());
                            respVo.setCustomerName(respVo.getCustomerList().get(0).getCustomerName());
                        }
                    }
                }
                if (UserTypeEnum.CUSTOMER_EMPLOYEE.getCode().equals(respVo.getUserType())) {
                    List<MdmCustomerMsgRespVo> customerListByUserName = mdmCustomerMsgService.getCustomerListByUserName(one.getUserName());
                    if (CollectionUtil.listNotEmptyNotSizeZero(customerListByUserName)) {
                        respVo.setCustomerCode(customerListByUserName.get(0).getCustomerCode());
                        respVo.setCustomerName(customerListByUserName.get(0).getCustomerName());
                    }
                }
                respVo.setRoleCodeList(mdmUserRelRoleService.findRoleCodeListByUserName(one.getUserName()));
                return respVo;
            }
        }
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmUserEditReqVo reqVo) {
        reqVo.setId(null);
        this.verifyData(reqVo);
        MdmUserEntity entity = CrmBeanUtil.copy(reqVo, MdmUserEntity.class);
        if (StringUtils.isEmpty(entity.getUserCode())) {
            entity.setUserCode(CodeUtil.generateCode(CodeRuleEnum.MDM_USER_CODE.getCode()));
        }
        entity.setLockState(CrmEnableStatusEnum.ENABLE.getCode());
        entity.setForceChangePassword(YesNoEnum.yesNoEnum.ONE.getValue());
        entity.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        this.save(entity);
        saveOrUpdateUserRelation(reqVo, entity);

        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuCodeObj != null) {
            MdmUserRespVo newObject = query(entity.getId(), null);
            crmLogSendUtil.sendForAdd(menuCodeObj.toString(), entity.getId(), entity.getUserName(), newObject);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmUserEditReqVo reqVo) {
        this.verifyData(reqVo);
        MdmUserEntity entity = this.getById(reqVo.getId());
        MdmUserRespVo oldObject = this.query(reqVo.getId(), null);
        CrmBeanUtil.copyProperties(reqVo, entity);
        this.updateById(entity);
        saveOrUpdateUserRelation(reqVo, entity);
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuCodeObj != null) {
            MdmUserRespVo newObject = this.query(reqVo.getId(), null);
            crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),
                    entity.getId(), entity.getUserName(), oldObject, newObject);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        Assert.notEmpty(ids, "id集合不能为空");

        List<MdmUserEntity> list = this.lambdaQuery()
                .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .in(MdmUserEntity::getId, ids)
                .list();
        Assert.notEmpty(list, "无效的id集合");
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuCodeObj != null) {
            list.forEach(item -> {
                crmLogSendUtil.sendForDel(menuCodeObj.toString(),
                        item.getId(), item.getUserName(), item);
            });
        }

        List<MdmUserEntity> collect = list.stream().filter(x -> !UserTypeEnum.CUSTOMER.getCode().equals(x.getUserType()) && !UserTypeEnum.CUSTOMER_EMPLOYEE.getCode().equals(x.getUserType())).collect(Collectors.toList());
        Assert.isTrue(collect.size() == 0, "存在其它不支持的用户类型");
        List<String> userNameList = list.stream().map(MdmUserEntity::getUserName).collect(Collectors.toList());
        mdmUserRelCustomerService.delByUserName(userNameList);
        mdmUserRelRoleService.delByUserName(userNameList);

        //删除用户
        this.lambdaUpdate()
                .in(MdmUserEntity::getId, list.stream().map(MdmUserEntity::getId).collect(Collectors.toSet()))
                .set(MdmUserEntity::getDelFlag, CrmDelFlagEnum.DELETE.getCode())
                .update();

    }

    /**
     * 校验新增编辑参数
     *
     * @param reqVo
     */
    private void verifyData(MdmUserEditReqVo reqVo) {
        String userName = reqVo.getUserName();
        Assert.hasText(userName, "用户帐号不能为空");
        Assert.hasText(reqVo.getUserName(), "用户姓名不能为空");
        Assert.hasText(reqVo.getUserType(), "缺失用户类型");
//        Assert.notNull(reqVo.getGender(), "性别不能为空");
//        Assert.hasText(reqVo.getUserPhone(), "电话不能为空");
//        Assert.hasText(reqVo.getStartTime(), "生效时间不能为空");
//        Assert.hasText(reqVo.getEndTime(), "失效时间不能为空");
        MdmUserEntity one = this.lambdaQuery()
                .eq(MdmUserEntity::getUserName, userName)
                .one();
        if (StringUtils.isNotEmpty(reqVo.getUserPhone())) {
            List<MdmUserEntity> list = this.lambdaQuery()
                    .eq(MdmUserEntity::getUserPhone, reqVo.getUserPhone())
                    .list()
                    .stream().filter(x -> StringUtils.isEmpty(reqVo.getId()) || !reqVo.getId().equals(x.getId()))
                    .collect(Collectors.toList());
            Assert.isTrue(CollectionUtil.listEmpty(list), "手机号码已经被占用");
        }
        if (StringUtils.isNotEmpty(reqVo.getEmail())) {
            List<MdmUserEntity> list = this.lambdaQuery()
                    .eq(MdmUserEntity::getEmail, reqVo.getEmail())
                    .list()
                    .stream().filter(x -> StringUtils.isEmpty(reqVo.getId()) || !reqVo.getId().equals(x.getId()))
                    .collect(Collectors.toList());
            Assert.isTrue(CollectionUtil.listEmpty(list), "邮箱已经被占用");
        }
        if (StringUtils.isNotEmpty(reqVo.getIdentityCardNumber())) {
            List<MdmUserEntity> list = this.lambdaQuery()
                    .eq(MdmUserEntity::getIdentityCardNumber, reqVo.getIdentityCardNumber())
                    .list()
                    .stream().filter(x -> StringUtils.isEmpty(reqVo.getId()) || !reqVo.getId().equals(x.getId()))
                    .collect(Collectors.toList());
            Assert.isTrue(CollectionUtil.listEmpty(list), "身份证号码已经存在");
        }
        if (StringUtils.isEmpty(reqVo.getId())) {
            //新增
            if (one != null) {
                String delMark = "";
                if (CrmDelFlagEnum.DELETE.getCode().equals(one.getDelFlag())) {
                    delMark = "（已逻辑删除）";
                }
                throw new BusinessException("帐号[" + userName + "]已存在" + delMark);
            }
            Assert.hasText(reqVo.getUserPassword(), "密码不能为空");
        }
        if (StringUtils.isEmpty(reqVo.getUserPassword())) {
            reqVo.setUserPassword(one.getUserPassword());
        } else {
            //密码加密
            reqVo.setUserPassword(Md5EncryptionAndDecryption.encryPwd(reqVo.getUserPassword()));
        }

        if (UserTypeEnum.CUSTOMER.getCode().equals(reqVo.getUserType())) {
            String parameterValue = ParamUtil.getParameterValue(ParameterParam.USER_CAN_RELATION_MANY_CUSTOMER);
            if (YesNoEnum.yesNoEnum.Y.getValue().equals(parameterValue)) {
                //客户用户（一对多）
                List<MdmUserCustomerEditReqVo> customerList = reqVo.getCustomerList();
                Assert.notEmpty(customerList, "客户用户必须关联客户");
                Map<String, MdmUserCustomerEditReqVo> customerUniqueMap = new HashMap<>(16);
                for (MdmUserCustomerEditReqVo item :
                        customerList) {
                    Assert.hasText(item.getCustomerCode(), "客户不能为空");
                    if (customerUniqueMap.containsKey(item.getCustomerCode())) {
                        throw new BusinessException("不能重复关联客户相同客户");
                    }
                    customerUniqueMap.put(item.getCustomerCode(), item);
                }
            } else if (YesNoEnum.yesNoEnum.N.getValue().equals(parameterValue)) {
                //客户用户（一对一）
                Assert.hasText(reqVo.getCustomerCode(), "客户编码不能为空");
            } else {
                throw new BusinessException("全局参数错误，请联系管理员处理");
            }
        } else if (UserTypeEnum.CUSTOMER_EMPLOYEE.getCode().equals(reqVo.getUserType())) {
            Assert.hasText(reqVo.getCustomerCode(), "客户编码不能为空");
        } else {
            throw new BusinessException("错误的类型");
        }
    }

    /**
     * 保存/更新用户关联关系
     *
     * @param reqVo
     * @param userEntity
     */
    private void saveOrUpdateUserRelation(MdmUserEditReqVo reqVo, MdmUserEntity userEntity) {
        mdmUserRelCustomerService.delByUserName(userEntity.getUserName());
        mdmUserRelRoleService.delByUserName(userEntity.getUserName());
        List<String> roleCodeList = reqVo.getRoleCodeList();
        if (CollectionUtil.listNotEmptyNotSizeZero(roleCodeList)) {
            mdmUserRelRoleService.add(userEntity.getUserName(), roleCodeList);
        }
        if (UserTypeEnum.CUSTOMER.getCode().equals(userEntity.getUserType())) {
            if (YesNoEnum.yesNoEnum.N.getValue().equals(ParamUtil.getParameterValue(ParameterParam.USER_CAN_RELATION_MANY_CUSTOMER))) {
                reqVo.setCustomerList(Collections.singletonList(new MdmUserCustomerEditReqVo().setCustomerCode(reqVo.getCustomerCode()).setRoleCode(reqVo.getRoleCode())));
            }
            Set<String> customerCodeSet = reqVo.getCustomerList().stream().map(MdmUserCustomerEditReqVo::getCustomerCode).collect(Collectors.toSet());
            mdmUserRelCustomerService.add(userEntity.getUserName(), new ArrayList<>(customerCodeSet));
        } else if (UserTypeEnum.CUSTOMER_EMPLOYEE.getCode().equals(userEntity.getUserType())) {
            mdmUserRelCustomerService.add(userEntity.getUserName(), reqVo.getCustomerCode());
        } else {

        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changeUserCurrentCustomer(MdmUserChangeCurrentCustomerReqVo reqVo) {
        Assert.hasText(reqVo.getUserName(), "缺失用户登录名");
        Assert.hasText(reqVo.getCustomerCode(), "缺失客户编码");
        MdmUserEntity one = this.lambdaQuery()
                .eq(MdmUserEntity::getUserName, reqVo.getUserName())
                .select(MdmUserEntity::getId, MdmUserEntity::getUserName, MdmUserEntity::getFullName, MdmUserEntity::getUserType)
                .one();
        Assert.notNull(one, "未找到用户");
        Assert.isTrue(UserTypeEnum.CUSTOMER.getCode().equals(one.getUserType()), "该用户不是客户用户");
        MdmCustomerMsgRespVo customer = mdmCustomerMsgService.queryByCustomerCode(reqVo.getCustomerCode());
        Assert.notNull(customer, "客户不存在");
        Assert.isTrue(CrmEnableStatusEnum.ENABLE.getCode().equals(customer.getEnableStatus()), "客户未启用");
        List<String> customerCodeList = mdmUserRelCustomerService.findCustomerCodeListByUserName(reqVo.getUserName());
        Assert.isTrue(customerCodeList.size() > 0, "用户未关联任何可用客户");
        Assert.isTrue(customerCodeList.contains(reqVo.getCustomerCode()), "用户未关联该客户");
        mdmUserRelCustomerService.setUniqueCurrentCustomerIfExist(reqVo.getUserName(), reqVo.getCustomerCode());
        UserRedis user = UserUtils.getUser();
        String token = UserUtils.getToken();
        if (user != null && StringUtils.isNotEmpty(user.getUsername()) && one.getUserName().equals(user.getUsername()) && StringUtils.isNotEmpty(token)) {
            //如果切换的是当前userName的客户，需要更新Redis缓存
            user.setCustcode(customer.getCustomerCode());
            user.setCustname(customer.getCustomerName());
            List<String> customerRelOrgList = mdmCustomerROrgService.findOrgCodeByCustomerCode(customer.getCustomerCode());
            if (CollectionUtil.listNotEmptyNotSizeZero(customerRelOrgList)) {
                final Map<String, String> orgNameMap = mdmOrgService.findOrgNameMap(customerRelOrgList);
                List<String> orgCodeList = customerRelOrgList.stream().filter(orgNameMap::containsKey).collect(Collectors.toList());
                if (CollectionUtil.listNotEmptyNotSizeZero(orgCodeList)) {
                    List<String> orgNameList = orgCodeList.stream().map(orgNameMap::get).collect(Collectors.toList());
                    user.setOrgcode(String.join(",", orgCodeList));
                    user.setOrgname(String.join(",", orgNameList));
                }
            }
            UserUtils.setUser(token, user);
        }
    }

    @Override
    public PageResult<MdmUserSelectRespVo> findCustomerUserSelectPageList(MdmUserSelectReqVo reqVo) {
        reqVo.setUserTypeList(Arrays.asList(UserTypeEnum.CUSTOMER.getCode(), UserTypeEnum.CUSTOMER_EMPLOYEE.getCode()));
        Page<MdmUserSelectRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmUserSelectRespVo> list = mdmUserMapper.findUserSelectList(page, reqVo);
        return PageResult.<MdmUserSelectRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public List<MdmUserRespVo> findCustomerRelCustomerUserList(MdmCustomerRelCustomerUserReqVo reqVo) {
        Assert.notEmpty(reqVo.getCustomerCodeList(), "缺失客户编码");
        List<String> userNameList = mdmUserRelCustomerService.findUserNameListByCustomerCodeList(reqVo.getCustomerCodeList());
        if (CollectionUtil.listNotEmptyNotSizeZero(userNameList)) {
            if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getRoleCodeList())) {
                List<String> roleRelUserNameList = mdmUserRelRoleService.findUserNameListByRoleCodeList(reqVo.getRoleCodeList());
                userNameList.retainAll(roleRelUserNameList);
            }
            if (CollectionUtil.listNotEmptyNotSizeZero(userNameList)) {
                return this.lambdaQuery()
                        .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                        .eq(MdmUserEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                        .in(MdmUserEntity::getUserName, userNameList)
                        .list()
                        .stream().map(x -> CrmBeanUtil.copy(x, MdmUserRespVo.class)).collect(Collectors.toList());
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmUserSelectRespVo> findCustomerUserSelectList(MdmUserSelectReqVo reqVo) {
        reqVo.setUserTypeList(Arrays.asList(UserTypeEnum.CUSTOMER.getCode(), UserTypeEnum.CUSTOMER_EMPLOYEE.getCode()));
        Integer pageSize = reqVo.getPageSize();
        if (pageSize == null || pageSize == 0) {
            pageSize = Integer.valueOf(ParamUtil.getParameterValue(ParameterParam.SELECT_BOX_OPTION_SIZE));
        }
        List<MdmUserSelectRespVo> list = new ArrayList<>();
        Set<String> codeList = new HashSet<>(16);
        if (StringUtils.isNotEmpty(reqVo.getSelectedCode())) {
            codeList.add(reqVo.getSelectedCode());
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSelectedCodeList())) {
            codeList.addAll(reqVo.getSelectedCodeList());
        }
        if (!codeList.isEmpty()) {
            reqVo.setSelectedCodeList(new ArrayList<>(codeList));
        }
        if (pageSize > 0) {
            list.addAll(mdmUserMapper.findUserSelectList(new Page<>(1L, pageSize, false), reqVo));
        }
        return list;
    }

    @Override
    public PageResult<MdmCustomerUserRelRolePageRespVo> findCustomerUserNotRelAnyRole(MdmCustomerUserRelRolePageReqVo reqVo) {
        reqVo.setUserTypeList(Arrays.asList(UserTypeEnum.CUSTOMER.getCode(), UserTypeEnum.CUSTOMER_EMPLOYEE.getCode()));
        Page<MdmCustomerUserRelRolePageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmCustomerUserRelRolePageRespVo> list = mdmCustomerUserMapper.findCustomerUserNotRelAnyRole(page, reqVo);
        return PageResult.<MdmCustomerUserRelRolePageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public PageResult<MdmCustomerUserRelRolePageRespVo> findCustomerUserNotRelCurRole(MdmCustomerUserRelRolePageReqVo reqVo) {
        reqVo.setUserTypeList(Arrays.asList(UserTypeEnum.CUSTOMER.getCode(), UserTypeEnum.CUSTOMER_EMPLOYEE.getCode()));
        Assert.hasText(reqVo.getRoleCode(),"缺失权限角色编码");
        Page<MdmCustomerUserRelRolePageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmCustomerUserRelRolePageRespVo> list = mdmCustomerUserMapper.findCustomerUserNotRelCurRole(page, reqVo);
        return PageResult.<MdmCustomerUserRelRolePageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public PageResult<MdmCustomerUserRelRolePageRespVo> findCustomerUserHasRelCurRole(MdmCustomerUserRelRolePageReqVo reqVo) {
        reqVo.setUserTypeList(Arrays.asList(UserTypeEnum.CUSTOMER.getCode(), UserTypeEnum.CUSTOMER_EMPLOYEE.getCode()));
        Assert.hasText(reqVo.getRoleCode(),"缺失权限角色编码");
        Page<MdmCustomerUserRelRolePageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmCustomerUserRelRolePageRespVo> list = mdmCustomerUserMapper.findCustomerUserHasRelCurRole(page, reqVo);
        return PageResult.<MdmCustomerUserRelRolePageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void bindUserRelRole(MdmCustomerUserRelRoleReqVo reqVo) {
        Assert.hasText(reqVo.getRoleCode(), "缺失权限角色编码");
        Assert.notEmpty(reqVo.getUserNameList(), "缺失用户编码集合");
        mdmUserRelRoleService.add(reqVo.getUserNameList(), reqVo.getRoleCode());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void replaceUserRelRole(MdmCustomerUserRelRoleReqVo reqVo) {
        Assert.hasText(reqVo.getRoleCode(), "缺失权限角色编码");
        Assert.notEmpty(reqVo.getIds(), "缺失关联id集合");
        mdmUserRelRoleService.replaceUserRelRole(reqVo.getIds(), reqVo.getRoleCode());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void unbindUserRelRole(List<String> ids) {
        Assert.notEmpty(ids, "缺失参数");
        mdmUserRelRoleService.removeByIds(ids);
    }

    @Override
    public PageResult<MdmCustomerUserRelTerminalPageRespVo> findTerminalNotRelAnyCustomerUser(MdmCustomerUserRelTerminalPageReqVo reqVo) {
        Assert.hasText(reqVo.getUserName(),"缺失用户账号");
        reqVo.setUserTypeList(Arrays.asList(UserTypeEnum.CUSTOMER.getCode(), UserTypeEnum.CUSTOMER_EMPLOYEE.getCode()));
        reqVo.setActApproveStatus(MdmBpmStatusEnum.PASS.getValue());
        reqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        Page<MdmCustomerUserRelTerminalPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmCustomerUserRelTerminalPageRespVo> list = mdmCustomerUserMapper.findTerminalNotRelAnyCustomerUser(page, reqVo);
        return PageResult.<MdmCustomerUserRelTerminalPageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public PageResult<MdmCustomerUserRelTerminalPageRespVo> findTerminalNotRelCurCustomerUser(MdmCustomerUserRelTerminalPageReqVo reqVo) {
        Assert.hasText(reqVo.getUserName(),"缺失用户账号");
        reqVo.setUserTypeList(Arrays.asList(UserTypeEnum.CUSTOMER.getCode(), UserTypeEnum.CUSTOMER_EMPLOYEE.getCode()));
        reqVo.setActApproveStatus(MdmBpmStatusEnum.PASS.getValue());
        reqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        Page<MdmCustomerUserRelTerminalPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmCustomerUserRelTerminalPageRespVo> list = mdmCustomerUserMapper.findTerminalNotRelCurCustomerUser(page, reqVo);
        return PageResult.<MdmCustomerUserRelTerminalPageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public PageResult<MdmCustomerUserRelTerminalPageRespVo> findTerminalHasRelCurCustomerUser(MdmCustomerUserRelTerminalPageReqVo reqVo) {
        Assert.hasText(reqVo.getUserName(),"缺失用户账号");
        reqVo.setUserTypeList(Arrays.asList(UserTypeEnum.CUSTOMER.getCode(), UserTypeEnum.CUSTOMER_EMPLOYEE.getCode()));
        reqVo.setActApproveStatus(MdmBpmStatusEnum.PASS.getValue());
        reqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        Page<MdmCustomerUserRelTerminalPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmCustomerUserRelTerminalPageRespVo> list = mdmCustomerUserMapper.findTerminalHasRelCurCustomerUser(page, reqVo);
        return PageResult.<MdmCustomerUserRelTerminalPageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void bindCustomerUserRelTerminal(MdmCustomerUserRelTerminalReqVo reqVo) {
        Assert.hasText(reqVo.getUserName(), "缺失用户");
        Assert.notEmpty(reqVo.getTerminalCodeList(), "缺失终端编码集合");
        mdmTerminalSupplyService.addCustomerUserTerminalSupply(reqVo.getTerminalCodeList(), reqVo.getUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void replaceCustomerUserRelTerminal(MdmCustomerUserRelTerminalReqVo reqVo) {
        Assert.hasText(reqVo.getUserName(), "缺失用户");
        Assert.hasText(reqVo.getTargetUserName(), "缺失目标用户");
        Assert.notEmpty(reqVo.getTerminalCodeList(), "缺失终端编码集合");
        mdmTerminalSupplyService.replaceCustomerUserTerminalSupply(reqVo.getTerminalCodeList(), reqVo.getUserName(), reqVo.getTargetUserName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void unbindCustomerUserRelTerminal(MdmCustomerUserRelTerminalReqVo reqVo) {
        Assert.hasText(reqVo.getUserName(), "缺失用户");
        Assert.notEmpty(reqVo.getTerminalCodeList(), "缺失终端编码集合");
        mdmTerminalSupplyService.delCustomerUserTerminalSupply(reqVo.getTerminalCodeList(), reqVo.getUserName());
    }

    /**
     * 设置列表数据的客户信息列表
     *
     * @param allList
     */
    protected void setCustomerCodeName(List<MdmCustomerUserPageRespVo> allList) {
        if (CollectionUtil.listEmpty(allList)) {
            return;
        }
        List<List<MdmCustomerUserPageRespVo>> partition = Lists.partition(allList, 500);
        for (List<MdmCustomerUserPageRespVo> list :
                partition) {
            final Map<String, List<String>> map = mdmUserRelCustomerService.getCustomerCodeListGroupByUserNameList(list.stream().map(MdmCustomerUserPageRespVo::getUserName).collect(Collectors.toList()));
            if (!map.isEmpty()) {
                Set<String> collect = map.values().stream().flatMap(Collection::stream).collect(Collectors.toSet());
                final Map<String, String> customerCodeNameMap = mdmCustomerMsgService.findCustomerCodeNameMapByCustomerCodeList(new ArrayList<>(collect));
                list.forEach(item -> {
                    if (map.containsKey(item.getUserName())) {
                        List<String> customerCodeList = map.get(item.getUserName()).stream().filter(customerCodeNameMap::containsKey).collect(Collectors.toList());
                        if (CollectionUtil.listNotEmptyNotSizeZero(customerCodeList)) {
                            String names = customerCodeList.stream().map(customerCodeNameMap::get).collect(Collectors.joining(","));
                            item.setCustomerCode(String.join(",", customerCodeList));
                            item.setCustomerName(names);
                        }
                    }
                });
            }
        }
    }

    /**
     * 设置列表数据的锁定状态
     *
     * @param list
     */
    protected void setLockState(List<MdmCustomerUserPageRespVo> list) {
        if (CollectionUtil.listEmpty(list)) {
            return;
        }
        final Set<String> lockUserNameSet = LoginHelpUtil.checkAndGetLockUserName(list.stream().map(MdmCustomerUserPageRespVo::getUserName).collect(Collectors.toSet()));
        list.forEach(item -> {
            item.setLockState(!lockUserNameSet.contains(item.getUserName()) ? CrmEnableStatusEnum.ENABLE.getCode() : CrmEnableStatusEnum.DISABLE.getCode());
        });
    }
}
