package com.biz.crm.user.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRelCustomerRespVo;
import com.biz.crm.user.mapper.MdmUserRelCustomerMapper;
import com.biz.crm.user.model.MdmUserRelCustomerEntity;
import com.biz.crm.user.service.MdmUserRelCustomerService;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.StringUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 客户用户与客户关联表接口实现
 *
 * @author Tao.Chen
 * @date 2021-04-19 14:55:59
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmUserRelCustomerServiceExpandImpl")
public class MdmUserRelCustomerServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmUserRelCustomerMapper, MdmUserRelCustomerEntity> implements MdmUserRelCustomerService {

    @Resource
    private MdmUserRelCustomerMapper mdmUserRelCustomerMapper;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void add(String userName, String customerCode) {
        if (StringUtils.isNotEmpty(userName) && StringUtils.isNotEmpty(customerCode)) {
            this.add(userName, Collections.singletonList(customerCode));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void add(String userName, String customerCode, String currentFlag) {
        if (StringUtils.isNotEmpty(userName) && StringUtils.isNotEmpty(customerCode)) {
            if (StringUtils.isEmpty(currentFlag)) {
                currentFlag = YesNoEnum.yesNoEnum.ZERO.getValue();
            } else if (!YesNoEnum.yesNoEnum.ONE.getValue().equals(currentFlag) && !YesNoEnum.yesNoEnum.ZERO.getValue().equals(currentFlag)) {
                throw new BusinessException("currentFlag参数不合法");
            } else {

            }
            MdmUserRelCustomerEntity one = this.lambdaQuery()
                    .eq(MdmUserRelCustomerEntity::getUserName, userName)
                    .eq(MdmUserRelCustomerEntity::getCustomerCode, customerCode)
                    .one();
            if (one == null) {
                MdmUserRelCustomerEntity entity = new MdmUserRelCustomerEntity();
                entity.setUserName(userName);
                entity.setCustomerCode(customerCode);
                entity.setCurrentFlag(currentFlag);
                this.save(entity);
            } else {
                if (currentFlag.equals(one.getCurrentFlag())) {
                    this.lambdaUpdate()
                            .eq(MdmUserRelCustomerEntity::getUserName, userName)
                            .eq(MdmUserRelCustomerEntity::getCustomerCode, customerCode)
                            .set(MdmUserRelCustomerEntity::getCurrentFlag, currentFlag)
                            .update();
                }
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void add(String userName, List<String> customerCodeList) {
        if (StringUtils.isNotEmpty(userName) && CollectionUtil.listNotEmptyNotSizeZero(customerCodeList)) {
            Set<String> existCodeSet = this.lambdaQuery()
                    .eq(MdmUserRelCustomerEntity::getUserName, userName)
                    .in(MdmUserRelCustomerEntity::getCustomerCode, customerCodeList)
                    .select(MdmUserRelCustomerEntity::getCustomerCode)
                    .list()
                    .stream().map(MdmUserRelCustomerEntity::getCustomerCode).collect(Collectors.toSet());
            Set<String> customerCodeSet = new HashSet<>(customerCodeList);
            List<MdmUserRelCustomerEntity> entityList = new ArrayList<>();
            for (String customerCode :
                    customerCodeSet) {
                if (existCodeSet.contains(customerCode)) {
                    continue;
                }
                MdmUserRelCustomerEntity entity = new MdmUserRelCustomerEntity();
                entity.setUserName(userName);
                entity.setCustomerCode(customerCode);
                entity.setCurrentFlag(YesNoEnum.yesNoEnum.ZERO.getValue());
                entityList.add(entity);
            }
            if (entityList.size() > 0) {
                this.saveBatch(entityList);
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void add(List<String> userNameList, String customerCode) {
        if (CollectionUtil.listNotEmptyNotSizeZero(userNameList) && StringUtils.isNotEmpty(customerCode)) {
            Set<String> existCodeSet = this.lambdaQuery()
                    .eq(MdmUserRelCustomerEntity::getCustomerCode, customerCode)
                    .in(MdmUserRelCustomerEntity::getUserName, userNameList)
                    .select(MdmUserRelCustomerEntity::getUserName)
                    .list()
                    .stream().map(MdmUserRelCustomerEntity::getUserName).collect(Collectors.toSet());
            Set<String> userNameSet = new HashSet<>(userNameList);
            List<MdmUserRelCustomerEntity> entityList = new ArrayList<>();
            for (String userName :
                    userNameSet) {
                if (existCodeSet.contains(userName)) {
                    continue;
                }
                MdmUserRelCustomerEntity entity = new MdmUserRelCustomerEntity();
                entity.setUserName(userName);
                entity.setCustomerCode(customerCode);
                entity.setCurrentFlag(YesNoEnum.yesNoEnum.ZERO.getValue());
                entityList.add(entity);
            }
            if (entityList.size() > 0) {
                this.saveBatch(entityList);
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void del(String userName, String customerCode) {
        if (StringUtils.isNotEmpty(userName) && StringUtils.isNotEmpty(customerCode)) {
            this.del(userName, Collections.singletonList(customerCode));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void del(String userName, List<String> customerCodeList) {
        if (StringUtils.isNotEmpty(userName) && CollectionUtil.listNotEmptyNotSizeZero(customerCodeList)) {
            this.lambdaUpdate()
                    .eq(MdmUserRelCustomerEntity::getUserName, userName)
                    .in(MdmUserRelCustomerEntity::getCustomerCode, customerCodeList)
                    .remove();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void del(List<String> userNameList, String customerCode) {
        if (CollectionUtil.listNotEmptyNotSizeZero(userNameList) && StringUtils.isNotEmpty(customerCode)) {
            this.lambdaUpdate()
                    .eq(MdmUserRelCustomerEntity::getCustomerCode, customerCode)
                    .in(MdmUserRelCustomerEntity::getUserName, userNameList)
                    .remove();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delByUserName(String userName) {
        if (StringUtils.isNotEmpty(userName)) {
            this.delByUserName(Collections.singletonList(userName));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delByUserName(List<String> userNameList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(userNameList)) {
            this.lambdaUpdate()
                    .in(MdmUserRelCustomerEntity::getUserName, userNameList)
                    .remove();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delByCustomerCode(String customerCode) {
        if (StringUtils.isNotEmpty(customerCode)) {
            this.delByCustomerCode(Collections.singletonList(customerCode));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delByCustomerCode(List<String> customerCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(customerCodeList)) {
            this.lambdaUpdate()
                    .in(MdmUserRelCustomerEntity::getCustomerCode, customerCodeList)
                    .remove();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void setUniqueCurrentCustomerIfExist(String userName, String customerCode) {
        if (StringUtils.isNotEmpty(userName) && StringUtils.isNotEmpty(customerCode)) {
            this.lambdaUpdate()
                    .eq(MdmUserRelCustomerEntity::getUserName, userName)
                    .set(MdmUserRelCustomerEntity::getCurrentFlag, YesNoEnum.yesNoEnum.ZERO.getValue())
                    .update();
            this.lambdaUpdate()
                    .eq(MdmUserRelCustomerEntity::getUserName, userName)
                    .eq(MdmUserRelCustomerEntity::getCustomerCode, customerCode)
                    .set(MdmUserRelCustomerEntity::getCurrentFlag, YesNoEnum.yesNoEnum.ONE.getValue())
                    .update();
        }
    }

    @Override
    public List<String> findCustomerCodeListByUserName(String userName) {
        if (StringUtils.isNotEmpty(userName)) {
            return this.lambdaQuery()
                    .eq(MdmUserRelCustomerEntity::getUserName, userName)
                    .select(MdmUserRelCustomerEntity::getCustomerCode)
                    .list()
                    .stream().map(MdmUserRelCustomerEntity::getCustomerCode).collect(Collectors.toList());
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmUserRelCustomerRespVo> findCustomerListByUserName(String userName) {
        if (StringUtils.isNotEmpty(userName)) {
            return this.lambdaQuery()
                    .eq(MdmUserRelCustomerEntity::getUserName, userName)
                    .list()
                    .stream().map(x -> CrmBeanUtil.copy(x, MdmUserRelCustomerRespVo.class)).collect(Collectors.toList());
        }
        return new ArrayList<>();
    }

    @Override
    public Map<String, List<String>> getCustomerCodeListGroupByUserNameList(List<String> userNameList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(userNameList)) {
            Map<String, List<String>> map = new HashMap<>(16);
            List<List<String>> listGroup = Lists.partition(userNameList, 500);
            for (List<String> group :
                    listGroup) {
                List<MdmUserRelCustomerEntity> list = this.lambdaQuery()
                        .in(MdmUserRelCustomerEntity::getUserName, group)
                        .select(MdmUserRelCustomerEntity::getUserName, MdmUserRelCustomerEntity::getCustomerCode)
                        .list();
                if (list.size() > 0) {
                    map.putAll(list.stream().collect(Collectors.groupingBy(MdmUserRelCustomerEntity::getUserName, Collectors.mapping(MdmUserRelCustomerEntity::getCustomerCode, Collectors.toList()))));
                }
            }
            return map;
        }
        return Collections.emptyMap();
    }

    @Override
    public List<String> findUserNameListByCustomerCode(String customerCode) {
        if (StringUtils.isNotEmpty(customerCode)) {
            return this.lambdaQuery()
                    .eq(MdmUserRelCustomerEntity::getCustomerCode, customerCode)
                    .select(MdmUserRelCustomerEntity::getUserName)
                    .list()
                    .stream().map(MdmUserRelCustomerEntity::getUserName).collect(Collectors.toList());
        }
        return new ArrayList<>();
    }

    @Override
    public List<String> findUserNameListByCustomerCodeList(List<String> customerCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(customerCodeList)) {
            return this.lambdaQuery()
                    .in(MdmUserRelCustomerEntity::getCustomerCode, customerCodeList)
                    .select(MdmUserRelCustomerEntity::getUserName)
                    .list()
                    .stream().map(MdmUserRelCustomerEntity::getUserName).distinct().collect(Collectors.toList());
        }
        return new ArrayList<>();
    }

}
