package com.biz.crm.user.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.mdm.LoginFromTypeEnum;
import com.biz.crm.login.service.MdmTerminalAppletLoginService;
import com.biz.crm.nebular.mdm.constant.UserTypeEnum;
import com.biz.crm.nebular.mdm.user.req.MdmUserRelWeChatReqVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRelWeChatRespVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRespVo;
import com.biz.crm.user.mapper.MdmUserRelWeChatMapper;
import com.biz.crm.user.model.MdmUserRelWeChatEntity;
import com.biz.crm.user.service.MdmUserRelWeChatService;
import com.biz.crm.user.service.MdmUserService;
import com.biz.crm.util.*;
import java.util.Objects;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 用户关联微信接口实现
 *
 * @author Tao.Chen
 * @date 2021-02-19 11:47:07
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmUserRelWeChatServiceExpandImpl")
public class MdmUserRelWeChatServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmUserRelWeChatMapper, MdmUserRelWeChatEntity> implements MdmUserRelWeChatService {

    @Resource
    private MdmUserRelWeChatMapper mdmUserRelWeChatMapper;
    @Resource
    private MdmUserService mdmUserService;
    @Autowired
    private MdmTerminalAppletLoginService mdmTerminalAppletLoginService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void bind(MdmUserRelWeChatReqVo reqVo) {
        Assert.hasText(reqVo.getUserName(), "缺失用户名");
        Assert.hasText(reqVo.getOpenId(), "缺失openId");
        Assert.hasText(reqVo.getOriginModule(), "缺失关联小程序模块");
        MdmUserRespVo mdmUserRespVo = mdmUserService.queryByUserName(reqVo.getUserName());
        Assert.notNull(mdmUserRespVo, "用户不存在");
        boolean uniqueCheck = false;
        if (LoginFromTypeEnum.APPLET_SFA.getValue().equals(reqVo.getOriginModule())) {
            //意义不明，应产品要求解开限制（提出人：雷环博 2021-12-29）
            /*Assert.isTrue(UserTypeEnum.USER.getCode().equals(mdmUserRespVo.getUserType()), "非企业用户不能绑定SFA小程序");*/
            uniqueCheck = YesNoEnum.yesNoEnum.Y.getValue().equals(ParamUtil.getParameterValue(ParameterParam.SFA_APPLET_USER_UNIQUE_FLAG));
        } else if (LoginFromTypeEnum.APPLET_DMS.getValue().equals(reqVo.getOriginModule())) {
            Assert.isTrue(UserTypeEnum.CUSTOMER.getCode().equals(mdmUserRespVo.getUserType()), "非客户用户不能绑定DMS小程序");
        }else if (LoginFromTypeEnum.APPLET_TERMINAL.getValue().equals(reqVo.getOriginModule())) {
            Assert.isTrue(UserTypeEnum.CUSTOMER.getCode().equals(mdmUserRespVo.getUserType()) || UserTypeEnum.TERMINAL.getCode().equals(mdmUserRespVo.getUserType()), "非客户用户或终端用户不能绑定终端小程序");
        }else {
            throw new BusinessException("小程序模块不存在");
        }
        if (uniqueCheck) {
            List<MdmUserRelWeChatEntity> userCheck = this.lambdaQuery()
                    .eq(MdmUserRelWeChatEntity::getOriginModule, reqVo.getOriginModule())
                    .eq(MdmUserRelWeChatEntity::getUserName, reqVo.getUserName())
                    .list()
                    .stream().filter(x -> !reqVo.getOpenId().equals(x.getOpenId())).collect(Collectors.toList());
            Assert.isTrue(CollectionUtil.listEmpty(userCheck), "该用户已经绑定了微信，不能重复绑定");
        }

        List<MdmUserRelWeChatEntity> collect = this.lambdaQuery()
                .eq(MdmUserRelWeChatEntity::getOriginModule, reqVo.getOriginModule())
                .eq(MdmUserRelWeChatEntity::getOpenId, reqVo.getOpenId())
                .select(MdmUserRelWeChatEntity::getId, MdmUserRelWeChatEntity::getUserName)
                .list()
                .stream().filter(x -> !reqVo.getUserName().equals(x.getUserName()))
                .collect(Collectors.toList());
        if (CollectionUtil.listNotEmptyNotSizeZero(collect)) {
            this.removeByIds(collect.stream().map(MdmUserRelWeChatEntity::getId).collect(Collectors.toList()));
        }
        List<MdmUserRelWeChatEntity> existCheck = this.lambdaQuery()
                .eq(MdmUserRelWeChatEntity::getOriginModule, reqVo.getOriginModule())
                .eq(MdmUserRelWeChatEntity::getUserName, reqVo.getUserName())
                .eq(MdmUserRelWeChatEntity::getOpenId, reqVo.getOpenId())
                .list();
        if (CollectionUtil.listEmpty(existCheck)) {
            if (StringUtils.isEmpty(reqVo.getLoginStatus())) {
                reqVo.setLoginStatus(YesNoEnum.yesNoEnum.ZERO.getValue());
            }
            this.save(CrmBeanUtil.copy(reqVo, MdmUserRelWeChatEntity.class));
        } else {
            if (!YesNoEnum.yesNoEnum.ONE.getValue().equals(existCheck.get(0).getLoginStatus())) {
                existCheck.get(0).setLoginStatus(YesNoEnum.yesNoEnum.ONE.getValue());
                this.updateById(existCheck.get(0));
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void unbindByOpenId(String originModule, String openId) {
        if (!YesNoEnum.yesNoEnum.Y.getValue().equals(ParamUtil.getParameterValue(ParameterParam.SFA_APPLET_USER_UNIQUE_FLAG))) {
            this.lambdaUpdate()
                    .eq(MdmUserRelWeChatEntity::getOriginModule, originModule)
                    .eq(MdmUserRelWeChatEntity::getOpenId, openId)
                    .remove();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void unbindByIds(List<String> ids) {
        if (CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            List<MdmUserRelWeChatEntity> list = this.lambdaQuery()
                    .in(MdmUserRelWeChatEntity::getId, ids)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                this.removeByIds(list.stream().map(MdmUserRelWeChatEntity::getId).collect(Collectors.toList()));
                UserUtils.deleteUser(list.stream().filter(x -> LoginFromTypeEnum.APPLET_SFA.getValue().equals(x.getOriginModule())).map(MdmUserRelWeChatEntity::getUserName).distinct().collect(Collectors.toList()), Collections.singletonList(LoginFromTypeEnum.APPLET_SFA.getValue()));
                UserUtils.deleteUser(list.stream().filter(x -> LoginFromTypeEnum.APPLET_DMS.getValue().equals(x.getOriginModule())).map(MdmUserRelWeChatEntity::getUserName).distinct().collect(Collectors.toList()), Collections.singletonList(LoginFromTypeEnum.APPLET_DMS.getValue()));
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void unbindByUserName(List<String> userNameList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(userNameList)) {
            List<MdmUserRelWeChatEntity> list = this.lambdaQuery()
                    .in(MdmUserRelWeChatEntity::getUserName, userNameList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                this.removeByIds(list.stream().map(MdmUserRelWeChatEntity::getId).collect(Collectors.toList()));
                UserUtils.deleteUser(list.stream().filter(x -> LoginFromTypeEnum.APPLET_SFA.getValue().equals(x.getOriginModule())).map(MdmUserRelWeChatEntity::getUserName).distinct().collect(Collectors.toList()), Collections.singletonList(LoginFromTypeEnum.APPLET_SFA.getValue()));
                UserUtils.deleteUser(list.stream().filter(x -> LoginFromTypeEnum.APPLET_DMS.getValue().equals(x.getOriginModule())).map(MdmUserRelWeChatEntity::getUserName).distinct().collect(Collectors.toList()), Collections.singletonList(LoginFromTypeEnum.APPLET_DMS.getValue()));
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changeLoginStatus(String originModule, String openId, String loginStatus) {
        this.lambdaUpdate()
                .eq(MdmUserRelWeChatEntity::getOriginModule, originModule)
                .eq(MdmUserRelWeChatEntity::getOpenId, openId)
                .set(MdmUserRelWeChatEntity::getLoginStatus, loginStatus)
                .update();
    }

    @Override
    public List<MdmUserRelWeChatRespVo> findUserRelWeChatList(MdmUserRelWeChatReqVo reqVo) {
        Assert.notNull(reqVo, "空参数");
        Assert.hasText(reqVo.getUserName(), "缺失用户名");
        List<MdmUserRelWeChatEntity> list = this.lambdaQuery()
                .eq(MdmUserRelWeChatEntity::getUserName, reqVo.getUserName())
                .eq(StringUtils.isNotEmpty(reqVo.getOriginModule()), MdmUserRelWeChatEntity::getOriginModule, reqVo.getOriginModule())
                .eq(StringUtils.isNotEmpty(reqVo.getLoginStatus()), MdmUserRelWeChatEntity::getLoginStatus, reqVo.getLoginStatus())
                .like(StringUtils.isNotEmpty(reqVo.getNickName()), MdmUserRelWeChatEntity::getNickName, reqVo.getNickName())
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            return list.stream().map(x -> {
                MdmUserRelWeChatRespVo copy = CrmBeanUtil.copy(x, MdmUserRelWeChatRespVo.class);
                copy.setOriginModuleName(LoginFromTypeEnum.getDesc(copy.getOriginModule()));
                return copy;
            }).collect(Collectors.toList());
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmUserRelWeChatRespVo> getBindUserListByOpenId(String openId, String originModule) {
        if (StringUtils.isNotEmpty(openId)) {
            List<MdmUserRelWeChatEntity> list = this.lambdaQuery()
                    .eq(MdmUserRelWeChatEntity::getOpenId, openId)
                    .eq(StringUtils.isNotEmpty(originModule), MdmUserRelWeChatEntity::getOriginModule, originModule)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return CrmBeanUtil.copyList(list, MdmUserRelWeChatRespVo.class);
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmUserRelWeChatRespVo> getBindWeChatListByUserName(String userName, String originModule) {
        if (StringUtils.isNotEmpty(userName)) {
            List<MdmUserRelWeChatEntity> list = this.lambdaQuery()
                    .eq(MdmUserRelWeChatEntity::getUserName, userName)
                    .eq(StringUtils.isNotEmpty(originModule), MdmUserRelWeChatEntity::getOriginModule, originModule)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return CrmBeanUtil.copyList(list, MdmUserRelWeChatRespVo.class);
            }
        }
        return new ArrayList<>();
    }

    @Override
    public void bindUserWeChatVerificationPhone(MdmUserRelWeChatReqVo bindReq, String phone, String code) {
        this.mdmTerminalAppletLoginService.VerificationPhoneByCode(phone,code);
        //获取用户
        UserRedis user = UserUtils.getUser();
        MdmUserRespVo userRespVo =mdmUserService.queryByUserName(user.getUsername());
        Validate.notNull(userRespVo,"用户不存在");
        Validate.isTrue(Objects.equals(phone,userRespVo.getUserPhone()),"手机号不是本账户手机号");
        this.bind(bindReq);

        String token = UserUtils.getToken();
        user.setOpenId(bindReq.getOpenId());
        UserUtils.setUser(token, user);
        HttpServletRequest request = HttpServletRequestUtil.getRequest();
        HttpServletResponse response = HttpServletRequestUtil.getResponse();
        CookiesUtil.doCoke(request, response, token, GlobalParam.TOKEN);
    }

}
