package com.biz.crm.activiti.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.activiti.entity.ActReModelEntity;
import com.biz.crm.activiti.entity.TaProcessBizRelationEntity;
import com.biz.crm.activiti.mapper.ActReModelMapper;
import com.biz.crm.activiti.mapper.TaProcessBizRelationMapper;
import com.biz.crm.activiti.service.TaProcessBizRelationService;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.GlobalWhetherEnum;
import com.biz.crm.eunm.activiti.Indicator;
import com.biz.crm.eunm.activiti.act.TaProcessBizRelationEnum;
import com.biz.crm.nebular.activiti.design.req.TaProcessFormReqVO;
import com.biz.crm.nebular.activiti.design.resp.TaProcessOrgRespVO;
import com.biz.crm.nebular.mdm.org.req.MdmOrgReqVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.util.*;
import com.biz.crm.vo.*;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 这是流程绘制-业务关系相关 业务接口实现类
 *
 * @author YuanZiJian
 * @date 2020/12/15 17:09
 * @version: V1.0
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "TaProcessBizRelationExpandServiceImpl")
public class TaProcessBizRelationServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<TaProcessBizRelationMapper, TaProcessBizRelationEntity> implements TaProcessBizRelationService {
//    @Resource
//    private TaProcessMapper taProcessMapper;
//
//    @Resource
//    private TaProcessService taProcessService;

    @Resource
    private TaProcessBizRelationMapper mapper;
    @Resource
    private ActReModelMapper actReModelMapper;

    /**
     * 根据流程key找到对应的业务关系
     *
     * @return com.biz.crm.act.vo.TaProcessBizRelationRespVo
     * @method getRelationByProcessKey
     * @date: 2020/12/15 17:04
     * @author: YuanZiJian
     */
    @Override
    public TaProcessBizRelationRespVo getRelationByProcessKey(String processKey) {
        AssertUtils.isNotEmpty(processKey,"流程Key不能为空");
        List<TaProcessBizRelationEntity> entityList = this.list(Wrappers.lambdaQuery(TaProcessBizRelationEntity.class)
                .eq(TaProcessBizRelationEntity::getProcessKey, processKey));
        //获取菜单列表
        List<TaProcessBizRelationEntity> menuList = entityList.stream()
                .filter(data -> TaProcessBizRelationEnum.BizRelationEnum.MENU.getVal().equals(data.getRelationType()))
                .collect(Collectors.toList());
        List<RelationVo> menuVoList = CrmBeanUtil.copyList(menuList, RelationVo.class);
        //获取组织列表
        List<TaProcessBizRelationEntity> orgList = entityList.stream()
                .filter(data -> TaProcessBizRelationEnum.BizRelationEnum.ORG.getVal().equals(data.getRelationType()))
                .collect(Collectors.toList());
        List<RelationVo> orgVoList = CrmBeanUtil.copyList(orgList, RelationVo.class);
        //封装返回结果Vo
        TaProcessBizRelationRespVo respVo = new TaProcessBizRelationRespVo();
        respVo.setProcessKey(entityList.get(0).getProcessKey());
        respVo.setMenuList(menuVoList);
        respVo.setOrgList(orgVoList);
        return respVo;
    }

    /**
     * 查出所有的流程关系信息
     *
     * @param taProcessBizRelationReqVo 接收前端参数
     * @return java.util.List<com.biz.crm.act.vo.TaProcessBizRelationRespVo>
     * @method getListRelation
     * @date: 2020/12/18 18:52
     * @author: YuanZiJian
     */
    @Override
    public List<TaProcessBizRelationRespVo> getListRelation(TaProcessBizRelationReqVo taProcessBizRelationReqVo) {
        List<TaProcessBizRelationEntity> entities = this.list(Wrappers.lambdaQuery(TaProcessBizRelationEntity.class)
                .eq(TaProcessBizRelationEntity::getProcessKey, taProcessBizRelationReqVo.getProcessKey()));
        if(CollectionUtils.isEmpty(entities)){
            return Lists.newArrayList();
        }
        //分组
        List<TaProcessBizRelationRespVo> result = new ArrayList<>();
        Map<String, List<TaProcessBizRelationEntity>> listGroupByProcessKey = entities.stream().collect(Collectors.groupingBy(TaProcessBizRelationEntity::getProcessKey));
        Set<String> keySet = listGroupByProcessKey.keySet();
        keySet.stream().forEach(data ->
        {
            TaProcessBizRelationRespVo respVo = new TaProcessBizRelationRespVo();
            respVo.setProcessKey(data);
            List<TaProcessBizRelationEntity> menuEntityList = listGroupByProcessKey.get(data).stream().filter(entity -> TaProcessBizRelationEnum.BizRelationEnum.MENU.getVal().equals(entity.getRelationType())).collect(Collectors.toList());
            List<RelationVo> menuList = CrmBeanUtil.copyList(menuEntityList, RelationVo.class);
            List<TaProcessBizRelationEntity> orgEntityList = listGroupByProcessKey.get(data).stream().filter(entity -> TaProcessBizRelationEnum.BizRelationEnum.ORG.getVal().equals(entity.getRelationType())).collect(Collectors.toList());
            List<RelationVo> orgList = CrmBeanUtil.copyList(orgEntityList, RelationVo.class);
            respVo.setMenuList(menuList);
            respVo.setOrgList(orgList);
            if (CollectionUtil.listNotEmptyNotSizeZero(orgList) && StringUtils.isNotEmpty(orgList.get(0).getIsOnlyCurrentOrg())) {
                respVo.setIsOnlyCurrentOrg(orgList.get(0).getIsOnlyCurrentOrg());
            } else {
                respVo.setIsOnlyCurrentOrg(GlobalWhetherEnum.YES.getCode());
            }
            result.add(respVo);
        });
        return result;
    }

    /**
     * 根据processKey删除对应Relation(逻辑删除)
     *
     * @return void
     * @method deleteRelation
     * @date: 2020/12/18 18:58
     * @author: YuanZiJian
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteRelation(String processKey) {
        AssertUtils.isNotEmpty(processKey,"流程编码[processKey]不能为空");
        //直接物理删除
        mapper.delete(Wrappers.lambdaQuery(TaProcessBizRelationEntity.class).eq(TaProcessBizRelationEntity::getProcessKey,processKey));
    }

    /**
     * 新增业务关系
     *
     * @param reqVo 接收参数的对象Vo
     * @return void
     * @method insertRelation
     * @date: 2020/12/15 17:07
     * @author: YuanZiJian
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void insertRelation(TaProcessBizRelationReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getProcessKey(),"流程编码[processKey]不能为空");
        //校验processKey对应的流程模型是否存在
        ActReModelEntity actReModelEntity = actReModelMapper.selectOne(Wrappers.lambdaQuery(ActReModelEntity.class).eq(ActReModelEntity::getKey, reqVo.getProcessKey()));
        AssertUtils.isNotNull(actReModelEntity,"流程编码:"+reqVo.getProcessKey()+"对应的流程数据不存在,请刷新页面");
        AssertUtils.isNotEmpty(reqVo.getMenuList(),"请选择可选页面[menuList]");
        AssertUtils.isNotEmpty(reqVo.getOrgList(),"请选择可选组织[orgList]");
        AssertUtils.isTrue(reqVo.getMenuList().size()== reqVo.getMenuList().stream().map(RelationVo::getBizCode).collect(Collectors.toSet()).size(),"请勿选择重复的可选页面菜单编码");
        AssertUtils.isTrue(reqVo.getOrgList().size()== reqVo.getOrgList().stream().map(RelationVo::getBizCode).collect(Collectors.toSet()).size(),"请勿选择重复的可选组织编码");
        //先删除
        this.remove(Wrappers.lambdaQuery(TaProcessBizRelationEntity.class)
                .eq(TaProcessBizRelationEntity::getProcessKey, reqVo.getProcessKey()));
        //转换需要保存的实体集合
        List<TaProcessBizRelationEntity> entityList = this.getEntityList(reqVo);
        this.saveOrUpdateBatch(entityList);
    }

    /**
     * 封装前端参数为 List<TaProcessBizRelationEntity>
     *
     * @param reqVo
     * @return java.util.List<com.biz.crm.act.model.TaProcessBizRelationEntity>
     * @method getEntityList
     * @date: 2020/12/15 17:35
     * @author: YuanZiJian
     */
    private List<TaProcessBizRelationEntity> getEntityList(TaProcessBizRelationReqVo reqVo) {
        List<TaProcessBizRelationEntity> saveList = Lists.newArrayList();
        List<String> needRemoveIds = Lists.newArrayList();
        Set<String> set = Sets.newHashSet();
        //先把数据库已经存在的关系查出来
        List<TaProcessBizRelationEntity> entities = mapper.selectList(Wrappers.lambdaQuery(TaProcessBizRelationEntity.class).eq(TaProcessBizRelationEntity::getProcessKey, reqVo.getProcessKey()));
        Map<String,TaProcessBizRelationEntity> map = Maps.newHashMap();
        if(CollectionUtils.isNotEmpty(entities)){
            entities.stream().forEach(o->{
                map.put(o.getBizCode()+o.getRelationType(),o);
            });
        }
        reqVo.getMenuList().forEach(o->{
            AssertUtils.isNotEmpty(o.getBizCode(),"可选页面的菜单编码不能为空");
            set.add(o.getBizCode()+TaProcessBizRelationEnum.BizRelationEnum.MENU.getVal());
            TaProcessBizRelationEntity entity = map.get(o.getBizCode() + TaProcessBizRelationEnum.BizRelationEnum.MENU.getVal());
            if(Objects.isNull(entity)){
                entity=new TaProcessBizRelationEntity();
                entity.setProcessKey(reqVo.getProcessKey());
                entity.setRelationType(TaProcessBizRelationEnum.BizRelationEnum.MENU.getVal());
                entity.setIsOnlyCurrentOrg(reqVo.getIsOnlyCurrentOrg());
                entity.setBizCode(o.getBizCode());
                entity.setBizName(o.getBizName());
            }else {
                entity.setIsOnlyCurrentOrg(Optional.ofNullable(reqVo.getIsOnlyCurrentOrg()).orElse(GlobalWhetherEnum.NO.getCode()));
                entity.setBizName(o.getBizName());
            }
            saveList.add(entity);
        });
        reqVo.getOrgList().forEach(o->{
            AssertUtils.isNotEmpty(o.getBizCode(),"可选组织编码不能为空");
            set.add(o.getBizCode()+TaProcessBizRelationEnum.BizRelationEnum.ORG.getVal());
            TaProcessBizRelationEntity entity = map.get(o.getBizCode() + TaProcessBizRelationEnum.BizRelationEnum.ORG.getVal());
            if(Objects.isNull(entity)){
                entity=new TaProcessBizRelationEntity();
                entity.setProcessKey(reqVo.getProcessKey());
                entity.setRelationType(TaProcessBizRelationEnum.BizRelationEnum.ORG.getVal());
                entity.setIsOnlyCurrentOrg(reqVo.getIsOnlyCurrentOrg());
                entity.setBizCode(o.getBizCode());
                entity.setBizName(o.getBizName());
            }else {
                entity.setIsOnlyCurrentOrg(Optional.ofNullable(reqVo.getIsOnlyCurrentOrg()).orElse(GlobalWhetherEnum.NO.getCode()));
                entity.setBizName(o.getBizName());
            }
            saveList.add(entity);
        });
        //哪些需要被删除
        entities.forEach(o->{
            if(!set.contains(o.getBizCode()+o.getRelationType())){
                needRemoveIds.add(o.getId());
            }
        });
        if(CollectionUtils.isNotEmpty(needRemoveIds)){
            mapper.deleteBatchIds(needRemoveIds);
        }
        return saveList;
    }

    /**
     * 根据当前登录人和菜单编码查找对应的已发布的流程
     * @return
     */
    @Override
    public SubmitActivitiResp findProcessByCurrentUserAndMenu() {
        SubmitActivitiResp submitActivitiResp = new SubmitActivitiResp();
        UserRedis userRedis = UserUtils.getUser();
        submitActivitiResp.setApplierName(userRedis.getUsername());
        submitActivitiResp.setOrgName(userRedis.getOrgname());
        //根据菜单编码和组织编码，获取对应的流程名字和流程key
        Object obj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        AssertUtils.isNotNull(obj, "未获取到菜单编码");
        String menuCode=String.valueOf(obj);
        AssertUtils.isNotEmpty(userRedis.getOrgcode(), "登录人组织不能为空");

        //查询出当前菜单可以选择的所有流程业务关系数据
        List<TaProcessBizRelationEntity> menuRelationEntities = mapper.selectList(Wrappers.lambdaQuery(TaProcessBizRelationEntity.class).eq(TaProcessBizRelationEntity::getBizCode,menuCode).eq(TaProcessBizRelationEntity::getRelationType, TaProcessBizRelationEnum.BizRelationEnum.MENU.getVal()));
        if(CollectionUtils.isEmpty(menuRelationEntities)){
            return submitActivitiResp;
        }
        //查询流程和组织的关系
        Set<String> processKeys = menuRelationEntities.stream().map(TaProcessBizRelationEntity::getProcessKey).collect(Collectors.toSet());
        List<TaProcessBizRelationEntity> orgRelationEntities = mapper.selectList(Wrappers.lambdaQuery(TaProcessBizRelationEntity.class).in(TaProcessBizRelationEntity::getProcessKey,processKeys).eq(TaProcessBizRelationEntity::getRelationType,TaProcessBizRelationEnum.BizRelationEnum.ORG.getVal()));
        //查找所有需要用到的组织做上下级判断
        Set<String> orgCodeSet = Sets.newHashSet();
        orgCodeSet.add(userRedis.getOrgcode());
        orgRelationEntities.stream().forEach(o->{
            orgCodeSet.add(o.getBizCode());
        });
        List<MdmOrgRespVo> orgList = OrgUtil.getOrgByCodeList(Lists.newArrayList(orgCodeSet));
        Map<String, MdmOrgRespVo> orgMap = orgList.stream().collect(Collectors.toMap(MdmOrgRespVo::getOrgCode, Function.identity()));
        Set<String> canUseProcessKeySet=Sets.newHashSet();
        MdmOrgRespVo loginUserOrg = orgMap.get(userRedis.getOrgcode());
        orgRelationEntities.forEach(o->{
            //使用层级编码判断是否可以用当前的流程
            String ruleCode = loginUserOrg.getRuleCode();
            MdmOrgRespVo orgRespVo = orgMap.get(o.getBizCode());
            //如果仅当前组织,判断值是否相等,否则使用降维编码判断
            if(org.apache.commons.lang3.StringUtils.equals(GlobalWhetherEnum.YES.getCode(),o.getIsOnlyCurrentOrg())){
                if(org.apache.commons.lang3.StringUtils.equals(loginUserOrg.getOrgCode(),o.getBizCode())){
                    canUseProcessKeySet.add(o.getProcessKey());
                }
            }else {
                if(org.apache.commons.lang3.StringUtils.contains(o.getBizCode(),loginUserOrg.getOrgCode())){
                    canUseProcessKeySet.add(o.getProcessKey());
                }
            }
        });
        if(CollectionUtils.isEmpty(canUseProcessKeySet)){
            return submitActivitiResp;
        }
        //根据流程key集合查询所有已发布符合条件的流程
        List<ActReModelEntity> processList = actReModelMapper.selectList(Wrappers.lambdaQuery(ActReModelEntity.class).in(ActReModelEntity::getKey, canUseProcessKeySet).eq(ActReModelEntity::getProcessStatus, Indicator.PROCESS_STATE_DEPLOY.getCode()));
        List<SubmitActivitiKeyDetail> detailList = processList.stream().map(o -> {
            SubmitActivitiKeyDetail submitActivitiKeyDetail = new SubmitActivitiKeyDetail();
            submitActivitiKeyDetail.setProcessKey(o.getKey());
            submitActivitiKeyDetail.setProcessName(o.getName());
            return submitActivitiKeyDetail;
        }).collect(Collectors.toList());
        submitActivitiResp.setDetails(detailList);
        return submitActivitiResp;
    }
}
