package com.biz.crm.design.service.impl;

import com.biz.crm.activiti.entity.ActReModelEntity;
import com.biz.crm.activiti.mapper.ActReModelMapper;
import com.biz.crm.base.BusinessException;
import com.biz.crm.design.service.DesignService;
import com.biz.crm.design.vo.NewModelVo;
import com.biz.crm.design.vo.SaveModelVo;
import com.biz.crm.eunm.activiti.Indicator;
import com.biz.crm.nebular.activiti.vo.ActReModel;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.UUIDGenerator;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.extern.slf4j.Slf4j;
import org.activiti.editor.constants.ModelDataJsonConstants;
import org.activiti.engine.ActivitiException;
import org.activiti.engine.RepositoryService;
import org.activiti.engine.repository.Model;
import org.apache.batik.transcoder.TranscoderInput;
import org.apache.batik.transcoder.TranscoderOutput;
import org.apache.batik.transcoder.image.PNGTranscoder;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.Objects;

/**
 * @author jianglong
 * @version V1.0
 * @Package com.biz.crm.design.service.impl
 * @Description: TODO
 * @date 2021/4/30 上午10:58
 */
@Service
@Slf4j
public class DesignServiceImpl implements DesignService {
    @Resource
    private ActReModelMapper actReModelMapper;

    @Resource
    private RepositoryService repositoryService;

    @Autowired
    private ObjectMapper objectMapper;

    @Override
    public ActReModel getByNameOrKey(String name, String key) {
        return actReModelMapper.getByNameOrKey(name, key);
    }

    /**
     * 新建一个空模型
     *
     * @param newModelVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public String newModel(NewModelVo newModelVo) {
        //初始化一个空模型
        Model model = repositoryService.newModel();
        //设置一些默认信息
        String name = newModelVo.getName();
        String description = "";
        int revision = 1;
        String key = newModelVo.getKey();
        ActReModel actReModel = this.getByNameOrKey(name, key);
        AssertUtils.isTrue(Objects.isNull(actReModel), "名字或者流程key重复[" + name + "][" + key + "]");

        ObjectNode modelNode = objectMapper.createObjectNode();
        modelNode.put(ModelDataJsonConstants.MODEL_NAME, name);
        modelNode.put(ModelDataJsonConstants.MODEL_DESCRIPTION, description);
        modelNode.put(ModelDataJsonConstants.MODEL_REVISION, revision);

        model.setName(name);
        model.setKey(key);
        model.setMetaInfo(modelNode.toString());
        //设置流程状态为未发布
        model.setVersion(1);

        repositoryService.saveModel(model);
        String id = model.getId();

        //完善ModelEditorSource
        ObjectNode editorNode = objectMapper.createObjectNode();
        editorNode.put("id", "canvas");
        editorNode.put("resourceId", "canvas");
        ObjectNode stencilSetNode = objectMapper.createObjectNode();
        stencilSetNode.put("namespace",
                "http://b3mn.org/stencilset/bpmn2.0#");
        editorNode.put("stencilset", stencilSetNode);
        try {
            //默认设置流程基础信息
            ObjectNode jsonObject = objectMapper.createObjectNode();
            jsonObject.put("process_id", key);
            jsonObject.put("name", name);
            jsonObject.put("documentation", "");
            jsonObject.put("process_author", "");
            jsonObject.put("process_version", "");
            jsonObject.put("process_namespace", "http://www.activiti.org/processdef");
            jsonObject.put("eventlisteners", "");
            jsonObject.put("signaldefinitions", "");
            jsonObject.put("messagedefinitions", "");
            setEndListener(jsonObject);
            editorNode.put("properties", jsonObject);
            setStartAndEnd(editorNode);
            setbounds(editorNode);
            editorNode.put("ssextensions",objectMapper.createArrayNode());
            log.info(editorNode.toString());
            repositoryService.addModelEditorSource(id, editorNode.toString().getBytes("utf-8"));
        } catch (UnsupportedEncodingException e) {
            log.error("{}", e);
            throw new BusinessException("调用工作流新建模型方法异常");
        }
        //修改流程发布状态
        ActReModelEntity entity = actReModelMapper.selectById(id);
        entity.setProcessStatus(Indicator.PROCESS_STATE_EDIT.getCode());
        actReModelMapper.updateById(entity);
        return id;
    }

    private void setbounds(ObjectNode jsonObject){
        ObjectNode bounds = objectMapper.createObjectNode();
        ObjectNode lowerRight = objectMapper.createObjectNode();
        lowerRight.put("x",1200);
        lowerRight.put("y",1050);
        ObjectNode upperLeft = objectMapper.createObjectNode();
        upperLeft.put("x",0);
        upperLeft.put("y",0);
        bounds.put("lowerRight",lowerRight);
        bounds.put("upperLeft",upperLeft);
        jsonObject.put("bounds",bounds);
    }

    //childShapes
    public void setStartAndEnd(ObjectNode jsonObject){
        ArrayNode childShapes =  objectMapper.createArrayNode();
        ObjectNode start1 = createStart();
        ObjectNode end = createEnd();
        childShapes.add(start1);
        childShapes.add(end);
        jsonObject.put("childShapes",childShapes);
    }

    /**
     * 创建开始节点
     * @return
     */
    private ObjectNode createStart(){
        ObjectNode start1 = objectMapper.createObjectNode();
        start1.put("resourceId", UUIDGenerator.generate());
        ObjectNode properties = objectMapper.createObjectNode();
        properties.put("overrideid","start1");
        properties.put("name","开始");
        properties.put("documentation","");
        properties.put("executionlisteners","");
        properties.put("initiator","");
        properties.put("formkeydefinition","");
        properties.put("formproperties","");
        start1.put("properties",properties);
        ObjectNode stencil = objectMapper.createObjectNode();
        stencil.put("id","StartNoneEvent");
        start1.put("stencil",stencil);
        start1.put("childShapes",objectMapper.createArrayNode());
        start1.put("outgoing",objectMapper.createArrayNode());
        ObjectNode bounds = objectMapper.createObjectNode();
        ObjectNode lowerRight = objectMapper.createObjectNode();
        lowerRight.put("x",210);
        lowerRight.put("y",90);
        ObjectNode upperLeft = objectMapper.createObjectNode();
        upperLeft.put("x",180);
        upperLeft.put("y",60);
        bounds.put("lowerRight",lowerRight);
        bounds.put("upperLeft",upperLeft);
        start1.put("bounds",bounds);
        start1.put("dockers",objectMapper.createArrayNode());
        return start1;
    }

    /**
     * 创建结束节点
     * @return
     */
    private ObjectNode createEnd(){
        ObjectNode end = objectMapper.createObjectNode();
        end.put("resourceId", UUIDGenerator.generate());
        ObjectNode properties = objectMapper.createObjectNode();
        properties.put("overrideid","end");
        properties.put("name","结束");
        properties.put("documentation","");
        properties.put("executionlisteners","");
        end.put("properties",properties);
        ObjectNode stencil = objectMapper.createObjectNode();
        stencil.put("id","EndNoneEvent");
        end.put("stencil",stencil);

        end.put("childShapes",objectMapper.createArrayNode());
        end.put("outgoing",objectMapper.createArrayNode());
        ObjectNode bounds = objectMapper.createObjectNode();
        ObjectNode lowerRight = objectMapper.createObjectNode();
        lowerRight.put("x",209);
        lowerRight.put("y",313);
        ObjectNode upperLeft = objectMapper.createObjectNode();
        upperLeft.put("x",181);
        upperLeft.put("y",285);
        bounds.put("lowerRight",lowerRight);
        bounds.put("upperLeft",upperLeft);
        end.put("bounds",bounds);
        end.put("dockers",objectMapper.createArrayNode());
        return end;
    }

    //设置设置监听器
    private void setEndListener(ObjectNode jsonObject){
        ObjectNode eventlisteners = objectMapper.createObjectNode();
        ArrayNode executionListeners =  objectMapper.createArrayNode();
        ObjectNode endListener = crateEndListener();
        executionListeners.add(endListener);
        eventlisteners.put("executionListeners",executionListeners);
        jsonObject.put("executionlisteners", eventlisteners);
    }

    //默认流程执行监听器
    private ObjectNode crateEndListener() {
        ObjectNode executionListener = objectMapper.createObjectNode();
        executionListener.put("event","end");
        executionListener.put("implementation","com.biz.crm.listener.GlobalExecutionListener");
        executionListener.put("className","com.biz.crm.listener.GlobalExecutionListener");
        executionListener.put("expression","");
        executionListener.put("delegateExpression","");
        return executionListener;
    }

    /**
     * 流程信息编辑保存
     *
     * @param saveModelVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateModel(SaveModelVo saveModelVo) {
        try {

            AssertUtils.isNotEmpty(saveModelVo.getModelId(), "流程模型id不能为空");
            Model model = repositoryService.getModel(saveModelVo.getModelId());

            ObjectNode modelJson = (ObjectNode) objectMapper.readTree(model.getMetaInfo());

            modelJson.put(ModelDataJsonConstants.MODEL_NAME, saveModelVo.getName());
            modelJson.put(ModelDataJsonConstants.MODEL_DESCRIPTION, saveModelVo.getDescription());
            model.setMetaInfo(modelJson.toString());
            model.setName(saveModelVo.getName());
            //修改转态为未发布
            model.setVersion(1);
            repositoryService.saveModel(model);

            repositoryService.addModelEditorSource(model.getId(), saveModelVo.getJson_xml().getBytes("utf-8"));

            InputStream svgStream = new ByteArrayInputStream(saveModelVo.getSvg_xml().getBytes("utf-8"));
            TranscoderInput input = new TranscoderInput(svgStream);

            PNGTranscoder transCoder = new PNGTranscoder();
            // Setup output
            ByteArrayOutputStream outStream = new ByteArrayOutputStream();
            TranscoderOutput output = new TranscoderOutput(outStream);

            // Do the transformation
            transCoder.transcode(input, output);
            final byte[] result = outStream.toByteArray();
            repositoryService.addModelEditorSourceExtra(model.getId(), result);
            outStream.close();
            //修改流程发布状态
            ActReModelEntity entity = actReModelMapper.selectById(saveModelVo.getModelId());
            entity.setProcessStatus(Indicator.PROCESS_STATE_EDIT.getCode());
            actReModelMapper.updateById(entity);

        } catch (Exception e) {
            log.error("Error saving model", e);
            throw new ActivitiException("Error saving model", e);
        }
    }

    /**
     * 根据流程模型id获取
     *
     * @param modelId
     * @return
     */
    @Override
    public ObjectNode getEditorJson(String modelId) {
        ObjectNode modelNode = null;

        Model model = repositoryService.getModel(modelId);

        if (model != null) {
            try {
                if (StringUtils.isNotEmpty(model.getMetaInfo())) {
                    modelNode = (ObjectNode) objectMapper.readTree(model.getMetaInfo());
                } else {
                    modelNode = objectMapper.createObjectNode();
                    modelNode.put(ModelDataJsonConstants.MODEL_NAME, model.getName());
                }
                modelNode.put(ModelDataJsonConstants.MODEL_ID, model.getId());
                ObjectNode editorJsonNode = (ObjectNode) objectMapper.readTree(
                        new String(repositoryService.getModelEditorSource(model.getId()), "utf-8"));
                modelNode.put("model", editorJsonNode);

            } catch (Exception e) {
                log.error("Error creating model JSON", e);
                throw new ActivitiException("Error creating model JSON", e);
            }
        }
        return modelNode;
    }
}
