package com.biz.crm.listener.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.ObjectUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.activiti.Indicator;
import com.biz.crm.exception.CommonException;
import com.biz.crm.listener.mapper.TaListenerMapper;
import com.biz.crm.listener.model.TaListenerEntity;
import com.biz.crm.listener.service.ITaListenerService;
import com.biz.crm.nebular.activiti.listener.req.TaListenerQueryVO;
import com.biz.crm.nebular.activiti.listener.req.TaListenerReqVo;
import com.biz.crm.nebular.activiti.listener.resp.TaListenerRespVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.PageUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;

/**
 * activiti监听器接口实现
 *
 * @author westion
 * @date 2020-12-03 09:46:08
 */
@Slf4j
@Service("taListenerService")
@ConditionalOnMissingBean(name="TaListenerServiceExpandImpl")
public class TaListenerServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<TaListenerMapper, TaListenerEntity> implements ITaListenerService {

    @Resource
    private TaListenerMapper taListenerMapper;


    @Autowired
    private ListenerHelper listenerHelper;

    /**
     * 列表
     *
     * @param reqVo 查询请求
     * @return page
     */
    @Override
    public PageResult<TaListenerRespVo> findList(TaListenerReqVo reqVo) {
        Page<TaListenerRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<TaListenerRespVo> list = taListenerMapper.findList(page, reqVo);
        list.forEach(l -> {
            listenerHelper.convertValue(l);
        });
        return PageResult.<TaListenerRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param id id
     * @return taListenerRespVo
     */
    @Override
    public TaListenerRespVo query(String id) {
        TaListenerEntity entity = this.taListenerMapper.selectById(id);
        if (ObjectUtils.isNotEmpty(entity)) {
            TaListenerRespVo taListenerRespVo = CrmBeanUtil.copy(entity, TaListenerRespVo.class);
            return listenerHelper.convertValue(taListenerRespVo);
        }
        return null;
    }

    /**
     * 新增
     *
     * @param reqVo 新增请求
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(TaListenerReqVo reqVo) {
        listenerHelper.saveCheck(reqVo);
        TaListenerEntity entity = CrmBeanUtil.copy(reqVo, TaListenerEntity.class);
        entity.setDefaultListener(0);
        this.save(entity);
    }

    /**
     * 更新
     *
     * @param reqVo 监听器实体
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(TaListenerReqVo reqVo) {
        AssertUtils.isNotNull(reqVo.getId(), "请传入需要更新的监听器编码！");
        listenerHelper.saveCheck(reqVo);
        TaListenerEntity entity = CrmBeanUtil.copy(reqVo, TaListenerEntity.class);
        this.updateById(entity);
    }

    /**
     * 删除
     *
     * @param ids id集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        List<TaListenerEntity> taListenerEntities = taListenerMapper.selectList(Wrappers.lambdaQuery(TaListenerEntity.class).in(TaListenerEntity::getId, ids));
        taListenerEntities.forEach(o->{
            AssertUtils.isTrue(!StringUtils.equals(CrmEnableStatusEnum.ENABLE.getCode(),o.getEnableStatus()),"启用的监听器不允许删除操作");
        });
        this.taListenerMapper.deleteBatchIds(ids);
    }

    /**
     * 条件查询
     *
     * @param queryVO
     * @return
     */
    @Override
    public List<TaListenerQueryVO> findExtListenerList(TaListenerQueryVO queryVO) {
        return this.taListenerMapper.findExtListenerList(queryVO);
    }

    /**
     * 启用
     *
     * @param ids 设置状态为启用
     */
    @Override
    public void enableBatch(List<String> ids) {
        //设置状态为启用
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        List<TaListenerEntity> entities = taListenerMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(entities)) {
            entities.forEach(o -> o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode()));
        }
        this.updateBatchById(entities);
    }

    /**
     * 设置状态为禁用
     *
     * @param ids id集合
     */
    @Override
    public void disableBatch(List<String> ids) {
        //设置状态为启用
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        List<TaListenerEntity> entities = taListenerMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(entities)) {
            entities.forEach(o -> o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode()));
        }
        this.updateBatchById(entities);
    }

    /**
     * 传入节点，查询尚未绑定该节点的监听器
     *
     * @param taListenerReqVo vo
     * @return 监听器列表
     */
    @Override
    public PageResult<TaListenerRespVo> queryBotBinding(TaListenerReqVo taListenerReqVo) {
        Page<TaListenerRespVo> page = PageUtil.buildPage(taListenerReqVo.getPageNum(), taListenerReqVo.getPageSize());
        List<TaListenerRespVo> list = taListenerMapper.queryNotBinding(page, taListenerReqVo);
        list.forEach(l -> {
            listenerHelper.convertValue(l);
        });
        return PageResult.<TaListenerRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }


}
