package com.biz.crm.activiti.service;

import com.biz.crm.activiti.entity.ActivitiBusinessEntity;
import com.biz.crm.design.util.WorkFlowGlobals;
import com.biz.crm.nebular.activiti.vo.CurrentVo;
import com.biz.crm.base.BusinessException;
import com.biz.crm.eunm.activiti.ActivitiOperateTypeEnum;
import com.biz.crm.util.CollectionUtil;
import lombok.extern.slf4j.Slf4j;
import org.activiti.engine.RepositoryService;
import org.activiti.engine.RuntimeService;
import org.activiti.engine.TaskService;
import org.activiti.engine.history.HistoricActivityInstance;
import org.activiti.engine.impl.RepositoryServiceImpl;
import org.activiti.engine.impl.persistence.entity.ProcessDefinitionEntity;
import org.activiti.engine.impl.persistence.entity.TaskEntity;
import org.activiti.engine.impl.pvm.PvmTransition;
import org.activiti.engine.impl.pvm.process.ActivityImpl;
import org.activiti.engine.runtime.ProcessInstance;
import org.activiti.engine.task.IdentityLink;
import org.activiti.engine.task.Task;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;

/**
 * @author jianglong
 * @version V1.0
 * @Package com.biz.crm.activiti.service
 * @Description: TODO
 * @date 2021/4/22 下午8:11
 */
@Service
@Slf4j
public class ActivityTaskUtil {
    @Autowired
    private ActivitiBusinessService activitiBusinessService;
    @Autowired
    private RuntimeService runtimeService;
    @Autowired
    private TaskService taskService;
    @Autowired
    private RepositoryService repositoryService;

    /**
     * 查询流程实例
     *
     * @return
     */
    public ProcessInstance getProcessInstance(String processInstanceId, String crmProcessInstanceId) {
        if (StringUtils.isEmpty(processInstanceId) && StringUtils.isEmpty(crmProcessInstanceId)) {
            throw new BusinessException("检查流程结束状态参数错误[参数不能全为空]");
        }
        if (StringUtils.isEmpty(processInstanceId)) {
            processInstanceId = activitiBusinessService.getByProcessInstanceId(crmProcessInstanceId).getProcessInstanceId();
        }
        ProcessInstance pi = runtimeService.createProcessInstanceQuery().processInstanceId(processInstanceId).singleResult();
        return pi;
    }

    public boolean checkEnd(String processInstanceId, String crmProcessInstanceId) {
        if (StringUtils.isEmpty(processInstanceId) && StringUtils.isEmpty(crmProcessInstanceId)) {
            throw new BusinessException("检查流程结束状态参数错误[参数不能全为空]");
        }
        if (StringUtils.isEmpty(processInstanceId)) {
            processInstanceId = activitiBusinessService.getByProcessInstanceId(crmProcessInstanceId).getProcessInstanceId();
        }
        ProcessInstance pi = runtimeService.createProcessInstanceQuery().processInstanceId(processInstanceId).singleResult();
        if (pi != null) {
            //未结束
            return false;
        } else {
            //已结束
            return true;
        }
    }

    /**
     * 查询当前处理人
     *
     * @param processInstanceId
     * @param crmProcessInstanceId
     * @return
     */
    public CurrentVo getCurrentVo(String processInstanceId, String crmProcessInstanceId) {
        if (StringUtils.isEmpty(processInstanceId) && StringUtils.isEmpty(crmProcessInstanceId)) {
            throw new BusinessException("检查流程结束状态参数错误[参数不能全为空]");
        }
        if (StringUtils.isEmpty(processInstanceId)) {
            processInstanceId = activitiBusinessService.getByProcessInstanceId(crmProcessInstanceId).getProcessInstanceId();
        }
        CurrentVo currentVo = null;
        List<Task> tasks = taskService.createTaskQuery().processInstanceId(processInstanceId).list();
        for (Task task : tasks) {
            List<IdentityLink> identityLinks = taskService.getIdentityLinksForTask(task.getId());
            if (CollectionUtil.listNotEmptyNotSizeZero(identityLinks)) {

                //职位
                Set<String> positionCodes = new HashSet<>();
                //流程角色
                Set<String> bpmRoleCodes = new HashSet<>();
                for (IdentityLink identityLink : identityLinks) {
                    if (StringUtils.isNotBlank(identityLink.getUserId())) {
                        positionCodes.add(identityLink.getUserId());
                    }
                    if (StringUtils.isNotBlank(identityLink.getGroupId())) {
                        bpmRoleCodes.add(identityLink.getGroupId());
                    }
                }
                if (!positionCodes.isEmpty() || !bpmRoleCodes.isEmpty()) {
                    currentVo = new CurrentVo();
                    currentVo.setBpmRoleCode(bpmRoleCodes);
                    currentVo.setPositionCode(positionCodes);
                }
            }
        }
        return currentVo;
    }

    /**
     * 检查是否重复提交 true 已经存在，false 不存在
     *
     * @return
     */
    public String checkRepeatCommit(String businessId) {
        List<String> bpmStatus = new ArrayList<>();
        bpmStatus.add(ActivitiOperateTypeEnum.COMMIT.getCode());
        List<ActivitiBusinessEntity> entities = activitiBusinessService.getListByBusinessIdOrNum(businessId, bpmStatus);
        if (CollectionUtil.listNotEmptyNotSizeZero(entities) && entities.size() > 1) {
            //正在进行中的流程超过1个，则抛出异常
            throw new BusinessException("该数据在流程[" + entities.get(0).getCrmProcessInstanceId() + "]审批中,请不要重复提交==>[" + businessId + "]");
        } else if (CollectionUtil.listNotEmptyNotSizeZero(entities) && entities.size() == 1) {
            //有一个正在进行中的流程，返回该流程的编码
            return entities.get(0).getCrmProcessInstanceId();
        } else {
            //没有正在进行中流程，返回null
            return null;
        }
    }

    /**
     * 获取需要高亮的线
     *
     * @param processDefinitionEntity
     * @param historicActivityInstances
     * @return
     */
    public List<String> getHighLightedFlows(
            ProcessDefinitionEntity processDefinitionEntity,
            List<HistoricActivityInstance> historicActivityInstances) {
        List<String> highFlows = new ArrayList<String>();// 用以保存高亮的线flowId
        for (int i = 0; i < historicActivityInstances.size() - 1; i++) {// 对历史流程节点进行遍历
            ActivityImpl activityImpl = processDefinitionEntity
                    .findActivity(historicActivityInstances.get(i)
                            .getActivityId());// 得到节点定义的详细信息
            List<ActivityImpl> sameStartTimeNodes = new ArrayList<ActivityImpl>();// 用以保存后需开始时间相同的节点
            ActivityImpl sameActivityImpl1 = processDefinitionEntity
                    .findActivity(historicActivityInstances.get(i + 1)
                            .getActivityId());
            // 将后面第一个节点放在时间相同节点的集合里
            sameStartTimeNodes.add(sameActivityImpl1);
            for (int j = i + 1; j < historicActivityInstances.size() - 1; j++) {
                HistoricActivityInstance activityImpl1 = historicActivityInstances
                        .get(j);// 后续第一个节点
                HistoricActivityInstance activityImpl2 = historicActivityInstances
                        .get(j + 1);// 后续第二个节点
                if (activityImpl1.getStartTime().equals(
                        activityImpl2.getStartTime())) {
                    // 如果第一个节点和第二个节点开始时间相同保存
                    ActivityImpl sameActivityImpl2 = processDefinitionEntity
                            .findActivity(activityImpl2.getActivityId());
                    sameStartTimeNodes.add(sameActivityImpl2);
                } else {
                    // 有不相同跳出循环
                    break;
                }
            }
            List<PvmTransition> pvmTransitions = activityImpl
                    .getOutgoingTransitions();// 取出节点的所有出去的线
            for (PvmTransition pvmTransition : pvmTransitions) {
                // 对所有的线进行遍历
                ActivityImpl pvmActivityImpl = (ActivityImpl) pvmTransition
                        .getDestination();
                // 如果取出的线的目标节点存在时间相同的节点里，保存该线的id，进行高亮显示
                if (sameStartTimeNodes.contains(pvmActivityImpl)) {
                    highFlows.add(pvmTransition.getId());
                }
            }
        }
        return highFlows;
    }


    /**
     * 根据任务编号获取流程定义
     *
     * @param taskId
     * @return
     */
    public ProcessDefinitionEntity findProcessDefinitionEntityByTaskId(String taskId) {
        TaskEntity task = (TaskEntity) taskService.createTaskQuery().taskId(taskId).singleResult();
        // 取得流程定义
        ProcessDefinitionEntity processDefinition = (ProcessDefinitionEntity) ((RepositoryServiceImpl) repositoryService)
                .getDeployedProcessDefinition(task.getProcessDefinitionId());
        if (processDefinition == null) {
            throw new BusinessException("流程定义未找到!");
        }
        return processDefinition;
    }

    /**
     * 清空当前流向
     *
     * @param activityImpl
     * @return
     */
    public List<PvmTransition> clearTransition(ActivityImpl activityImpl) {
        // 存储当前节点所有流向临时变量
        List<PvmTransition> oriPvmTransitionList = new ArrayList<PvmTransition>();
        // 获取当前节点所有流向，存储到临时变量，然后清空
        List<PvmTransition> pvmTransitionList = activityImpl.getOutgoingTransitions();
        for (PvmTransition pvmTransition : pvmTransitionList) {
            oriPvmTransitionList.add(pvmTransition);
        }
        pvmTransitionList.clear();

        return oriPvmTransitionList;
    }

    /**
     * 还原以前流向
     *
     * @param activityImpl
     * @param oriPvmTransitionList
     */
    public void restoreTransition(ActivityImpl activityImpl, List<PvmTransition> oriPvmTransitionList) {
        // 清空现有流向
        List<PvmTransition> pvmTransitionList = activityImpl.getOutgoingTransitions();
        pvmTransitionList.clear();
        // 还原以前流向
        for (PvmTransition pvmTransition : oriPvmTransitionList) {
            pvmTransitionList.add(pvmTransition);
        }
    }

    /**
     * 设置流程操作状态
     *
     * @param variable
     * @param approveOpt
     */
    public void setbpmOperation(Map<String, Object> variable, String approveOpt) {
        variable.put(WorkFlowGlobals.BPM_OPERATION, approveOpt);
    }


}
