package com.biz.crm.activiti.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.activiti.entity.ActivitiBusinessEntity;
import com.biz.crm.activiti.mapper.ActivitiBusinessMapper;
import com.biz.crm.activiti.service.ActivitiBusinessService;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.activiti.ActivitiOperateTypeEnum;
import com.biz.crm.eunm.activiti.ProcessInstanceStatusEnum;
import com.biz.crm.nebular.activiti.vo.ActivitiBusinessVo;
import com.biz.crm.nebular.activiti.vo.ProcessInstanceQueryReqVo;
import com.biz.crm.nebular.activiti.vo.ProcessInstanceQueryRespVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionUserOrgRespVo;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.activiti.engine.runtime.ProcessInstance;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @author jianglong
 * @version V1.0
 * @Package com.biz.crm.activiti.service.impl
 * @Description: TODO
 * @date 2021/4/22 下午3:03
 */
@Service
@Slf4j
@ConditionalOnMissingBean(name = "ActivitiBusinessExpandServiceImpl")
public class ActivitiBusinessServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<ActivitiBusinessMapper, ActivitiBusinessEntity> implements ActivitiBusinessService {

    @Resource
    private ActivitiBusinessMapper activitiBusinessMapper;
    @Autowired
    private ActivitiBusinessHelper helper;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ActivitiBusinessEntity save(ProcessInstance processInstance, ActivitiBusinessVo activitiBusinessVo) {
        ActivitiBusinessEntity activitiBusinessEntity = CrmBeanUtil.copy(activitiBusinessVo, ActivitiBusinessEntity.class);
        activitiBusinessEntity.setRemarks(activitiBusinessVo.getRemark());
        activitiBusinessEntity.setCrmProcessInstanceId(CodeUtil.getCodeYyyymmddHHmm());
        activitiBusinessEntity.setProcessInstanceId(processInstance.getId());
        activitiBusinessEntity.setProcessVersionKey(processInstance.getProcessDefinitionId());
        activitiBusinessEntity.setBpmStatus(ActivitiOperateTypeEnum.COMMIT.getCode());
        activitiBusinessEntity.setProcessInstanceStatus(ProcessInstanceStatusEnum.ENABLE.getCode());
        activitiBusinessEntity.setCommitDate(DateUtil.date2Str(new Date(), DateUtil.datetimeFormat));
        this.save(activitiBusinessEntity);
        return activitiBusinessEntity;
    }

    @Override
    public ActivitiBusinessEntity getByProcessInstanceId(String crmProcessInstanceId) {
        List<ActivitiBusinessEntity> activitiBusinessEntities = this.list(Wrappers.lambdaQuery(ActivitiBusinessEntity.class)
                .eq(ActivitiBusinessEntity::getCrmProcessInstanceId, crmProcessInstanceId));
        AssertUtils.isNotEmpty(activitiBusinessEntities, "流程编号已不存在[" + crmProcessInstanceId + "]");
        return activitiBusinessEntities.get(0);
    }

    @Override
    public ActivitiBusinessEntity getByCrmProcessInstanceId(String processInstanceId) {
        List<ActivitiBusinessEntity> activitiBusinessEntities = this.list(Wrappers.lambdaQuery(ActivitiBusinessEntity.class)
                .eq(ActivitiBusinessEntity::getProcessInstanceId, processInstanceId));
        AssertUtils.isNotEmpty(activitiBusinessEntities, "流程实例已不存在[" + processInstanceId + "]");
        return activitiBusinessEntities.get(0);
    }

    @Override
    public List<ActivitiBusinessEntity> getByBusinessId(String businessId) {
        List<ActivitiBusinessEntity> activitiBusinessEntities = this.list(Wrappers.lambdaQuery(ActivitiBusinessEntity.class)
                .eq(ActivitiBusinessEntity::getBusinessId, businessId)
                .orderByDesc(ActivitiBusinessEntity::getCreateDateAll));
        return activitiBusinessEntities;
    }

    @Override
    public ActivitiBusinessEntity getActivitiBusiness(String crmProcessInstanceId, String processInstanceId) {
        if (StringUtils.isEmpty(crmProcessInstanceId) && StringUtils.isEmpty(processInstanceId)) {
            throw new BusinessException("参数不能全为空[crmProcessInstanceId][processInstanceId]");
        }
        if (StringUtils.isNotBlank(crmProcessInstanceId)) {
            List<ActivitiBusinessEntity> activitiBusinessEntities = this.list(Wrappers.lambdaQuery(ActivitiBusinessEntity.class)
                    .eq(ActivitiBusinessEntity::getCrmProcessInstanceId, crmProcessInstanceId));
            AssertUtils.isNotEmpty(activitiBusinessEntities, "流程编号已不存在[" + crmProcessInstanceId + "]");
            return activitiBusinessEntities.get(0);
        }
        if (StringUtils.isNotBlank(processInstanceId)) {
            List<ActivitiBusinessEntity> activitiBusinessEntities = this.list(Wrappers.lambdaQuery(ActivitiBusinessEntity.class)
                    .eq(ActivitiBusinessEntity::getProcessInstanceId, processInstanceId));
            AssertUtils.isNotEmpty(activitiBusinessEntities, "流程实例已不存在[" + processInstanceId + "]");
            return activitiBusinessEntities.get(0);
        }
        throw new BusinessException("逻辑异常");
    }

    @Override
    public List<ActivitiBusinessEntity> getListByBusinessIdOrNum(String businessIdOrNum, List<String> bpmStatus) {
        List<ActivitiBusinessEntity> activitiBusinessEntities = this.list(Wrappers.lambdaQuery(ActivitiBusinessEntity.class)
                .eq(ActivitiBusinessEntity::getBusinessId, businessIdOrNum)
                .in(CollectionUtils.isNotEmpty(bpmStatus), ActivitiBusinessEntity::getBpmStatus, bpmStatus)
        );
        return activitiBusinessEntities;
    }

    /**
     * 分页查询流程实例
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<ProcessInstanceQueryRespVo> processInstancePage(ProcessInstanceQueryReqVo reqVo) {
        Page<ProcessInstanceQueryRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<ProcessInstanceQueryRespVo> list = activitiBusinessMapper.processInstancePage(page, reqVo);
        if (CollectionUtils.isNotEmpty(list)) {
            list.forEach(o -> {
                helper.setCurrentPositionAndUserByProcessInstanceId(o);
                o.setBpmStatusName(ActivitiOperateTypeEnum.getStatusNameByCode(o.getBpmStatus()));
                o.setProcessInstanceStatusName(ProcessInstanceStatusEnum.getStatusNameByCode(o.getProcessInstanceStatus()));
                //计算总耗时
                String commitElapsedTime = null;
                if (StringUtils.equals(ActivitiOperateTypeEnum.COMMIT.getCode(), o.getBpmStatus())) {
                    //审批中结束时间取当前时间
                    commitElapsedTime = helper.computeProcessTotalHandleTime(o.getCreateDateAll(), DateUtil.date2Str(new Date(), DateUtil.datetimeFormat));
                } else {
                    //只要流程实例的审批状态不是审批中,结束时间就取更新时间作为结束时间
                    commitElapsedTime = helper.computeProcessTotalHandleTime(o.getCreateDateAll(), o.getUpdateDateAll());
                    o.setProcessEndDate(o.getUpdateDateAll());
                }
                o.setCommitElapsedTime(commitElapsedTime);
            });
        }
        return PageResult.<ProcessInstanceQueryRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ActivitiBusinessEntity updateBpmStatus(String processInstanceId, String bpmStatus, String processInstanceStatus) {
        log.info("修改流程状态");
        //校验状态值正确性
        AssertUtils.isNotEmpty(bpmStatus, "流程状态值不能为空");
        AssertUtils.isNotEmpty(processInstanceId, "流程实例id不能为空");
        boolean checkCodeExist = ActivitiOperateTypeEnum.checkCodeExist(bpmStatus);
        if (!checkCodeExist) {
            throw new BusinessException("流程状态编码值:" + bpmStatus + ",不存在");
        }
        List<ActivitiBusinessEntity> activitiBusinessEntities = this.list(Wrappers.lambdaQuery(ActivitiBusinessEntity.class)
                .eq(ActivitiBusinessEntity::getProcessInstanceId, processInstanceId));
        AssertUtils.isNotEmpty(activitiBusinessEntities, "流程实例已不存在[" + processInstanceId + "]");
        if (activitiBusinessEntities.size() > 1) {
            throw new BusinessException("流程实例重复，请联系程序管理员[" + processInstanceId + "]");
        }
        ActivitiBusinessEntity entity = activitiBusinessEntities.get(0);
        entity.setBpmStatus(bpmStatus);
        entity.setProcessInstanceStatus(processInstanceStatus);
        this.updateById(entity);
        return entity;
    }

    /**
     * 分页查询已办列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<ProcessInstanceQueryRespVo> taskDonePage(ProcessInstanceQueryReqVo reqVo) {
        UserRedis user = UserUtils.getUser();
        reqVo.setCreateCode(user.getUsername());
        Page<ProcessInstanceQueryRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        if(StringUtils.isNotEmpty(reqVo.getCommitBeginDate())){
            reqVo.setCommitBeginDate(reqVo.getCommitBeginDate()+" 00:00:00");
        }
        if(StringUtils.isNotEmpty(reqVo.getCommitEndDate())){
            reqVo.setCommitEndDate(reqVo.getCommitEndDate()+" 23:59:59");
        }
        if(StringUtils.isNotEmpty(reqVo.getDoneBeginDate())){
            reqVo.setDoneBeginDate(reqVo.getDoneBeginDate()+" 00:00:00");
        }
        if(StringUtils.isNotEmpty(reqVo.getDoneEndDate())){
            reqVo.setDoneEndDate(reqVo.getDoneEndDate()+" 23:59:59");
        }
        List<ProcessInstanceQueryRespVo> list = activitiBusinessMapper.taskDonePage(page, reqVo);
        if (CollectionUtils.isNotEmpty(list)) {
            list.forEach(o -> {
                o.setBpmStatusName(ActivitiOperateTypeEnum.getStatusNameByCode(o.getBpmStatus()));
                o.setProcessInstanceStatusName(ProcessInstanceStatusEnum.getStatusNameByCode(o.getProcessInstanceStatus()));
            });
        }
        return PageResult.<ProcessInstanceQueryRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }
}
