package com.biz.crm.activiti.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.activiti.entity.ActivitiBusinessEntity;
import com.biz.crm.activiti.entity.TaFuncRoleEntity;
import com.biz.crm.activiti.entity.TaNodeConfigEntity;
import com.biz.crm.activiti.mapper.ActReModelMapper;
import com.biz.crm.activiti.mapper.TaFuncRoleMapper;
import com.biz.crm.activiti.mapper.TaNodeConfigMapper;
import com.biz.crm.activiti.mapper.TaTaskCommunicateMapper;
import com.biz.crm.activiti.service.ITaNodeConfigService;
import com.biz.crm.activiti.util.XmlActivitiUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.activiti.IndicatorStr;
import com.biz.crm.listener.mapper.TaListenerProcMapper;
import com.biz.crm.listener.model.TaListenerProcEntity;
import com.biz.crm.nebular.activiti.design.req.TaNodeConfigReqVo;
import com.biz.crm.nebular.activiti.design.resp.OptBtnVO;
import com.biz.crm.nebular.activiti.design.resp.TaNodeConfigRespVo;
import com.biz.crm.nebular.activiti.start.req.ProcessInfoReqVO;
import com.biz.crm.nebular.activiti.vo.ActReModel;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.PageUtil;
import com.biz.crm.util.StringUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.activiti.engine.task.Task;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 接口实现
 *
 * @author jerry7
 * @date 2020-12-04 10:33:17
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="TaNodeConfigExpandServiceImpl")
public class TaNodeConfigServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<TaNodeConfigMapper, TaNodeConfigEntity> implements ITaNodeConfigService {

    @Resource
    private TaNodeConfigMapper taNodeConfigMapper;

    @Autowired
    private ActivitiBusinessHelper helper;
    @Resource
    private TaFuncRoleMapper funcRoleMapper;
    @Resource
    private TaTaskCommunicateMapper communicateMapper;


    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<TaNodeConfigRespVo> findList(TaNodeConfigReqVo reqVo) {
        Page<TaNodeConfigRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<TaNodeConfigRespVo> list = taNodeConfigMapper.findList(page, reqVo);
        return PageResult.<TaNodeConfigRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param id 查询id
     * @return taNodeConfigRespVo
     */
    @Override
    public TaNodeConfigRespVo query(String id) {
        TaNodeConfigEntity entity = this.taNodeConfigMapper.selectById(id);
        AssertUtils.isNotNull(entity,"节点数据不存在");
        TaNodeConfigRespVo configRespVo = CrmBeanUtil.copy(entity, TaNodeConfigRespVo.class);
        if(org.apache.commons.lang3.StringUtils.isNotEmpty(configRespVo.getBtnAuthRoleId())){
            TaFuncRoleEntity taFuncRoleEntity = funcRoleMapper.selectById(configRespVo.getBtnAuthRoleId());
            if(Objects.nonNull(taFuncRoleEntity)){
                configRespVo.setBtnAuthRoleName(taFuncRoleEntity.getRoleName());
            }
        }
        return configRespVo;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(TaNodeConfigReqVo reqVo) {
        TaNodeConfigEntity entity = CrmBeanUtil.copy(reqVo, TaNodeConfigEntity.class);
        this.save(entity);
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(TaNodeConfigReqVo reqVo) {
        TaNodeConfigEntity entity = this.getById(reqVo.getId());
        AssertUtils.isNotNull(entity,"节点数据不存在");
        TaNodeConfigEntity copy = CrmBeanUtil.copy(reqVo, TaNodeConfigEntity.class);
        this.saveOrUpdate(copy);

    }

    /**
     * 删除
     *
     * @param ids id集合
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        List<TaNodeConfigEntity> taNodeConfigEntities = taNodeConfigMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(taNodeConfigEntities)) {
            this.taNodeConfigMapper.deleteBatchIds(ids);
        }
        this.updateBatchById(taNodeConfigEntities);
    }

    /***
     * 初始化流程节点配置
     * 如果某一个KEY对应的流程是第一次发布，解析XML文件获得TASK，并保存初始化配置
     * @param taProcess
     * @param versionKey
     */
//    @Override
//    public void initNodeConfig(ActReModel actReModel, String versionKey) {
//        List<TaNodeConfigEntity> configEntities = null;
//        try {
//            configEntities = XmlActivitiUtil.getNodes(taProcess.getProcessXml());
//        } catch (Exception e) {
//            e.printStackTrace();
//            throw new BusinessException("解析xml获取task失败!");
//        }
//        if (CollectionUtils.isNotEmpty(configEntities)) {
//            for (TaNodeConfigEntity configEntity : configEntities) {
//                configEntity.setProcessNodeId(actReModel.getKey_() + ":" + configEntity.getProcessNodeCode());
//                configEntity.setProcessVersionKey(versionKey);
//                taNodeConfigMapper.insert(configEntity);
//            }
//        }
//    }

    /**
     * 根据流程版本号查询所有的配置
     *
     * @param processVersionKey
     * @return
     */
    @Override
    public List<TaNodeConfigEntity> findNodeByProcessVersionKey(String processVersionKey) {
        return taNodeConfigMapper.findNodeByProcessVersionKey(processVersionKey);
    }


    /**
     * 根据processkey和processVersionKey获取节点列表
     *
     * @param taNodeConfigReqVo
     * @return
     */
    @Override
    public PageResult<TaNodeConfigRespVo> queryByProcessKey(TaNodeConfigReqVo taNodeConfigReqVo) {
        if (StringUtils.isNotEmpty(taNodeConfigReqVo.getProcessVersionKey())) {
            return findList(taNodeConfigReqVo);
        } else {
            String processVersionKey = taNodeConfigMapper.getMaxProcessVersionByKey(taNodeConfigReqVo.getProcessKey());
            if (StringUtils.isEmpty(processVersionKey)) {
                return PageResult.<TaNodeConfigRespVo>builder()
                        .data(Lists.newArrayList())
                        .count((long) 0)
                        .build();
            }
            taNodeConfigReqVo.setProcessVersionKey(processVersionKey);
            return findList(taNodeConfigReqVo);
        }

    }

    /**
     * 跟格流程版本KEY和节点KEY搜索节点
     *
     * @param processVersionKey
     * @param taskDefKey
     * @return
     */
    @Override
    public TaNodeConfigEntity findNodeByTaskVersionKey(String processVersionKey, String taskDefKey) {
        return taNodeConfigMapper.findNodeByTaskVersionKey(processVersionKey, taskDefKey);
    }

    /**
     * 将节点配置List转成Map，节点KEY做KEY
     * @param configList
     * @return
     */
    @Override
    public Map<String, TaNodeConfigEntity> configListToMap( List<TaNodeConfigEntity> configList){
        if(CollectionUtils.isNotEmpty(configList)){
            Map<String, TaNodeConfigEntity> configMap = new HashMap<>();
            configList.forEach(config->{
                configMap.put(config.getProcessNodeCode(), config);
            });
            return configMap;
        }
        return null;
    }

    /**
     * 获取流程的历史版本ProcessVersion
     * @param processKey
     * @return
     */
    @Override
    public List<TaNodeConfigRespVo> getAllProcessVersionKey(String processKey) {
        AssertUtils.isNotEmpty(processKey,"流程key[processKey]不能为空");
        return taNodeConfigMapper.getAllProcessVersionByKey(processKey);
    }

    /**
     * 获取流程对应的按钮权限数据
     * @param reqVo
     * @return
     */
    @Override
    public List<OptBtnVO> getButtonsByCrmProcessInstanceId(ProcessInfoReqVO reqVo) {
        AssertUtils.isNotEmpty(reqVo.getProcessNo(), "流程编码[processNo]不能为空");
        ActivitiBusinessEntity businessEntity = helper.findActivitiBusinessEntityByCrmInstanceId(reqVo.getProcessNo());
        List<OptBtnVO> respVos=Lists.newArrayList();

        //抄送和传阅获取时固定指给一个已阅的按钮
        if(StringUtils.isNotEmpty(reqVo.getCopyTaskId())){
            OptBtnVO btnVO = new OptBtnVO();
            btnVO.setBtnCode(IndicatorStr.PROCESS_BTN_ISSUE.getCode());
            btnVO.setBtnName(IndicatorStr.PROCESS_BTN_ISSUE.getLabel());
            respVos.add(btnVO);
            return respVos;
        }
        if(StringUtils.isNotEmpty(reqVo.getCommunicateId())){
            OptBtnVO btnVO = new OptBtnVO();
            btnVO.setBtnCode(IndicatorStr.PROCESS_BTN_REPLAY_MSG.getCode());
            btnVO.setBtnName(IndicatorStr.PROCESS_BTN_REPLAY_MSG.getLabel());
            respVos.add(btnVO);
            return respVos;
        }
        //
        AssertUtils.isNotNull(businessEntity,"流程不存在:"+reqVo.getProcessNo());
        //查询当前流程实例的任务集合
        if(StringUtils.isNotEmpty(reqVo.getTaskId())){
            Task currentTask = helper.findCurrentTaskByTaskId(reqVo.getTaskId());
            if(Objects.nonNull(currentTask)){
                List<TaNodeConfigEntity> nodeConfigEntities = helper.findNodesByProcessVersionKey(businessEntity.getProcessVersionKey());
                List<TaNodeConfigEntity> currentNodes = nodeConfigEntities.stream().filter(o -> org.apache.commons.lang3.StringUtils.equals(currentTask.getTaskDefinitionKey(),o.getProcessNodeCode())).collect(Collectors.toList());
                if(CollectionUtils.isEmpty(currentNodes)){
                    return respVos;
                }
                TaNodeConfigEntity nodeConfigEntity = currentNodes.get(0);
                if(StringUtils.isEmpty(nodeConfigEntity.getBtnAuthRoleId())){
                    return respVos;
                }
                TaFuncRoleEntity funcRoleEntity = funcRoleMapper.selectById(nodeConfigEntity.getBtnAuthRoleId());
                List<OptBtnVO> optBtnVOS = helper.parseBtnList(funcRoleEntity.getRoleAuths());
                respVos.addAll(optBtnVOS);
            }
        }
        return respVos;
    }
}
