package com.biz.crm.listener.service.impl;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.base.BusinessException;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.activiti.Column;
import com.biz.crm.eunm.activiti.ColumnStr;
import com.biz.crm.listener.mapper.TaListenerMapper;
import com.biz.crm.listener.model.TaListenerEntity;
import com.biz.crm.nebular.activiti.listener.req.TaListenerProcReqVo;
import com.biz.crm.nebular.activiti.listener.req.TaListenerQueryVO;
import com.biz.crm.nebular.activiti.listener.req.TaListenerReqVo;
import com.biz.crm.nebular.activiti.listener.resp.TaListenerRespVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;

/**
 * 监听器校验工具类
 *
 * @author jerry7
 * @date 2020/12/3 15:42
 */
@Slf4j
@Service
public class ListenerHelper {

    @Resource
    private TaListenerMapper taListenerMapper;

    /**
     * 监听器保存校验
     *
     * @param vo 监听器实体
     */
    public void saveCheck(TaListenerReqVo vo) {
        AssertUtils.isNotEmpty(vo.getListenerName(), "监听器名称不能为空!");
        AssertUtils.isNotEmpty(vo.getListenerEvent(), "监听方法不能为空！");
        AssertUtils.isNotEmpty(vo.getListenerValue(), "监听器路径不能为空！");
        AssertUtils.isNotEmpty(vo.getListenerType(), "监听器类型不能为空！");
        this.checkListenerValue(vo.getListenerValue());
        List<TaListenerEntity> taListenerEntities = taListenerMapper.selectList(Wrappers.lambdaQuery(TaListenerEntity.class).eq(TaListenerEntity::getListenerValue,vo.getListenerValue()).ne(StringUtils.isNotEmpty(vo.getId()),TaListenerEntity::getId,vo.getId()));
        AssertUtils.isTrue(CollectionUtils.isEmpty(taListenerEntities),"当前监听器已经存在,请勿重复添加");

        //默认启用
        if(StringUtils.isEmpty(vo.getEnableStatus())){
            vo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        }
    }

    /**
     * 校验监听器的类路径是否合法
     * @param listenerValue
     */
    public void checkListenerValue(String listenerValue){
        try {
            Class<?> listenerClass = Class.forName(listenerValue);
        } catch (ClassNotFoundException e) {
            throw new BusinessException("请填写正确的监听器类路径");
        }
    }
    /**
     * 监听器节点绑定保存校验
     *
     * @param reqVo 监听器实体
     */
    public void procSaveCheck(TaListenerProcReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getListenerId(), "监听器ID不能为空！");
        AssertUtils.isNotEmpty(reqVo.getNodeCode(), "节点ID不能为空！");
        AssertUtils.isNotEmpty(reqVo.getProcessVersionKey(), "流程版本不能为空！");
        //检查监听器是否存在
        AssertUtils.isNotNull(taListenerMapper.selectById(reqVo.getListenerId()), reqVo.getListenerId() + "指向的监听器不存在！");
    }


    /**
     * 转换返回的监听器状态，监听器类型，监听器方法
     *
     * @param taListenerRespVo 监听器实体
     * @return 带状态返回值的监听器实体
     */
    public TaListenerRespVo convertValue(TaListenerRespVo taListenerRespVo) {
        taListenerRespVo.setEnableStatusName(CrmEnableStatusEnum.getDesc(taListenerRespVo.getEnableStatus()));
        taListenerRespVo.setTypeIdValue(Column.LISTENER_TYPE.getLabelByCode(taListenerRespVo.getTypeId()));
        taListenerRespVo.setListenerTypeValue(ColumnStr.LISTENER_VALUE_TYPE.getLabelByCode(taListenerRespVo.getListenerType()));
        return taListenerRespVo;
    }


    /**
     * 转换返回的监听器状态，监听器类型，监听器方法
     *
     * @param taListenerQueryVO
     * @return
     */
    public TaListenerQueryVO convertValue(TaListenerQueryVO taListenerQueryVO) {
        taListenerQueryVO.setEnableStatusName(CrmEnableStatusEnum.getDesc(taListenerQueryVO.getEnableStatus()));
        taListenerQueryVO.setTypeIdValue(Column.LISTENER_TYPE.getLabelByCode(taListenerQueryVO.getTypeId()));
        taListenerQueryVO.setListenerTypeValue(ColumnStr.LISTENER_VALUE_TYPE.getLabelByCode(taListenerQueryVO.getListenerType()));
        return taListenerQueryVO;
    }
}
