package com.biz.crm.util;

import org.aspectj.apache.bcel.classfile.Code;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

/**
 * @Author zhangyuzhu
 * @Description code工具类
 * @Created data 2018/8/24 12:40
 */
public class CodeUtil {

    private final long workerIdBits = 5L;
    private final long datacenterIdBits = 5L;
    private final long maxWorkerId = ~(-1L << workerIdBits);
    private final long maxDatacenterId = ~(-1L << datacenterIdBits);
    private final long sequenceBits = 10L;
    private final long timestampLeftShift = sequenceBits;
    /** 生成序列的掩码，这里为4095 (0b111111111111=0xfff=4095) */
    private final long sequenceMask = ~(-1L << sequenceBits);
    /** 工作机器ID(0~31) */
    private long workerId;
    /** 数据中心ID(0~31) */
    private long datacenterId;
    /** 毫秒内序列(0~4095) */
    private long sequence = 0L;
    /** 上次生成ID的时间截 */
    private long lastTimestamp = -1L;
    private static CodeUtil idWorker;
    private final long TWEPOCH = 1596037883039L;

    public static String getCode() {
        if(idWorker == null) {
            synchronized (CodeUtil.class) {
                if(idWorker == null) {
                    idWorker = new CodeUtil(0, 0);
                }
            }
        }
        return to62RadixString(idWorker.nextId());
    }
    private CodeUtil(long workerId, long datacenterId) {
        if (workerId > maxWorkerId || workerId < 0) {
            throw new IllegalArgumentException(String.format("worker Id can't be greater than %d or less than 0", maxWorkerId));
        }
        if (datacenterId > maxDatacenterId || datacenterId < 0) {
            throw new IllegalArgumentException(String.format("datacenter Id can't be greater than %d or less than 0", maxDatacenterId));
        }
        this.workerId = workerId;
        this.datacenterId = datacenterId;
    }
    protected long tilNextMillis(long lastTimestamp) {
        long timestamp = System.currentTimeMillis();
        while (timestamp <= lastTimestamp) {
            timestamp = System.currentTimeMillis();
        }
        return timestamp;
    }
    protected synchronized long nextId() {
        long timestamp = System.currentTimeMillis();
        if (timestamp < lastTimestamp) {
            throw new RuntimeException(
                    String.format("Clock moved backwards.  Refusing to generate id for %d milliseconds", lastTimestamp - timestamp));
        }
        //如果是同一时间生成的，则进行毫秒内序列
        if (lastTimestamp == timestamp) {
            sequence = (sequence + 1) & sequenceMask;
            //毫秒内序列溢出
            if (sequence == 0) {
                //阻塞到下一个毫秒,获得新的时间戳
                timestamp = tilNextMillis(lastTimestamp);
            }
        }
        //时间戳改变，毫秒内序列重置
        else {
            sequence = 0L;
        }
        //上次生成ID的时间截
        lastTimestamp = timestamp;
        //移位并通过或运算拼到一起组成64位的ID
        return ((timestamp - TWEPOCH) << timestampLeftShift) | sequence;
    }


    static final char[] DIGITS =
            { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
                    'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j',
                    'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't',
                    'u', 'v', 'w', 'x', 'y', 'z', 'A', 'B', 'C', 'D',
                    'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
                    'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z' };

    private static String to62RadixString(long seq) {
        StringBuilder sb = new StringBuilder();
        do {
            int remainder = (int) (seq % 62);
            sb.append(DIGITS[remainder]);
            seq = seq / 62;
        } while (seq != 0);
        return sb.reverse().toString();
    }
    /**
     *  生成三位数的code（前面用0填充）
     *  用于生成树结构的规则code
     * @param parentCode
     * @param rank
     * @return
     */
    public static String fullThree(String parentCode,int rank){
        String rankStr = "";
        if(rank < 10){
            rankStr = "00"+ rank;
        }else if(rank < 100){
            rankStr = "0"+ rank;
        }else{
            rankStr = ""+ rank;
        }
        if(parentCode == null || parentCode.equals("")){
            return rankStr;
        }
        return new StringBuffer(parentCode).append(rankStr).toString();
    }


    /**
     * 生成编码(临时用)
     * @return
     */
    public static String createCode(){
        return getCode();
    }

    /**
     * 传入一个功能名称,通过功能名称查询编码规则表获取编码规则,然后生成对应的编码
     * @param functionName
     * @return
     */
    public static String createOneCode(String functionName){
        return getCode();
    }

    /**
     * 传入一个功能名称,和需要的编码个数通过功能名称查询编码规则表获取编码规则,然后生成对应的编码集合
     * @param functionName 功能代码
     * @param num 编码个数
     * @return
     */
    public static List<String> createCodeList(String functionName, int num){
        List<String> list = new ArrayList<>(num);
        for (int i =0 ; i<num ; i++ ){
            list.add(getCode());
        }
        return list;
    }

}
