package com.biz.crm.eunm.dms;

import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.StringUtils;
import com.google.common.collect.Lists;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import lombok.Getter;
import lombok.SneakyThrows;

/**
 * 促销规则类型枚举（自定义动态枚举）
 * @Author: chenrong
 * @Date: 2021/4/13 14:03
 */
@Getter
public class RuleTypeDynamicEnum {

  //促销规则类型字典类型编码
  public static final String PROMOTION_POLICY_RULE_TYPE = "promotion_policy_rule_type";

  //条件规则-枚举
  public static final RuleTypeDynamicEnum CONDITION = initEnum("condition", "条件规则");
  //计算规则-枚举
  public static final RuleTypeDynamicEnum CALCULATE = initEnum("calculate", "计算规则");
  //限量规则-枚举
  public static final RuleTypeDynamicEnum LIMITED = initEnum("limited", "限量规则");

  //编码，必须与相关字典中的编码保持一致
  private String code;
  //描述
  private String desc;
  //对象中的属性名称
  private String objectName;

  /**
   * 初始化静态枚举
   * @param code
   * @param desc
   * @return
   */
  private static RuleTypeDynamicEnum initEnum(String code, String desc) {
    return new RuleTypeDynamicEnum(code, desc, buildObjectName(code));
  }

  /**
   * 构建参数对象名称
   * @param code
   * @return
   */
  public static String buildObjectName(String code) {
    return code.concat("s");
  }

  /**
   * 获取字典中所有的规则，以枚举的类型返回
   * @return
   */
  public static List<RuleTypeDynamicEnum> getRuleTypeEnums() {
    Map<String, String> dictMap = DictUtil.dictMap(PROMOTION_POLICY_RULE_TYPE);
    if(!CollectionUtil.mapNotEmpty(dictMap)) {
      return Lists.newArrayList();
    }
    List<RuleTypeDynamicEnum> ruleTypeDynamicEnums = Lists.newArrayList();
    dictMap.forEach((k, v) -> {
      RuleTypeDynamicEnum rule = new RuleTypeDynamicEnum(k, v, buildObjectName(k));
      ruleTypeDynamicEnums.add(rule);
    });
    return ruleTypeDynamicEnums;
  }

  /**
   * 根据字典编码获取枚举
   * @param code
   * @return
   */
  public static RuleTypeDynamicEnum getRuleTypeEnumByCode(String code) {
    if(StringUtils.isEmpty(code)) {
      return null;
    }
    String value = DictUtil.dictValue(PROMOTION_POLICY_RULE_TYPE, code);
    return new RuleTypeDynamicEnum(code, value, buildObjectName(code));
  }

  /**
   * 根据参数对象名称获取枚举
   * @param objectName
   * @return
   */
  public static RuleTypeDynamicEnum getRuleTypeEnumByObjectName(String objectName) {
    List<RuleTypeDynamicEnum> ruleTypeDynamicEnums = getRuleTypeEnums();
    ruleTypeDynamicEnums = ruleTypeDynamicEnums.stream().filter(rule ->
            Objects.equals(rule.getObjectName(), objectName)).collect(Collectors.toList());
    if(CollectionUtil.listEmpty(ruleTypeDynamicEnums)) {
      return null;
    }
    return ruleTypeDynamicEnums.stream().findFirst().get();
  }

  /**
   * 获取所有静态枚举
   * @return
   */
  @SneakyThrows
  public static List<RuleTypeDynamicEnum> getAllStaticEnums(){
    Class cls = RuleTypeDynamicEnum.class;
    Field[] declaredFields = cls.getDeclaredFields();
    List<RuleTypeDynamicEnum> enums = Lists.newArrayList();
    for(Field f : declaredFields) {
      String modifier = Modifier.toString(f.getModifiers());
      if(Objects.equals(modifier, "public static final")) {
        f.setAccessible(true);
        enums.add((RuleTypeDynamicEnum) f.get(null));
      }
    }
    return enums;
  }

  public RuleTypeDynamicEnum(String code, String desc, String objectName) {
    this.code = code;
    this.desc = desc;
    this.objectName = objectName;
  }



}
