package com.biz.crm.util;

import com.biz.crm.common.param.RedisParam;
import com.biz.crm.mdm.bpmrole.MdmBpmRoleRedisFeign;
import com.biz.crm.mdm.org.MdmOrgRedisFeign;
import com.biz.crm.nebular.mdm.bpmrole.resp.MdmBpmRoleRedisRespVo;
import com.biz.crm.nebular.mdm.bpmrole.resp.MdmBpmRoleRespVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRedisVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgReloadRedisVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.service.RedisService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @Description: 通过缓存查询流程角色工具类
 * @project：crm-core
 * @package：com.biz.crm.util
 * @author：longyu
 * @create：2021/1/14 15:29
 */
@Slf4j
@Component
public class BpmRoleUtil {

    private static RedisService redisService;

    private static MdmBpmRoleRedisFeign mdmbpmRoleRedisFeign;

    @Autowired
    public void setRedisService(RedisService redis) {
        BpmRoleUtil.redisService = redis;
    }

    @Autowired
    public void setMdmBpmRoleRedisFeign(MdmBpmRoleRedisFeign bpmRoleRedisFeign) {
        BpmRoleUtil.mdmbpmRoleRedisFeign = bpmRoleRedisFeign;
    }

    /**
     * 根据流程角色获取详情
     * @param code 流程角色编码
     * @return
     */
    public static MdmBpmRoleRespVo getBpmRoleCode(String code){
        if (code != null && !"".equals(code)) {
            Map<String, MdmBpmRoleRedisRespVo> bpmRoleRedisMap = getBpmRoleSelfRedisMapFromRedisFirst(Collections.singletonList(code));
            if (bpmRoleRedisMap.containsKey(code)) {
                MdmBpmRoleRedisRespVo mdmBpmRoleRedisRespVo = bpmRoleRedisMap.get(code);
                return CrmBeanUtil.copy(mdmBpmRoleRedisRespVo, MdmBpmRoleRespVo.class);
            }
        }
        return null;
    }

    /**
     * 根据流程角色集合获取详情
     * @param codeList 编码集合
     * @return
     */
    public static List<MdmBpmRoleRespVo> getBpmRoleCodeList(List<String> codeList){
        List<MdmBpmRoleRespVo> list = new ArrayList<>();
        if (codeList != null && codeList.size() > 0) {
            Map<String, MdmBpmRoleRedisRespVo> bpmRoleRedisMap = getBpmRoleSelfRedisMapFromRedisFirst(codeList);
            if (!bpmRoleRedisMap.isEmpty()) {
                list.addAll(bpmRoleRedisMap.values().stream().map(item -> CrmBeanUtil.copy(item, MdmBpmRoleRespVo.class)).collect(Collectors.toList()));
            }
        }
        return list;
    }

    /**
     * 获取流程角色缓存数据从redis获取
     * @param bpmRoleCodeList 流程角色编码集合
     * @return
     */
    private static Map<String, MdmBpmRoleRedisRespVo> getBpmRoleSelfRedisMapFromRedisFirst(List<String> bpmRoleCodeList){
        Map<String, MdmBpmRoleRedisRespVo> map = new HashMap<>(16);
        try {
            if (bpmRoleCodeList != null && bpmRoleCodeList.size() > 0){
                List<MdmBpmRoleRedisRespVo> redisVoList = new ArrayList<>();
                Set<String> lockCodeList = new HashSet<>(16);
                List<Object> lockListFromHash = redisService.hmget(RedisParam.MDM_BPM_ROLE_LOCK_HASH,new HashSet<Object>(bpmRoleCodeList));
                if (lockListFromHash != null && lockListFromHash.size() > 0){
                    lockCodeList.addAll(lockListFromHash.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.toSet()));
                }
                Set<String> notLockGetFromRedisCodeSet = bpmRoleCodeList.stream().filter(item -> !lockCodeList.contains(item)).collect(Collectors.toSet());
                Set<String> redisHasCodeSet = new HashSet<>(16);
                if (!notLockGetFromRedisCodeSet.isEmpty()){
                    List<Object> bpmRoleListFromRedis = redisService.hmget(RedisParam.MDM_BPM_ROLE_SELF_HASH,new HashSet<Object>(notLockGetFromRedisCodeSet));
                    if (bpmRoleListFromRedis != null && bpmRoleListFromRedis.size() > 0){
                        List<MdmBpmRoleRedisRespVo> collect = bpmRoleListFromRedis.stream().filter(Objects::nonNull).map(item -> CrmBeanUtil.copy(item,MdmBpmRoleRedisRespVo.class)).collect(Collectors.toList());
                        if (collect.size() > 0) {
                            redisVoList.addAll(collect);
                            redisHasCodeSet.addAll(collect.stream().map(MdmBpmRoleRedisRespVo::getBpmRoleCode).collect(Collectors.toSet()));
                        }
                    }
                }
                Set<String> getByFeignCodeSet = notLockGetFromRedisCodeSet.stream().filter(item -> !redisHasCodeSet.contains(item)).collect(Collectors.toSet());
                if (!getByFeignCodeSet.isEmpty()) {
                    Result<List<MdmBpmRoleRedisRespVo>> result = mdmbpmRoleRedisFeign.findRedisListByBpmRoleCodeList(new ArrayList<>(getByFeignCodeSet));
                    if (result.isSuccess()) {
                        Map<String, String> lockCodeMap = new HashMap<>(16);
                        List<MdmBpmRoleRedisRespVo> list = result.getResult();
                        if (list != null && list.size() > 0) {
                            Map<String, MdmBpmRoleRedisRespVo> collect = list.stream().collect(Collectors.toMap(MdmBpmRoleRedisRespVo::getBpmRoleCode, v -> v));

                            Map<String, MdmBpmRoleRedisRespVo> cacheMap = new HashMap<>(16);
                            for (String bpmRoleCode :
                                    getByFeignCodeSet) {
                                if (!collect.containsKey(bpmRoleCode)) {
                                    lockCodeMap.put(bpmRoleCode, bpmRoleCode);
                                } else {
                                    redisVoList.add(collect.get(bpmRoleCode));
                                    cacheMap.put(bpmRoleCode, collect.get(bpmRoleCode));
                                }
                            }
                            if (!cacheMap.isEmpty()) {
                                redisService.hmset(RedisParam.MDM_BPM_ROLE_SELF_HASH, new HashMap<Object, Object>(cacheMap));
                            }
                        } else {
                            lockCodeMap.putAll(getByFeignCodeSet.stream().collect(Collectors.toMap(k -> k, v -> v)));
                        }
                        if (!lockCodeMap.isEmpty()) {
                            redisService.hmset(RedisParam.MDM_BPM_ROLE_LOCK_HASH, new HashMap<Object, Object>(lockCodeMap), RedisParam.TIME3);
                        }
                    }
                }
                if (redisVoList.size() > 0) {
                    map.putAll(redisVoList.stream().collect(Collectors.toMap(MdmBpmRoleRedisRespVo::getBpmRoleCode, v -> v)));
                }
            }
        }catch (Exception e){
            log.error("组织查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 根据流程角色编码删除单个redis缓存
     */
    public static void deleteByBpmRoleCode(List<String> code){
        if (code != null && code.size() > 0){
            redisService.hdel(RedisParam.MDM_BPM_ROLE_SELF_HASH,code.toArray());
        }
    }

    /**
     * 清楚全部流程角色redis缓存
     */
    public static void deleteAllCache() {

        redisService.del(RedisParam.MDM_BPM_ROLE_LOCK_HASH);
        redisService.del(RedisParam.MDM_BPM_ROLE_SELF_HASH);

    }

    /**
     * 重新加载redis缓存
     */
    public static void reloadAllCache(){
        deleteAllCache();
        Result<List<MdmBpmRoleRedisRespVo>> result = mdmbpmRoleRedisFeign.findRedisReloadList();
        if (result.isSuccess()){
            List<MdmBpmRoleRedisRespVo> list = result.getResult();
            if (list != null && list.size() > 0){
                List<List<MdmBpmRoleRedisRespVo>> partition = Lists.partition(list,500);
                for ( List<MdmBpmRoleRedisRespVo> item :
                partition){
                    Map<String,MdmBpmRoleRedisRespVo> map = item.stream().collect(Collectors.toMap(MdmBpmRoleRedisRespVo::getBpmRoleCode,v -> v));
                    redisService.hmset(RedisParam.MDM_BPM_ROLE_SELF_HASH,new HashMap<Object, Object>(map));
                }
            }
        }
    }
}
