package com.biz.crm.util;

import com.biz.crm.base.BusinessException;
import com.biz.crm.base.CrmBaseEntity;
import com.biz.crm.nebular.mdm.CrmBaseVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import lombok.SneakyThrows;
import org.apache.commons.lang3.reflect.FieldUtils;
import org.apache.poi.ss.formula.functions.T;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 处理，继承自{@link CrmBaseVo}的公共字段
 * @Author: chenrong
 * @Date: 2020/11/24 10:23
 */
public class CommonFieldUtil {

  private static final Logger LOGGER = LoggerFactory.getLogger(CommonFieldUtil.class);

  /**
   * 新增时初始化公共字段，一旦出错就抛异常，避免数据错乱
   * @param vo
   */
  public static void initCreateFields(Object vo) {
//    if (vo instanceof Collection) {
//      ((Collection) vo).forEach(v -> initCreateFields(v));
//    }
//    if (vo == null || !(vo instanceof CrmBaseVo || vo instanceof CrmBaseEntity)) {
//      return;
//    }
//    initDecimalZero(vo);
//    try {
//      String[] timeStr = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date()).split(" ");
//      UserRedis user = UserUtils.getUser();
//      if (user == null) {
//        return;
//      }
//      Map<String, Object> attributes = Maps.newHashMap();
//      attributes.put("delFlag", "1");
//      attributes.put("enableStatus", "1");
//      attributes.put("createDate", timeStr[0]);
//      attributes.put("createDateSecond", timeStr[1]);
//      attributes.put("createCode", user.getCustcode());
//      attributes.put("createName", user.getCustname());
//      attributes.put("createOrgCode", user.getOrgcode());
//      attributes.put("createOrgName", user.getOrgname());
//      attributes.put("createPosCode", user.getPoscode());
//      attributes.put("createPosName", user.getPosname());
//      setValue(attributes, vo);
//    } catch (Exception e) {
//      throw new BusinessException("处理公共字段失败", e);
//    }
  }

  /**
   * 设置字段值
   * @param attributes
   * @param vo
   * @throws IllegalAccessException
   */
  private static void setValue(Map<String, Object> attributes, Object vo) throws IllegalAccessException {
    List<Field> fields = getFields(vo.getClass());
    Map<String, Field> fieldMap = Maps.newHashMap();
    fields.forEach(f -> fieldMap.put(f.getName(), f));
    for (Map.Entry<String, Object> entry : attributes.entrySet()) {
      Field field = fieldMap.get(entry.getKey());
      if (field == null) {
        continue;
      }
      field.setAccessible(true);
      field.set(vo, entry.getValue());
    }
  }


  /**
   * 获取所有字段，包括所有父类
   * @param c
   * @return
   */
  public static List<Field> getFields(Class c) {
    List<Field> fields = Lists.newArrayList();
    while (c != null) {
      fields.addAll(Arrays.asList(c.getDeclaredFields()));
      c = c.getSuperclass();
    }
    return fields;
  }

  /**
   * 修改时初始化公共字段，一旦出错就抛异常，避免数据错乱
   * @param vo
   */
  public static void initUpdateFields(Object vo) {
    if (vo instanceof Collection) {
      ((Collection) vo).forEach(v -> initUpdateFields(v));
    }
    if (vo == null || !(vo instanceof CrmBaseVo || vo instanceof CrmBaseEntity)) {
      return;
    }
    try {
      String[] timeStr = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date()).split(" ");
      UserRedis user = UserUtils.getUser();
      if (user == null) {
        return;
      }
      Map<String, Object> attributes = Maps.newHashMap();
      attributes.put("updateDate", timeStr[0]);
      attributes.put("updateDateSecond", timeStr[1]);
      attributes.put("updateCode", user.getCustcode());
      attributes.put("updateName", user.getCustname());
      attributes.put("updateOrgCode", user.getOrgcode());
      attributes.put("updateOrgName", user.getOrgname());
      attributes.put("updatePosCode", user.getPoscode());
      attributes.put("updatePosName", user.getPosname());
      setValue(attributes, vo);
    } catch (Exception e) {
      LOGGER.error("data:{}", vo);
      throw new BusinessException("处理公共字段失败", e);
    }
  }

  /**
   * 获取对象中指定的字段值
   * @param o
   * @param fieldNames
   * @return
   */
  public static Map<String, Object> fieldsValue(Object o, String... fieldNames) {
    Map<String, Object> result = Maps.newHashMap();
    try {
      Class c = o.getClass();
      for (String fieldName : fieldNames) {
        Field field = c.getDeclaredField(fieldName);
        field.setAccessible(true);
        result.put(fieldName, field.get(o));
      }
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return Maps.newHashMap();
    }
    return result;
  }

  /**
   * java字段名转化成数据库字段名
   * @param sourceName
   * @return
   */
  public static String fieldHumpToLine(String sourceName) {
    Pattern humpPattern = Pattern.compile("[A-Z]");
    Matcher matcher = humpPattern.matcher(sourceName);
    StringBuffer sb = new StringBuffer();
    while (matcher.find()) {
      matcher.appendReplacement(sb, "_" + matcher.group(0).toLowerCase());
    }
    matcher.appendTail(sb);
    return sb.toString();
  }

  /**
   * 数据库字段名转化成java字段名
   * @param sourceName
   * @return
   */
  public static String fieldLineToHump(String sourceName) {
    Pattern linePattern = Pattern.compile("_(\\w)");
    Matcher matcher = linePattern.matcher(sourceName.toLowerCase());
    StringBuffer sb = new StringBuffer();
    while (matcher.find()) {
      matcher.appendReplacement(sb, matcher.group(1).toUpperCase());
    }
    matcher.appendTail(sb);
    return sb.toString();
  }

  /**
   * 初始化为null的，{@link BigDecimal}类型字段
   * @param v
   */
  public static void initDecimalZero(Object v) {
    Class c = v.getClass();
    try {
      Field[] fields = c.getDeclaredFields();
      if (fields == null) {
        return;
      }
      for (Field field : fields) {
        field.setAccessible(true);
        if (field.getType() == BigDecimal.class && field.get(v) == null) {
          field.set(v, BigDecimal.ZERO);
        }
      }
    } catch (IllegalAccessException e) {
      LOGGER.error("data:{}", v);
      throw new BusinessException("处理空数字字段异常", e);
    }
  }

  /**
   * 根据某个属性值计算两个集合差集
   * result = source - target
   * 注：传入的两个集合中，泛型，均必须为java对象
   * @param source
   * @param target
   * @param keys  唯一约束，暂时最多传入一个有效，如果不传，默认为‘id’
   * @return
   */
  public static List<String> difference(Collection source, Collection target, String... keys) {
    ValidateUtils.validate(source, "传入源集合不能为空");
    ValidateUtils.validate(target, "传入目标集合不能为空");
    String key = "id";
    if(keys != null && keys.length > 0) {
      key = keys[0];
    }
    Set<String> sourceKeys = getKeyValues(source, key);
    Set<String> targetKeys = getKeyValues(target, key);
    Set<String> differenceIds = Sets.difference(sourceKeys, targetKeys);
    if (!CollectionUtil.collectionNotEmpty(differenceIds)) {
      return Lists.newArrayList();
    }
    return Lists.newArrayList(differenceIds);
  }

  /**
   * 获取指定属性值
   * @param collection
   * @param key
   * @return
   */
  @SneakyThrows
  public static Set<String> getKeyValues(Collection<Object> collection, String key) {
    ValidateUtils.validate(collection, "传入集合不能为空");
    Set<String> keys = Sets.newHashSet();
    for (Object s : collection) {
      List<Field> fields = getFields(s.getClass());
      List<Field> fieldList = fields.stream().filter(f -> f.getName().equals(key)).collect(Collectors.toList());
      fieldList.get(0).setAccessible(true);
      Object value = fieldList.get(0).get(s);
      keys.add(value == null ? "" : value.toString());
    }
    return keys.stream().filter(k -> Objects.nonNull(k)).collect(Collectors.toSet());
  }

}
