package com.biz.crm.util;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.BeanWrapper;
import org.springframework.beans.BeanWrapperImpl;

import java.beans.PropertyDescriptor;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 对象处理
 *
 * @Author huang.long
 * @Date Created in 2020-03-12 10:15
 */
@Slf4j
public class CrmBeanUtil {
    /**
     * 复制bean的属性, ******* 注意：source的空值属性会覆盖target ********
     *
     * @param source 源 要复制的对象
     * @param target 目标 复制到此对象
     */
    public static void copyProperties(Object source, Object target) {
        if(null == source || null == target){
            return;
        }
        BeanUtils.copyProperties(source, target);
    }

    /**
     * 复制bean的属性,忽略空值属性
     *
     * @param source 源 要复制的对象
     * @param target 目标 复制到此对象
     */
    public static void copyPropertiesIgnoreEmpty(Object source, Object target) {
        if(null == source || null == target){
            return;
        }
        BeanUtils.copyProperties(source, target, getEmptyPropertyNames(source));
    }
    private static String[] getEmptyPropertyNames(Object source) {
        final BeanWrapper src = new BeanWrapperImpl(source);
        java.beans.PropertyDescriptor[] pds = src.getPropertyDescriptors();

        Set<String> emptyNames = new HashSet<>();
        for(java.beans.PropertyDescriptor pd : pds) {
            Object srcValue = src.getPropertyValue(pd.getName());
            if (srcValue == null || StringUtils.isBlank(srcValue.toString())) {
                emptyNames.add(pd.getName());
            }
        }
        String[] result = new String[emptyNames.size()];
        return emptyNames.toArray(result);
    }



    /**
     * 复制对象
     *
     * @param source 源 要复制的对象
     * @param target 目标 复制到此对象
     * @param <T>
     * @return
     */
    public static <T> T copy(Object source, Class<T> target) {
        if (source==null) {
            return null;
        }
        try {
            T newInstance = target.newInstance();
            BeanUtils.copyProperties(source, newInstance);
            return newInstance;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 复制list
     *
     * @param source
     * @param target
     * @param <T>
     * @param <K>
     * @return
     */
    public static <T, K> List<K> copyList(List<T> source, Class<K> target) {

        if (null == source || source.isEmpty()) {
            return new ArrayList<>();
        }
        return source.stream().map(e -> copy(e, target)).collect(Collectors.toList());
    }

    /**
     * 复制bean的扩展字段1~10
     *
     * @param source 源 要复制的对象
     * @param target 目标 复制到此对象
     */
    public static void copyExt10(Object source, Object target) {
        if (source == null || target == null) {
            return;
        }
        Set<String> props = new HashSet<>(16);
        for (int i = 1; i <= 10; i++) {
            props.add("ext" + i);
        }
        String[] filteredPropertyNames =
                Arrays.stream(BeanUtils.getPropertyDescriptors(source.getClass()))
                        .map(PropertyDescriptor::getName)
                        .filter(item -> !props.contains(item))
                        .toArray(String[]::new);
        if (filteredPropertyNames.length > 0) {
            BeanUtils.copyProperties(source, target, filteredPropertyNames);
        }
    }

    /**
     * 复制bean的扩展字段1~50
     *
     * @param source 源 要复制的对象
     * @param target 目标 复制到此对象
     */
    public static void copyExt50(Object source, Object target) {
        if (source == null || target == null) {
            return;
        }
        Set<String> props = new HashSet<>(64);
        for (int i = 1; i <= 50; i++) {
            props.add("ext" + i);
        }
        String[] filteredPropertyNames =
                Arrays.stream(BeanUtils.getPropertyDescriptors(source.getClass()))
                        .map(PropertyDescriptor::getName)
                        .filter(item -> !props.contains(item))
                        .toArray(String[]::new);
        if (filteredPropertyNames.length > 0) {
            BeanUtils.copyProperties(source, target, filteredPropertyNames);
        }
    }
}
