package com.biz.crm.util;

import com.biz.crm.common.param.RedisParam;
import com.biz.crm.dict.feign.MdmDictFeign;
import com.biz.crm.nebular.mdm.dict.resp.DictDataVo;
import com.biz.crm.nebular.mdm.dict.resp.DictGroupVo;
import com.biz.crm.service.RedisService;
import lombok.extern.slf4j.Slf4j;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author zxw
 **/
@Slf4j
public class DictUtil {

    private static RedisService redisService;

    private static MdmDictFeign mdmDictFeign;

    public static void setRedisService(RedisService redis) {
        redisService = redis;
    }

    public static void setMdmDictFeign(MdmDictFeign mdmDictFeign) {
        DictUtil.mdmDictFeign = mdmDictFeign;
    }

    /**
     * 根据字典类型编码集合查询对应的value的map集合
     *
     * @param dictTypeCodes 分类编码集合
     * @return map
     */
    public static Map<String, Map<String, String>> getDictValueMapsByCodes(List<String> dictTypeCodes) {
        Map<String, Map<String, String>> maps = new HashMap<>(16);
        try {
            if (dictTypeCodes != null && dictTypeCodes.size() > 0) {
                Map<String, DictGroupVo> groupMaps = getDictGroupMap(dictTypeCodes);
                for (String dictTypeCode :
                        dictTypeCodes) {
                    if (groupMaps.containsKey(dictTypeCode)) {
                        Map<String, String> map = new LinkedHashMap<>(16);
                        DictGroupVo dictGroup = groupMaps.get(dictTypeCode);
                        if (dictGroup != null) {
                            for (DictDataVo item :
                                    dictGroup.getDataList()) {
                                map.put(item.getDictCode(), item.getDictValue());
                            }
                        }
                        maps.put(dictTypeCode, map);
                    }
                }
            }
        } catch (Exception e) {
            log.error("获取字典数据异常!!!!!!!!");
            e.printStackTrace();
        }
        return maps;
    }

    /**
     * 根据字典类型编码查询对应的值的集合
     *
     * @param dictTypeCode 分类编码
     * @return map
     */
    public static Map<String, String> getDictValueMapsByCodes(String dictTypeCode) {
        return dictMap(dictTypeCode);
    }


    /**
     * 根据字典类型编码查询对应的值的集合
     *
     * @param dictTypeCode 字典类型编码
     * @return map
     */
    public static Map<String, String> dictMap(String dictTypeCode) {
        Map<String, String> map = new LinkedHashMap<>(16);
        if (StringUtils.isNotEmpty(dictTypeCode)) {
            DictGroupVo dict = getDictGroup(dictTypeCode);
            if (dict != null) {
                for (DictDataVo item :
                        dict.getDataList()) {
                    map.put(item.getDictCode(), item.getDictValue());
                }
            }
        }
        return map;
    }

    private static DictGroupVo getDictGroup(String dictTypeCode) {
        if (dictTypeCode != null && !"".equals(dictTypeCode)) {
            Map<String, DictGroupVo> dictGroupMap = getDictGroupMap(Collections.singletonList(dictTypeCode));
            if (!dictGroupMap.isEmpty()) {
                return dictGroupMap.get(dictTypeCode);
            }
        }
        return null;
    }

    private static Map<String, DictGroupVo> getDictGroupMap(List<String> dictTypeCodeList) {
        Map<String, DictGroupVo> map = new HashMap<>(16);
        try {
            if (redisService != null) {
                if (dictTypeCodeList != null && dictTypeCodeList.size() > 0) {
                    List<String> unlockCodeList = dictTypeCodeList.stream().filter(
                            dictTypeCode -> redisService.get(RedisParam.DICT_TYPE_LOCK + dictTypeCode) == null
                    ).collect(Collectors.toList());
                    List<String> notCacheCodeList = new ArrayList<>();
                    for (String dictTypeCode :
                            unlockCodeList) {
                        Object o = redisService.get(RedisParam.DICT_TYPE_GROUP + dictTypeCode);
                        if (o == null) {
                            notCacheCodeList.add(dictTypeCode);
                        } else {
                            map.put(dictTypeCode, (DictGroupVo) o);
                        }
                    }
                    if (notCacheCodeList.size() > 0) {
                        Result<List<DictGroupVo>> result = mdmDictFeign.getDictGroupList(dictTypeCodeList);
                        if (result.isSuccess()) {
                            List<DictGroupVo> list = result.getResult();
                            if (list != null && list.size() > 0) {
                                Map<String, DictGroupVo> collect = list.stream().collect(Collectors.toMap(DictGroupVo::getDictTypeCode, v -> v));
                                for (String dictTypeCode :
                                        notCacheCodeList) {
                                    if (!collect.containsKey(dictTypeCode)) {
                                        redisService.setSeconds(RedisParam.DICT_TYPE_LOCK + dictTypeCode, dictTypeCode, RedisParam.TIME5);
                                    } else {
                                        addRedisCache(collect.get(dictTypeCode));
                                        map.put(dictTypeCode, collect.get(dictTypeCode));
                                    }
                                }
                            }
                        }
                    }
                }
            } else {
                List<DictGroupVo> dictGroupList = getDictGroupList(dictTypeCodeList);
                map = dictGroupList.stream().collect(Collectors.toMap(DictGroupVo::getDictTypeCode, v -> v));
            }
        } catch (Exception e) {
            log.error("数据字典工具类异常：{}", e);
        }
        return map;
    }

    private static List<DictGroupVo> getDictGroupList(List<String> dictTypeCodeList) {
        Result<List<DictGroupVo>> result = mdmDictFeign.getDictGroupList(dictTypeCodeList);
        if (result.isSuccess()) {
            List<DictGroupVo> list = result.getResult();
            if (list != null && list.size() > 0) {
                return list;
            }
        }
        return new ArrayList<>();
    }

    /**
     * 删除数据字典缓存
     *
     * @param dictTypeCode
     */
    public static void deleteRedisCache(String dictTypeCode) {
        if (StringUtils.isNotEmpty(dictTypeCode)) {
            redisService.del(RedisParam.DICT_TYPE_GROUP + dictTypeCode);
        }
    }

    /**
     * 添加数据字典缓存
     *
     * @param group
     */
    private static void addRedisCache(DictGroupVo group) {
        if (group != null) {
            redisService.set(RedisParam.DICT_TYPE_GROUP + group.getDictTypeCode(), group);
        }
    }

}
