package com.biz.crm.util;

import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.config.SpringApplicationContextUtil;
import com.biz.crm.feign.MdmCommonUserFeign;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRespVo;
import com.biz.crm.service.RedisService;
import io.jsonwebtoken.lang.Assert;
import lombok.extern.slf4j.Slf4j;

import javax.servlet.http.HttpServletRequest;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author jianglong
 * @version V1.0
 * @Package com.biz.crm.util
 * @Description: 登录信息
 * @date 2020/8/26 下午3:04
 */
@Slf4j
public class UserUtils {
    private static UserAutoConfig userAutoConfig = null;
    private static RedisService redisService;
    private static MdmCommonUserFeign mdmCommonUserFeign;

    private static String USER_GROUP_PREFIX = "userGroup:";

    private static void initUserFeign() {
        if (mdmCommonUserFeign == null) {
            mdmCommonUserFeign = SpringApplicationContextUtil.getApplicationContext().getBean(MdmCommonUserFeign.class);
        }
    }

    private static void initRedis() {
        if (redisService == null) {
            redisService = SpringApplicationContextUtil.getApplicationContext().getBean(RedisService.class);
        }
    }

    //得到当前登录人
    public static UserRedis getUser() {
        UserRedis user = ThreadLocalUtil.getUser();
        if (user != null) {
            return user;
        }
        UserRedis userRedis = null;
        try {
            String token = getToken();
            if (StringUtils.isEmpty(token)) {
                return null;
            } else {
                initRedis();
                Object o = redisService.get(token);
                if (o == null) {
                    redisService.setSeconds(token, null, RedisParam.TIME10);
                } else {
                    userRedis = (UserRedis) o;
                    ThreadLocalUtil.setUser(userRedis);
                }
            }
        } catch (Exception e) {
            log.error("获取当前用户:", e);
        }
        return userRedis;
    }

    /**
     * 获取当前token
     *
     * @return
     */
    public static String getToken() {
        HttpServletRequest request = HttpServletRequestUtil.getRequest();
        Object tokenLock = ThreadLocalUtil.getObj(GlobalParam.TOKEN);
        String token = tokenLock == null ? "" : tokenLock.toString();
        if (StringUtils.isNotEmpty(token) || request == null) {
            return token;
        }
        if(request!=null){
            token = request.getHeader(GlobalParam.TOKEN);
            if(StringUtils.isEmpty(token)){
                token = request.getHeader(GlobalParam.WEBSOCT_TOKEN);
            }
        }
        return token;
    }

    public static String getToken(String key) {
        initRedis();
        return redisService.getAndSet(key);
    }

    /**
     * 默认设置当前登录权限，如果没有登录情况
     */
    public static void doTokenForNull() {
        initConfig();
        if (StringUtils.isEmpty(userAutoConfig.getUsername())) {
            return;
        }
        try {
            ConcurrentHashMap<String, Object> map = new ConcurrentHashMap<>();
            String token = UUID.randomUUID().toString().replaceAll("-", "");
            map.put(GlobalParam.TOKEN, token);
//            UserRedis userRedis = new UserRedis();
//            userRedis.setUsername(userAutoConfig.getUsername());
            ThreadLocalUtil.stObj(map);
            setUser(token, userAutoConfig.getUsername());
//            userRedis.setId("admin");
//            redisService.setHours(token, userRedis, RedisParam.TIME1);
        } catch (Exception e) {
            log.error("默认设置权限错误", e);
        }
    }

    private static void initConfig() {
        if (userAutoConfig == null) {
            userAutoConfig = SpringApplicationContextUtil.getApplicationContext().getBean(UserAutoConfig.class);
        }
    }

    public static void setUser(String token, UserRedis userRedis) {
        Assert.hasText(token, "token不能为空");
        Assert.notNull(userRedis, "userRedis不能为空");
        initRedis();
        String username = userRedis.getUsername();
        Assert.hasText(username, "用户帐号不能为空");
        redisService.setDays(token, userRedis, RedisParam.TIME1);
//        List<Object> objects = redisService.lGet(USER_GROUP_PREFIX + username, 0, -1);
//        List<String> userTokenList = new ArrayList<>();
//        if (!CollectionUtils.isEmpty(objects)) {
//            for (Object object : objects) {
//                userTokenList.add(((String) object));
//            }
//        }
//        if (!userTokenList.contains(token)) {
//            userTokenList.add(token);
//        }
//        redisService.lSet(USER_GROUP_PREFIX + username, userTokenList,60 * 60 * 24);
//        for (String s : userTokenList) {
//            redisService.setDays(s, userRedis, RedisParam.TIME1);
//        }
    }

    public static void setUser(String token, String userName) {
        initRedis();
        Assert.hasText(token, "token不能为空");
        Assert.hasText(userName, "用户帐号不能为空");
        initUserFeign();
        UserRedis userRedis = new UserRedis();
        userRedis.setUsername(userName);
        //预先缓存会话信息，给下面的feign请求使用
        redisService.setDays(token, userRedis, RedisParam.TIME1);
        Result<MdmUserRespVo> detail = mdmCommonUserFeign.detail(null, userName);
        MdmUserRespVo mdmUserRespVo = Optional.ofNullable(detail)
                .map(Result::getResult)
                .orElseThrow(() -> new BusinessException("未查询到用户:" + userName));
        userRedis.setRealname(mdmUserRespVo.getFullName());
        userRedis.setPoscode(mdmUserRespVo.getPositionCode());
        userRedis.setPosname(mdmUserRespVo.getPositionName());
        userRedis.setOrgname(mdmUserRespVo.getOrgName());
        userRedis.setOrgcode(mdmUserRespVo.getOrgCode());
        redisService.setDays(token, userRedis, RedisParam.TIME1);
    }

    public static void remove() {
        initRedis();
        String token = getToken();
        if (!StringUtils.isEmpty(token)) {
            redisService.del(token);
        }
    }

}
