package com.biz.crm.util;

import com.biz.crm.common.param.RedisParam;
import com.biz.crm.mdm.feign.MdmDictFeign;
import com.biz.crm.nebular.mdm.dict.resp.DictAttrConfRedisVo;
import com.biz.crm.nebular.mdm.dict.resp.DictDataRedisVo;
import com.biz.crm.nebular.mdm.dict.resp.DictDataVo;
import com.biz.crm.nebular.mdm.dict.resp.DictGroupRedisVo;
import com.biz.crm.service.RedisService;
import lombok.extern.slf4j.Slf4j;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 数据字典工具类
 **/
@Slf4j
public class DictUtil {

    public static final String PATH_SPLIT = "/";

    private static RedisService redisService;

    private static MdmDictFeign mdmDictFeign;

    public static void setRedisService(RedisService redis) {
        redisService = redis;
    }

    public static void setMdmDictFeign(MdmDictFeign feign) {
        mdmDictFeign = feign;
    }

    /**
     * 根据字典类型编码查询对应的值的集合，查询全部的
     *
     * @param dictTypeCode 字典类型编码
     * @return map
     */
    public static Map<String, String> dictMap(String dictTypeCode) {
        if (StringUtils.isNotEmpty(dictTypeCode)) {
            return fullMap(dictTypeCode);
        }
        return new HashMap<>(16);
    }

    /**
     * 根据字典类型编码集合查询对应的value的map集合，传字典类型编码获取第一层，传路径获取路径下一层
     *
     * @param dictTypeCode
     * @return
     */
    public static Map<String, String> dictSingleMap(String dictTypeCode) {
        Map<String, String> map = new LinkedHashMap<>(16);
        if (StringUtils.isNotEmpty(dictTypeCode)) {
            List<DictDataVo> list = list(dictTypeCode);
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return list.stream().collect(Collectors.toMap(DictDataVo::getDictCode, DictDataVo::getDictValue));
            }
        }
        return map;
    }

    /**
     * 根据字典类型编码集合查询对应的value的map集合，传字典类型编码获取第一层，传路径获取路径下一层
     *
     * @param dictTypeCodes 分类编码集合
     * @return map
     */
    public static Map<String, Map<String, String>> getDictValueMapsByCodes(List<String> dictTypeCodes) {
        Map<String, Map<String, String>> maps = new HashMap<>(16);
        try {
            if (dictTypeCodes != null && dictTypeCodes.size() > 0) {
                for (String dictTypeCode :
                        dictTypeCodes) {
                    maps.put(dictTypeCode, dictMap(dictTypeCode));
                }
            }
        } catch (Exception e) {
            log.error("获取字典数据异常!!!!!!!!");
            e.printStackTrace();
        }
        return maps;
    }

    /**
     * 根据字典类型编码查询对应的值的集合，传字典类型编码获取第一层，传路径获取路径下一层
     *
     * @param dictTypeCode 字典类型编码
     * @return map
     */
    public static Map<String, String> getDictValueMapsByCodes(String dictTypeCode) {
        return dictMap(dictTypeCode);
    }

    /**
     * 获取字典的直接下级Map
     *
     * @param dictTypeCode 数据字典类型编码
     * @param dictCode     字典编码
     * @return
     */
    public static Map<String, String> getChildrenMap(String dictTypeCode, String dictCode) {
        Map<String, String> map = new LinkedHashMap<>(16);
        if (dictTypeCode != null && !"".equals(dictTypeCode) && dictCode != null && !"".equals(dictCode)) {
            map = dictMap(dictTypeCode + PATH_SPLIT + dictCode);
        }
        return map;
    }

    /**
     * 获取扩展字段Map
     *
     * @param dictTypeCode 数据字典类型编码
     * @param dictCode     字典编码
     * @return
     */
    public static Map<String, String> getExtendMap(String dictTypeCode, String dictCode) {
        Map<String, String> map = new LinkedHashMap<>(16);
        if (dictTypeCode != null && !"".equals(dictTypeCode) && dictCode != null && !"".equals(dictCode)) {
            String realDictTypeCode = getRealDictTypeCode(dictTypeCode);
            DictGroupRedisVo group = getDictGroup(realDictTypeCode);
            if (group != null) {
                List<DictAttrConfRedisVo> confList = group.getConfList();
                if (confList != null && confList.size() > 0) {
                    for (DictDataRedisVo data :
                            group.getDataList()) {
                        if (dictCode.equals(data.getDictCode())) {
                            for (DictAttrConfRedisVo conf :
                                    confList) {
                                Object obj = ReflectUtil.getFiledValueByName(data, conf.getExtField());
                                map.put(conf.getFieldCode(), obj == null ? "" : (String) obj);
                            }
                            break;
                        }
                    }
                }
            }
        }
        return map;
    }

    /**
     * 树形，传字典类型编码从第一层开始，传路径编码从路径下级开始
     *
     * @param dictTypeCode
     * @return
     */
    public static List<DictDataVo> tree(String dictTypeCode) {
        List<DictDataVo> tree = new ArrayList<>();
        if (dictTypeCode != null && !"".equals(dictTypeCode)) {
            String realDictTypeCode = getRealDictTypeCode(dictTypeCode);
            List<DictDataVo> fullTree = fullTree(realDictTypeCode);
            if (realDictTypeCode.equals(dictTypeCode)) {
                return fullTree;
            } else {
                return getChildrenFromTree(dictTypeCode, fullTree);
            }
        }
        return tree;
    }

    /**
     * 获取值的list,传字典类型编码获取第一层，传路径获取路径下一层
     *
     * @param dictTypeCode
     * @return
     */
    public static List<DictDataVo> list(String dictTypeCode) {
        List<DictDataVo> tree = tree(dictTypeCode);
        if (CollectionUtil.listNotEmptyNotSizeZero(tree)) {
            tree.forEach(item -> {
                item.setChildren(null);
            });
            return tree;
        }
        return new ArrayList<>();
    }

    /**
     * 获取字典的直接下级List
     *
     * @param dictTypeCode 数据字典类型编码
     * @param dictCode     字典编码
     * @return
     */
    public static List<DictDataVo> getChildrenList(String dictTypeCode, String dictCode) {
        if (dictTypeCode != null && !"".equals(dictTypeCode) && dictCode != null && !"".equals(dictCode)) {
            return list(dictTypeCode + PATH_SPLIT + dictCode);
        }
        return new ArrayList<>();
    }

    /**
     * 根据字典类型编码和字典编码获取字典值
     *
     * @param dictTypeCode 字典类型编码
     * @param dictCode     字典编码
     * @return
     */
    public static String dictValue(String dictTypeCode, String dictCode) {
        if (StringUtils.isNotEmpty(dictTypeCode) && StringUtils.isNotEmpty(dictCode)) {
            Map<String, String> map = fullMap(dictTypeCode);
            if (map.containsKey(dictCode)) {
                return map.get(dictCode);
            }
        }
        return null;
    }

    /**
     * 删除数据字典缓存
     *
     * @param dictTypeCode
     */
    public static void deleteRedisCache(String dictTypeCode) {
        if (StringUtils.isNotEmpty(dictTypeCode)) {
            redisService.del(RedisParam.DICT_TYPE_GROUP + dictTypeCode);
            redisService.del(RedisParam.DICT_TYPE_LOCK + dictTypeCode);
        }
    }

    /*--------------------------------------------------以下为私有方法--------------------------------------------------*/

    /**
     * 根据编码获取缓存group
     *
     * @param dictTypeCode
     * @return
     */
    private static DictGroupRedisVo getDictGroup(String dictTypeCode) {
        if (dictTypeCode != null && !"".equals(dictTypeCode)) {
            Map<String, DictGroupRedisVo> dictGroupMap = getDictGroupMap(Collections.singletonList(dictTypeCode));
            if (!dictGroupMap.isEmpty()) {
                return dictGroupMap.get(dictTypeCode);
            }
        }
        return null;
    }

    /**
     * 根据编码批量获取group集合，先从缓存取，缓存没有再在数据库取，数据库也没有就加锁
     *
     * @param dictTypeCodeList
     * @return
     */
    private static Map<String, DictGroupRedisVo> getDictGroupMap(List<String> dictTypeCodeList) {
        Map<String, DictGroupRedisVo> map = new HashMap<>(16);
        try {
            if (dictTypeCodeList != null && dictTypeCodeList.size() > 0) {
                List<String> unlockCodeList = dictTypeCodeList.stream().filter(
                        dictTypeCode -> !checkLock(dictTypeCode)
                ).collect(Collectors.toList());
                List<String> notCacheCodeList = new ArrayList<>();
                for (String dictTypeCode :
                        unlockCodeList) {
                    Object o = redisService.get(RedisParam.DICT_TYPE_GROUP + dictTypeCode);
                    if (o == null) {
                        notCacheCodeList.add(dictTypeCode);
                    } else {
                        map.put(dictTypeCode, (DictGroupRedisVo) o);
                    }
                }
                if (notCacheCodeList.size() > 0) {
                    Result<List<DictGroupRedisVo>> result = mdmDictFeign.getDictGroupRedisList(dictTypeCodeList);
                    if (result.isSuccess()) {
                        List<DictGroupRedisVo> list = result.getResult();
                        if (list != null && list.size() > 0) {
                            Map<String, DictGroupRedisVo> collect = list.stream().collect(Collectors.toMap(DictGroupRedisVo::getDictTypeCode, v -> v));
                            for (String dictTypeCode :
                                    notCacheCodeList) {
                                if (!collect.containsKey(dictTypeCode)) {
                                    lock(dictTypeCode);
                                } else {
                                    addRedisCache(collect.get(dictTypeCode));
                                    map.put(dictTypeCode, collect.get(dictTypeCode));
                                }
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            log.error("数据字典工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 添加数据字典缓存
     *
     * @param group
     */
    private static void addRedisCache(DictGroupRedisVo group) {
        if (group != null) {
            redisService.set(RedisParam.DICT_TYPE_GROUP + group.getDictTypeCode(), group);
        }
    }

    /**
     * 加锁
     *
     * @param dictTypeCode
     */
    private static void lock(String dictTypeCode) {
        if (dictTypeCode != null && !"".equals(dictTypeCode)) {
            redisService.setSeconds(RedisParam.DICT_TYPE_LOCK + dictTypeCode, dictTypeCode, RedisParam.TIME5);
        }
    }

    /**
     * 检查是否被锁
     *
     * @param dictTypeCode
     * @return
     */
    private static boolean checkLock(String dictTypeCode) {
        if (dictTypeCode != null && !"".equals(dictTypeCode)) {
            return redisService.get(RedisParam.DICT_TYPE_LOCK + dictTypeCode) != null;
        }
        return false;
    }

    /**
     * 根据路径编码分离出真实的字典类型编码
     *
     * @param dictTypeCode
     * @return
     */
    private static String getRealDictTypeCode(String dictTypeCode) {
        if (StringUtils.isNotEmpty(dictTypeCode)) {
            if (dictTypeCode.contains(PATH_SPLIT)) {
                String[] split = dictTypeCode.split(PATH_SPLIT);
                return split[0];
            }
        }
        return dictTypeCode;
    }

    /**
     * 从树中获取某个节点开始的子树
     *
     * @param path
     * @param dataTreeList
     * @return
     */
    private static List<DictDataVo> getChildrenFromTree(String path, List<DictDataVo> dataTreeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(dataTreeList)) {
            for (DictDataVo item :
                    dataTreeList) {
                if (path.equals(item.getPath())) {
                    return item.getChildren();
                }
                return getChildrenFromTree(path, item.getChildren());
            }
        }
        return new ArrayList<>();
    }

    /**
     * 获取字典全部值，从指定节点开始
     *
     * @param dictTypeCode
     * @return
     */
    private static Map<String, String> fullMap(String dictTypeCode) {
        Map<String, String> map = new LinkedHashMap<>(16);
        if (StringUtils.isNotEmpty(dictTypeCode)) {
            String realDictTypeCode = getRealDictTypeCode(dictTypeCode);
            DictGroupRedisVo group = getDictGroup(realDictTypeCode);
            if (group != null) {
                List<DictDataRedisVo> dataList = group.getDataList();
                if (CollectionUtil.listNotEmptyNotSizeZero(dataList)) {
                    map = dataList.stream().collect(Collectors.toMap(DictDataRedisVo::getDictCode, DictDataRedisVo::getDictValue));
                }
            }
        }
        return map;
    }

    /**
     * 获取整棵树，从根节点开始
     *
     * @param dictTypeCode
     * @return
     */
    private static List<DictDataVo> fullTree(String dictTypeCode) {
        List<DictDataVo> tree = new ArrayList<>();
        if (StringUtils.isNotEmpty(dictTypeCode)) {
            String realDictTypeCode = getRealDictTypeCode(dictTypeCode);
            DictGroupRedisVo group = getDictGroup(realDictTypeCode);
            if (group != null) {
                List<DictDataRedisVo> dataList = group.getDataList();
                if (CollectionUtil.listNotEmptyNotSizeZero(dataList)) {
                    tree.addAll(generateTree(dataList, group.getConfList()));
                }
            }
        }
        return tree;
    }

    /**
     * 构建树形结构
     *
     * @param total
     * @return
     */
    private static List<DictDataVo> generateTree(List<DictDataRedisVo> total, List<DictAttrConfRedisVo> confList) {

        if (CollectionUtil.listEmpty(total)) {
            return new ArrayList<>();
        }

        List<DictDataVo> totalList = new ArrayList<>();
        if (CollectionUtil.listNotEmptyNotSizeZero(confList)) {
            totalList.addAll(total.stream().map(data -> {
                DictDataVo copy = CrmBeanUtil.copy(data, DictDataVo.class);
                Map<String, String> extendMap = new LinkedHashMap<>(16);
                for (DictAttrConfRedisVo conf :
                        confList) {
                    Object obj = ReflectUtil.getFiledValueByName(data, conf.getExtField());
                    extendMap.put(conf.getFieldCode(), obj == null ? "" : (String) obj);
                }
                copy.setExtendMap(extendMap);
                return copy;
            }).collect(Collectors.toList()));
        } else {
            totalList.addAll(CrmBeanUtil.copyList(total, DictDataVo.class));
        }

        //构建树list
        List<DictDataVo> treeList = new ArrayList<>();
        //当前操作层级数据
        List<DictDataVo> curLevelList = new ArrayList<>();
        //未操作数据
        List<DictDataVo> restList = new ArrayList<>();

        //key:id
        Map<String, DictDataVo> totalMap = totalList.stream().collect(Collectors.toMap(DictDataVo::getDictCode, v -> v));

        //查找第一层
        for (DictDataVo item : totalList) {
            if (StringUtils.isEmpty(item.getParentDictCode()) || !totalMap.containsKey(item.getParentDictCode())) {
                treeList.add(item);
                curLevelList.add(item);
            } else {
                restList.add(item);
            }
        }

        //构建数据，从第二层开始
        while (curLevelList.size() > 0 && restList.size() > 0) {
            List<DictDataVo> restTempList = new ArrayList<>();
            List<DictDataVo> curLevelTempList = new ArrayList<>();
            Map<String, String> curLevelMap = curLevelList.stream().collect(Collectors.toMap(DictDataVo::getDictCode, DictDataVo::getDictCode));
            Map<String, List<DictDataVo>> curLevelChildrenMap = new HashMap<>();

            for (DictDataVo item : restList) {
                if (curLevelMap.containsKey(item.getParentDictCode())) {
                    curLevelTempList.add(item);

                    List<DictDataVo> childrenList = new ArrayList<>();
                    if (curLevelChildrenMap.containsKey(item.getParentDictCode())) {
                        childrenList.addAll(curLevelChildrenMap.get(item.getParentDictCode()));
                    }
                    childrenList.add(item);
                    curLevelChildrenMap.put(item.getParentDictCode(), childrenList);
                } else {
                    restTempList.add(item);
                }
            }

            for (DictDataVo item : curLevelList) {
                if (curLevelChildrenMap.containsKey(item.getDictCode())) {
                    item.setChildren(curLevelChildrenMap.get(item.getDictCode()));
                }
            }

            curLevelList.clear();
            curLevelList.addAll(curLevelTempList);
            restList.clear();
            restList.addAll(restTempList);
        }

        return treeList;
    }

}
