package com.biz.crm.util;

import com.biz.crm.base.BusinessException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.StringUtils;

import java.beans.PropertyEditorSupport;
import java.sql.Timestamp;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @Author zhangyuzhu
 * @Description
 * @Created data 2018/10/8 11:41
 */
@Slf4j
public abstract class DateUtil extends PropertyEditorSupport {

    public static final SimpleDateFormat date_sdf_yyyy_m = new SimpleDateFormat("yyyy-M");
    // 各种时间格式
    public static final SimpleDateFormat date_sdf = new SimpleDateFormat("yyyy-MM-dd");
    // 各种时间格式
    public static final SimpleDateFormat date_sdf_2 = new SimpleDateFormat("yyyy-MM-dd");
    // 各种时间格式
    public static final SimpleDateFormat yyyyMMdd = new SimpleDateFormat("yyyyMMdd");
    // 各种时间格式
    public static final SimpleDateFormat yyyyM = new SimpleDateFormat("yyyyM");
    // 各种时间格式
    public static final SimpleDateFormat yyyy = new SimpleDateFormat("yyyy");
    // 各种时间格式
    public static final SimpleDateFormat yyyyMM = new SimpleDateFormat("yyyyMM");
    // 各种时间格式
    public static final SimpleDateFormat MM = new SimpleDateFormat("MM");

    public static final SimpleDateFormat yyyy_MM = new SimpleDateFormat("yyyy-MM");
    // 各种时间格式
    public static final SimpleDateFormat date_sdf_wz = new SimpleDateFormat("yyyy年MM月dd日");
    public static final SimpleDateFormat time_sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm");
    public static final SimpleDateFormat yyyymmddhhmmss = new SimpleDateFormat("yyyyMMddHHmmss");
    public static final SimpleDateFormat short_time_sdf = new SimpleDateFormat("HH:mm:ss");
    public static final SimpleDateFormat datetimeFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    public static final SimpleDateFormat time_ymdh = new SimpleDateFormat("yyyy-MM-dd HH");
    // 以毫秒表示的时间
    public static final long DAY_IN_MILLIS = 24 * 3600 * 1000;
    private static final long HOUR_IN_MILLIS = 3600 * 1000;
    private static final long MINUTE_IN_MILLIS = 60 * 1000;
    private static final long SECOND_IN_MILLIS = 1000;
    public static final int MINIMAL_DAYS_IN_FIRSTWEEK = 4;
    public static final int FIRST_DAY_OF_WEEK = 2;


    public static final long ONE_DAY_MILLISECONDS = 24 * 60 * 60 * 1000;
    public static final String DEFAULT_DATE_PATTERN = "yyyy-MM-dd HH:mm";
    public static final String DEFAULT_DAY_PATTERN = "yyyy-MM-dd";
    public static final String DEFAULT_MONTH_PATTERN = "yyyy年MM月";
    public static final String DEFAULT_MONTH_NO_CH_PATTERN = "yyyyMM";
    public static final String DEFAULT_MONTH_DAY_PATTERN = "yyyyMMdd";
    public static final String DEFAULT_MONTH_DAY_PATTERN_ = "yyyy-MM";
    public static final String DEFAULT_TIME_PATTERN = "HH:mm";
    public static final String DEFAULT_DATE_ALL_PATTERN = "yyyy-MM-dd HH:mm:ss";
    public static final String DEFAULT_TIME_ALL_PATTERN = "HH:mm:ss";
    public static final String DEFAULT_DATE_TIME_PATTERN = "yyyyMMddHHmmss";

    /**
     * 每天最早的时刻
     */
    public static final String DAY_EARLIEST_TIME = "00:00:00";

    /**
     * 每天最晚的时刻
     */
    public static final String DAY_LATEST_TIME = "23:59:59";


    public static String getYYYYM(Date date) {
        return yyyyM.format(date);
    }

    public static String getYYYYMMDD() {
        return new SimpleDateFormat(DEFAULT_DAY_PATTERN).format(new Date());
    }

    public static String getYYYYMMDD(Date date) {
        return date_sdf.format(date);
    }

    public static Date getDateYYYYMMDD(String date) {
        try {
            return date_sdf.parse(date);
        } catch (ParseException e) {
            // TODO Auto-generated catch block
            log.error("日期处理异常", e);
        }
        return null;
    }

    public static Date getDateYYYYMD(String date) throws ParseException {
        return date_sdf_2.parse(date);
    }

    // 指定模式的时间格式
    public static SimpleDateFormat getSDFormat(String pattern) {
        return new SimpleDateFormat(pattern);
    }

    public static String getFormatDateStr(Date date, String pattern) {
        if (date == null) {
            return null;
        }
        DateFormat dtFmt = new SimpleDateFormat(pattern, Locale.US);
        return dtFmt.format(date);
    }

    /**
     * 当前日历，这里用中国时间表示
     *
     * @return 以当地时区表示的系统当前日历
     */
    public static Calendar getCalendar() {
        return Calendar.getInstance();
    }

    /**
     * 指定毫秒数表示的日历
     *
     * @param millis 毫秒数
     * @return 指定毫秒数表示的日历
     */
    public static Calendar getCalendar(long millis) {
        Calendar cal = Calendar.getInstance();
        // --------------------cal.setTimeInMillis(millis);
        cal.setTime(new Date(millis));
        return cal;
    }

    // ////////////////////////////////////////////////////////////////////////////
    // getDate
    // 各种方式获取的Date
    // ////////////////////////////////////////////////////////////////////////////

    /**
     * 当前日期
     *
     * @return 系统当前时间
     */
    public static Date getDate() {
        return new Date();
    }

    /**
     * 指定毫秒数表示的日期
     *
     * @param millis 毫秒数
     * @return 指定毫秒数表示的日期
     */
    public static Date getDate(long millis) {
        return new Date(millis);
    }

    /**
     * 时间戳转换为字符串
     *
     * @param time
     * @return
     */
    public static String timestamptoStr(Timestamp time) {
        Date date = null;
        if (null != time) {
            date = new Date(time.getTime());
        }
        return date2Str(date_sdf);
    }

    /**
     * 字符串转换时间戳
     *
     * @param str
     * @return
     */
    public static Timestamp str2Timestamp(String str) {
        Date date = str2Date(str, date_sdf);
        return new Timestamp(date.getTime());
    }

    /**
     * 字符串转换成日期
     *
     * @param str
     * @param sdf
     * @return
     */
    public static Date str2Date(String str, SimpleDateFormat sdf) {
        if (null == str || "".equals(str)) {
            return null;
        }
        Date date = null;
        try {
            date = sdf.parse(str);
            return date;
        } catch (ParseException e) {
            log.error("日期处理异常", e);
        }
        return null;
    }

    /**
     * 计算两个时间的中间的月数
     *
     * @param date1
     * @param date2
     * @return
     */
    public static int getMonthSpace(String date1, String date2, String format) {
        try {
            int result = 0;
            SimpleDateFormat sdf = new SimpleDateFormat(format);
            Calendar c1 = Calendar.getInstance();
            Calendar c2 = Calendar.getInstance();
            c1.setTime(sdf.parse(date1));
            c2.setTime(sdf.parse(date2));
            result = (c1.get(Calendar.YEAR) - c2.get(Calendar.YEAR)) * 12 + c1.get(Calendar.MONTH)
                    - c2.get(Calendar.MONTH);
            return result == 0 ? 1 : Math.abs(result);
        } catch (Exception e) {
            log.error("日期处理异常", e);
            throw new BusinessException("时间处理失败");
        }
    }


    /**
     * 日期转换为字符串
     *
     * @param date_sdf 日期格式
     * @return 字符串
     */
    public static String date2Str(SimpleDateFormat date_sdf) {
        Date date = getDate();
        if (null == date) {
            return null;
        }
        return date_sdf.format(date);
    }

    /**
     * 得到当前时间
     */
    public static String dateNow2Str() {
        return date2Str(date_sdf);
    }

    /**
     * 得到当前时间 HH:mm:ss
     */
    public static String dateNowStrHms() {
        return date2Str(short_time_sdf);
    }

    /**
     * 格式化时间
     *
     * @param date
     * @param format
     * @return
     */
    public static String dateformat(String date, String format) {
        SimpleDateFormat sformat = new SimpleDateFormat(format);
        Date _date = null;
        try {
            _date = sformat.parse(date);
        } catch (ParseException e) {
            // TODO Auto-generated catch block
            log.error("日期处理异常", e);
        }
        return sformat.format(_date);
    }

    /**
     * 日期转换为字符串
     *
     * @param date     日期
     * @param date_sdf 日期格式
     * @return 字符串
     */
    public static String date2Str(Date date, SimpleDateFormat date_sdf) {
        if (null == date) {
            return null;
        }
        return date_sdf.format(date);
    }

    /**
     * 日期转换为字符串
     *
     * @param format 日期格式
     * @return 字符串
     */
    public static String getDate(String format) {
        Date date = new Date();
        if (null == date) {
            return null;
        }
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        return sdf.format(date);
    }

    /**
     * 日期转换为字符串
     *
     * @param format 日期格式
     * @return 字符串
     */
    public static Date getDate(SimpleDateFormat format) {
        Date date = new Date();
        String nowDate = format.format(date);

        try {
            return format.parse(nowDate);
        } catch (ParseException e) {
            log.error("日期处理异常", e);
        }
        return null;
    }

    /**
     * 指定毫秒数的时间戳
     *
     * @param millis 毫秒数
     * @return 指定毫秒数的时间戳
     */
    public static Timestamp getTimestamp(long millis) {
        return new Timestamp(millis);
    }

    /**
     * 以字符形式表示的时间戳
     *
     * @param time 毫秒数
     * @return 以字符形式表示的时间戳
     */
    public static Timestamp getTimestamp(String time) {
        return new Timestamp(Long.parseLong(time));
    }

    /**
     * 系统当前的时间戳
     *
     * @return 系统当前的时间戳
     */
    public static Timestamp getTimestamp() {
        return new Timestamp(new Date().getTime());
    }

    /**
     * 指定日期的时间戳
     *
     * @param date 指定日期
     * @return 指定日期的时间戳
     */
    public static Timestamp getTimestamp(Date date) {
        return new Timestamp(date.getTime());
    }

    /**
     * 指定日历的时间戳
     *
     * @param cal 指定日历
     * @return 指定日历的时间戳
     */
    public static Timestamp getCalendarTimestamp(Calendar cal) {
        // ---------------------return new Timestamp(cal.getTimeInMillis());
        return new Timestamp(cal.getTime().getTime());
    }

    public static Timestamp gettimestamp() {
        Date dt = new Date();
        DateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        String nowTime = df.format(dt);
        Timestamp buydate = Timestamp.valueOf(nowTime);
        return buydate;
    }

    public static Timestamp getDateTimestamp() {
        Date dt = new Date();
        DateFormat df = new SimpleDateFormat("yyyy-MM-dd");
        String nowTime = df.format(dt);

        return str2Timestamp(nowTime);
    }

    // ////////////////////////////////////////////////////////////////////////////
    // getMillis
    // 各种方式获取的Millis
    // ////////////////////////////////////////////////////////////////////////////

    /**
     * 系统时间的毫秒数
     *
     * @return 系统时间的毫秒数
     */
    public static long getMillis() {
        return new Date().getTime();
    }

    /**
     * 指定日历的毫秒数
     *
     * @param cal 指定日历
     * @return 指定日历的毫秒数
     */
    public static long getMillis(Calendar cal) {
        // --------------------return cal.getTimeInMillis();
        return cal.getTime().getTime();
    }

    /**
     * 指定日期的毫秒数
     *
     * @param date 指定日期
     * @return 指定日期的毫秒数
     */
    public static long getMillis(Date date) {
        return date.getTime();
    }

    /**
     * 指定时间戳的毫秒数
     *
     * @param ts 指定时间戳
     * @return 指定时间戳的毫秒数
     */
    public static long getMillis(Timestamp ts) {
        return ts.getTime();
    }

    // ////////////////////////////////////////////////////////////////////////////
    // formatDate
    // 将日期按照一定的格式转化为字符串
    // ////////////////////////////////////////////////////////////////////////////

    /**
     * 默认方式表示的系统当前日期，具体格式：年-月-日
     *
     * @return 默认日期按“年-月-日“格式显示
     */
    public static String formatDate() {
        return new SimpleDateFormat("yyyy-MM-dd").format(getCalendar().getTime());
    }

    /**
     * 获取时间字符串
     */
    public static String getDataString(SimpleDateFormat formatstr) {
        return formatstr.format(getCalendar().getTime());
    }

    /**
     * 指定日期的默认显示，具体格式：年-月-日
     *
     * @param cal 指定的日期
     * @return 指定日期按“年-月-日“格式显示
     */
    public static String formatDate(Calendar cal) {
        return date_sdf.format(cal.getTime());
    }

    /**
     * 指定日期的默认显示，具体格式：年-月-日
     *
     * @param date 指定的日期
     * @return 指定日期按“年-月-日“格式显示
     */
    public static String formatDate(Date date) {
        return date_sdf.format(date);
    }

    /**
     * 指定毫秒数表示日期的默认显示，具体格式：年-月-日
     *
     * @param millis 指定的毫秒数
     * @return 指定毫秒数表示日期按“年-月-日“格式显示
     */
    public static String formatDate(long millis) {
        return date_sdf.format(new Date(millis));
    }

    /**
     * 默认日期按指定格式显示
     *
     * @param pattern 指定的格式
     * @return 默认日期按指定格式显示
     */
    public static String formatDate(String pattern) {
        return getSDFormat(pattern).format(getCalendar().getTime());
    }

    /**
     * 指定日期按指定格式显示
     *
     * @param cal     指定的日期
     * @param pattern 指定的格式
     * @return 指定日期按指定格式显示
     */
    public static String formatDate(Calendar cal, String pattern) {
        return getSDFormat(pattern).format(cal.getTime());
    }

    /**
     * 指定日期按指定格式显示
     *
     * @param date    指定的日期
     * @param pattern 指定的格式
     * @return 指定日期按指定格式显示
     */
    public static String formatDate(Date date, String pattern) {
        return getSDFormat(pattern).format(date);
    }

    // ////////////////////////////////////////////////////////////////////////////
    // formatTime
    // 将日期按照一定的格式转化为字符串
    // ////////////////////////////////////////////////////////////////////////////

    /**
     * 默认方式表示的系统当前日期，具体格式：年-月-日 时：分
     *
     * @return 默认日期按“年-月-日 时：分“格式显示
     */
    public static String formatTime() {
        return time_sdf.format(getCalendar().getTime());
    }

    /**
     * 指定毫秒数表示日期的默认显示，具体格式：年-月-日 时：分
     *
     * @param millis 指定的毫秒数
     * @return 指定毫秒数表示日期按“年-月-日 时：分“格式显示
     */
    public static String formatTime(long millis) {
        return time_sdf.format(new Date(millis));
    }

    /**
     * 指定日期的默认显示，具体格式：年-月-日 时：分
     *
     * @param cal 指定的日期
     * @return 指定日期按“年-月-日 时：分“格式显示
     */
    public static String formatTime(Calendar cal) {
        return time_sdf.format(cal.getTime());
    }

    /**
     * 指定日期的默认显示，具体格式：年-月-日 时：分
     *
     * @param date 指定的日期
     * @return 指定日期按“年-月-日 时：分“格式显示
     */
    public static String formatTime(Date date) {
        return time_sdf.format(date);
    }

    // ////////////////////////////////////////////////////////////////////////////
    // formatShortTime
    // 将日期按照一定的格式转化为字符串
    // ////////////////////////////////////////////////////////////////////////////

    /**
     * 默认方式表示的系统当前日期，具体格式：时：分
     *
     * @return 默认日期按“时：分“格式显示
     */
    public static String formatShortTime() {
        return new SimpleDateFormat("HH:mm:ss").format(getCalendar().getTime());
    }

    /**
     * 指定毫秒数表示日期的默认显示，具体格式：时：分
     *
     * @param millis 指定的毫秒数
     * @return 指定毫秒数表示日期按“时：分“格式显示
     */
    public static String formatShortTime(long millis) {
        return short_time_sdf.format(new Date(millis));
    }

    /**
     * 指定日期的默认显示，具体格式：时：分
     *
     * @param cal 指定的日期
     * @return 指定日期按“时：分“格式显示
     */
    public static String formatShortTime(Calendar cal) {
        return short_time_sdf.format(cal.getTime());
    }

    /**
     * 指定日期的默认显示，具体格式：时：分
     *
     * @param date 指定的日期
     * @return 指定日期按“时：分“格式显示
     */
    public static String formatShortTime(Date date) {
        return short_time_sdf.format(date);
    }

    // ////////////////////////////////////////////////////////////////////////////
    // parseDate
    // parseCalendar
    // parseTimestamp
    // 将字符串按照一定的格式转化为日期或时间
    // ////////////////////////////////////////////////////////////////////////////

    /**
     * 根据指定的格式将字符串转换成Date 如输入：2003-11-19 11:20:20将按照这个转成时间
     *
     * @param src     将要转换的原始字符窜
     * @param pattern 转换的匹配格式
     * @return 如果转换成功则返回转换后的日期
     * @throws ParseException
     * @throws ParseException
     */
    public static Date parseDate(String src, String pattern) {
        try {
            return getSDFormat(pattern).parse(src);
        } catch (Exception e) {
            log.error("日期处理异常", e);
            throw new BusinessException("时间处理失败");
        }


    }

    /**
     * 根据指定的格式将字符串转换成Date 如输入：2003-11-19 11:20:20将按照这个转成时间
     *
     * @param src     将要转换的原始字符窜
     * @param pattern 转换的匹配格式
     * @return 如果转换成功则返回转换后的日期
     * @throws ParseException
     * @throws ParseException
     */
    public static Calendar parseCalendar(String src, String pattern) throws ParseException {

        Date date = parseDate(src, pattern);
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        return cal;
    }

    public static String formatAddDate(String src, String pattern, int amount) throws ParseException {
        Calendar cal;
        cal = parseCalendar(src, pattern);
        cal.add(Calendar.DATE, amount);
        return formatDate(cal);
    }

    public static String formatAddMonth(String src, String pattern, int month) throws ParseException {
        Calendar cal;
        cal = parseCalendar(src, pattern);
        cal.add(Calendar.MONTH, month);
        return formatDate(cal);
    }

    public static Date formatAddMonth(Date src, int month) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(src);
        calendar.add(Calendar.MONTH, month);
        return calendar.getTime();
    }


    /**
     * 根据指定的格式将字符串转换成Date 如输入：2003-11-19 11:20:20将按照这个转成时间
     *
     * @param src     将要转换的原始字符窜
     * @param pattern 转换的匹配格式
     * @return 如果转换成功则返回转换后的时间戳
     * @throws ParseException
     * @throws ParseException
     */
    public static Timestamp parseTimestamp(String src, String pattern) throws ParseException {
        Date date = parseDate(src, pattern);
        return new Timestamp(date.getTime());
    }

    // ////////////////////////////////////////////////////////////////////////////
    // dateDiff
    // 计算两个日期之间的差值
    // ////////////////////////////////////////////////////////////////////////////

    /**
     * 计算两个时间之间的差值，根据标志的不同而不同
     *
     * @param flag   计算标志，表示按照年/月/日/时/分/秒等计算
     * @param calSrc 减数
     * @param calDes 被减数
     * @return 两个日期之间的差值
     */
    public static int dateDiff(char flag, Calendar calSrc, Calendar calDes) {

        long millisDiff = getMillis(calSrc) - getMillis(calDes);

        if (flag == 'y') {
            return (calSrc.get(Calendar.YEAR) - calDes.get(Calendar.YEAR));
        }

        if (flag == 'd') {
            return (int) (millisDiff / DAY_IN_MILLIS);
        }

        if (flag == 'h') {
            return (int) (millisDiff / HOUR_IN_MILLIS);
        }

        if (flag == 'm') {
            return (int) (millisDiff / MINUTE_IN_MILLIS);
        }

        if (flag == 's') {
            return (int) (millisDiff / SECOND_IN_MILLIS);
        }

        return 0;
    }

    /**
     * String类型 转换为Date, 如果参数长度为10 转换格式”yyyy-MM-dd“ 如果参数长度为19 转换格式”yyyy-MM-dd HH:mm:ss“ * @param text String类型的时间值
     */
    @Override
    public void setAsText(String text) throws IllegalArgumentException {
        if (StringUtils.hasText(text)) {
            try {
                if (text.indexOf(":") == -1 && text.length() == 10) {
                    setValue(DateUtil.date_sdf.parse(text));
                } else if (text.indexOf(":") > 0 && text.length() == 19) {
                    setValue(DateUtil.datetimeFormat.parse(text));
                } else {
                    throw new IllegalArgumentException("Could not parse date, date format is error ");
                }
            } catch (ParseException ex) {
                IllegalArgumentException iae = new IllegalArgumentException("Could not parse date: " + ex.getMessage(), ex);
                throw iae;
            }
        } else {
            setValue(null);
        }
    }

    /**
     * 获取当前年份
     *
     * @return
     */
    public static int getCurrentYear() {
        GregorianCalendar calendar = new GregorianCalendar();
        calendar.setTime(getDate());
        return calendar.get(Calendar.YEAR);
    }

    /**
     * 获取当前月份
     *
     * @return
     */
    public static int getCurrentMonth() {
        GregorianCalendar calendar = new GregorianCalendar();
        calendar.setTime(getDate());
        return calendar.get(Calendar.MONTH);
    }

    /**
     * 两个时间减多少天
     */
    public static long subtractDay(Date endDate, Date startDate) {
        return (endDate.getTime() - startDate.getTime()) / (24 * 60 * 60 * 1000);
    }

    /**
     * 返回Date参数的 前后N天日期 取决于第二个参数
     * 返回date
     *
     * @param date
     * @param i
     * @return
     */
    public static Date ReturnTheDay(Date date, int i) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.DAY_OF_MONTH, i);
        return calendar.getTime();
    }

    public static String format(Date date, String pattern) {
        if (date != null) {
            try {
                return new SimpleDateFormat(pattern).format(date);
            } catch (Exception ex) {
                log.error("日期处理异常", ex);
            }
        }
        return "";
    }

    public static String format(Object value, String pattern) {
        if (value != null) {
            try {
                Date date = null;
//				if(value instanceof oracle.sql.TIMESTAMP) {
//					date = new Date(((oracle.sql.TIMESTAMP)value).dateValue().getTime());
//				} else {
//					date = (Date)value;
//				}
                return new SimpleDateFormat(pattern).format(date);
            } catch (Exception ex) {
                log.error("日期处理异常", ex);
            }
        }
        return "";
    }

    public static Date getDateTime(String dateTime) {
        Date date = new Date();
        String datePart = DateUtil.format(date, "yyyy-MM-dd");
        String dateTimePart = datePart + " " + dateTime;
        Date newDate = DateUtil.parse(dateTimePart, "yyyy-MM-dd HH:mm");
        return newDate;
    }

    public static Date parse(String date, String pattern) {
        Date newDate = null;
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(pattern);
            sdf.setLenient(false);
            newDate = sdf.parse(date);
        } catch (Exception ex) {
            log.error("日期处理异常", ex);
            throw new BusinessException("日期转换错误：" + date + "，转换为：" + pattern);
        }
        return newDate;
    }

    /**
     * @描述: 得到date的昨天
     */
    public static Date getYesterdayByDate(String date, String pattern) {
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(pattern);
            sdf.setLenient(false);
            Date temp = sdf.parse(date);
            Calendar cal = Calendar.getInstance();
            cal.setTime(temp);
            cal.add(Calendar.DAY_OF_MONTH, -1);
            return cal.getTime();
        } catch (Exception ex) {
        }
        return null;
    }

    /**
     * @描述: 得到某一月的天数
     */
    public static int getDaysOfMonth(int year, int month) {
        Calendar cal = Calendar.getInstance();
        cal.set(year, month - 1, 1);
        return cal.getActualMaximum(Calendar.DAY_OF_MONTH);
    }

    /**
     * @描述: 得到某一月的天数
     */
    public static int getDaysOfMonth(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        return cal.getActualMaximum(Calendar.DAY_OF_MONTH);
    }

    /**
     * @描述: 得到某一季的天数
     */
    public static int getDaysOfQuarter(int year, int quarter) {
        Calendar cal = Calendar.getInstance();
        int days = 0;
        int firstMonth = (quarter - 1) * 3;
        for (int i = firstMonth; i < firstMonth + 3; i++) {
            cal.set(year, i, 1);
            days += cal.getActualMaximum(Calendar.DAY_OF_MONTH);
        }
        return days;
    }

    /**
     * @描述: 得到某一年的天数
     */
    public static int getDaysOfYear(int year) {
        Calendar cal = Calendar.getInstance();
        cal.set(year, 0, 1);
        return cal.getActualMaximum(Calendar.DAY_OF_YEAR);
    }

    /**
     * @描述: 得到某一年的天数
     */
    public static int getDaysOfYear(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        return cal.getActualMaximum(Calendar.DAY_OF_YEAR);
    }

    /**
     * @描述: 得到某一周的星期一的时间
     */
    public static Date getFirstDayOfWeek(int year, int week) {
        Calendar cal = Calendar.getInstance();
        cal.setFirstDayOfWeek(FIRST_DAY_OF_WEEK);
        cal.setMinimalDaysInFirstWeek(MINIMAL_DAYS_IN_FIRSTWEEK);
        cal.set(year, Calendar.JANUARY, 1);
        int addDays = week * 7;
        if (cal.get(Calendar.WEEK_OF_YEAR) == 1) {
            addDays = (week - 1) * 7;
        }
        cal.add(Calendar.DATE, addDays);
        cal.set(Calendar.DAY_OF_WEEK, cal.getFirstDayOfWeek());
        return cal.getTime();

    }

    /**
     * @描述: 得到某一周的第一天
     */
    public static Date getFirstDayOfWeek(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setFirstDayOfWeek(FIRST_DAY_OF_WEEK);
        cal.setMinimalDaysInFirstWeek(MINIMAL_DAYS_IN_FIRSTWEEK);
        cal.setTime(date);
        int dayOfWeek = cal.get(Calendar.DAY_OF_WEEK);
        if (dayOfWeek == 1) {
            dayOfWeek = 8;
        }

        cal.setTimeInMillis((cal.getTimeInMillis() + (FIRST_DAY_OF_WEEK - dayOfWeek) * 86400000L));
        return cal.getTime();
    }

    /**
     * @描述: 得到某一月的第一天
     */
    public static Date getFirstDayOfMonth(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.set(Calendar.DAY_OF_MONTH, 1);
        return cal.getTime();
    }

    /**
     * @描述: 得到某一月的最后一天
     */
    public static Date getLastDayOfMonth(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.set(Calendar.DAY_OF_MONTH, cal.getActualMaximum(Calendar.DAY_OF_MONTH));
        return cal.getTime();
    }

    /**
     * @描述: 得到上月的第一天
     */
    public static Date getFirstDayOfLastMonth() {
        Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(cal.getTimeInMillis() - cal.get(Calendar.DAY_OF_MONTH) * 86400000L);
        cal.set(Calendar.DAY_OF_MONTH, 1);
        return cal.getTime();
    }

    /**
     * @描述: 得到上月的最后一天
     */
    public static Date getLastDayOfLastMonth() {
        Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(cal.getTimeInMillis() - cal.get(Calendar.DAY_OF_MONTH) * 86400000L);
        return cal.getTime();
    }

    /**
     * @描述: 得到上月的最后一天
     */
    public static Date getLastDayOfLastMonth(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.setTimeInMillis(cal.getTimeInMillis() - cal.get(Calendar.DAY_OF_MONTH) * 86400000L);
        return cal.getTime();
    }

    /**
     * @描述: 得到下月的第一天
     */
    public static Date getFirstDayOfNextMonth() {
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.DAY_OF_MONTH, 1);
        cal.setTimeInMillis(cal.getTimeInMillis() + cal.getActualMaximum(Calendar.DAY_OF_MONTH) * 86400000L);
        return cal.getTime();
    }

    /**
     * @描述: 得到下月的第一天
     */
    public static Date getFirstDayOfNextMonth(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.set(Calendar.DAY_OF_MONTH, 1);
        cal.setTimeInMillis(cal.getTimeInMillis() + cal.getActualMaximum(Calendar.DAY_OF_MONTH) * 86400000L);
        return cal.getTime();
    }

    /**
     * @描述: 得到下月的最后一天
     */
    public static Date getLastDayOfNextMonth() {
        Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(cal.getTimeInMillis() + cal.getActualMaximum(Calendar.DAY_OF_MONTH) * 86400000L);
        cal.set(Calendar.DAY_OF_MONTH, 1);
        cal.setTimeInMillis(cal.getTimeInMillis() + (cal.getActualMaximum(Calendar.DAY_OF_MONTH) - 1) * 86400000L);
        return cal.getTime();
    }


    /**
     * @描述: 得到某一日期的周数
     */
    public static int getWeekOfDate(int year, int month, int day) {
        Calendar cal = Calendar.getInstance();
        cal.setFirstDayOfWeek(FIRST_DAY_OF_WEEK);
        cal.setMinimalDaysInFirstWeek(MINIMAL_DAYS_IN_FIRSTWEEK);
        cal.set(year, month - 1, day, 0, 0, 0);
        return cal.get(Calendar.WEEK_OF_YEAR);

    }

    /**
     * @描述: 得到某一日期的天数
     */
    public static int getDayOfDate(int year, int month, int day) {
        Calendar cal = Calendar.getInstance();
        cal.set(year, month - 1, day, 0, 0, 0);
        return cal.get(Calendar.DAY_OF_YEAR);
    }

    public static Date getDate2() {
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);
        return cal.getTime();
    }

    public static Date getDate(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);
        return cal.getTime();
    }

    public static Date yesterday() {
        return new Date(System.currentTimeMillis() - 86400000L);
    }

    public static Date tomorrow() {
        return new Date(System.currentTimeMillis() + 86400000L);
    }

    public static Date getEndDate() {
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.YEAR, 2099);
        cal.set(Calendar.MONTH, 12 - 1);
        cal.set(Calendar.DATE, 31);
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);
        return cal.getTime();
    }

    public static Date parseToDate(Object value) {
        if (value instanceof Timestamp) {
            return (Date) value;
        }
        if (value instanceof java.sql.Date) {
            return (Date) value;
        }
        return (Date) value;

    }

    /**
     * 获取指定日期时间
     *
     * @param calendarConstant 指定时间类型
     * @param appointNum       指定时间
     * @return
     */
    public static Date getAppointDate(int calendarConstant, int appointNum) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        calendar.add(calendarConstant, appointNum);
        return calendar.getTime();
    }

    public static String calSpendTimes(String startTime, String endTime) {
        Date startTimeDateType = DateUtil.parse(startTime, "yyyy-MM-dd HH:mm:ss");

        Date endTimeDateType = null;
        if (endTime != null) {
            endTimeDateType = DateUtil.parse(endTime, "yyyy-MM-dd HH:mm:ss");
        }

        long totalTimes = (endTime == null) ?
                (Calendar.getInstance().getTimeInMillis() - startTimeDateType.getTime())
                :
                (endTimeDateType.getTime() - startTimeDateType.getTime());

        long dayCount = totalTimes / (1000 * 60 * 60 * 24);//计算天
        long restTimes = totalTimes % (1000 * 60 * 60 * 24);//剩下的时间用于计于小时
        long hourCount = restTimes / (1000 * 60 * 60);//小时
        restTimes = restTimes % (1000 * 60 * 60);
        long minuteCount = restTimes / (1000 * 60);

        String spendTimes = dayCount + "天" + hourCount + "小时" + minuteCount + "分";

        return spendTimes;
    }

    /**
     * 拆分两个日期获取月份
     *
     * @param minDate
     * @param maxDate
     * @return
     * @throws ParseException
     */
    public static List<String> getMonthBetween(String minDate, String maxDate) {
        ArrayList<String> result = new ArrayList<String>();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM");// 格式化为年月

        Calendar min = Calendar.getInstance();
        Calendar max = Calendar.getInstance();

        try {
            min.setTime(sdf.parse(minDate));
            min.set(min.get(Calendar.YEAR), min.get(Calendar.MONTH), 1);

            max.setTime(sdf.parse(maxDate));
            max.set(max.get(Calendar.YEAR), max.get(Calendar.MONTH), 2);

            Calendar curr = min;
            while (curr.before(max)) {
                result.add(sdf.format(curr.getTime()));
                curr.add(Calendar.MONTH, 1);
            }
        } catch (ParseException e) {
            log.error("日期处理异常", e);
        }
        return result;
    }

    static int[] DAYS = {0, 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

    /**
     * @param date yyyy-MM-dd HH:mm:ss
     * @return
     */
    public static boolean isValidDateTime(String date) {
        try {
            int year = Integer.parseInt(date.substring(0, 4));
            if (year <= 0)
                return false;
            int month = Integer.parseInt(date.substring(5, 7));
            if (month <= 0 || month > 12)
                return false;
            int day = Integer.parseInt(date.substring(8, 10));
            if (day <= 0 || day > DAYS[month])
                return false;
            if (month == 2 && day == 29 && isGregorianLeapYear(year)) {
                return false;
            }
            int hour = Integer.parseInt(date.substring(11, 13));
            if (hour < 0 || hour > 23)
                return false;
            int minute = Integer.parseInt(date.substring(14, 16));
            if (minute < 0 || minute > 59)
                return false;
            int second = Integer.parseInt(date.substring(17, 19));
            if (second < 0 || second > 59)
                return false;

        } catch (Exception e) {
            log.error("日期处理异常", e);
            return false;
        }
        return true;
    }

    /**
     * @param date yyyy-MM-dd
     * @return
     */
    public static boolean isValidMoth(String date) {
        try {
            int year = Integer.parseInt(date.substring(0, 4));
            if (year <= 0)
                return false;
            int month = Integer.parseInt(date.substring(5, 7));
            if (month <= 0 || month > 12)
                return false;
        } catch (Exception e) {
            log.error("日期处理异常", e);
            return false;
        }
        return true;
    }

    /**
     * @param date yyyy-MM-dd
     * @return
     */
    public static boolean isValidDate(String date) {
        try {
            int year = Integer.parseInt(date.substring(0, 4));
            if (year <= 0)
                return false;
            int month = Integer.parseInt(date.substring(5, 7));
            if (month <= 0 || month > 12)
                return false;
            int day = Integer.parseInt(date.substring(8, 10));
            if (day <= 0 || day > DAYS[month])
                return false;
            if (month == 2 && day == 29 && isGregorianLeapYear(year)) {
                return false;
            }
        } catch (Exception e) {
            log.error("日期处理异常", e);
            return false;
        }
        return true;
    }

    public static boolean isGregorianLeapYear(int year) {
        return year % 4 != 0 || (year % 100 == 0 && year % 400 != 0);
    }

    /**
     * 时间转换器 月末与月初
     *
     * @param fromdate
     * @param type     0表示当前月第一天 1当前月最后一天
     */
    public static Date switchDate(Date fromdate, int type) {
        if (fromdate != null) {
            if (type == 0) {
                GregorianCalendar gcLast = (GregorianCalendar) Calendar
                        .getInstance();
                gcLast.setTime(fromdate);
                gcLast.set(Calendar.HOUR_OF_DAY, 0);
                gcLast.set(Calendar.MINUTE, 0);
                gcLast.set(Calendar.SECOND, 0);
                // 设置为第一天
                gcLast.set(Calendar.DAY_OF_MONTH, 1);
                return gcLast.getTime();
            } else {
                // 获取Calendar
                Calendar calendar = Calendar.getInstance();
                calendar.setTime(fromdate);
                calendar.set(Calendar.HOUR_OF_DAY, 23);
                calendar.set(Calendar.MINUTE, 59);
                calendar.set(Calendar.SECOND, 59);
                // 设置日期为本月最大日期
                calendar.set(Calendar.DATE,
                        calendar.getActualMaximum(Calendar.DATE));
                // 设置日期格式
                return calendar.getTime();
            }
        }
        return null;
    }

    /**
     * 获取指定月的前一月（年）或后一月（年）
     *
     * @param dateStr
     * @param addYear
     * @param addMonth 月份为 实际月份减一
     * @param addDate
     * @param ymd      1月 2 年 3月最后一天 4年最后一天
     * @return
     * @throws Exception 方法的简述.
     * @throws Exception
     */
    public static String getAnyDay(String dateStr, int addYear, int addMonth,
                                   int addDate, int ymd) throws Exception {
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(
                    "yyyy");
            Date sourceDate = sdf.parse(dateStr);
            SimpleDateFormat sdf_ = new SimpleDateFormat(
                    "yyyy-MM");
            Date sourceDate_ = sdf_.parse(dateStr);
            Calendar cal = Calendar.getInstance();
            cal.clear();
            cal.setTime(sourceDate);
            cal.add(Calendar.YEAR, addYear);
            cal.add(Calendar.MONTH, sourceDate_.getMonth() + addMonth);
            if (ymd == 1) {// 月
                cal.add(Calendar.DAY_OF_MONTH, 0);
            } else if (ymd == 2) {
                cal.set(Calendar.DAY_OF_YEAR, 1);
            } else if (ymd == 3) {
                cal.set(Calendar.DAY_OF_MONTH,
                        cal.getActualMaximum(Calendar.DAY_OF_MONTH));
                System.out.println(cal.getTime().getMonth());
            } else {
                cal.set(Calendar.YEAR, cal.getTime().getYear() + 1900);
                cal.set(Calendar.DAY_OF_YEAR,
                        cal.getActualMaximum(Calendar.DAY_OF_YEAR));
            }
            cal.add(Calendar.DATE, addDate);

            SimpleDateFormat returnSdf = new SimpleDateFormat(
                    "yyyy-MM-dd");

            String dateTmp = returnSdf.format(cal.getTime());
            return dateTmp;
        } catch (Exception e) {
            log.error("日期处理异常", e);
            throw new Exception(e.getMessage());
        }
    }

    /**
     * 获取指定月的第一天
     * <p>指定月的第一天<br>
     * 方法的详细说明第二行
     *
     * @param date String格式yyyy-MM
     * @return
     * @author Warren Wang
     */
    public static String getFirstDayOfMonth(String date) {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM");

        Calendar calendar = Calendar.getInstance();// 获取当前日期
        try {
            calendar.setTime(format.parse(date));
        } catch (ParseException e) {
            log.error("日期处理异常", e);
            throw new IllegalArgumentException("参数错误");
        }
        calendar.set(Calendar.DATE, calendar.getMinimum(Calendar.DATE));
        SimpleDateFormat format2 = new SimpleDateFormat("yyyy-MM-dd");
        return format2.format(calendar.getTime());
    }

    /**
     * 获取指定月的最后一天
     * <p>指定月的最后一天<br>
     *
     * @param date String格式yyyy-MM
     * @return
     * @author Warren Wang
     */
    public static String getFinalDayOfMonth(String date) {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM");

        Calendar calendar = Calendar.getInstance();
        try {
            calendar.setTime(format.parse(date));
        } catch (ParseException e) {
            log.error("日期处理异常", e);
            throw new IllegalArgumentException("参数错误");
        }
        calendar.set(Calendar.DATE, calendar.getActualMaximum(Calendar.DATE));
        SimpleDateFormat format2 = new SimpleDateFormat("yyyy-MM-dd");
        return format2.format(calendar.getTime());
    }

    public static final String DEFAULT_DATE_FORMAT = "yyyy-MM-dd";

    public static boolean isDate(String aDate) {
        return isDate(aDate, DEFAULT_DATE_FORMAT);
    }

    public static boolean isDate(String aDate, String fmt) {
        if (aDate == null || aDate.trim().equals(""))
            return false;
        SimpleDateFormat sdf = new SimpleDateFormat(fmt);
        sdf.setLenient(false);
        try {
            sdf.parse(aDate);
        } catch (ParseException e) {
            return false;
        }
        return true;
    }

    /**
     * 获取指定月的前一月（年）或后一月（年）
     *
     * @param dateStr
     * @param addYear
     * @param addMonth 月份为 实际月份减一
     * @param addDay
     * @param pattern  例如yyyy-MM-dd
     * @return
     * @throws Exception 方法的简述.
     * @throws Exception
     */
    public static String getAnyDay(String dateStr, String pattern, int addYear, int addMonth,
                                   int addDay) {
        try {
            Date date = DateUtil.parseDate(dateStr, pattern);
            Calendar cal = Calendar.getInstance();
            cal.setTime(date);
            if (addYear != 0) {
                cal.add(Calendar.YEAR, addYear);
            }
            if (addMonth != 0) {
                cal.add(Calendar.MONTH, addMonth);
            }
            if (addDay != 0) {
                cal.add(Calendar.DATE, addDay);
            }
            return DateUtil.formatDate(cal);
        } catch (Exception e) {
            log.error("日期处理异常", e);
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * 过滤特殊字符
     */
    public static String filterChar(String date) {
        Pattern p = Pattern.compile("\\s*|\t|\r|\n");
        Matcher m = p.matcher(date);
        return m.replaceAll("");
    }

    public static int getDayOfMonth(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        return cal.get(Calendar.DATE);
    }

    public static String convertDateStr(long millSec) {
        DateFormat fmtDateTime = new SimpleDateFormat("yyyy-MM-dd");
        return fmtDateTime.format(new Date(millSec * 1000));
    }

    public static Date extractionDate(long millSec) {
        Date in = new Date(millSec * 1000);
        DateFormat fmtDateTime = new SimpleDateFormat("yyyy-MM-dd");
        String text = fmtDateTime.format(in);
        ParsePosition pos = new ParsePosition(0);
        return fmtDateTime.parse(text, pos);
    }

    public static int[] monthToYear(int month) {
        int[] ret = new int[]{0, 0};
        ret[0] = month / 12;
        if (month % 12 != 0) {
            ret[1] = month % 12;
        }
        return ret;
    }

    public static String monthToYearStr(int month) {
        int[] ret = monthToYear(month);
        String str = ret[0] + "岁";
        if (ret[0] <= 5) {
            if (ret[1] != 0) {
                str += ret[1] + "个月";
            }
        }
        return str;
    }

    /**
     * ** 根据传入的时间和格式获得字符串 *****
     */
    public static String getDateStrByFormat(Date date, String format) {
        if (date == null)
            return "";
        DateFormat dtFmt = new SimpleDateFormat(format);
        return dtFmt.format(date);
    }

    public static boolean isSameDate(Date date1, Date date2) {
        Calendar cal1 = Calendar.getInstance();
        cal1.setTime(date1);

        Calendar cal2 = Calendar.getInstance();
        cal2.setTime(date2);

        boolean isSameYear = cal1.get(Calendar.YEAR) == cal2
                .get(Calendar.YEAR);
        boolean isSameMonth = isSameYear
                && cal1.get(Calendar.MONTH) == cal2.get(Calendar.MONTH);
        boolean isSameDate = isSameMonth
                && cal1.get(Calendar.DAY_OF_MONTH) == cal2
                .get(Calendar.DAY_OF_MONTH);

        return isSameDate;
    }

    public static boolean isSameDay(Date date1, Date date2) {
        long between = Math.abs(date2.getTime() - date1.getTime());

        if (between > ONE_DAY_MILLISECONDS) {
            return false;
        } else if (between == ONE_DAY_MILLISECONDS) {
            Calendar cal1 = Calendar.getInstance();
            cal1.setTime(date1);

            Calendar cal2 = Calendar.getInstance();
            cal2.setTime(date2);

            if (cal1.get(Calendar.HOUR_OF_DAY) == 0 && cal2.get(Calendar.HOUR_OF_DAY) == 0) {
                return true;
            }
            if (cal1.get(Calendar.HOUR_OF_DAY) == 24 && cal2.get(Calendar.HOUR_OF_DAY) == 24) {
                return true;
            }
            return false;
        }
        return true;
    }

    /**
     * ** 根据传入的时间字符串和格式获得Date对象 *****
     */
    public static Date getDateByFormat(String date, String format) {
        if (date == null)
            return null;
        DateFormat dtFmt = new SimpleDateFormat(format);
        Date dateTime = null;
        try {
            dateTime = dtFmt.parse(date);
        } catch (ParseException e) {
            log.error("日期处理异常", e);
        }
        return dateTime;
    }

    /**
     * *** 传入时间戳(秒),计算2个时间差多少天(自然天，不是24小时，如今天凌晨0点1分和昨天23点59分相差为1天) *******
     */
    public static long getDistanceDay(long date1, long date2) {
        String dateStr1 = getDateStrByFormat(new Date(date1 * 1000), "yyyy-MM-dd");
        String dateStr2 = getDateStrByFormat(new Date(date2 * 1000), "yyyy-MM-dd");

        Date date3 = getDateByFormat(dateStr1, "yyyy-MM-dd");
        Date date4 = getDateByFormat(dateStr2, "yyyy-MM-dd");

        long newDate1 = convertTimeStamp(date3);
        long newDate2 = convertTimeStamp(date4);
        long diff;
        if (newDate1 >= newDate2) {
            diff = newDate1 - newDate2;
        } else {
            diff = newDate2 - newDate1;
        }
        return diff / (60 * 60 * 24);
    }

    /**
     * 传入时间返回时间戳(秒) **
     */
    public static long convertTimeStamp(Date date) {
        if (date == null) {
            return 0;
        }
        return date.getTime() / 1000;
    }

    /**
     * *** 传入时间戳,计算2个时间差多少秒 *******
     */
    public static int getDistanceSecond(long date1, long date2) {
        long diff = date1 - date2;
        double diffDouble = diff / 1000;
        return (int) diffDouble;
    }


    /**
     * *** 根据生日算年龄 *******
     */
    public static Integer getAge(Date birthday) {
        if (birthday == null) {
            return null;
        }
        Calendar calendar = Calendar.getInstance();
        int yearNow = calendar.get(Calendar.YEAR);
        calendar.setTime(birthday);
        int yearBirth = calendar.get(Calendar.YEAR);
        if (yearNow - yearBirth < 0) {
            return null;
        }
        return yearNow - yearBirth;
    }

    /**
     * *** 传人两个时间求之间天数 *******
     */
    public static int getDayCount(Date startDate, Date endDate) {
        if (!startDate.before(endDate)) {
            return 0;
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(startDate);
        int startDayCount = (int) calendar.getTimeInMillis() / 86400000;
        calendar.setTime(endDate);
        int endDayCount = (int) calendar.getTimeInMillis() / 86400000;
        return endDayCount - startDayCount;
    }

    /**
     * 获得周一到周七的文本 **
     */
    public static String getWeekDayStr(Integer weekDay) {
        String str = "周";
        switch (weekDay) {
            case 1:
                str += "一";
                break;
            case 2:
                str += "二";
                break;
            case 3:
                str += "三";
                break;
            case 4:
                str += "四";
                break;
            case 5:
                str += "五";
                break;
            case 6:
                str += "六";
                break;
            case 7:
                str += "日";
                break;
        }
        return str;
    }

    /**
     * 获得周上午下午晚上的文本 **
     */
    public static String getDayTypeStr(Integer dayType) {
        String str = "";
        switch (dayType) {
            case 1:
                str += "上午";
                break;
            case 2:
                str += "下午";
                break;
            case 3:
                str += "晚上";
                break;
        }
        return str;
    }

    /**
     * *** 根据传入时间判断是否为当天 *******
     */
    public static boolean isToday(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        int startDayCount = (int) calendar.getTimeInMillis() / 86400000;
        calendar.setTime(date);
        int endDayCount = (int) calendar.getTimeInMillis() / 86400000;
        return endDayCount == startDayCount;
    }

    /**
     * *** 根据传入时间和相差天数计算结果Date *******
     */
    public static Date getDayByDateDistance(Date date, int distanceDay) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.DATE, distanceDay); // 日期减1
        Date lastDay = calendar.getTime(); // 结果
        return lastDay;
    }

    /**
     * *** 根据传入时间和相差秒数计算结果Date *******
     */
    public static Date getTimeByTimeAndDiffSecond(Date date, int second) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.SECOND, second);
        Date lastDay = calendar.getTime(); // 结果
        return lastDay;
    }

    public static int getCurrentTime() {
        return Long.valueOf(new Date().getTime() / 1000).intValue();
    }

    public static String getDateStrByTimeInt(Integer time) {
        SimpleDateFormat sdf = new SimpleDateFormat("MM/dd/yyyy HH:mm:ss");
        Long timeLong = Long.parseLong(time + "");
        Date dt = new Date(timeLong * 1000);
        String sDateTime = sdf.format(dt);
        return sDateTime.toString();
    }

    public static Date increaseDate(Date time, int unit, int value) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(time);
        calendar.set(unit, calendar.get(unit) + value);
        return calendar.getTime();
    }

    public static Date increaseDate(Date time, int value) {
        return increaseDate(time, Calendar.MINUTE, value);
    }


    public static boolean isOverlap(Date dt1Start, Date dt1End, Date dt2Start, Date dt2End) {
        if (dt1Start == null || dt1End == null || dt2Start == null || dt2End == null) {
            throw new IllegalArgumentException("param error");
        }
        long dt1StartLong = dt1Start.getTime();
        long dt1EndLong = dt1End.getTime();
        long dt2StartLong = dt2Start.getTime();
        long dt2EndLong = dt2End.getTime();

        if (dt1StartLong >= dt1EndLong) {
            throw new IllegalArgumentException("param error,param1 must before param2");
        }
        if (dt2StartLong >= dt2EndLong) {
            throw new IllegalArgumentException("param error,param3 must before param4");
        }
        if (dt1EndLong <= dt2StartLong) {
            return false;
        } else if (dt1StartLong >= dt2EndLong) {
            return false;
        }
        return true;
    }

    public static boolean isDateBetween(Date start, Date end, Date paramDate) {
        if (start == null || end == null || paramDate == null) {
            throw new IllegalArgumentException("param error");
        }
        long startLong = start.getTime();
        long endLong = end.getTime();
        long paramLong = paramDate.getTime();

        if (paramLong < startLong) {
            return false;
        } else if (paramLong > endLong) {
            return false;
        }
        return true;
    }

    /**
     * ***** 比较一个时间是否比另一个大 *******
     */
    public static boolean equalIsDateMoreThanAnother(Date date1, Date date2) {
        if (date1.getTime() >= date2.getTime()) {
            return true;
        } else {
            return false;
        }
    }

    public static boolean isDateExpired(Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        return c.getTime().before(new Date());
    }

    public static Date getTodayStartTime(Date date) {
        if (date == null) {
            date = new Date();
        }

        Calendar c = Calendar.getInstance();
        c.setTime(date);
        c.set(Calendar.HOUR_OF_DAY, 0);
        c.set(Calendar.MINUTE, 0);
        c.set(Calendar.SECOND, 0);
        return c.getTime();
    }

    public static Date getTodayEndTime(Date date) {
        if (date == null) {
            date = new Date();
        }

        Calendar c = Calendar.getInstance();
        c.setTime(date);
        c.set(Calendar.HOUR_OF_DAY, 23);
        c.set(Calendar.MINUTE, 59);
        c.set(Calendar.SECOND, 59);
        return c.getTime();
    }

    /**
     *      * 日期格式字符串转换成时间戳
     *      *
     *      * @param dateStr
     *      *            字符串日期
     *      * @param format
     *      *            如：yyyy-MM-dd HH:mm:ss
     *      *
     *      * @return
     *     
     */
    public static long Date2TimeStamp(String dateStr, String format) {
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(format);
            return sdf.parse(dateStr).getTime() / 1000;
        } catch (Exception e) {
            log.error("日期处理异常", e);
        }
        return 0;
    }

    /**
     * 拆分时间格式到年月日
     *
     * @param date
     * @return 拆分时间
     */
    public static Map<String, Integer> splitTime(String date) {
        int index1 = date.indexOf("-");
        int index2 = date.lastIndexOf("-");
        int month;
        int day;
        //季度
        int quater;
        int year = Integer.parseInt(date.substring(0, index1));
        if ("0".equals(date.substring(index1 + 1, index1 + 2))) {
            month = Integer.parseInt(date.substring(index1 + 2, index1 + 3));
        } else {
            month = Integer.parseInt(date.substring(index1 + 1, index1 + 3));
        }
        if ("0".equals(date.substring(index2 + 1, index2 + 2))) {
            day = Integer.parseInt(date.substring(index2 + 2, index2 + 3));
        } else {
            day = Integer.parseInt(date.substring(index2 + 1, index2 + 3));
        }
        //根据月份获取季度
        quater = getQuaterByMonth(month);
        Map<String, Integer> map = new HashMap<>(2);
        map.put("year", year);
        map.put("month", month);
        map.put("day", day);
        map.put("quater", quater);
        return map;
    }

    /**
     * 根据月份获取对应的季度
     *
     * @param month
     * @return
     */
    public static int getQuaterByMonth(int month) {
        int quater;
        if (month >= 1 && month <= 3) {
            quater = 1;
        } else if (month >= 4 && month <= 6) {
            quater = 2;
        } else if (month >= 4 && month <= 6) {
            quater = 3;
        } else {
            quater = 4;
        }
        return quater;
    }

    /**
     * 增加秒
     *
     * @param date
     * @param second
     * @return
     */
    public static Date dateAddSecond(Date date, int second) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.SECOND, second);
        return cal.getTime();
    }

    //将java8 的 java.time.LocalDateTime 转换为 java.util.Date，默认时区为东8区
    public static Date localDateTimeConvertToDate(LocalDateTime localDateTime) {
        return Date.from(localDateTime.toInstant(ZoneOffset.of("+8")));
    }

    //将java.util.Date 转换为java8 的java.time.LocalDateTime,默认时区为东8区
    public static LocalDateTime dateConvertToLocalDateTime(Date date) {
        return date.toInstant().atOffset(ZoneOffset.of("+8")).toLocalDateTime();
    }

    /**
     * 得到日期为星期几
     * @param dates
     * @return
     */
    public static String getWeek(String dates) {

        Calendar cal = Calendar.getInstance();
        Date d = null;
        try {
            d = date_sdf.parse(dates);
        } catch (ParseException e) {

            e.printStackTrace();
        }
        cal.setTime(d);
        int w = cal.get(Calendar.DAY_OF_WEEK) - 1;
        if (w == 0) w = 7;
        return String.valueOf(w);
    }

    /**
     * 通过时间并传入数字得到几天后
     * @param date
     * @param num
     * @return
     */
    public static String getFewDaysLater(Date date, Integer num){
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        c.add(Calendar.DAY_OF_MONTH, num);
        return date_sdf.format(c.getTime());
    }

    /**
     *
     * @param dateStr 2020-11
     * @param beginDate 2020-11-01
     * @param endDate 2020-12-31
     * @return 判断时间是否在范围内
     */
    public static Boolean judgeDateStrBetween(String dateStr,String beginDate,String endDate){
        String substring = dateStr.substring(0, 7);
        String substring1 = beginDate.substring(0, 7);
        String substring2 = endDate.substring(0, 7);
        return substring.compareTo(substring1)>=0 && substring.compareTo(substring2)<=0;
    }

}
